/*
 * Date  : Jun 28, 2011
 * Author: K.Jaensch, klausj@phonetik.uni-muenchen.de
 */
 
package ips.audio.pulse;

import java.nio.ByteBuffer;
import java.util.Vector;

import javax.sound.sampled.AudioFormat;
import javax.sound.sampled.AudioSystem;
import javax.sound.sampled.Control;
import javax.sound.sampled.DataLine;
import javax.sound.sampled.LineEvent;
import javax.sound.sampled.LineListener;
import javax.sound.sampled.LineUnavailableException;
import javax.sound.sampled.Control.Type;

/**
 * @author K.Jaensch, klausj@phonetik.uni-muenchen.de
 *
 */

public abstract class PulseDataLine implements DataLine {

    public static final boolean DEBUG=false;
 // Default time to wait for asynchronous callbacks before throwing ann error
 	// Should not be necessary, but avoids freezing application
 	protected final static int DEFAULT_ASYNC_TIMEOUT_MS=6000;
 	protected final static int DEFAULT_NOTIFY_WAIT_MS=1000;
    protected String name;
    protected PulseMixer mixer;
    protected DataLine.Info dlInfo;
    
    protected AudioFormat audioFormat;
    
    protected volatile ByteBuffer nativePointer;
    private native ByteBuffer nnew();
    protected Vector<LineListener> listeners;
    
    protected long bytesWritten;
    
    /**
     * controls of this line (none implemented) 
     */
    private Control[] controls = new Control[0];
    protected volatile boolean opening=false;
    protected volatile boolean closing=false;
    protected volatile LineUnavailableException lineUnavailableException=null;
    protected volatile boolean open=false;
    protected boolean active=false;
    protected volatile boolean running=false;
    
    public abstract void release(ByteBuffer nativeBuffer); 
    
    public PulseDataLine(PulseMixer mixer, Info dlInfo,String name){
        this.mixer=mixer;
        this.dlInfo=dlInfo;
        this.name=name;
        listeners = new Vector<LineListener>();
//        nativePointer=nnew();
    }
    
   
    public void streamReady(){
    	if (DEBUG)System.out.println("Stream ready");
    	synchronized(this) {
    		opening=false;
    		notifyAll();
    	}
    }
   
    public void streamFailed(){
    	System.err.println("PulseAudio: Stream failed!");
    	synchronized(this) {
    		lineUnavailableException=new LineUnavailableException("PulseAudio: Opening stream failed!");
    		opening=false;
    		notifyAll();
    	}
    }
   
   public void streamTerminated(){
   	if (DEBUG)System.out.println("Stream terminated");
   	release(nativePointer);
  	 	synchronized(this) {
  		 closing=false;
  		 notifyAll();
       }
  }
  

    @Override
    public void close(){
        if(open){
            open = false;
            update(new LineEvent(this, LineEvent.Type.CLOSE, getLongFramePosition()));
        }
    }
   
    @Override
    public Control getControl(Type arg0) {
        return null;
    }

    @Override
    public Control[] getControls() {
        return controls;
    }

    @Override
    public javax.sound.sampled.Line.Info getLineInfo() {
       return dlInfo;
    }

    @Override
    public boolean isControlSupported(Type arg0) {
        return false;
    }

    @Override
    public boolean isOpen() {
        
        return open;
    }

    @Override
    public void open() throws LineUnavailableException {
        if (open)
            return;
        open = true;
       
        update(new LineEvent(this, LineEvent.Type.OPEN, 0));
    }
    @Override
    public int available(){
    	return 0;
    }

    @Override
    public void drain() {
       // implemented in Sourecdataline
    }

    @Override
    public void flush() {
        // TODO Auto-generated method stub

    }

   

    @Override
    public AudioFormat getFormat() {
       return audioFormat;
    }

    @Override
    public int getFramePosition() {
       
        return (int)getLongFramePosition();
    }

    @Override
    public float getLevel() {
        return AudioSystem.NOT_SPECIFIED;
    }

    @Override
    public long getLongFramePosition(){
    	return 0;
    }
    
    
    @Override
    public long getMicrosecondPosition() {
        long framePos=getLongFramePosition();
        double sPos = ((double) framePos * 1000000);
        double msPos = sPos / audioFormat.getFrameRate();
        if (DEBUG)
            System.out.println(
                framePos + " " + audioFormat.getFrameRate() + " " + sPos + " " + msPos);

        return (long) msPos;
    }

    @Override
    public boolean isActive() {
        return active;
    }

    @Override
    public boolean isRunning() {
        // TODO not the same
        return active;
    }

    @Override
    public void start(){
    	if(open){
    		if(!active){
    			active=true;
    			update(new LineEvent(this, LineEvent.Type.START, getLongFramePosition()));
    		}
    	}
    }
    
    	
    
//    public native boolean nStart();

    @Override
    public void stop(){
    	
        if(active){
        	active=false;
        	if(open){
        		update(new LineEvent(this, LineEvent.Type.STOP, getLongFramePosition()));
        	}
        }
    }

    public ByteBuffer getNativePointer() {
        return nativePointer;
    }
    @Override
    public void addLineListener(LineListener listener) {
        synchronized(listeners){
            if (listener != null && !listeners.contains(listener)) {
                listeners.addElement(listener);
            }
        }
    }
    
    
    protected void update(LineEvent le) {
        synchronized(listeners){
            for(LineListener ll:listeners){
                ll.update(le);
            }
        }
    }
    @Override
    public void removeLineListener(LineListener listener) {
        synchronized(listeners){
            if (listener != null) {
                listeners.removeElement(listener);
            }
        }
    }
    public void finalize() throws Throwable{
       if(DEBUG) System.out.println("finalize");
        release();
        super.finalize();
    }

    private native void release();


	@Override
	public int getBufferSize() {
		// TODO Auto-generated method stub
		return 0;
	}

}
