/*
 * Date  : Jan 12, 2011
 * Author: K.Jaensch, klausj@phonetik.uni-muenchen.de
 */

package ips.audio.pulse;

import java.io.File;
import java.net.URL;
import java.nio.ByteBuffer;
import java.util.ArrayList;
import java.util.Hashtable;
import java.util.List;
import java.util.Properties;
import java.util.Vector;

import javax.sound.sampled.AudioFormat;
import javax.sound.sampled.AudioFormat.Encoding;
import javax.sound.sampled.AudioSystem;
import javax.sound.sampled.DataLine;
import javax.sound.sampled.Mixer;
import javax.sound.sampled.Mixer.Info;
import javax.sound.sampled.SourceDataLine;
import javax.sound.sampled.TargetDataLine;
import javax.sound.sampled.spi.MixerProvider;

/**
 * @author K.Jaensch, klausj@phonetik.uni-muenchen.de
 *
 */

public class PulseMixerProvider extends MixerProvider {

	final static boolean DEBUG = false;
	
	public final static String SERVICE_DESCRIPTION_FILE_PATH = "ipsk.audio.ajs.MixerProviderServiceDescriptor.xml";
	final static String NATIVE_LIB_NAME="ips.ajs.pulseaudio.native";
	final static String NATIVE_LIB_LOCATION_KEY = "ips.pulseaudiojavasound.nativelib";

	// Default time to wait for asynchronous callbacks before throwing ann error
	// Should not be necessary, but avoids freezing application
	private final static int DEFAULT_ASYNC_TIMEOUT_MS=6000;
	private final static String META_PATH = "META-INF";
	private final static String META_SERVICES_PATH = META_PATH + "/services/";
	private boolean useAJSSuffix = false;

	private volatile List<PulseMixerInfo> infoList = new ArrayList<PulseMixerInfo>();
	private volatile boolean sourcesEnumerated = false;
	private volatile boolean sinksEnumerated = false;
	private volatile List<Mixer> targetMixersList = new Vector<Mixer>();
	private volatile List<Mixer> sourceMixersList = new Vector<Mixer>();
	private Hashtable<PulseMixerInfo, PulseMixer> mixers = new Hashtable<PulseMixerInfo, PulseMixer>();
	private volatile ByteBuffer np;
	private Properties ajsProperties;
	private volatile boolean connected = false;
	private volatile Object connectionLock = new Object();

	/**
	 * @return the ajsProperties
	 */
	public Properties getAJSProperties() {
		return ajsProperties;
	}

	/**
	 * @param ajsProperties
	 *            the ajsProperties to set
	 */
	public void setAJSProperties(Properties ajsProperties) {
		this.ajsProperties = ajsProperties;
	}

	/**
	 * @return the np
	 */
	ByteBuffer getNp() {
		return np;
	}

	private Thread mainLoop = null;

	private AudioFormat[] pcmAudioFormats;
	private String defaultSourceName;
	private String defaultSinkName;

	private native ByteBuffer init(String appName, String iconName);

	private native void mainLoop(ByteBuffer np) throws PulseException;

	private native void connect(ByteBuffer np) throws PulseException;

	private native void disconnect(ByteBuffer np);

	private native void mainLoopQuit(ByteBuffer np);

	private native void release(ByteBuffer np);

	private native void serverInfo(ByteBuffer np);

	private native void enumerateCards(ByteBuffer np);

	static {
		String dllLocation = null;
		try {
			dllLocation = System.getProperty(NATIVE_LIB_LOCATION_KEY);
		} catch (SecurityException se) {
			// OK no debug mode, continue
		}

		if (dllLocation != null) {
			if (DEBUG)
				System.out.println("Pulse-Java library loading ... ");
			File dsLib = new File(dllLocation);
			if (!dsLib.exists()) {
				System.err.println("PulseJavaSound library not found!");
			}
			System.load(dllLocation);

		} else {

			try{
				// Load lib (64-bit only)
				System.loadLibrary(NATIVE_LIB_NAME);
			}catch(Error e2){
				System.err.println("PulseJavaSound native library "+NATIVE_LIB_NAME+" (lib"+NATIVE_LIB_NAME+".so) could not be loaded!");
			}

		}
		if (DEBUG)
			System.out.println("Pulse driver loaded");

	}

	public PulseMixerProvider() {
		super();
		// path for the declarative XML service descriptor
		String ajsXmlServiceDescriptorPath = META_SERVICES_PATH + SERVICE_DESCRIPTION_FILE_PATH;
		// URL sdURL=ClassLoader.getSystemResource(ajsXmlServiceDescriptorPath);
		ClassLoader classLoader = getClass().getClassLoader();
		if (classLoader != null) {
			URL sdURL = classLoader.getResource(ajsXmlServiceDescriptorPath);
			if (sdURL == null) {
				// if no XML descriptor found we are in standard JavaSound mode
				// and need the suffix " (AJS)"
				useAJSSuffix = true;
				if (DEBUG)
					System.out.println(ajsXmlServiceDescriptorPath + " not found. Switching suffix on");
			}
		}

		List<AudioFormat> pcmAudioFormatsList = new ArrayList<AudioFormat>();
		pcmAudioFormatsList.add(new AudioFormat(Encoding.PCM_UNSIGNED, AudioSystem.NOT_SPECIFIED, 8,
				AudioSystem.NOT_SPECIFIED, AudioSystem.NOT_SPECIFIED, AudioSystem.NOT_SPECIFIED, true));
		pcmAudioFormatsList.add(new AudioFormat(Encoding.PCM_UNSIGNED, AudioSystem.NOT_SPECIFIED, 8,
				AudioSystem.NOT_SPECIFIED, AudioSystem.NOT_SPECIFIED, AudioSystem.NOT_SPECIFIED, false));

		int[] supportedSampleSizes = new int[] { 16, 24, 32 };
		for (int sz : supportedSampleSizes) {
			pcmAudioFormatsList
					.add(new AudioFormat(AudioSystem.NOT_SPECIFIED, sz, AudioSystem.NOT_SPECIFIED, true, true));
			pcmAudioFormatsList
					.add(new AudioFormat(AudioSystem.NOT_SPECIFIED, sz, AudioSystem.NOT_SPECIFIED, true, false));
		}
		pcmAudioFormats = pcmAudioFormatsList.toArray(new AudioFormat[pcmAudioFormatsList.size()]);

	}

	private synchronized void initialize() {
		if (np == null) {
			String appName = null;
			String iconName = null;
			if (ajsProperties != null) {
				appName = ajsProperties.getProperty("ips.audio.ajs.application.name");
				iconName = ajsProperties.getProperty("ips.audio.ajs.freedesktop.application.icon.name");
			}
			np = init(appName, iconName);
			connected = false;
			try {
				connect();
				mainLoop(np);

			} catch (PulseException e) {
				e.printStackTrace();
				release();
			}
			Runtime rt = Runtime.getRuntime();
			rt.addShutdownHook(new Thread(new Runnable() {
				@Override
				public void run() {
					disconnect();
					// disconnect happens immediately, no terminate state
					// callback seen
					// synchronized(connectionLock) {
					//
					// int waited=0;
					// // shutdown threads should not block to long
					// while(connected && waited<40000){
					// if (DEBUG)System.out.println("Wait for context
					// disconnect...");
					// try {
					// connectionLock.wait(500);
					// waited+=500;
					// } catch (InterruptedException e) {
					// // OK
					// }
					// }
					// }
					mainLoopQuit();
					release();
				}
			}));
		}
	}

	public void contextReady() {
		// connected to context and context is ready
		synchronized (connectionLock) {
			connected = true;
			connectionLock.notifyAll();
		}
		// next step: get server info
		serverInfo();

	}

	
	public void contextFailed() {
		System.err.println("Could not get PulseAudio context!");
		// No context
		synchronized (connectionLock) {
			connected = false;
			connectionLock.notifyAll();
		}
		// do not enumerate cards
		synchronized (infoList) {
			sourcesEnumerated = true;
			sinksEnumerated = true;
			infoList.notifyAll();
		}
	}

	public void contextTerminated() {
		if (DEBUG)
			System.out.println("PA: Context terminated");
		synchronized (connectionLock) {
			connected = false;
			connectionLock.notifyAll();
		}
	}

	private void connect() throws PulseException {
		connect(np);
	}

	private void disconnect() {
		disconnect(np);
	}

	private void mainLoopQuit() {
		mainLoopQuit(np);
	}

	private void release() {
		release(np);
	}

	/*
	 * Callback from native: Server info, contains names of default source/sink.
	 */
	public void serverInfo(String defaultSourceName, String defaultSinkName) {
		if (DEBUG)
			System.out.println("Def. source: " + defaultSourceName + ", def. sink: " + defaultSinkName);
		this.defaultSourceName = defaultSourceName;
		this.defaultSinkName = defaultSinkName;
		enumerateCards();
	}

	/*
	 * Callback from native for each enumerated recording line
	 */
	public void sourceInfo(int cardIdx, String sourceName, String sourceDescr) {
		if (DEBUG)
			System.out.println("Source (java): " + cardIdx + " " + sourceName + " " + sourceDescr);

		PulseMixerInfo pmi = new PulseMixerInfo(cardIdx, sourceName, sourceDescr, false);
		PulseMixer pm = new PulseMixer(this, pmi);

		DataLine.Info tdlInfo = new DataLine.Info(TargetDataLine.class, pcmAudioFormats, AudioSystem.NOT_SPECIFIED,
				AudioSystem.NOT_SPECIFIED);
		TargetDataLine tdl = new PulseTargetDataLine(pm, tdlInfo, sourceName);
		pm.setTargetDataLine(tdl);
		synchronized (infoList) {
			mixers.put(pmi, pm);
			if (pmi.getKeyName().equals(defaultSourceName)) {
				infoList.add(0, pmi);
				targetMixersList.add(0, pm);
			} else {
				infoList.add(pmi);
				targetMixersList.add(pm);
			}

		}
	}
	/*
	 * Native callback: Source info enumeration has finished.
	 */
	public void sourceInfoListEnd() {
		synchronized (infoList) {
			sourcesEnumerated = true;
			infoList.notifyAll();
		}
	}

	/*
	 * Native callback for each enumerated playback line
	 */
	public void sinkInfo(int cardIdx, String sinkName, String sinkDescr) {
		if (DEBUG)
			System.out.println("Sink (java): " + cardIdx + " " + sinkName + " " + sinkDescr);

		PulseMixerInfo pmi = new PulseMixerInfo(cardIdx, sinkName, sinkDescr, false);

		PulseMixer pm = new PulseMixer(this, pmi);

		DataLine.Info sdlInfo = new DataLine.Info(SourceDataLine.class, pcmAudioFormats, AudioSystem.NOT_SPECIFIED,
				AudioSystem.NOT_SPECIFIED);
		SourceDataLine sdl = new PulseSourceDataLine(pm, sdlInfo, sinkName);
		pm.setSourceDataLine(sdl);
		synchronized (infoList) {
			// confusing names here:
			// PulseAudio uses source for capture lines
			// JavaSound uses source for playback lines
			mixers.put(pmi, pm);
			if (pmi.getKeyName().equals(defaultSinkName)) {
				// set default playback mixer to first position
				infoList.add(0, pmi);
				sourceMixersList.add(0, pm);
			} else {
				infoList.add(pmi);
				sourceMixersList.add(pm);
			}

		}

	}

	/*
	 * Native callback: sink info enumeration finished.
	 */
	public void sinkInfoListEnd() {
		synchronized (infoList) {
			sinksEnumerated = true;
			infoList.notifyAll();
		}
	}

	/*
	 * Wait for mixer info list ready.
	 */
	private void waitForInfoList() {
		
		int maxNotifyWait=1000;
		int totalWaitMs=0;
		while (totalWaitMs<DEFAULT_ASYNC_TIMEOUT_MS && (!sourcesEnumerated || !sinksEnumerated)) {
			try {
				synchronized (infoList) {
					if (DEBUG)
						System.out.println("Wait for enum");
					infoList.wait(maxNotifyWait);
					totalWaitMs+=maxNotifyWait;
				}
			} catch (InterruptedException e) {
				// OK
			}
		}
	}

	@Override
	public synchronized Mixer getMixer(Info info) {
		PulseMixer mixer = null;
		initialize();
		waitForInfoList();
		
		for (int i = 0; i < infoList.size(); i++) {
			PulseMixerInfo mInfo = infoList.get(i);

			if (mInfo != null && mInfo.equals(info)) {
				// lookup
				mixer = mixers.get(mInfo);
				return mixer;
			}
		}

		throw new IllegalArgumentException();

	}

	@Override
	public synchronized Info[] getMixerInfo() {
		Info[] mixerInfos = null;
		initialize();

		waitForInfoList();
		synchronized (infoList) {
			mixerInfos = infoList.toArray(new Info[0]);
		}
		return mixerInfos;
	}

	/**
	 * Test
	 * 
	 * @param args
	 */
	public static void main(String[] args) {
		final PulseMixerProvider amp = new PulseMixerProvider();

		amp.enumerateCards();
		try {
			Thread.sleep(4000);
		} catch (InterruptedException e) {
			// TODO Auto-generated catch block
			e.printStackTrace();
		}
		System.exit(0);

	}

	private void serverInfo() {
		serverInfo(np);
	}

	private void enumerateCards() {
		synchronized (infoList) {
			sourcesEnumerated = false;
			sinksEnumerated = false;
			infoList.clear();
			targetMixersList.clear();
			sourceMixersList.clear();
		}
		enumerateCards(np);
	}

}
