/*
 * Date  : Jun 28, 2011
 * Author: K.Jaensch, klausj@phonetik.uni-muenchen.de
 */

package ips.audio.pulse;

import java.nio.ByteBuffer;

import javax.sound.sampled.AudioFormat;
import javax.sound.sampled.LineEvent;
import javax.sound.sampled.LineUnavailableException;
import javax.sound.sampled.SourceDataLine;

/**
 * @author K.Jaensch, klausj@phonetik.uni-muenchen.de
 *
 */

public class PulseSourceDataLine extends PulseDataLine implements SourceDataLine {

	// Special timeouts for drain
	// Draining needs a much higher timeout
	// for large buffers draining may take some seconds
	protected final static int DEFAULT_DRAIN_ASYNC_TIMEOUT_MS = 120000;

	private int bufferSize;

	private boolean blockingRead;
	private volatile boolean drained = false;
	private volatile boolean draining = false;
	private volatile boolean flushed = false;

	public PulseSourceDataLine(PulseMixer mixer, Info slDataLineInfo, String sinkName) {
		super(mixer, slDataLineInfo, sinkName);

		audioFormat = new AudioFormat(44100, 16, 2, true, false);
		ByteBuffer mpNp = mixer.getMixerProvider().getNp();
		nativePointer = init(mpNp);
	}

	public native ByteBuffer init(ByteBuffer nativePmp);

	public native void open(ByteBuffer np, String sinkName, int samplerate, int validSampleSizeInBits,
			int sampleSizeInBits, int channels, boolean bigEndian) throws PulseException;

	public native int available(ByteBuffer np);

	public native void write(ByteBuffer np, byte[] data, int offset, int len) throws PulseException;

	private native void flushBegin(ByteBuffer np);

	private native void drainBegin(ByteBuffer np);

	public native void cork(ByteBuffer nativeBuffer, boolean cork);

	public native void release(ByteBuffer nativeBuffer);

	private volatile byte[] buffer = new byte[0];
	private volatile int bufAvail = 0;

	private volatile boolean triggered = false;

	// @Override
	// public int available() {
	// // TODO Auto-generated method stub
	// return 0;
	// }

	public void start() {
		if (DEBUG) {
			System.out.println("PA Source dataline started");
		}
		if (open && !running) {
			triggered = false;
			active = true;
			if (DEBUG) {
				System.out.println("PA Source dataline uncork ...");
			}
			cork(nativePointer, false);
			update(new LineEvent(this, LineEvent.Type.START, getLongFramePosition()));
			if (DEBUG) {
				System.out.println("PA Source dataline wait for running...");
			}
			waitForRunningState(true);
		}
		if (DEBUG) {
			System.out.println("PA Source dataline started");
		}
	}

	public void corkStatusChanged(int res) {
		if (DEBUG)
			System.out.println("Cork cb result:" + res);
		if (res > 0) {

			if (open) {

				if (active) {
					synchronized (this) {
						running = true;
						notifyAll();
						if (DEBUG)
							System.out.println("Started.");
					}
					// update(new LineEvent(this, LineEvent.Type.START,
					// getLongFramePosition()));
				} else {
					running = false;
					// update(new LineEvent(this, LineEvent.Type.STOP,
					// getLongFramePosition()));
				}
			}
		}
	}

	private void waitForRunningState(boolean running) {
		int totalWaitMs = 0;
		synchronized (this) {
			while (active && !running && totalWaitMs < DEFAULT_ASYNC_TIMEOUT_MS) {
				if (DEBUG)
					System.out.println("Wait for running==" + running + " ...");
				try {
					wait(DEFAULT_NOTIFY_WAIT_MS);
					totalWaitMs += DEFAULT_NOTIFY_WAIT_MS;
				} catch (InterruptedException e) {
					// TODO Auto-generated catch block
					e.printStackTrace();
				}
			}
		}
	}

	public void stop() {
		if (open && running) {
			waitForNotDraining();
			active = false;
			cork(nativePointer, true);
			waitForRunningState(false);
			update(new LineEvent(this, LineEvent.Type.STOP, getLongFramePosition()));
		}
		if (DEBUG) {
			System.out.println("PA Source dataline stopped");
		}
	}

	@Override
	public void flush() {
		if (!open || bytesWritten == 0) {
			return;
		}
		flushed = false;
		bufAvail = buffer.length;
		// waitForNotDraining();
		flushBegin(nativePointer);
		if (DEBUG)
			System.out.println("Flushing ...");
		int totalWaitMS = 0;
		synchronized (this) {
			while (!flushed && totalWaitMS < DEFAULT_ASYNC_TIMEOUT_MS) {
				if (DEBUG)
					System.out.println("Wait for flushed");
				try {
					wait(DEFAULT_NOTIFY_WAIT_MS);
					totalWaitMS += DEFAULT_NOTIFY_WAIT_MS;
				} catch (InterruptedException e) {
					// OK
				}
			}
		}
	}

	public void flushed(int result) {
		if (DEBUG)
			System.out.println("Flushed cb: " + result);
		if (result > 0) {
			synchronized (this) {
				flushed = true;
				notifyAll();
				if (DEBUG)
					System.out.println("Flushed.");
			}
		}
	}

	private void waitForDrained() {

		int totalWaitMs = 0;
		synchronized (this) {
			while (!drained && totalWaitMs < DEFAULT_DRAIN_ASYNC_TIMEOUT_MS) {
				if (DEBUG)
					System.out.println("Wait for drained");
				try {
					wait(DEFAULT_NOTIFY_WAIT_MS);
					totalWaitMs += DEFAULT_NOTIFY_WAIT_MS;
				} catch (InterruptedException e) {
					// OK
				}
			}
		}
	}

	private void waitForNotDraining() {

		int totalWaitMs = 0;
		synchronized (this) {
			while (draining && totalWaitMs < DEFAULT_DRAIN_ASYNC_TIMEOUT_MS) {
				if (DEBUG)
					System.out.println("Wait for not draining...");
				try {
					wait(DEFAULT_NOTIFY_WAIT_MS);
					totalWaitMs += DEFAULT_NOTIFY_WAIT_MS;
				} catch (InterruptedException e) {
					// OK
				}
			}
		}
	}

	@Override
	public void drain() {
		synchronized (this) {
			drained = false;
			draining = true;

			drainBegin(nativePointer);
			if (DEBUG)
				System.out.println("Draining ...");
			waitForDrained();

		}
	}

	private native void trigger(ByteBuffer nativePointer);

	public void drained(int result) {
		if (DEBUG)
			System.out.println("Drained cb: " + result);
		synchronized (this) {
			draining = false;
			drained = true;
			notifyAll();
			if (DEBUG)
				System.out.println("Drained.");
		}
	}

	@Override
	public AudioFormat getFormat() {
		return audioFormat;
	}

	@Override
	public void open(AudioFormat af, int bufferSize) throws LineUnavailableException {
		if (open)
			return;
		
		drained = false;
		bytesWritten = 0;
		blockingRead = true;

		int chs = af.getChannels();
		int sampleSize = af.getFrameSize() / chs;
		float sr=af.getSampleRate();
		if(bufferSize<0){
			// default one second buffer size
			bufferSize=(int)sr*chs*sampleSize;
		}
		this.bufferSize = bufferSize;
		lineUnavailableException = null;
		opening = true;
		int sampleSizeInBits=8*sampleSize;
		try {
			open(nativePointer, name, (int) af.getSampleRate(), af.getSampleSizeInBits(), sampleSizeInBits, chs,
					af.isBigEndian());
		} catch (PulseException e1) {
			throw new IllegalArgumentException(e1); 
		}

		synchronized (this) {
			int totalWaitMs = 0;
			while (opening && totalWaitMs < DEFAULT_ASYNC_TIMEOUT_MS) {
				if (DEBUG)
					System.out.println("Wait for stream ready...");
				try {
					wait(DEFAULT_NOTIFY_WAIT_MS);
					totalWaitMs += DEFAULT_NOTIFY_WAIT_MS;
				} catch (InterruptedException e) {
					// OK
				}
			}
			if (totalWaitMs >= DEFAULT_ASYNC_TIMEOUT_MS) {
				throw new LineUnavailableException(
						"PulseAudio: Timeout: Waited " + DEFAULT_ASYNC_TIMEOUT_MS + " ms for playback line!");
			}
		}
		if (lineUnavailableException != null) {
			throw lineUnavailableException;
		}
		super.open();
		audioFormat = af;
	}

	@Override
	public void open(AudioFormat arg0) throws LineUnavailableException {
		open(arg0, -1);
	}

	public int getBufferSize() {
		return bufferSize;
	}

	private int byteLenToMs(int len) {
		int ms = (int) (1000 * (len / (audioFormat.getFrameRate() * (float) audioFormat.getFrameSize())));
		return ms;
	}

	public void finalize() throws Throwable {
		if (DEBUG)
			System.out.println("finalize");
		super.finalize();

	}

	public int available() {
		int nativeAvail = available(nativePointer);
		if (bufAvail > nativeAvail) {
			return bufAvail;
		} else {
			if (nativeAvail < 0) {
				return 0;
			}
			return nativeAvail;
		}
	}

	@Override
	public int write(byte[] b, int off, int len) {
		if (len > buffer.length) {
			buffer = new byte[len];
			bufAvail = len;
		}
		for (int i = 0; i < len; i++) {
			buffer[i] = b[off + i];
		}
		bufAvail -= len;

		int written = 0;
		int pos = 0;
		while (active && written < len) {
			int toWrite = len - written;
			int nav = 0;

			do {
				nav = available(nativePointer);
				if (nav <= 0) {
					try {
						Thread.sleep(10);
					} catch (InterruptedException e) {
						// OK
					}
				}
			} while (nav <= 0 && active);
			if (toWrite > nav)
				toWrite = nav;
			try {
				write(nativePointer, buffer, pos, toWrite);
			} catch (PulseException e) {
				e.printStackTrace();
			}
			pos += toWrite;
			written += toWrite;
		}
		bufAvail = buffer.length;
		bytesWritten += len;
		return (len);
	}

	private native long bytePosition(ByteBuffer np);

	private native long usecPosition(ByteBuffer np);

	public long getLongFramePosition() {
		if (nativePointer == null || !open) {
			return 0;
		} else {
			long bytePos = bytePosition(nativePointer);
			long framePos = bytePos / audioFormat.getFrameSize();
			return framePos;
		}
	}

	private native void close(ByteBuffer nativePointer);

	public void close() {
		if (open) {
			waitForNotDraining();
			close(nativePointer);
			synchronized (this) {
				int totalWaitMs = 0;
				while (closing && totalWaitMs < DEFAULT_ASYNC_TIMEOUT_MS) {
					if (DEBUG)
						System.out.println("Wait for stream termination...");
					try {
						wait(DEFAULT_NOTIFY_WAIT_MS);
						totalWaitMs += DEFAULT_NOTIFY_WAIT_MS;
					} catch (InterruptedException e) {
						// OK
					}
				}
			}
		}
		super.close();
	}

}
