/*
 * Date  : Jun 28, 2011
 * Author: K.Jaensch, klausj@phonetik.uni-muenchen.de
 */

package ips.audio.pulse;

import java.nio.ByteBuffer;

import javax.sound.sampled.AudioFormat;
import javax.sound.sampled.LineUnavailableException;
import javax.sound.sampled.TargetDataLine;

/**
 * @author K.Jaensch, klausj@phonetik.uni-muenchen.de
 *
 */

public class PulseTargetDataLine extends PulseDataLine implements TargetDataLine {

	private static double DEFAULT_LATENCY_TIME = 0.05; // 50 ms latency
														// (fragsize)
	
	// Apply Linux workaround: 
	// For mono recordings PulseAudio mixes both stereo channels
	// 2016-03-15, K.Jaensch
	private static boolean USE_MONO_CAPTURE_STEREO_HACK = true;

	
	private volatile boolean pickFirstChannel = false;
	private volatile int nativeChannels;
	private byte[] nativeBuf;
	private volatile int bufferSize;
	private volatile int sampleSize;

	public native void release(ByteBuffer nativeBuffer);

	public PulseTargetDataLine(PulseMixer mixer, Info tlDataLineInfo, String sourceName) {
		super(mixer, tlDataLineInfo, sourceName);
		audioFormat = new AudioFormat(44100, 16, 2, true, false);
		ByteBuffer mpNp = mixer.getMixerProvider().getNp();
		nativePointer = init(mpNp);
	}

	private native ByteBuffer init(ByteBuffer mpNp);

	public native void open(ByteBuffer np, String sinkName, int samplerate, int validSampleSizeInBits,
			int sampleSizeInBits, int channels, boolean bigEndian, int i, int bufferSize);

	public native int available(ByteBuffer np);

	public int available() {
		if (!active) {
			// stream not ready
			return 0;
		}
		int nativeAvail = available(nativePointer);
		if (nativeAvail < 0) {
			// stream not ready yet or corked (?)
//			System.err.println("PulseTargetDataLine nAvailable returned: " + nativeAvail);
			return 0;
		}

		if (nativeAvail > bufferSize) {
			return bufferSize;
		}
		return nativeAvail;
	}

	@Override
	public void drain() {
		// drain on capture makes no sense to me
	}

	@Override
	public void flush() {
		// Not implemented
	}

	@Override
	public AudioFormat getFormat() {
		return audioFormat;
	}

	@Override
	public void open(AudioFormat af, int bufferSize) throws LineUnavailableException {
		if (open)
			return;
		int frameSize = af.getFrameSize();
		// Calculate frames of latency time
		int latencyFrames = (int) (af.getSampleRate() * DEFAULT_LATENCY_TIME);
		// PulseAudio buffer attribute fragSize for latency
		int fragSize = latencyFrames * frameSize;
		int chs = af.getChannels();
		if (chs == 1 && USE_MONO_CAPTURE_STEREO_HACK) {
			nativeChannels = 2;
			pickFirstChannel = true;
			nativeBuf = new byte[0];
		} else {
			nativeChannels = chs;
			pickFirstChannel=false;
		}
		sampleSize = af.getFrameSize() / chs;
		open(nativePointer, name, (int) af.getSampleRate(), af.getSampleSizeInBits(), sampleSize, nativeChannels,
				af.isBigEndian(), fragSize, bufferSize);
		// for Pulseaudio we could return here and notify the application
		// asynchronously with the OPEN LineEvent, but I think many applications expect teh line to be open
		// when the opne() method returns, so we implement it synchronously 
		synchronized (this) {
			int totalWaitMs = 0;
			while (opening && totalWaitMs < DEFAULT_ASYNC_TIMEOUT_MS) {
				if (DEBUG)
					System.out.println("Wait for stream ready...");
				try {
					wait(DEFAULT_NOTIFY_WAIT_MS);
				} catch (InterruptedException e) {
					// OK
				}
			}
			if (totalWaitMs >= DEFAULT_ASYNC_TIMEOUT_MS) {
				throw new LineUnavailableException(
						"PulseAudio: Timeout: Waited " + DEFAULT_ASYNC_TIMEOUT_MS + " ms for capture line!");
			}
		}
		if (lineUnavailableException != null) {
			throw lineUnavailableException;
		}
		  
		super.open();
		audioFormat = af;
		this.bufferSize = bufferSize;
	}

	@Override
	public void open(AudioFormat arg0) throws LineUnavailableException {
		open(arg0, -1);
	}

	private int byteLenToMs(int len) {
		int ms = (int) (1000 * (len / (audioFormat.getFrameRate() * (float) audioFormat.getFrameSize())));
		return ms;
	}

	@Override
	public int getBufferSize() {

		return bufferSize;
	}

	public native int read(ByteBuffer np, byte[] arg0, int arg1, int arg2);

	// @Override
	public int read(byte[] buf, int off, int len) {
		if (!open || !active) {
			return 0;
		}
		int nLen;
		if (pickFirstChannel) {
			nLen = len * 2;
		} else {
			nLen = len;
		}
		if (available() == 0) {
			try {
				Thread.sleep(100);
			} catch (InterruptedException e) {
				// OK
			}
		}

		int r;
		
		if (pickFirstChannel) {
			// pick the samples for first channel from interleaved stereo signal
			if (nativeBuf.length < nLen) {
				nativeBuf = new byte[nLen];
			}
			// read stereo
			int nr = read(nativePointer, nativeBuf, 0, nLen);
			
			// bytes read in requested mono format 
			r = nr / 2;
			
			// sample count
			int samples = r/sampleSize;
			
			// iterate over read samples
			for (int si = 0; si < samples; si++) {
				// iterate bytes of one sample of first channel only
				for (int bi = 0; bi < sampleSize; bi++) {
					int samplePos = si * sampleSize;
					// target buffer position
					int bufPos=off + samplePos + bi;
					// source buffer position
					int nBufPos=samplePos * 2 + bi;
					
					// copy byte
					buf[bufPos] = nativeBuf[nBufPos];
				}
			}
		} else {
			r = read(nativePointer, buf, off, nLen);
		}
		return r;
	}

	private native void close(ByteBuffer nativePointer);

	public void close() {
		if (open) {
			if(active){
				stop();
			}
			close(nativePointer);
			synchronized (this) {
				int totalWaitMs = 0;
				while (closing && totalWaitMs < DEFAULT_ASYNC_TIMEOUT_MS) {
					if (DEBUG)
						System.out.println("Wait for stream termination...");
					try {
						wait(DEFAULT_NOTIFY_WAIT_MS);
						totalWaitMs += DEFAULT_NOTIFY_WAIT_MS;
					} catch (InterruptedException e) {
						// OK
					}
				}
			}
		}
		super.close();
	}

	public void finalize() throws Throwable {
		if (DEBUG)
			System.out.println("finalize");
		super.finalize();

	}

}
