//    IPS Java Utils
// 	  (c) Copyright 2015
// 	  Institute of Phonetics and Speech Processing,
//    Ludwig-Maximilians-University, Munich, Germany
//
//
//    This file is part of IPS Java Utils
//
//
//    IPS Java Utils is free software: you can redistribute it and/or modify
//    it under the terms of the GNU Lesser General Public License as published by
//    the Free Software Foundation, version 3 of the License.
//
//    IPS Java Utils is distributed in the hope that it will be useful,
//    but WITHOUT ANY WARRANTY; without even the implied warranty of
//    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
//    GNU Lesser General Public License for more details.
//
//    You should have received a copy of the GNU Lesser General Public License
//    along with IPS Java Utils.  If not, see <http://www.gnu.org/licenses/>.

package ipsk.util;

import java.awt.Window;
import java.nio.file.Path;
import java.nio.file.Paths;
import java.util.regex.Pattern;


/**
 * @author klausj
 *
 */
public class SystemHelper {

	private String osName;
	private String javaVersion;
	private String javaVendor;
	
	private static volatile SystemHelper instance=null;
	
	private static final String OS_MAC_OS_X_NAME= "Mac OS X";
	private static final String OS_WINDOWS_NAME= "Windows";
	private static final String OS_LINUX_NAME= "Linux";
	
    private static final String FREE_DESKTOP_DATA_HOME_ENV_VAR_NAME="XDG_DATA_HOME";
    private static final String FREE_DESKTOP_DATA_HOME_DEFAULT_DIRNAME=".local/share";
    
    private static final String FREE_DESKTOP_CONFIG_HOME_ENV_VAR_NAME="XDG_CONFIG_HOME";
    private static final String FREE_DESKTOP_CONFIG_HOME_DEFAULT_DIRNAME=".config";
    
    private static final String MAC_OS_X_LIBRARY_NAME="Library";
    private static final String MAC_OS_X_APPLICATION_SUPPORT_NAME="Application Support";
    
   
    private static final String WINDOWS_APPLICATION_DATA_ENV_NAME="AppData";
	
	
	private boolean isWindows=false;
	
	public SystemHelper(){
		super();
		osName=System.getProperty("os.name");
		// "Windows 7" ... "Windows 10"
		// "Mac OS X"
		// "Linux"
		javaVersion=System.getProperty("java.version");
		javaVendor=System.getProperty("java.vendor");
		Pattern winPattern = Pattern.compile("^"+OS_WINDOWS_NAME+".*", Pattern.CASE_INSENSITIVE | Pattern.UNICODE_CASE);
		isWindows=winPattern.matcher(osName).matches();
	}
	public synchronized static SystemHelper getInstance(){
		if(instance==null){
			instance=new SystemHelper();
		}
		return instance;
	}
	
	public boolean isMacOSX(){
		return(OS_MAC_OS_X_NAME.equalsIgnoreCase(osName));
	}
	
	public boolean isWindows(){
        return(isWindows);
    }
	
	public boolean isLinux(){
        return(OS_LINUX_NAME.equalsIgnoreCase(osName));
    }
	
	
	public Path homeDirPath(){
	    String userHomePropStr=System.getProperty("user.home");
        if(userHomePropStr!=null){
            return Paths.get(userHomePropStr);
        }else{
            return null;
        }
        
	}
	
	 public Path applicationUserDataFolder(){
	        
	         Path homeDir=homeDirPath();
	       if(isLinux()){
	           String relDataHomePath=System.getenv(FREE_DESKTOP_DATA_HOME_ENV_VAR_NAME);
	           if(relDataHomePath==null){
	               relDataHomePath=FREE_DESKTOP_DATA_HOME_DEFAULT_DIRNAME;
	           }
	           return homeDir.resolve(relDataHomePath);
	           
	       }else if(isMacOSX()){
	           // Mac OS X seems to use fixed folder pathes
	           Path libraryPath=homeDir.resolve(MAC_OS_X_LIBRARY_NAME);
	           Path appSupportPath=libraryPath.resolve(MAC_OS_X_APPLICATION_SUPPORT_NAME);
	           return appSupportPath;
	       }else if(isWindows()){
	           String appDataStr=System.getenv(WINDOWS_APPLICATION_DATA_ENV_NAME);
	           return Paths.get(appDataStr);
	           
	       }else{
	           return null;
	       }
	    }
	    
	    public Path applicationUserConfigFolder(){
	        
	       if(isLinux()){
	           String relCfgHomePath=System.getenv(FREE_DESKTOP_CONFIG_HOME_ENV_VAR_NAME);
	           if(relCfgHomePath==null){
	               relCfgHomePath=FREE_DESKTOP_CONFIG_HOME_DEFAULT_DIRNAME;
	           }
	           Path homeDir=homeDirPath();
	           return homeDir.resolve(relCfgHomePath);
	           
	       }else{
	           return applicationUserDataFolder();
	       }
	    }
	    
	
	
	public boolean avoidWindowDispose(){
		return isMacOSX();
	}
	
	public static void disposeWindowForReuse(Window window){
		SystemHelper sh=getInstance();
		if(sh.avoidWindowDispose()){
			window.setVisible(false);
		}else{
			window.dispose();
		}
	}
	
	public static void main(String[] args){
	    SystemHelper sh=SystemHelper.getInstance();
	    System.out.print("Detected operating system: ");
	    if(sh.isLinux()){
           System.out.println("Linux");
        }else if(sh.isMacOSX()){
            System.out.println("Mac OS X ");
        }else if(sh.isWindows()){
            System.out.println("Linux");
        }else{
            System.out.println("(None)");
            System.err.println("Could not determine operating system type!");
        }
	    System.out.println("User applicaton data folder: "+sh.applicationUserDataFolder());
	    System.out.println("User application config folder: "+sh.applicationUserConfigFolder());
        
	}
	
	
}
