//    IPS Java Utils
//    (c) Copyright 2023
//    Institute of Phonetics and Speech Processing,
//    Ludwig-Maximilians-University, Munich, Germany
//
//
//    This file is part of IPS Java Utils
//
//
//    IPS Java Utils is free software: you can redistribute it and/or modify
//    it under the terms of the GNU Lesser General Public License as published by
//    the Free Software Foundation, version 3 of the License.
//
//    IPS Java Utils is distributed in the hope that it will be useful,
//    but WITHOUT ANY WARRANTY; without even the implied warranty of
//    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
//    GNU Lesser General Public License for more details.
//
//    You should have received a copy of the GNU Lesser General Public License
//    along with IPS Java Utils.  If not, see <http://www.gnu.org/licenses/>.

package ipsk.text.xml;

import javax.xml.XMLConstants;
import javax.xml.parsers.DocumentBuilderFactory;
import javax.xml.parsers.ParserConfigurationException;
import javax.xml.transform.TransformerFactory;

// See OWASP Cheatsheet https://cheatsheetseries.owasp.org/cheatsheets/XML_External_Entity_Prevention_Cheat_Sheet.html
/**
 * Helper class to initialize XML builder factories more safely.
 */
public class XMLFactoryBuilder {

	/**
	 * Build new instance of a document builder factory and configure it more safely. 
	 * @throws ParserConfigurationException thrown when configuration fails
	 * @return builder factory
	 * 
	*/
	public static DocumentBuilderFactory safeDocumentBuilderFactoryInstance() throws ParserConfigurationException {
		return XMLFactoryBuilder.safeDocumentBuilderFactoryInstance(false, false);
	}

	/**
	 * Build new instance of a document builder factory and configure it more safely.
	 * @param needDTD set to true if you need DTDs
	 * @throws ParserConfigurationException thrown when configuration fails
	 * @return builder factory
	 * 
	*/
	public static DocumentBuilderFactory safeDocumentBuilderFactoryInstance(boolean needDTD) throws ParserConfigurationException {
		return XMLFactoryBuilder.safeDocumentBuilderFactoryInstance(true, false);
	}

	
	/**
	 * Build new instance of a document builder factory and configure it more safely. 
	 * @throws ParserConfigurationException thrown when configuration fails
	 * @param needDTD set to true if you need DTDs
	 * @param needExternalDTD set to true if you need external DTDs
	 * @return builder factory
	 * 
	*/
	
	public static DocumentBuilderFactory safeDocumentBuilderFactoryInstance(boolean needDTD,boolean needExternalDTD) throws ParserConfigurationException {

		DocumentBuilderFactory dbf = DocumentBuilderFactory.newInstance();
	
		if(!needDTD) {
			String disallowDocTypeFeat = "http://apache.org/xml/features/disallow-doctype-decl";
			dbf.setFeature(disallowDocTypeFeat, true);
		}
		if(!needExternalDTD) {
			dbf.setFeature(XMLConstants.FEATURE_SECURE_PROCESSING, true);
		}
		
		String loadExtDTDFeat = "http://apache.org/xml/features/nonvalidating/load-external-dtd";
		dbf.setFeature(loadExtDTDFeat, false);

		String extGenEntFeat ="http://xml.org/sax/features/external-general-entities";
		dbf.setFeature(extGenEntFeat, false);
			
		String extParamEntFeat="http://xml.org/sax/features/external-parameter-entities";	
		dbf.setFeature(extParamEntFeat, false);

		dbf.setExpandEntityReferences(false);
		
		dbf.setXIncludeAware(false);

		return dbf;

	}
	
	public static TransformerFactory safeTransformerFactoryInstance() {
	  TransformerFactory tff = TransformerFactory.newInstance();
	  tff.setAttribute(XMLConstants.ACCESS_EXTERNAL_DTD, "");
	  tff.setAttribute(XMLConstants.ACCESS_EXTERNAL_STYLESHEET, "");
	  return tff;
	}
	

}
