//    IPS Java Utils
// 	  (c) Copyright 2016
// 	  Institute of Phonetics and Speech Processing,
//    Ludwig-Maximilians-University, Munich, Germany
//
//
//    This file is part of IPS Java Utils
//
//
//    IPS Java Utils is free software: you can redistribute it and/or modify
//    it under the terms of the GNU Lesser General Public License as published by
//    the Free Software Foundation, version 3 of the License.
//
//    IPS Java Utils is distributed in the hope that it will be useful,
//    but WITHOUT ANY WARRANTY; without even the implied warranty of
//    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
//    GNU Lesser General Public License for more details.
//
//    You should have received a copy of the GNU Lesser General Public License
//    along with IPS Java Utils.  If not, see <http://www.gnu.org/licenses/>.

package ips.incubator.dsp;

import java.util.ArrayList;
import java.util.List;

import org.apache.commons.math3.analysis.solvers.LaguerreSolver;
import org.apache.commons.math3.complex.Complex;



/**
 * @author klausj
 *
 */
public class LPCFormantEstimator implements FormantEstimator {

	private LaguerreSolver solver;
	private ZPlane zPlane;
	private int estimatedNrOfFormants;
	/**
	 * 
	 */
	public LPCFormantEstimator(double sampleRate) {
		super();
		solver=new LaguerreSolver();
		zPlane=new ZPlane(sampleRate);
		estimatedNrOfFormants=estimateNrOfFormants(sampleRate);
	}
	
	public int estimateNrOfFormants(double sampleRate){
		// use rule of thumb for number of filter coeffs/formants
		return(int)Math.floor(2+(sampleRate/1000.0));
	}

	/* (non-Javadoc)
	 * @see ips.incubator.dsp.FormantEstimator#estimate(double[])
	 */
	@Override
	public Complex[] estimatePoles(double[] x,int nrPoles) {
		double[] lpcFilterCoeffs=LPC.process(x, nrPoles);
		double[] revCoeffs=new double[lpcFilterCoeffs.length];
		for(int i=0;i<revCoeffs.length;i++) {
			revCoeffs[i]=lpcFilterCoeffs[revCoeffs.length-i-1];
		}
		// Coeff and roots are in reverse order
		Complex[] revLpcRoots=solver.solveAllComplex(revCoeffs, 0.0);
		Complex[] lpcRoots=new Complex[revLpcRoots.length];
		for(int j=0;j<lpcRoots.length;j++) {
			lpcRoots[j]=revLpcRoots[lpcRoots.length-j-1];
		}
		return lpcRoots;
	}
	
	/* (non-Javadoc)
	 * @see ips.incubator.dsp.FormantEstimator#estimate(double[])
	 */
	@Override
	public Complex[] estimatePoles(double[] x,double sampleRate) {
		return estimatePoles(x,estimateNrOfFormants(sampleRate));
	}

	/* (non-Javadoc)
	 * @see ips.incubator.dsp.FormantEstimator#estimateFormantFrequencies(double[], int)
	 */
	@Override
	public List<Double> estimateFormantFrequencies(double[] x, int nrFormants) {
		Complex[] poles=estimatePoles(x,nrFormants);
		List<Double> freqs=new ArrayList<Double>();
		for(int i=0;i<poles.length;i++){
			Complex pole=poles[i];
			if(pole.getImaginary()>0.0) {
				freqs.add(zPlane.frequency(pole));
			}
		}
		return freqs;
	}

	/* (non-Javadoc)
	 * @see ips.incubator.dsp.FormantEstimator#estimateFormantFrequencies(double[], double)
	 */
	@Override
	public List<Double> estimateFormantFrequencies(double[] x, double sampleRate) {
		return estimateFormantFrequencies(x,estimateNrOfFormants(sampleRate));
	}
	
	private double toFrequency(org.apache.commons.math3.complex.Complex c,double sampleRate) {
		double f=Math.atan2(c.getImaginary(),c.getReal())*sampleRate/(2*Math.PI);
		return f;
	}
	

	public static void main(String[] args) {
		//double[] testX=new double[] {0.2669,0.1695,0.1554,0.4680,0.4190, 0.3466, 0.9530,0.9225};
		double[] testX=new double[] {0.266900000000000  , 0.169500000000000  , 0.155400000000000 ,  0.468000000000000  , 0.419000000000000 ,  0.346600000000000  , 0.953000000000000 ,  0.922500000000000};
		
		
		LPCFormantEstimator lfe=new LPCFormantEstimator(44100);
//		Complex[] roots=lfe.estimatePoles(testX,testX.length-1);
//		for(Complex root:roots){
//			System.out.println(root);
//		}
//		System.out.println();
		List<Double> fmts=lfe.estimateFormantFrequencies(testX,8);
		for(double fmt:fmts){
			System.out.println(fmt);
		}
		System.out.println();
		
	}
	

}
