/*
 * Date  : 25.03.2017
 * Author: K.Jaensch, klausj@phonetik.uni-muenchen.de
 */
 
package ipsk.db.speech;



import java.math.BigDecimal;

import org.w3c.dom.Attr;
import org.w3c.dom.Document;
import org.w3c.dom.Element;
import org.w3c.dom.Node;
import org.w3c.dom.NodeList;

/**
 * @author K.Jaensch, klausj@phonetik.uni-muenchen.de
 *
 */

/**
 * @author K.Jaensch, klausj@phonetik.uni-muenchen.de
 *
 */
public class Font extends TextFormatElement{

	protected static final String ELEMENT_NAME = "font";
	
	protected static final String ATT_SIZE = "size";
	
	private static final BigDecimal HUNDRED=new BigDecimal(100);
	
	public static class FontSize{
		
		private BigDecimal value=null;
		private boolean relative=false;
		
		public boolean isRelative() {
			return relative;
		}
		public FontSize(BigDecimal absSize) {
			this(absSize,false);
		}
		public FontSize(BigDecimal value,boolean relative) {
			super();
			this.value = value;
			this.relative=relative;
		}
		
		public BigDecimal getValue() {
			return value;
		}
		public String toString() {
			if(relative) {
				return value.multiply(HUNDRED).toString()+"%";
			}else {
				return value.toString();
			}
		}
	}
	
	
	private String size=null;
	
	public String getSize() {
        return size;
    }

    public void setSize(String size) {
        this.size = size;
    }

    public static String NORMAL="normal";
	
	protected static final String ATT_STYLE = "style";	
	public static String ITALIC="italic";
	private String style=null;


	public String getStyle() {
		return style;
	}

	public void setStyle(String style) {
		this.style = style;
	}

	protected static final String ATT_WEIGHT = "weight";
	public static String BOLD="bold";
	private String weight=null;

	public String getWeight() {
		return weight;
	}

	public void setWeight(String weight) {
		this.weight = weight;
	}
	
	public Text text=null;
	

	public Text getText() {
        return text;
    }

    public void setText(Text text) {
        this.text = text;
    }

    /**
	 * 
	 */
	public Font() {
		super();
	}

	/**
	 * @param n
	 */
	public Font(Element n) {
		super();
		Attr attr = n.getAttributeNode(ATT_SIZE);
        if (attr != null) {
            setSize(attr.getValue());
        }
		attr = n.getAttributeNode(ATT_STYLE);
		if (attr != null) {
			setStyle(attr.getValue());
		}
		attr = n.getAttributeNode(ATT_WEIGHT);
		if (attr != null) {
			setWeight(attr.getValue());
		}
		NodeList textElNl=n.getElementsByTagName("text");
		for(int i=0;i<textElNl.getLength();i++){
		    Element textEl=(Element)textElNl.item(i);
		    text=new Text(textEl);
		}
	}
	
	public Element toElement(Document d){
		Element e=d.createElement(ELEMENT_NAME);
		if(size!=null){
            e.setAttribute(ATT_SIZE, getSize());
        }
		if(style!=null){
		    e.setAttribute(ATT_STYLE, getStyle());
		}
		if(weight!=null){
		    e.setAttribute(ATT_WEIGHT, getWeight());
		}
        if(text!=null){
            e.appendChild(text.toElement(d));
        }
        
        return e;
    }
	
	public Object clone() throws CloneNotSupportedException{
        Object c=super.clone();
        Font b=(Font)c;
        return b;
    }
	
	public String toPlainTextString(){
	   return  getText().getText();
	}
	
	@javax.persistence.Transient
	@java.beans.Transient
	public FontSize fontSize(){
	    if(size==null){
	        return new FontSize(new BigDecimal(1));
	    }else{
	        // parse
	        String trSizeStr=size.trim();
	        if(trSizeStr.endsWith("em")){
	            String relSizeStr=trSizeStr.replaceFirst("\\s*em","");
	             return new FontSize(new BigDecimal(relSizeStr),true);
	           
	        }else if(trSizeStr.endsWith("%")){
	            String relPercSizeStr=trSizeStr.replaceFirst("\\s*%","");
	            BigDecimal relPercSize=new BigDecimal(relPercSizeStr);
	            return new FontSize(relPercSize.divide(new BigDecimal(100)),true);
               
	        }else if(trSizeStr.endsWith("px")){
	            // absolute size px (px is default)
	            String absSizeStr=trSizeStr.replaceFirst("\\s*px","");
	            BigDecimal absSize=new BigDecimal(absSizeStr);
	            return new FontSize(absSize);
	        }else{
	            // absolute size default is pixel px
	        	BigDecimal absSize=new BigDecimal(trSizeStr);
	        	 return new FontSize(absSize);
	        }
	    }
	}
	
	@javax.persistence.Transient
	@java.beans.Transient
	public double toRealFontSize(double parentSize){

	    FontSize fs=fontSize();
	    if(fs!=null) {
	    	double fsVal=fs.getValue().doubleValue();
	    	if(fs.isRelative()) {
	    		return fsVal*parentSize;
	    	}else {
	    		return fsVal;
	    	}
	    }else {
	    	return parentSize;
	    }
	}
	
	
	public String toString(){

		StringBuffer sb=new StringBuffer();
		sb.append("<"+ELEMENT_NAME);
		if(size!=null){
            sb.append(" "+ATT_SIZE+"=\"");
            sb.append(size);
            sb.append("\"");
        }
		if(style!=null){
			sb.append(" "+ATT_STYLE+"=\"");
			sb.append(style);
			sb.append("\"");
		}
		if(weight!=null){
			sb.append(" "+ATT_WEIGHT+"=\"");
			sb.append(weight);
			sb.append("\"");
		}
		sb.append(">");
		sb.append(getText());
		sb.append("</"+ELEMENT_NAME+">");
		return sb.toString();
	}

}
