//    IPS Java Speech Database
//    (c) Copyright 2011
//    Institute of Phonetics and Speech Processing,
//    Ludwig-Maximilians-University, Munich, Germany
//
//
//    This file is part of IPS Java Speech Database
//
//
//    IPS Java Speech Database is free software: you can redistribute it and/or modify
//    it under the terms of the GNU Lesser General Public License as published by
//    the Free Software Foundation, version 3 of the License.
//
//    IPS Java Speech Database is distributed in the hope that it will be useful,
//    but WITHOUT ANY WARRANTY; without even the implied warranty of
//    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
//    GNU Lesser General Public License for more details.
//
//    You should have received a copy of the GNU Lesser General Public License
//    along with IPS Java Speech Database.  If not, see <http://www.gnu.org/licenses/>.

package ipsk.db.speech;


import ipsk.beans.HiddenProperties;
import ipsk.beans.LinkID;
import ipsk.beans.PreferredDisplayOrder;
import ipsk.beans.dom.DOMAttributes;
import ipsk.beans.dom.DOMCollectionElement;
import ipsk.beans.dom.DOMElements;
import ipsk.util.EnumResourceKeys;
import ipsk.util.MemberResourceKey;
import ipsk.util.ResourceBundleName;
import ipsk.util.ResourceKey;
import ipsk.util.annotations.TextAreaView;

import java.time.Duration;
import java.time.Instant;
import java.time.ZoneId;
import java.util.ArrayList;
import java.util.Date;
import java.util.HashSet;
import java.util.List;
import java.util.Set;

import javax.persistence.CascadeType;
import javax.persistence.Column;
import javax.persistence.Entity;
import javax.persistence.EnumType;
import javax.persistence.Enumerated;
import javax.persistence.FetchType;
import javax.persistence.GeneratedValue;
import javax.persistence.Id;
import javax.persistence.JoinColumn;
import javax.persistence.JoinTable;
import javax.persistence.ManyToMany;
import javax.persistence.ManyToOne;
import javax.persistence.NamedQueries;
import javax.persistence.NamedQuery;
import javax.persistence.OneToMany;
import javax.persistence.PrePersist;
import javax.persistence.PreUpdate;
import javax.persistence.Table;
import javax.persistence.Temporal;
import javax.persistence.TemporalType;
import javax.persistence.Transient;
import javax.xml.bind.annotation.XmlAttribute;
import javax.xml.bind.annotation.XmlElement;
import javax.xml.bind.annotation.XmlEnum;
import javax.xml.bind.annotation.XmlID;
import javax.xml.bind.annotation.XmlIDREF;
import javax.xml.bind.annotation.XmlRootElement;
import javax.xml.bind.annotation.XmlTransient;
import javax.xml.bind.annotation.XmlType;
import javax.xml.bind.annotation.adapters.XmlJavaTypeAdapter;


/**
 * Recording Session 
 */
@Entity
@Table(name = "session")
@NamedQueries(value = { 
@NamedQuery(
        name="session.bySpeaker",
        query="SELECT sess FROM Session sess WHERE ?1 MEMBER OF sess.speakers"
       
),
@NamedQuery(
        name="sessions.forProject",
        query="SELECT s FROM Session s WHERE s.project = ?1"
       
)
})
@ResourceBundleName("ipsk.db.speech.PropertyNames")
@HiddenProperties({"httpSessionId"})
@PreferredDisplayOrder("sessionId,uuid,project,status,type,date,sealed,exportForAnnotation,script,code,speakers,recordingFiles,comment")
@DOMAttributes("sessionId")
@DOMElements({"state","date","sealed","loadedDate","startedTrainingDate","startedDate","completedDate","code","speakers","script","comment"})
@XmlType(name="session",namespace="session")
public class Session implements java.io.Serializable {
	
	public enum Status{
		/**
		 * The session object is created.
		 */
		CREATED,
		/**
		 * The session is loaded.
		 */
		LOADED,
		/**
		 * A session training section is started.
		 */
		STARTED_TRAINING,
		/**
		 * The session is started.
		 */
		STARTED,
		/**
		 * The session is complete. All required recordings were done.
		 */
		COMPLETED
	}
	
	@XmlType(name="sessionType")
	@EnumResourceKeys(memberResourceKeys = {
			@MemberResourceKey(name = "TEST", key = "test"),
			@MemberResourceKey(name = "TEST_DEF_A", key = "test.any_audio_device"),
			@MemberResourceKey(name = "SINE_TEST", key = "test.sinus"),
			@MemberResourceKey(name = "NORM", key = "normal")}
	)
	public enum Type{
		/**
		 * A test session using required audio devices
		 */
		TEST,
		/**
		 * A test session using default audio devices even if project configuration requires a certain device 
		 */
		TEST_DEF_A,
		/**
		 * Sinus tone recording reliability test
		 */
		SINE_TEST,
		/**
		 * A normal session
		 */
		NORM}
	
	public static enum LogLevel{SEVERE,WARNING,INFO,CONFIG,FINE,FINER,FINEST};
	
	// 
//	public final static String TYPE_NORM="NORM";
//	public final static String TEST="TEST";
//	public final static String TYPE_SINE_TEST="SINE_TEST";

	// Fields    

	private Integer sessionId;
	
	private String uuid;
	
	private Script script;
	
	private Project project;
	
	private Organisation organisation;
	
	private String code;

	private String environment;
	
	//private Location location;

	private String comment;

	private Date date;
	
	private Date loadedDate;
	
	private Date startedTrainingDate;
	
	private Date startedDate;
	
	private Date completedDate;
	
	private Date lastRestartDate;
	
	private Date sealedDate;

	private Type type;

	private Status status=Status.CREATED;
	
	private boolean sealed=false;

	private String httpSessionId;
	
	private String storageDirectoryURL;
	
//	private LogLevel logLevel;


	private boolean exportForAnnotation=false;

	@Column(nullable=false)
	@ResourceKey("session.export_for_annotation")
	public boolean isExportForAnnotation() {
		return exportForAnnotation;
	}

	public void setExportForAnnotation(boolean exportForAnnotation) {
		this.exportForAnnotation = exportForAnnotation;
	}

	private Set<RecordingFile> recordingFiles = new HashSet<RecordingFile>(0);
	
	private Boolean hasRecordings=null;

	private Set<Speaker> speakers = new HashSet<Speaker>(0);

	// Constructors

	/** default constructor */
	public Session() {
		super();
	}

	/** minimal constructor */
	public Session(int sessionId) {
		this.sessionId = sessionId;
	}

	

	// Property accessors
	@Id
	@Column(name = "session_id", unique = true, nullable = false)
	//@SequenceGenerator(name="ID_SEQ",sequenceName="id_seq")
    @GeneratedValue(generator="id_gen")
    @ResourceKey("id")
	@LinkID
	@XmlID
	@XmlJavaTypeAdapter(XMLIntegerAdapter.class)
    public Integer getSessionId() {
		return this.sessionId;
	}

	public void setSessionId(Integer sessionId) {
		this.sessionId = sessionId;
	}

	@Column(name = "uuid",length=36, unique = true, nullable = true, updatable=false)
	@ResourceKey("uuid")
	public String getUuid() {
		return uuid;
	}

	public void setUuid(String uuid) {
		this.uuid = uuid;
	}

	
	@ManyToOne
	@JoinColumn(name = "script_id")
	@ResourceKey("script")
	@XmlIDREF
	public Script getScript() {
		return this.script;
	}

	public void setScript(Script script) {
		this.script = script;
	}
	
	@Column(name = "code", length = 100)
	@ResourceKey("code")
	public String getCode() {
		return this.code;
	}

	public void setCode(String code) {
		this.code = code;
	}

	@Column(name = "environment", length = 100)
	@ResourceKey("environment")
	public String getEnvironment() {
		return this.environment;
	}

	public void setEnvironment(String environment) {
		this.environment = environment;
	}
	
//	public Location getLocation() {
//		return location;
//	}
//
//	public void setLocation(Location location) {
//		this.location = location;
//	}

	@Column(name = "comment", length = 1000)
	@ResourceKey("comments")
	@TextAreaView
	public String getComment() {
		return this.comment;
	}

	public void setComment(String comment) {
		this.comment = comment;
	}

	@ManyToOne
	@JoinColumn(name = "project")
	@ResourceKey("project")
	@XmlIDREF
	public Project getProject() {
		return this.project;
	}

	public void setProject(Project project) {
		this.project = project;
	}

	@Column(updatable=false)
	@Temporal(TemporalType.TIMESTAMP)
	@ResourceKey("point_in_time")
	public Date getDate() {
		return this.date;
	}

	public void setDate(Date date) {
		this.date = date;
	}
	
	@Column()
	@Temporal(TemporalType.TIMESTAMP)
	@ResourceKey("loaded")
	public Date getLoadedDate() {
		return loadedDate;
	}

	public void setLoadedDate(Date loadedDate) {
		this.loadedDate = loadedDate;
	}
	
	@Column()
	@Temporal(TemporalType.TIMESTAMP)
	@ResourceKey("started_training")
	public Date getStartedTrainingDate() {
		return startedTrainingDate;
	}

	public void setStartedTrainingDate(Date startedTrainingDate) {
		this.startedTrainingDate = startedTrainingDate;
	}
	
	@Column()
	@Temporal(TemporalType.TIMESTAMP)
	@ResourceKey("started")
	public Date getStartedDate() {
		return startedDate;
	}

	public void setStartedDate(Date startedDate) {
		this.startedDate = startedDate;
	}

	@Column()
	@Temporal(TemporalType.TIMESTAMP)
	@ResourceKey("completed")
	public Date getCompletedDate() {
		return completedDate;
	}

	public void setCompletedDate(Date completedDate) {
		this.completedDate = completedDate;
	}

	@Column()
	@Temporal(TemporalType.TIMESTAMP)
	@ResourceKey("last_restart")
	public Date getLastRestartDate() {
		return lastRestartDate;
	}

	public void setLastRestartDate(Date lastRestartDate) {
		this.lastRestartDate = lastRestartDate;
	}

	@Column()
	@Temporal(TemporalType.TIMESTAMP)
	@ResourceKey("sealed")
	/**
	 * Get date at which the session was sealed by the project administrator to 
	 * @return
	 */
	public Date getSealedDate() {
		return sealedDate;
	}

	public void setSealedDate(Date sealedDate) {
		this.sealedDate = sealedDate;
	}
	
	@OneToMany(cascade = { CascadeType.ALL }, fetch = FetchType.LAZY, mappedBy = "session")
	@ResourceKey("recording_files")
	@XmlTransient
	public Set<RecordingFile> getRecordingFiles() {
		return this.recordingFiles;
	}

	public void setRecordingFiles(Set<RecordingFile> recordingFiles) {
		this.recordingFiles = recordingFiles;
	}
	
	@Transient
	public void markHasRecordings(Boolean hasRecordings){
	    this.hasRecordings=hasRecordings;
	}
	@Transient
	public boolean hasRecordings(){
	    if(hasRecordings!=null){
	        return hasRecordings;
	    }else{
	        if(recordingFiles!=null){
	            return recordingFiles.size()>0;
	        }
	    }
	    return false;
	}

	//@ManyToMany(cascade = { CascadeType.ALL }, fetch = FetchType.LAZY, mappedBy = "sessions")
	@ManyToMany(fetch = FetchType.LAZY)
	@JoinTable(
		        name="participates",
		        joinColumns={@JoinColumn(name="session_id")},
		        inverseJoinColumns={@JoinColumn(name="speaker_id")}
		    )
	@DOMCollectionElement(collectionElementName="speaker")
    @ResourceKey("participants")
   // @XmlTransient
	@XmlIDREF
	public Set<Speaker> getSpeakers() {
		return this.speakers;
	}

	public void setSpeakers(Set<Speaker> speakers) {
		this.speakers = speakers;
	}
	
	
	@Column(name = "type", length = 10)
	@Enumerated(EnumType.STRING)
	@ResourceKey("type")
	public Type getType() {
		return type;
	}

	public void setType(Type type) {
		this.type = type;
	}
	
	@Column(name = "status", length = 100)
	@Enumerated(EnumType.STRING)
	@ResourceKey("status")
	public Status getStatus() {
		return status;
	}

	public void setStatus(Status status) {
		this.status = status;
	}
	
	@Column()
	@ResourceKey("sealed")
	public boolean getSealed() {
		return sealed;
	}

	public void setSealed(boolean sealed) {
		this.sealed = sealed;
	}
	
	/**
	 * The HTTP web session ID
	 * Used for authentication.
	 * @return HTTP session ID
	 */
	@Column(name="jsessionid",length = 100)
	@XmlTransient
	public String getHttpSessionId() {
		return httpSessionId;
	}
	
	/**
	 * Set HTTP session ID.
	 * If a new web recording session is created the current HTTP session ID of the
	 * authenticated user is stored here. If the user logs out, the  web application is 
	 * redeployed or the server is restarted the web recorder can still upload data, though
	 * the corresponding HTTP session is invalidated.
	 * @param httpSessionId HTTP session ID
	 */
	public void setHttpSessionId(String httpSessionId) {
		this.httpSessionId = httpSessionId;
	}
	
	@PrePersist
	public void setCurrentDate(){
		if(date==null){
			setDate(new Date());
		}
	}
	
	@PreUpdate
	public void setSealedDate() {
		if(sealedDate==null && sealed) {
			sealedDate=new Date();
		}
	}
	
	public String toString(){
		return "Session: "+sessionId;
	}
	
	@Column(name = "storage_dir_url", length = 1000)
	@ResourceKey("storage.dir")
	@XmlTransient
	public String getStorageDirectoryURL() {
		return storageDirectoryURL;
	}

	public void setStorageDirectoryURL(String storageDirectoryURL) {
		this.storageDirectoryURL = storageDirectoryURL;
	}
	
//	@Column(name = "log_level", length = 10)
//	@Enumerated(EnumType.STRING)
//	@ResourceKey("logging.level")
//	public LogLevel getLogLevel() {
//		return logLevel;
//	}
//
//	public void setLogLevel(LogLevel logLevel) {
//		this.logLevel = logLevel;
//	}
	
	@ManyToOne()
	@JoinColumn(name = "organisation_id")
	@ResourceKey("organisation")
	@XmlTransient
	public Organisation getOrganisation() {
		return organisation;
	}

	public void setOrganisation(Organisation organisation) {
		this.organisation = organisation;
	}
	
	/**
	 * Returns amximum recording level of all recording items.
	 * Training sections are ignored. 
	 * @return Maximum level 0.0...1.0
	 */
	@Transient
	public Double getMaxLevel(){
		Double maxLevel=null;
		Set<RecordingFile> recordingFiles=getRecordingFiles();
		if(recordingFiles!=null){
			for(RecordingFile rf:recordingFiles){
				Recording r=rf.getRecording();
				if(r!=null){
				    Group g=r.getGroup();
				    
					Section section=g.getSection();
					if(section!=null && section.isTraining()){
						continue;
					}
				}
				Double rfML=rf.getMaxLevel();
				if(rfML!=null){
				if(maxLevel!=null){
					if(rfML > maxLevel)maxLevel=rfML;
				}else{
					maxLevel=rfML;
				}
				}
			}
		}
		return maxLevel;
	}
	
	/**
	 * Returns missing (not yet recorded) recording items.
	 * Training sections are ignored. 
	 * @return list of missing recording items
	 */
	@Transient
//	@XmlTransient
	public List<Recording> getMissingRecordingItems(){
		ArrayList<Recording> missingRecordingItems=new ArrayList<Recording>();
		Set<Recording> recordedItems=new HashSet<Recording>();
		Set<RecordingFile> rfs=getRecordingFiles();
		for(RecordingFile rf:rfs){
			Recording rr=rf.getRecording();
			recordedItems.add(rr);
		}
		
		// add all recording items of script sorted
		Script script=getScript();
		if(script==null)return null;
//		Section[] sections=script.getSections();
		List<Section> sections=script.getSections();
		for(Section s:sections){
			// Ignore training sections
		    if(!s.isTraining()){
		        //			PromptItem[] pis=s.getPromptItems();
		        List<Group> gs=s.getGroups();
		        for(Group g:gs){
		            List<PromptItem>gPis=g.getPromptItems();
		            for(PromptItem pi:gPis){
		                if(pi instanceof Recording){
		                	Recording r=(Recording)pi;
		                	if(!recordedItems.contains(r)) {
		                		missingRecordingItems.add(r);
		                	}
		                }
		            }
		        }
		    }
		}
		
		return missingRecordingItems;
	}
	
	
	@Transient
	public Long getSessionDuration() {
		Long dur=null;
		if(startedDate!=null && completedDate!=null) {
			dur=completedDate.getTime()-startedDate.getTime();
		}
		return dur;
	}
	

	public void applyDTO(SessionDTO dto) {
		Integer dtoSessId=dto.getSessionId();
		Integer sessId=getSessionId();
		if(dtoSessId!=null && sessId!=null && !sessId.equals(dtoSessId)) {
			throw new IllegalArgumentException("DTO object to apply has different ID: "+sessId+", DTO ID: "+dtoSessId);
		}
		setStatus(dto.getStatus());
		setLoadedDate(dto.getLoadedDate());
		setStartedTrainingDate(dto.getStartedTrainingDate());
		setStartedDate(dto.getStartedDate());
		setCompletedDate(dto.getCompletedDate());
	}
	
	
}
