//    IPS Java Speech Database
//    (c) Copyright 2011
//    Institute of Phonetics and Speech Processing,
//    Ludwig-Maximilians-University, Munich, Germany
//
//
//    This file is part of IPS Java Speech Database
//
//
//    IPS Java Speech Database is free software: you can redistribute it and/or modify
//    it under the terms of the GNU Lesser General Public License as published by
//    the Free Software Foundation, version 3 of the License.
//
//    IPS Java Speech Database is distributed in the hope that it will be useful,
//    but WITHOUT ANY WARRANTY; without even the implied warranty of
//    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
//    GNU Lesser General Public License for more details.
//
//    You should have received a copy of the GNU Lesser General Public License
//    along with IPS Java Speech Database.  If not, see <http://www.gnu.org/licenses/>.

package ipsk.db.speech;



import ipsk.beans.PreferredDisplayOrder;
import ipsk.beans.Unit;
import ipsk.beans.dom.DOMAttributes;
import ipsk.beans.dom.DOMElements;
import ipsk.beans.dom.DOMRoot;
import ipsk.beans.validation.Input;
import ipsk.persistence.Delegate;
import ipsk.persistence.ObjectImmutableIfReferenced;
import ipsk.util.ResourceBundleName;
import ipsk.util.ResourceKey;

import java.util.ArrayList;
import java.util.HashSet;
import java.util.Set;

import javax.persistence.CascadeType;
import javax.persistence.Column;
import javax.persistence.Entity;
import javax.persistence.FetchType;
import javax.persistence.Inheritance;
import javax.persistence.InheritanceType;
import javax.persistence.JoinColumn;
import javax.persistence.ManyToMany;
import javax.persistence.ManyToOne;
import javax.persistence.NamedQueries;
import javax.persistence.NamedQuery;
import javax.persistence.OneToMany;
import javax.persistence.PrimaryKeyJoinColumn;
import javax.persistence.Table;
import javax.persistence.Transient;
import javax.persistence.UniqueConstraint;
import javax.xml.bind.annotation.XmlSeeAlso;
import javax.xml.bind.annotation.XmlTransient;
import javax.xml.bind.annotation.XmlType;

import org.w3c.dom.Document;
import org.w3c.dom.Element;

/**
 * Speaker
 */
@Entity
@Table(name = "speaker")
@PrimaryKeyJoinColumn(name="person_id")
//@Inheritance(strategy=InheritanceType.JOINED)
@DOMRoot
@ResourceBundleName("ipsk.db.speech.PropertyNames")
// Changed sequence in FROM clause
// See http://forum.java.sun.com/thread.jspa?threadID=780197
// TopLink bug ?
//@NamedQuery(
//	        name="speakers.byOrganisation",
//	        query="SELECT speaker FROM Speaker speaker,Organisation orga WHERE orga = ?1 AND speaker MEMBER OF orga.persons"  
//)
// OK this works :)
@NamedQueries(value={
@NamedQuery(
        name="speakers.byOrganisation",
        query="SELECT spk FROM Organisation orga,Speaker spk WHERE orga = ?1 AND spk MEMBER OF orga.persons"  
),

@NamedQuery(
        name="speakers.byOrganisation.orderedByRegisteredDesc",
        query="SELECT spk FROM Organisation orga,Speaker spk WHERE orga = ?1 AND spk MEMBER OF orga.persons ORDER BY spk.registered DESC"  
) })
@DOMAttributes("personId")
@DOMElements({"name","forename","code","address","sex","dateOfBirth","birthPlace","profession","dialectRegion","additionalLanguage","comments"})
@PreferredDisplayOrder("personId,name,forename,code,informedConsents,adress,zipcode,sex,birthYear,dateOfBirth,height,weight,smoker,brace,mouthPiercing,birthPlace,schoolInCity,schoolInCountry,city,country,profession,dialectRegion,motherTongue,motherTongueMother,motherTongueFather,additionalLanguage,comments")
//@XmlType(name="speaker",propOrder={"name","forename","address","sex","dateOfBirth","height","weight","smoker","brace","mouthPiercing","birthPlace","zipcode","profession","dialectRegion","motherTongue","motherTongueMother","motherTongueFather","additionalLanguage","comments"})
@XmlType(name="speaker",namespace="speaker")
public class Speaker extends Person {
	public final static String ELEMENT_NAME="speaker";
	// Fields    

	private String code;

	private String accent;

	private Boolean smoker;

	private Double height;

	private String motherTongue;

	private DialectRegion dialectRegion;
	
	private Double weight;

	private Boolean brace;

	private Boolean mouthPiercing;

	private String motherTongueMother;

	private String motherTongueFather;
	
    private String schoolInCity;
   
	private String schoolInCountry;
	
	private String additionalLanguage;
	
	private String musicalInstrument;

	private String type;
	
	private Account registeredByAccount;
	
	private Set<InformedConsent> informedConsents = new HashSet<InformedConsent>(0);
	
	private String informedConsent;
	
	private String informedBroadConsent;

	private boolean informedConsentInPaperFormSigned=false;
	
	@ResourceKey("consent.informed.paperform.signed")
	public boolean isInformedConsentInPaperFormSigned() {
		return informedConsentInPaperFormSigned;
	}

	public void setInformedConsentInPaperFormSigned(boolean informedConsentInPaperFormSigned) {
		this.informedConsentInPaperFormSigned = informedConsentInPaperFormSigned;
	}

	private Set<Session> sessions = new HashSet<Session>(0);


	// Constructors

	/** default constructor */
	public Speaker() {
		super();
	}

	/** minimal constructor */
	public Speaker(int personId) {
	    this();
		this.personId = personId;
	}

	@Column(name = "code", length = 10)
	 @ResourceKey("code")
	 public String getCode() {
		return this.code;
	}

	public void setCode(String code) {
		this.code = code;
	}

	@Column(name = "accent",  length = 100)
	 @ResourceKey("accent")
	 public String getAccent() {
		return this.accent;
	}

	public void setAccent(String accent) {
		this.accent = accent;
	}

	@Column(name = "smoker")
	@ResourceKey("smoker")
	public Boolean getSmoker() {
		return this.smoker;
	}

	public void setSmoker(Boolean smoker) {
		this.smoker = smoker;
	}

	
	@Column(name = "mother_tongue", length = 100)
	@ResourceKey("mother_tongue")
	public String getMotherTongue() {
		return this.motherTongue;
	}

	public void setMotherTongue(String motherTongue) {
		this.motherTongue = motherTongue;
	}

	
	@ManyToOne
	@JoinColumn(name = "dialect_region")
	@ResourceKey("dialect_region")
	public DialectRegion getDialectRegion() {
		return this.dialectRegion;
	}

	public void setDialectRegion(DialectRegion dialectRegion) {
		this.dialectRegion = dialectRegion;
	}

	
	@Column(name = "brace")
	@ResourceKey("brace")
	public Boolean getBrace() {
		return this.brace;
	}

	public void setBrace(Boolean brace) {
		this.brace = brace;
	}

	@Column(name = "mouth_piercing")
	@ResourceKey("mouth_piercing")
	public Boolean getMouthPiercing() {
		return this.mouthPiercing;
	}

	public void setMouthPiercing(Boolean mouthPiercing) {
		this.mouthPiercing = mouthPiercing;
	}

	@Column(name = "mother_tongue_mother", length = 100)
	@ResourceKey("mother_tongue_of_mother")
	public String getMotherTongueMother() {
		return this.motherTongueMother;
	}

	public void setMotherTongueMother(String motherTongueMother) {
		this.motherTongueMother = motherTongueMother;
	}

	@Column(name = "mother_tongue_father", length = 100)
	@ResourceKey("mother_tongue_of_father")
	public String getMotherTongueFather() {
		return this.motherTongueFather;
	}

	public void setMotherTongueFather(String motherTongueFather) {
		this.motherTongueFather = motherTongueFather;
	}

	@Column(name = "type", length = 8)
	@ResourceKey("type")
	public String getType() {
		return this.type;
	}

	public void setType(String type) {
		this.type = type;
	}

	@ManyToOne
	@JoinColumn(name = "registered_by_account")
	@ResourceKey("registered_by_account")
	@XmlTransient
	public Account getRegisteredByAccount() {
		return registeredByAccount;
	}

	public void setRegisteredByAccount(Account registeredByAccount) {
		this.registeredByAccount = registeredByAccount;
	}
	
	
	@OneToMany( fetch = FetchType.LAZY,cascade=CascadeType.ALL, mappedBy = "speaker")
	@ResourceKey("consents.informed")
	@XmlTransient
	public Set<InformedConsent> getInformedConsents() {
		return informedConsents;
	}

	public void setInformedConsents(Set<InformedConsent> informedConsents) {
		this.informedConsents = informedConsents;
	}

	
	@ManyToMany(fetch = FetchType.LAZY, mappedBy = "speakers")
//	@JoinTable(
//		        name="participates",
//		        joinColumns={@JoinColumn(name="speaker_id")},
//		        inverseJoinColumns={@JoinColumn(name="session_id")}
//		    )
	@ResourceKey("sessions")
	@ObjectImmutableIfReferenced
	@XmlTransient
	public Set<Session> getSessions() {
		return this.sessions;
	}

	public void setSessions(Set<Session> sessions) {
		this.sessions = sessions;
	}
	
//	@ManyToOne
//	public Account getAdminAccount() {
//		return adminAccount;
//	}
//
//	public void setAdminAccount(Account adminAccount) {
//		this.adminAccount = adminAccount;
//	}
//	
//	
//	@ManyToMany(mappedBy="speakers",fetch = FetchType.EAGER)
//	// inverse side
//	@ResourceKey("projects")
//	public Set<Project> getProjects() {
//		return this.projects;
//	}
//
//	public void setProjects(Set<Project> projects) {
//		this.projects = projects;
//	}
	
	
	@Unit("m")
	@Column(name = "height",precision = 8)
	@ResourceKey("height")
	public Double getHeight() {
		return height;
	}

	public void setHeight(Double height) {
		this.height = height;
	}
	
	@Unit("kg")
	@Column(name = "weight",precision = 8)
	@ResourceKey("weight")
	public Double getWeight() {
		return weight;
	}

	public void setWeight(Double weight) {
		this.weight = weight;
	}

	@Column(name = "school_in_city",length=1000)
	@ResourceKey("school_in_city")
	public String getSchoolInCity() {
		return schoolInCity;
	}

	public void setSchoolInCity(String schoolInCity) {
		this.schoolInCity = schoolInCity;
	}
	
	@Column(name = "school_in_country",length=1000)
	@ResourceKey("school_in_country")
	public String getSchoolInCountry() {
		return schoolInCountry;
	}

	public void setSchoolInCountry(String schoolInCountry) {
		this.schoolInCountry = schoolInCountry;
	}

	public void setAdditionalLanguage(String additionalLanguage) {
		this.additionalLanguage = additionalLanguage;
	}

	@Column(name = "additional_language",length=100)
	@ResourceKey("language.additional")
	@Input(required=false)
	public String getAdditionalLanguage() {
		return additionalLanguage;
	}
	
	@Column
	@ResourceKey("musical_instrument")
	@Input(required=false)
	public String getMusicalInstrument() {
		return musicalInstrument;
	}

	public void setMusicalInstrument(String musicalInstrument) {
		this.musicalInstrument = musicalInstrument;
	}
	
	
	@Column(name = "informed_consent",length=10000)
	@ResourceKey("consent.informed")
	//@Input(required=true)
	public String getInformedConsent() {
		return informedConsent;
	}

	public void setInformedConsent(String informedConsent) {
		this.informedConsent = informedConsent;
	}

	@Column(name = "informed_consent_broad",length=10000)
	@ResourceKey("consent.broad.informed")
	@Input(required=false)
	public String getInformedBroadConsent() {
		return informedBroadConsent;
	}

	public void setInformedBroadConsent(String informedBroadConsent) {
		this.informedBroadConsent = informedBroadConsent;
	}

//	@Transient
//	private Project currentProject;
//	

//	@XmlTransient
//	public Project currentProject() {
//		return currentProject;
//	}
	public void applyCurrentProject(Project currentProject) {
		//this.currentProject = currentProject;
		InformedConsent ic=currentInformedConsent(currentProject);
		if(ic==null) {
			ic=new InformedConsent();
			getInformedConsents().add(ic);
		}
		ic.setInformedConsentInPaperFormSigned(getInformedConsentInPaperFormSignedDelegate());
		ic.setInformedConsent(getInformedConsentDelegate());
		ic.setInformedBroadConsent(getInformedBroadConsentDelegate());
		ic.setProject(currentProject);
		ic.setSpeaker(this);
		

	}
	
	private InformedConsent currentInformedConsent(Project currentProject) {
		if(currentProject!=null) {
			Set<InformedConsent> cnsts=this.getInformedConsents();
			for(InformedConsent cnst:cnsts) {
				Project cnstPrj=cnst.getProject();
				if(currentProject.equals(cnstPrj)) {
					return cnst;
				}
			}
		}
		return null;
	}

	private String informedBroadConsentDelegate;
	@Transient
	@XmlTransient
	@Delegate
	@ResourceKey("consent.broad.informed")
	public String getInformedBroadConsentDelegate() {
		return informedBroadConsentDelegate;
	}
	
	public void setInformedBroadConsentDelegate(String informedBroadConsentDelegate) {
		this.informedBroadConsentDelegate=informedBroadConsentDelegate;
	}
	
	private String informedConsentDelegate;
	@Transient
	@XmlTransient
	@Delegate
	@ResourceKey("consent.informed")
	public String getInformedConsentDelegate() {
		return informedConsentDelegate;
	}
	public void setInformedConsentDelegate(String informedConsentDelegate) {
		this.informedConsentDelegate=informedConsentDelegate;
	}
	
	private boolean informedConsentInPaperFormSignedDelegate;
	
	@Transient
	@XmlTransient
	@Delegate
	@ResourceKey("consent.informed.paperform.signed")
	public boolean getInformedConsentInPaperFormSignedDelegate() {
		return informedConsentInPaperFormSignedDelegate;
	}
	
	@Transient
	public void setInformedConsentInPaperFormSignedDelegate(boolean informedConsentInPaperFormSignedDelegate) {
		this.informedConsentInPaperFormSignedDelegate=informedConsentInPaperFormSignedDelegate;
	}
	
	
	public Element toElement(Document d){
		Element e=d.createElement(ELEMENT_NAME);
        e.setAttribute("personId", Integer.toString(personId));
        if(code!=null){
        	Element codeElement=d.createElement("code");
        	codeElement.setTextContent(code);
        	e.appendChild(codeElement);
        }
        if(code!=null){
        	Element codeElement=d.createElement("code");
        	codeElement.setTextContent(code);
        	e.appendChild(codeElement);
        }
        
        
		return e;
	}
	
	
	public String toString() {
		String s=super.toString();
		String name=getName();
		String code=getCode();
		if((name==null || "".equals(name)) && code!=null && ! "".equals(code)) {
			s=s+", Code: "+code;
		}
		return s;
	}
	
	
}
