//    IPS Java Speech Database
//    (c) Copyright 2011
//    Institute of Phonetics and Speech Processing,
//    Ludwig-Maximilians-University, Munich, Germany
//
//
//    This file is part of IPS Java Speech Database
//
//
//    IPS Java Speech Database is free software: you can redistribute it and/or modify
//    it under the terms of the GNU Lesser General Public License as published by
//    the Free Software Foundation, version 3 of the License.
//
//    IPS Java Speech Database is distributed in the hope that it will be useful,
//    but WITHOUT ANY WARRANTY; without even the implied warranty of
//    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
//    GNU Lesser General Public License for more details.
//
//    You should have received a copy of the GNU Lesser General Public License
//    along with IPS Java Speech Database.  If not, see <http://www.gnu.org/licenses/>.

package ipsk.db.speech;

import java.util.ArrayList;

import javax.persistence.Column;
import javax.persistence.Entity;
import javax.persistence.GeneratedValue;
import javax.persistence.Id;
import javax.persistence.JoinColumn;
import javax.persistence.OneToOne;
import javax.persistence.Table;
import javax.xml.bind.annotation.XmlTransient;

import org.w3c.dom.Attr;
import org.w3c.dom.Document;
import org.w3c.dom.Element;
import org.w3c.dom.Node;
import org.w3c.dom.NodeList;

import ipsk.util.ResourceBundleName;
import ipsk.util.ResourceKey;

/**
 * Represents a annotationtemplate element of the recording script.
 */
@Entity
@Table(name = "annotationtemplate")
@ResourceKey("annotation.template")
@ResourceBundleName("ipsk.db.speech.PropertyNames")
public class AnnotationTemplate extends BasicPropertyChangeSupport implements java.io.Serializable {

    public static final String ELEMENT_NAME = "annotationtemplate";

    public final static String ATTLANGCODE = "languageISO639code";
    public final static String ATTCOUNTRYCODE = "countryISO3166code";
    // Fields

    private int annotationtemplateId;

    private String annotationtemplate;
    private String languageISO639code;
    private String countryISO3166code;

    private Recording recording;

    private String[] comments = new String[0];

    // Constructors

    /** default constructor */
    public AnnotationTemplate() {
        super();
    }

    /** minimal constructor */
    public AnnotationTemplate(int annotationtemplateId) {
        this();
        this.annotationtemplateId = annotationtemplateId;
    }

    public AnnotationTemplate(Element e) {
        this();
        NodeList childs = e.getChildNodes();
        ArrayList<String> commentsArrList = new ArrayList<String>();
        for (int ci = 0; ci < childs.getLength(); ci++) {
            Node n = childs.item(ci);
            if (n.getNodeType() == Node.COMMENT_NODE) {
                commentsArrList.add(n.getNodeValue());
            }
        }
        comments = commentsArrList.toArray(new String[0]);
        Attr attLangCode=e.getAttributeNode(ATTLANGCODE);
        if(attLangCode!=null){
            languageISO639code=attLangCode.getValue();
        }
        Attr attCountryCode=e.getAttributeNode(ATTCOUNTRYCODE);
        if(attCountryCode!=null){
            countryISO3166code=attCountryCode.getValue();
        } 
        setAnnotationtemplate(e.getTextContent());
    }

    // Property accessors
    @Id
    @Column(name = "annotationtemplate_id", unique = true, nullable = false)
    @GeneratedValue(generator = "id_gen")
    @ResourceKey("id")
    @XmlTransient
    public int getAnnotationtemplateId() {
        return this.annotationtemplateId;
    }

    public void setAnnotationtemplateId(int annotationtemplateId) {
        this.annotationtemplateId = annotationtemplateId;
    }

    @Column(name = "annotationtemplate", length = 100000)
    @ResourceKey("annotation.template.text")
    public String getAnnotationtemplate() {
        return this.annotationtemplate;
    }

    public void setAnnotationtemplate(String annotationtemplate) {
        String oldannotationtemplate = this.annotationtemplate;
        this.annotationtemplate = annotationtemplate;
        propertyChangeSupport.firePropertyChange("annotationtemplate", oldannotationtemplate, this.annotationtemplate);
    }

    @Column(name = "languageISO639code", length = 3)
    public String getLanguageISO639code() {
        return languageISO639code;
    }

    public void setLanguageISO639code(String languageISO639code) {
        this.languageISO639code = languageISO639code;
    }

    /**
     * Get country ISO 3166 code 
     * @return country ISO 3166 code 
     */
    @Column(name = "countryISO3166code", length = 7)
    public String getCountryISO3166code() {
        return countryISO3166code;
    }

    public void setCountryISO3166code(String countryISO3166code) {
        this.countryISO3166code = countryISO3166code;
    }

    @XmlTransient
    @OneToOne
    @JoinColumn(name = "recording_id")
    public Recording getRecording() {
        return this.recording;
    }

    public void setRecording(Recording recording) {
        this.recording = recording;
    }

    public String toString() {
        if (annotationtemplate != null) {
            return annotationtemplate;
        } else {
            return "";
        }
    }

    public boolean equals(Object o) {
        if (o == null)
            return false;
        if (!(o instanceof AnnotationTemplate))
            return false;
        if (o == this)
            return true;

        AnnotationTemplate other = (AnnotationTemplate) o;
        if (other.getAnnotationtemplateId() != annotationtemplateId)
            return false;

        if (other.getAnnotationtemplate() == null) {
            if (annotationtemplate != null)
                return false;
        } else {
            if (!other.getAnnotationtemplate().equals(annotationtemplate))
                return false;
        }
        return true;
    }

    public Element toElement(Document d) {
        Element e = d.createElement(ELEMENT_NAME);
        for (String comm : comments) {
            e.appendChild(d.createComment(comm));
        }
        if (languageISO639code != null) {
            e.setAttribute(ATTLANGCODE, languageISO639code);
        }
        if (countryISO3166code != null) {
            e.setAttribute(ATTCOUNTRYCODE, countryISO3166code);
        }
        e.appendChild(d.createTextNode(getAnnotationtemplate()));

        return e;
    }

}
