//    IPS Java Speech Database
//    (c) Copyright 2011
//    Institute of Phonetics and Speech Processing,
//    Ludwig-Maximilians-University, Munich, Germany
//
//
//    This file is part of IPS Java Speech Database
//
//
//    IPS Java Speech Database is free software: you can redistribute it and/or modify
//    it under the terms of the GNU Lesser General Public License as published by
//    the Free Software Foundation, version 3 of the License.
//
//    IPS Java Speech Database is distributed in the hope that it will be useful,
//    but WITHOUT ANY WARRANTY; without even the implied warranty of
//    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
//    GNU Lesser General Public License for more details.
//
//    You should have received a copy of the GNU Lesser General Public License
//    along with IPS Java Speech Database.  If not, see <http://www.gnu.org/licenses/>.

package ipsk.db.speech.script;


import java.beans.PropertyChangeEvent;
import java.beans.PropertyChangeListener;
import java.beans.PropertyChangeSupport;
import java.io.Serializable;
import java.util.ArrayList;
import java.util.List;
import java.util.Random;

import javax.persistence.Cacheable;
import javax.persistence.CascadeType;
import javax.persistence.Column;
import javax.persistence.Entity;
import javax.persistence.EnumType;
import javax.persistence.Enumerated;
import javax.persistence.FetchType;
import javax.persistence.GeneratedValue;
import javax.persistence.Id;
import javax.persistence.Inheritance;
import javax.persistence.InheritanceType;
import javax.persistence.JoinColumn;
import javax.persistence.ManyToOne;
import javax.persistence.OneToMany;
import javax.persistence.OrderBy;
import javax.persistence.OrderColumn;
import javax.persistence.Table;
import javax.persistence.Transient;
import javax.xml.bind.annotation.XmlTransient;
import javax.xml.bind.annotation.XmlType;

import org.w3c.dom.Attr;
import org.w3c.dom.Document;
import org.w3c.dom.Element;
import org.w3c.dom.Node;
import org.w3c.dom.NodeList;

import ipsk.beans.PreferredDisplayOrder;
import ipsk.db.speech.BasicPropertyChangeSupport;
import ipsk.persistence.ImmutibilityProvider;
import ipsk.persistence.IntegerSequenceGenerator;
import ipsk.util.PluralResourceKey;
import ipsk.util.ResourceBundleName;
import ipsk.util.ResourceKey;

/**
 * Represents a group element of the recording script.
 * The group element contains a list of prompt items (recordings or non-recordings)
 * @author K.Jaensch, klausj@phonetik.uni-muenchen.de
 *
 */
@Entity
@Table(name = "recgroup")
@Cacheable
@XmlType(name="group",namespace="group",propOrder={"order","promptItems"})
//@XmlType(name="group",namespace="group")
@ResourceBundleName("ipsk.db.speech.PropertyNames")
@ResourceKey("group")
@PluralResourceKey("groups")
@PreferredDisplayOrder("groupId,section,order,*")
public class Group extends BasicPropertyChangeSupport implements Serializable,PropertyChangeListener,ImmutibilityProvider{
    public final static String ELEMENT_NAME="group";
   
    @XmlType(namespace="group")
    public enum Order {
        SEQUENTIAL("sequential"), RANDOM("random"), RANDOMIZED("randomized");

        Order(String value) {
            this.value = value;
        }
        private final String value;

        public String value() {
            return value; 
        }
        public String toString() {
            return value; 
        }
        public static Order getByValue(String value){
            for(Order pp:Order.values()){
                if(pp.value.equals(value)){
                    return pp;
                }
            }
            return null;
        }
    }
    
    private int groupId;
	
    private Order order;
    
    protected Section section;
    
    private List<PromptItem> shuffledPromptItems=null;

	/**
	 * @return the groupId
	 */
    @Id
    @Column(name = "group_id", unique = true, nullable = false)
    @GeneratedValue(generator="id_gen")
    @ResourceKey("id")
    @XmlTransient
	public int getGroupId() {
		return groupId;
	}

	/**
	 * @param groupId the groupId to set
	 */
	public void setGroupId(int groupId) {
		this.groupId = groupId;
	}

	private List<PromptItem> promptItems=new ArrayList<PromptItem>();
	private String[] comments=new String[0];
	
    public Group() {
        super();
    }
    
    /**
     * Create cloned group.
     * The prompt items are not deep cloned.
     * @param group Existing group to clone.
     */
    public Group(Group group) {
        super();
        promptItems.addAll(group.promptItems);
    }
   
    public Group(Element e){
        this(null,e,null);
    }
    
   
    public Group(IntegerSequenceGenerator seqGen, Element e,Section section){
    	this();
        if(seqGen!=null){
            setGroupId(seqGen.getAndIncrement());
        }
        Attr attr = e.getAttributeNode("order");
        if (attr != null){
            setOrder(Order.getByValue(attr.getValue()));
        }
    	NodeList childs=e.getChildNodes();
    	ArrayList<String>commentsArrList=new ArrayList<String>();
    	for(int ci=0;ci<childs.getLength();ci++){
    		Node n=childs.item(ci);
    		if(n.getNodeType()==Node.COMMENT_NODE){
    			commentsArrList.add(n.getNodeValue());
    		}
    	}
    	comments=commentsArrList.toArray(new String[0]);

    	ArrayList<PromptItem> promptItemArrList=new ArrayList<PromptItem>();
    	NodeList eNodeList=e.getChildNodes();
    	int promptItemPosition=0;
    	for(int i=0;i<eNodeList.getLength();i++){
    		Node n=eNodeList.item(i);
    		if(n.getNodeType()==Node.ELEMENT_NODE){
    			Element piE=(Element)n;
    			if(piE.getTagName().equals(Recording.ELEMENT_NAME)){
    				Recording recording = new Recording(seqGen,piE);
//    				recording.setPosition(promptItemPosition++);
//    				recording.setSection(getSection());
    				recording.setGroup(this);
    				promptItemArrList.add(recording);
    			}else if(piE.getTagName().equals(Nonrecording.ELEMENT_NAME)){
    				Nonrecording nonrecording = new Nonrecording(seqGen,piE);
//    				nonrecording.setPosition(promptItemPosition++);
//    				nonrecording.setSection(getSection());
    				nonrecording.setGroup(this);
    				promptItemArrList.add(nonrecording);
    			}
    		}
    	}
    	setPromptItems(promptItemArrList);
        setSection(section);
    }
    
    public Group(IntegerSequenceGenerator seqGen,PromptItem pi,Section section){
        this();
        List<PromptItem> pis=new ArrayList<PromptItem>(1);
        pi.setGroup(this);
        pis.add(pi);
        setPromptItems(pis);
        setSection(section);
    }
    
    public Object clone() throws CloneNotSupportedException{
        Group cg=new Group();
        cg.setOrder(getOrder());
        cg.propertyChangeSupport=new PropertyChangeSupport(this);
        List<PromptItem> pis=promptItems;
        ArrayList<PromptItem> cPis=new ArrayList<PromptItem>();
        for(PromptItem pi:pis){
            PromptItem cPi=(PromptItem)pi.clone();
            cPis.add(cPi);
        }
        cg.promptItems=cPis;
       return cg;
   }
    
    
    @Column(name = "ordering", length = 100)
    @Enumerated(EnumType.STRING)
    @ResourceKey("order")
    public Order getOrder() {
        return this.order;
    }

    public void setOrder(Order order) {
        Order oldOrder = this.order;
        this.order = order;
        propertyChangeSupport.firePropertyChange("order", oldOrder, this.order);
    }
    
    @ManyToOne
    @JoinColumn(name = "section_id")
    @ResourceKey("section")
    @XmlTransient
    public Section getSection() {
        return this.section;
    }

    public void setSection(Section section) {
        Section oldSection=this.section;
        this.section = section;
        propertyChangeSupport.firePropertyChange("section", oldSection,this.section);
    }

    @Transient
    @XmlTransient
    public Order getNNOrder() {
        if(order==null)return Order.SEQUENTIAL;
        return this.order;
    }
    
    /**
     * Shuffles the prompt items.
     */
    @Transient
    //@XmlTransient
    public void shuffleItems() {
        List<PromptItem> shuffledPis = new ArrayList<PromptItem>(promptItems);
        
        Random rnd = new Random();
        for (int i = promptItems.size(); i > 1; i--) {
                swap(shuffledPis, i - 1, rnd.nextInt(i));
        }
        shuffledPromptItems=shuffledPis;
    }
    
    private void swap(List<PromptItem> items, int i, int j) {
        PromptItem tmp = items.get(i);
        items.set(i, items.get(j));
        items.set(j, tmp);
    }
    
    @Transient
    //@XmlTransient
    public List<PromptItem> shuffledPromptItems(){
        if(getNNOrder().equals(Order.RANDOM)){
            if(shuffledPromptItems==null){
                shuffleItems();
            }
            return shuffledPromptItems;
        }else{
            return promptItems;
        }
    }
    
    
//    public void updatePositions(){
//        int idx=0;
//        for(PromptItem pi:promptItems){
//            pi.setPosition(idx++);
//        }
//    }
//    
    @Transient
    public void updateUpwardsRelations(){
        for(PromptItem pi:promptItems){
            pi.setGroup(this);

        }
    }
 

		@OneToMany(cascade={CascadeType.REMOVE,CascadeType.MERGE},fetch = FetchType.LAZY, mappedBy = "group")
		@OrderColumn(name="position")
		@ResourceKey("promptitems")
		public List<PromptItem> getPromptItems(){
			return promptItems;
		}

		public void setPromptItems(List<PromptItem> promptItems){
		    this.promptItems = promptItems;
		    shuffledPromptItems=null;
		}
		
		

    public Element toElement(Document d) {
        Element e =d.createElement(ELEMENT_NAME);
        Order order = getOrder();
        if (order != null){
            e.setAttribute("order", order.value());
        }
        for(String comm:comments){
			e.appendChild(d.createComment(comm));
		}
        List<PromptItem> pis=getPromptItems();
        for(PromptItem pi:pis){
            e.appendChild(pi.toElement(d));
        }
        return e;
    }
    
    public void propertyChange(PropertyChangeEvent evt) {
		String propName=evt.getPropertyName();
		String hPropName=ELEMENT_NAME+"."+propName;
		propertyChangeSupport.firePropertyChange(hPropName, evt.getOldValue(), evt.getNewValue());
	}

    @Transient
    @XmlTransient
	public List<String> getMIMETypes() {
		List<String> piMms=new ArrayList<String>();
		for(PromptItem pi:promptItems){
			piMms.addAll(pi.getMIMETypes());
		}
		return piMms;
	}
    
    
    public String toString(){
    	return "Group "+getGroupId();
    }

	/**
	 * 
	 */
    void apply() {

    	for(PromptItem pi:getPromptItems()){
    		pi.apply();
    	}
    	if(getNNOrder().equals(Order.RANDOM)){
    		if(shuffledPromptItems==null){
    			shuffleItems();
    		}
    		setPromptItems(shuffledPromptItems);
//    		updatePositions();
    		setOrder(Order.RANDOMIZED);
    	}

    }

    /**
     * Apply default pre-recording delay.
     * @param defaultPrerecdelay default pre-recording delay in milliseconds
     */
    @Transient
    public void defaultPrerecdelay(int defaultPrerecdelay) {
        List<PromptItem> pis=getPromptItems();
        for(PromptItem pi:pis){
            if(pi instanceof Recording){
                Recording r=(Recording)pi;
                r.setDefaultPrerecdelay(defaultPrerecdelay);
                       
            }
        }
    }

    /**
     * Apply default post-recording delay.
     * @param defaultPostrecdelay default post-recording delay in milliseconds
     */
    @Transient
    public void defaultPostrecdelay(int defaultPostrecdelay) {
        List<PromptItem> pis=getPromptItems();
        for(PromptItem pi:pis){
            if(pi instanceof Recording){
                Recording r=(Recording)pi;
                r.setDefaultPostrecdelay(defaultPostrecdelay);
                       
            }
        }
    }
    
    @Transient
    public void defaultAutoplay(boolean defaultAutoplay){
        List<PromptItem> pis=getPromptItems();
        for(PromptItem pi:pis){
            pi.defaultAutoplay(defaultAutoplay);
        }
    }
    
    @Transient
    public void defaultVirtualViewBox(VirtualViewBox defaultVirtualViewBox) {
		for(PromptItem pi:promptItems) {
			pi.defaultVirtualViewBox(defaultVirtualViewBox);
		}
	}
    
    /* (non-Javadoc)
	 * @see ipsk.persistence.ImmutibilityProvider#isImmutable()
	 */
	@Override
	@Transient
	public boolean isImmutable() {
		
		Section s=getSection();
		return (s!=null && s.isImmutable()); 
	}
	
	@Override
	@Transient
	public boolean isRemovable() {
		return !isImmutable();
	}

}
