//    IPS Java Speech Database
//    (c) Copyright 2011
//    Institute of Phonetics and Speech Processing,
//    Ludwig-Maximilians-University, Munich, Germany
//
//
//    This file is part of IPS Java Speech Database
//
//
//    IPS Java Speech Database is free software: you can redistribute it and/or modify
//    it under the terms of the GNU Lesser General Public License as published by
//    the Free Software Foundation, version 3 of the License.
//
//    IPS Java Speech Database is distributed in the hope that it will be useful,
//    but WITHOUT ANY WARRANTY; without even the implied warranty of
//    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
//    GNU Lesser General Public License for more details.
//
//    You should have received a copy of the GNU Lesser General Public License
//    along with IPS Java Speech Database.  If not, see <http://www.gnu.org/licenses/>.

package ipsk.db.speech.script;


import java.awt.datatransfer.DataFlavor;
import java.awt.datatransfer.Transferable;
import java.awt.datatransfer.UnsupportedFlavorException;
import java.beans.PropertyChangeSupport;
import java.io.IOException;
import java.io.Serializable;
import java.net.URI;
import java.util.ArrayList;
import java.util.HashSet;
import java.util.List;
import java.util.Set;
import java.util.regex.Matcher;
import java.util.regex.Pattern;

import javax.persistence.Cacheable;
import javax.persistence.Column;
import javax.persistence.Entity;
import javax.persistence.FetchType;
import javax.persistence.GeneratedValue;
import javax.persistence.Id;
import javax.persistence.Inheritance;
import javax.persistence.InheritanceType;
import javax.persistence.JoinColumn;
import javax.persistence.JoinTable;
import javax.persistence.ManyToMany;
import javax.persistence.ManyToOne;
import javax.persistence.Table;
import javax.persistence.Transient;
import javax.xml.bind.annotation.XmlSeeAlso;
import javax.xml.bind.annotation.XmlTransient;

import org.w3c.dom.Document;
import org.w3c.dom.Element;

import ipsk.db.speech.BasicPropertyChangeSupport;
import ipsk.db.speech.LocalizedText;
import ipsk.db.speech.script.prompt.Mediaitem;
import ipsk.db.speech.script.prompt.Presenter;
import ipsk.persistence.ImmutibilityProvider;
import ipsk.util.ResourceBundleName;
import ipsk.util.ResourceKey;

/**
 * Generic element which represents a sequential prompt in a section.
 * Base class for Recording and Nonrecording elements.
 * 
 * @author K.Jaensch, klausj@phonetik.uni-muenchen.de
 *
 */
@Entity
@Table(name = "recording", uniqueConstraints = {})
@Cacheable
@ResourceBundleName("ipsk.db.speech.PropertyNames")
@Inheritance(strategy=InheritanceType.SINGLE_TABLE)
@XmlSeeAlso({Recording.class,Nonrecording.class})
public abstract class PromptItem extends BasicPropertyChangeSupport implements Serializable,Cloneable,Transferable,ImmutibilityProvider{
	public static final DataFlavor CLASS_DATA_FLAVOR=new DataFlavor(PromptItem.class,null);
	protected int recpromptId;
	private Group group;
	protected Presenter presenter;
	protected List<Mediaitem> mediaitems=new ArrayList<Mediaitem>();

    protected int recscriptIndex;
    private static final Pattern FILENAME_FROM_PATH_PATTERN = Pattern.compile("([^/]*)$");
    
   
    public PromptItem() {
        super();
    }
    public PromptItem(boolean initialize) {
		super();
		if (initialize) {
//			mediaitem = new Mediaitem();
		}
	}
    public abstract Element toElement(Document d);
    
    @Transient
    public List<String> getMIMETypes(){
        List<Mediaitem> misList=getMediaitems();
        ArrayList<String> mimeList=new ArrayList<String>();
        for(Mediaitem mi:misList){
            mimeList.add(mi.getMimetype());
        }
        return mimeList;
    }
    
//  Property accessors
	@Id
	@Column(name = "recording_id", unique = true, nullable = false)
    @GeneratedValue(generator="id_gen")
    @ResourceKey("id")
	@XmlTransient
    public int getRecpromptId() {
		return this.recpromptId;
	}

	public void setRecpromptId(int recpromptId) {
		int oldRecpromptId=this.recpromptId;
		this.recpromptId = recpromptId;
		propertyChangeSupport.firePropertyChange("recpromptId", oldRecpromptId, this.recpromptId);
	}

	/**
	 * @return the group
	 */
	@ManyToOne
	@JoinColumn(name = "group_id")
	@ResourceKey("group")
	@XmlTransient
	public Group getGroup() {
		return group;
	}
	/**
	 * @param group the group to set
	 */
	public void setGroup(Group group) {
		this.group = group;
	}

	
	// Note changed the type of promptItems in MediaItem class from List to set 
	@ManyToMany(fetch = FetchType.LAZY)
	 @JoinTable(
		      name="recording_mediaitem",
		      joinColumns=@JoinColumn(name="promptitems_recording_id", referencedColumnName="recording_id")
		      )
	@ResourceKey("media.item")
	//@XmlElement(name="mediaitems")
	public List<Mediaitem> getMediaitems() {
		return this.mediaitems;
	}

	public void setMediaitems(List<Mediaitem> mediaitems) {
		this.mediaitems = mediaitems;
	}

    @Transient
    public String getDescription(){
        StringBuffer description=new StringBuffer();
        List<Mediaitem> mis=getMediaitems();
        int misSize=mis.size();
        for(int i=0;i<misSize;i++){
            Mediaitem mi=mis.get(i);
            if(mi.getAlt() !=null) {
                description.append(mi.getAlt());
            }else{
                URI src=mi.getSrc();
                String mimeType=mi.getNNMimetype();
                if (mimeType.startsWith("image")) {
                    description.append("IMAGE: ");
                } else if (mimeType.startsWith("audio")) {
                    description.append("AUDIO: ");
                } else if (mimeType.startsWith("video")) {
                    description.append("VIDEO: ");
                }
                if (mi.getPromptText() !=null) {
                    description.append(mi.getPromptText());
                } 
                if (src != null) {
                	String srcPath=src.getPath();
                	if(srcPath!=null) {
                		Matcher matcher = FILENAME_FROM_PATH_PATTERN.matcher(srcPath);
                		if(matcher.find() && matcher.groupCount()==1) {
                			String srcFn=matcher.group(1);
                			String srcFnNm=srcFn.replaceFirst("[.][^.]*$","");
                			description.append(srcFnNm);
                		}
                	}
                } 
            }
            if(i+1<misSize){
                // not last item
                description.append(", ");
            }
        }
        return description.toString();
    }

    @Transient
    @XmlTransient
    public int getRecscriptIndex() {
        return recscriptIndex;
    }

    public void setRecscriptIndex(int recscriptIndex) {
    	int oldRecscriptIndex=this.recscriptIndex;
        this.recscriptIndex = recscriptIndex;
        propertyChangeSupport.firePropertyChange("recscriptIndex", oldRecscriptIndex, this.recscriptIndex);
    }
    
    protected Integer position;
    
//    /**
//     * Get the position index inside a section respectively a group.
//     * @return position index or null if unknown
//     */
//    @Column(name = "position")
//    @ResourceKey("position")
//    @XmlTransient
//    public Integer getPosition() {
//        return position;
//    }
//
//    /**
//     * Set the position index of this prompt item inside a section respectively a group.
//     * @param position position index or null if unknown 
//     */
//    public void setPosition(Integer position) {
//        this.position=position;
//    }

    @Transient
    protected Mediaitem annotationTemplateMediaitem(){ 

        for(Mediaitem mi:getMediaitems()){
            if(mi.getAnnotationTemplate()){
                return mi;
            }
        }
        return null;
    }
    
    private String annotationTemplateText(){
        if( this instanceof Recording){
            Recording thisR=(Recording)this;
            AnnotationTemplate annoTempl=thisR.getAnnotationTemplate();
            if(annoTempl!=null){
                String annoTemplTxt=annoTempl.getAnnotationtemplate();
                if(annoTemplTxt!=null){
                    return annoTemplTxt;
                }
            }
        }
        Mediaitem ami=annotationTemplateMediaitem();
        String amiTxt=null;
        if(ami!=null){
            amiTxt=ami.getText();
        }
        return amiTxt;
    }
    
    
    
    
    public LocalizedText annotationTemplateLocalizedText(){


        if( this instanceof Recording){
            Recording thisR=(Recording)this;
            AnnotationTemplate annoTempl=thisR.getAnnotationTemplate();
            if(annoTempl!=null){
                String annoTemplTxt=annoTempl.getAnnotationtemplate();
                if(annoTemplTxt!=null){
                    java.util.Locale lLoc=null;
                    String langCode=annoTempl.getLanguageISO639code();
                    if(langCode!=null && !langCode.isEmpty()){
                        String countryCode=annoTempl.getCountryISO3166code();
                        if(countryCode!=null && !countryCode.isEmpty()){
                            lLoc= new java.util.Locale(langCode,countryCode);
                        }else{
                            lLoc= new java.util.Locale(langCode);
                        }
                    }
                    //                    else{
                    //                        // default current locale
                    //                        lLoc=java.util.Locale.getDefault();
                    //                    }
                    LocalizedText lt=new LocalizedText(annoTemplTxt, lLoc);
                    return lt;
                }
            }
        }
        Mediaitem ami=annotationTemplateMediaitem();
        String amiTxt=null;
        if(ami!=null){
            amiTxt=ami.getText();
            if(amiTxt!=null){
                java.util.Locale lLoc=null;
                String langCode=ami.getLanguageISO639code();
                if(langCode!=null && !langCode.isEmpty()){
                    String countryCode=ami.getCountryISO3166code();
                    if(countryCode!=null && !countryCode.isEmpty()){
                        lLoc= new java.util.Locale(langCode,countryCode);
                    }else{
                        lLoc= new java.util.Locale(langCode);
                    }
                }
                //            else{
                //                // default current locale
                //                lLoc=java.util.Locale.getDefault();
                //            }
                LocalizedText lt=new LocalizedText(amiTxt, lLoc);
                return lt;
            }
        }
        return null;
    }

    public Object clone() throws CloneNotSupportedException{
	   
         Object c=super.clone();
         PromptItem pi=(PromptItem)c;
         pi.propertyChangeSupport=new PropertyChangeSupport(this);
         List<Mediaitem> mis=mediaitems;
         ArrayList<Mediaitem> cMis=new ArrayList<Mediaitem>();
         for(Mediaitem mi:mis){
             Mediaitem cMi=(Mediaitem)mi.clone();
             cMis.add(cMi);
         }
         pi.mediaitems=cMis;
        return pi;
	}
	
	@Transient
	public Object getTransferData(DataFlavor flavor) throws UnsupportedFlavorException, IOException {
		 if (!isDataFlavorSupported(flavor)) {
           throw new UnsupportedFlavorException(flavor);
       }
       return this;
	}
	@Transient
	public DataFlavor[] getTransferDataFlavors() {
		
		return new DataFlavor[]{CLASS_DATA_FLAVOR};
	}
	@Transient
	public boolean isDataFlavorSupported(DataFlavor flavor) {
		return(CLASS_DATA_FLAVOR.equals(flavor));
	}
	@Transient
	@XmlTransient
    public Presenter getPresenter() {
        return presenter;
    }
    public void setPresenter(Presenter presenter) {
        this.presenter = presenter;
    }

    @Transient
    public Set<URI> resourceURIs(){
    	Set<URI> usedUris=new HashSet<URI>();
    	for(Mediaitem mi:mediaitems){
    		URI srcUri=mi.getSrc();
    		if(srcUri!=null){
    			usedUris.add(srcUri);
    		}
    	}
    	return usedUris;
    }
    
    @Transient
    public void defaultAutoplay(boolean defaultAutoplay){
        for(Mediaitem mi:mediaitems){
            mi.setDefaultAutoplay(defaultAutoplay);
        }
    }
    
	/**
	 * 
	 */
	protected void apply() {
		for(Mediaitem mi:mediaitems){
			mi.apply();
		}
	}
	
	/* (non-Javadoc)
	 * @see ipsk.persistence.ImmutibilityProvider#isImmutable()
	 */
	@Override
	 @Transient
	public boolean isImmutable() {
		Group g=getGroup();
		return g!=null && g.isImmutable();
	}
	
	@Override
	@Transient
	public boolean isRemovable() {
		return !isImmutable();
	}
	
	public void defaultVirtualViewBox(VirtualViewBox defaultVirtualViewBox) {
		for(Mediaitem mi:mediaitems) {
			mi.setDefaultVirtualViewBox(defaultVirtualViewBox);
		}
	}
	
}
