//    IPS Java Speech Database
//    (c) Copyright 2011
//    Institute of Phonetics and Speech Processing,
//    Ludwig-Maximilians-University, Munich, Germany
//
//
//    This file is part of IPS Java Speech Database
//
//
//    IPS Java Speech Database is free software: you can redistribute it and/or modify
//    it under the terms of the GNU Lesser General Public License as published by
//    the Free Software Foundation, version 3 of the License.
//
//    IPS Java Speech Database is distributed in the hope that it will be useful,
//    but WITHOUT ANY WARRANTY; without even the implied warranty of
//    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
//    GNU Lesser General Public License for more details.
//
//    You should have received a copy of the GNU Lesser General Public License
//    along with IPS Java Speech Database.  If not, see <http://www.gnu.org/licenses/>.

package ipsk.db.speech.script;


import ipsk.beans.PreferredDisplayOrder;
import ipsk.beans.Unit;
import ipsk.persistence.IntegerSequenceGenerator;
import ipsk.persistence.ObjectImmutableIfReferenced;
import ipsk.db.speech.RecordingFile;
import ipsk.db.speech.script.Section.Order;
import ipsk.db.speech.script.prompt.Mediaitem;
import ipsk.db.speech.script.prompt.Recprompt;
import ipsk.db.speech.utils.BooleanValue;
import ipsk.util.PluralResourceKey;
import ipsk.util.ResourceBundleName;
import ipsk.util.ResourceKey;

import java.beans.PropertyChangeEvent;
import java.beans.PropertyChangeListener;
import java.util.ArrayList;
import java.util.HashSet;
import java.util.List;
import java.util.Set;

import javax.persistence.Cacheable;
import javax.persistence.CascadeType;
import javax.persistence.Column;
import javax.persistence.DiscriminatorValue;
import javax.persistence.Entity;
import javax.persistence.FetchType;
import javax.persistence.JoinColumn;
import javax.persistence.JoinTable;
import javax.persistence.ManyToMany;
import javax.persistence.ManyToOne;
import javax.persistence.NamedQuery;
import javax.persistence.OneToMany;
import javax.persistence.OneToOne;
import javax.persistence.Table;
import javax.persistence.Transient;
import javax.xml.bind.annotation.XmlElement;
import javax.xml.bind.annotation.XmlSeeAlso;
import javax.xml.bind.annotation.XmlTransient;
import javax.xml.bind.annotation.XmlType;

import org.w3c.dom.Attr;
import org.w3c.dom.Document;
import org.w3c.dom.Element;
import org.w3c.dom.Node;
import org.w3c.dom.NodeList;

/**
 * Represents a recording element of the recording script.
 * Recording generated by hbm2java
 */
@Entity
@Cacheable
@Table(name = "recording")
@DiscriminatorValue("recording")
@ResourceBundleName("ipsk.db.speech.PropertyNames")
@ResourceKey("recording")
@PluralResourceKey("recordings")
@PreferredDisplayOrder("recpromptId,group,position,itemcode,mediaitems,recduration,rectype,recordingFiles")
@XmlType(name="recording",namespace="promptitem",propOrder={"itemcode","rectype","prerecdelay","postrecdelay","recduration","recinstructions","reccomment","beep","blocked","finalsilence","annotationTemplate","mediaitems"})
//@XmlType(name="recording",namespace="promptitem")
//@XmlSeeAlso(Recprompt.class)
public class Recording extends Recprompt implements PropertyChangeListener{
public final static String ELEMENT_NAME="recording";

public final static boolean DEF_BLOCKED=false;

public final static Integer DEF_RECDURATION=null; // unlimited now
public final static int DEF_PRERECDELAY=1000;
public final static int DEF_POSTRECDELAY=500;
public final static String DEF_RECTYPE="audio/wav";

// prompt file attribute names

public final static String ATTPREREC = "prerecdelay";
public final static String ATTRECDUR = "recduration";

public final static String ATTPOSTREC = "postrecdelay";


public final static String ATTITEMCODE = "itemcode";

public final static String ATTTYPE = "rectype";

public final static String ATTSILENCE = "finalsilence";
public final static String ATTBEEP = "beep";
public final static String ATTBLOCKED = "blocked";



	// Fields    

	
	private Reccomment reccomment;
	
	private Recinstructions recinstructions;

	private AnnotationTemplate annotationTemplate;

	private String itemcode;

	private Integer recduration;

	private Integer prerecdelay;

	private Integer postrecdelay;

	private Integer finalsilence;

	private String beep;

	private String rectype;
    
    private Boolean blocked;
	//private Integer position;


    
	private Set<RecordingFile> recordingFiles = new HashSet<RecordingFile>(0);
	
	private int defaultPrerecdelay=DEF_PRERECDELAY;
	

    private int defaultPostrecdelay=DEF_POSTRECDELAY;

	private String[] comments=new String[0];
	// Constructors

	/** default constructor */
	public Recording() {
		super();
		// itemcode and recduration are required
		itemcode="";
		recduration=DEF_RECDURATION;
		rectype=DEF_RECTYPE;
	}

	public Recording(IntegerSequenceGenerator seqGen, Element e) {
		super(seqGen,(Element)e.getElementsByTagName(Recprompt.ELEMENT_NAME).item(0));
		
		NodeList childs=e.getChildNodes();
		ArrayList<String>commentsArrList=new ArrayList<String>();
		for(int ci=0;ci<childs.getLength();ci++){
			Node n=childs.item(ci);
			if(n.getNodeType()==Node.COMMENT_NODE){
				commentsArrList.add(n.getNodeValue());
			}
		}
		comments=commentsArrList.toArray(new String[0]);
		
		initialize();
		
		// REQUIRED attributes		
//		setRecduration(Integer.parseInt(e.getAttribute(ATTRECDUR)));
		
		setItemcode(e.getAttribute(ATTITEMCODE));

		// IMPLIED attributes; if no attribute is given, default values are used
		
		
		Attr attr = e.getAttributeNode(ATTPREREC);
		if (attr != null) {
			setPrerecdelay(Integer.parseInt(attr.getValue()));
		}
		attr = e.getAttributeNode(ATTRECDUR);
        if (attr != null) {
            setRecduration(Integer.parseInt(attr.getValue()));
        }
		attr = e.getAttributeNode(ATTPOSTREC);
		if (attr != null) {
			setPostrecdelay(Integer.parseInt(attr.getValue()));
		}
		attr = e.getAttributeNode(ATTBEEP);
		if (attr != null) {
			setBeep(attr.getValue());
		}
		attr = e.getAttributeNode(ATTSILENCE);
		if (attr != null) {
			setFinalsilence(Integer.parseInt(attr.getValue()));
		} 
		attr = e.getAttributeNode(ATTTYPE);
		if (attr != null) {
			setRectype(attr.getValue());
		} 
        attr = e.getAttributeNode(ATTBLOCKED);
        if (attr != null) {
            setBlocked(BooleanValue.parseExtendedBoolean(attr));
        } 
       
		NodeList instrE=e.getElementsByTagName(Recinstructions.ELEMENT_NAME);
		if(instrE.getLength()>0){
			setRecinstructions(new Recinstructions((Element)(instrE.item(0))));
		}
		
		
		ArrayList<Mediaitem> mis=new ArrayList<Mediaitem>();
		NodeList miE=e.getElementsByTagName(Mediaitem.ELEMENT_NAME);
		int miELen=miE.getLength();
		for(int i=0;i<miELen;i++){
		    Mediaitem mi=new Mediaitem((Element)miE.item(i));
		    mis.add(mi);
		}
//		if(miE.getLength()>0){
//			setMediaitem(new Mediaitem((Element)(miE.item(0))));
//		}
		setMediaitems(mis);
		
		NodeList commentE=e.getElementsByTagName(Reccomment.ELEMENT_NAME);
		if(commentE.getLength()>0){
			setReccomment(new Reccomment((Element)(commentE.item(0))));
		}
		
		NodeList atE=e.getElementsByTagName(AnnotationTemplate.ELEMENT_NAME);
        if(atE.getLength()>0){
            setAnnotationTemplate(new AnnotationTemplate((Element)atE.item(0)));
        }
		
	}
	
	private void initialize(){
		itemcode="";
		//prerecdelay=0;
		//postrecdelay=0;
		recduration=DEF_RECDURATION;
		//beep="0";
		//finalsilence=0;
		//rectype="audio/wave";
	}
	
	
	// This method is deprecated!
	// It does not consider project default values for pre and post-delay ! 
	
//	 /**
//     * getTotalRecTime() returns the sum of the pre-recording delay time, the 
//     * recording duration and the post-recording delay time.
//     * The time base is milliseconds.
//     * 
//     * @return total recording time
//     */
//	@Transient
//    public Integer getTotalRecTime() {
//	    Integer recDuartion=getRecduration();
//	    if(recDuartion==null) return null;
//    	return getNNPrerecdelay() + recduration + getNNPostrecdelay();
//    }

//	@ManyToOne(fetch = FetchType.LAZY)
//	@JoinColumn(name = "section_id", unique = false, nullable = true, insertable = true, updatable = true)
//	public Section getSection() {
//		return this.section;
//	}
//
//	public void setSection(Section section) {
//		this.section = section;
//	}
	
	@ManyToOne(fetch=FetchType.LAZY)
	@JoinColumn(name = "recinstructions_id", unique = false, nullable = true, insertable = true, updatable = true)
	@ResourceKey("recinstructions")
	public Recinstructions getRecinstructions() {
		return this.recinstructions;
	}

	public void setRecinstructions(Recinstructions recinstructions) {
		Recinstructions oldRecinstructions=this.recinstructions;
		if(oldRecinstructions != null)oldRecinstructions.removePropertyChangeListener((PropertyChangeListener)this);
		this.recinstructions = recinstructions;
		if(this.recinstructions!=null)this.recinstructions.addPropertyChangeListener((PropertyChangeListener)this);
		propertyChangeSupport.firePropertyChange("recinstructions", oldRecinstructions, this.recinstructions);
	}
	
	@ManyToOne(fetch=FetchType.LAZY)
	@JoinColumn(name = "reccomment_id")
	@ResourceKey("reccomment")
	public Reccomment getReccomment() {
		return this.reccomment;
	}

	public void setReccomment(Reccomment reccomment) {
		Reccomment oldReccomment=this.reccomment;
		if(oldReccomment!=null)oldReccomment.removePropertyChangeListener((PropertyChangeListener)this);
		this.reccomment = reccomment;
		if(this.reccomment!=null)this.reccomment.addPropertyChangeListener((PropertyChangeListener)this);
		propertyChangeSupport.firePropertyChange("reccomment", oldReccomment, this.reccomment);
	}

	@OneToOne(mappedBy="recording",cascade= {CascadeType.PERSIST,CascadeType.MERGE,CascadeType.REMOVE},fetch=FetchType.LAZY)
    @ResourceKey("annotation.template")
	public AnnotationTemplate getAnnotationTemplate() {
	    return annotationTemplate;
	}

	public void setAnnotationTemplate(AnnotationTemplate annotationTemplate) {
	    this.annotationTemplate = annotationTemplate;
	}

	//@Column(name = "itemcode", length = 10)
	@Column(name = "itemcode",length=1000)
	@ResourceKey("code")
	public String getItemcode() {
		return this.itemcode;
	}

	public void setItemcode(String itemcode) {
		String oldItemcode=this.itemcode;
		this.itemcode = itemcode;
		propertyChangeSupport.firePropertyChange("itemcode", oldItemcode, this.itemcode);
	}

	@Column(name = "recduration")
	@ResourceKey("recording.duration")
	@Unit("ms")
	public Integer getRecduration() {
		return this.recduration;
	}

	public void setRecduration(Integer recduration) {
		Integer oldRecduration=this.recduration;
		this.recduration = recduration;
		propertyChangeSupport.firePropertyChange("recduration", oldRecduration, this.recduration);
	}

	@Column(name = "prerecdelay")
	@ResourceKey("prerecdelay")
    @Unit("ms")
	public Integer getPrerecdelay() {
		return this.prerecdelay;
	}
	
	@Transient
	//@XmlTransient
	public int getNNPrerecdelay(){
		if (prerecdelay==null)return defaultPrerecdelay;
		return prerecdelay;
	}

	public void setPrerecdelay(Integer prerecdelay) {
		Integer oldPrerecdelay=this.prerecdelay;
		this.prerecdelay = prerecdelay;
		propertyChangeSupport.firePropertyChange("prerecdelay", oldPrerecdelay, this.prerecdelay);
	}

	@Column(name = "postrecdelay")
	@ResourceKey("postrecdelay")
	@Unit("ms")
	public Integer getPostrecdelay() {
		return this.postrecdelay;
	}
	@Transient
	//@XmlTransient
	public int getNNPostrecdelay(){
		if(postrecdelay==null)return defaultPostrecdelay;
		return postrecdelay;
	}

	public void setPostrecdelay(Integer postrecdelay) {
		Integer oldPostrecdelay=this.postrecdelay;
		this.postrecdelay = postrecdelay;
		propertyChangeSupport.firePropertyChange("postrecdelay", oldPostrecdelay, this.postrecdelay);
	}

	@Column(name = "finalsilence")
	@ResourceKey("finalsilence")
	@Unit("ms")
	public Integer getFinalsilence() {
		return this.finalsilence;
	}

	public void setFinalsilence(Integer finalsilence) {
		Integer oldFinalsilence=this.finalsilence;
		this.finalsilence = finalsilence;
		propertyChangeSupport.firePropertyChange("finalsilence", oldFinalsilence, this.finalsilence);
	}

	@Column(name = "beep", length = 10)
	@ResourceKey("beep")
	public String getBeep() {
		return this.beep;
	}

	public void setBeep(String beep) {
		String oldBeep=this.beep;
		this.beep = beep;
		propertyChangeSupport.firePropertyChange("beep", oldBeep, this.beep);
	}

	@Column(name = "rectype", length = 10)
	@ResourceKey("type")
	public String getRectype() {
		return this.rectype;
	}

	public void setRectype(String rectype) {
		String oldRectype=this.rectype;
		this.rectype = rectype;
		propertyChangeSupport.firePropertyChange("rectype", oldRectype, this.rectype);
	}

	@OneToMany(fetch = FetchType.LAZY, mappedBy = "recording")
	@ResourceKey("recording_files")
	@ObjectImmutableIfReferenced
	@XmlTransient
	public Set<RecordingFile> getRecordingFiles() {
		return this.recordingFiles;
	}

	public void setRecordingFiles(Set<RecordingFile> recordingFiles) {
		this.recordingFiles = recordingFiles;
	}
	
	
	
		// Repeat getter and setter to get XmlType propOrder to work
	// https://stackoverflow.com/questions/6790168/can-should-i-list-inherited-properties-for-a-jaxb-mapped-bean-in-the-proporder
		@ManyToMany(fetch = FetchType.LAZY)
		@JoinTable(
			      name="recording_mediaitem",
			      joinColumns=@JoinColumn(name="promptitems_recording_id", referencedColumnName="recording_id")
			      )
		@ResourceKey("media.item")
		@XmlElement(name="mediaitems")
		public List<Mediaitem> getMediaitems() {
			return super.getMediaitems();
		}

		public void setMediaitems(List<Mediaitem> mediaitems) {
			super.setMediaitems(mediaitems);
		}

	
	protected void apply(){
		super.apply();
		
		if(prerecdelay==null){
		    setPrerecdelay(getNNPrerecdelay());
		}
		if(postrecdelay==null){
		    setPostrecdelay(getNNPostrecdelay());
		}
		
	}


	public Element toElement(Document d){
		Element e=d.createElement(ELEMENT_NAME);
		for(String comm:comments){
			e.appendChild(d.createComment(comm));
		}
		Integer recDuration=getRecduration();
		if(recDuration!=null){
		    e.setAttribute(ATTRECDUR, recduration.toString());
		}
		e.setAttribute(ATTITEMCODE, getItemcode());
		Integer prerecdelay=getPrerecdelay();
		if(prerecdelay!=null)e.setAttribute(ATTPREREC, prerecdelay.toString());
		Integer postrecdelay=getPostrecdelay();
		if(postrecdelay!=null)e.setAttribute(ATTPOSTREC, postrecdelay.toString());
		
		if(beep!=null){
		    e.setAttribute(ATTBEEP, beep);
        }
        if(blocked!=null){
            e.setAttribute(ATTBLOCKED, blocked.toString());
        }
        if(finalsilence!=null){
            e.setAttribute(ATTSILENCE, finalsilence.toString());
        }
       
		Recinstructions recInstr=getRecinstructions();
		if(recInstr!=null){
		e.appendChild(recInstr.toElement(d));
		}
		
		e.appendChild(super.toElement(d));
		
		Reccomment recComment=getReccomment();
		if(recComment!=null){
		e.appendChild(recComment.toElement(d));
		}
		
		AnnotationTemplate at=getAnnotationTemplate();
        if(at!=null){
            e.appendChild(at.toElement(d));
        }
		
		
		return e;
	}
//	/**
//	 * getDescription() returns descriptive information about an item. For this,
//	 * the item attributes are checked in the following order:
//	 * <ol>
//	 * <li>ALT-tag text</li>
//	 * <li>text contents of item</li>
//	 * <li>comments text</li>
//	 * <li>file name of item URL</li>
//	 * </ol>
//	 *  
//	 * @param pi prompt item
//	 * @return String descriptive text
//	 */
//	@Transient
//	public String getDescription() {
//
//		// same code as in superclass PromptItem except the Recccoment option
//		String description = "";
//		Mediaitem mi = getMediaitem();
//		if (mi != null) {
//			if (mi.getAlt() != null) {
//				return mi.getAlt();
//			} else if (mi.getPromptText() != null) {
//				description = mi.getPromptText();
//			} else if (mi.getSrc() != null) {
//				description = mi.getSrc().getFile();
//			}
//			String mimeType = mi.getNNMimetype();
//			if (mimeType.startsWith("image")) {
//				return "IMAGE: " + description;
//			} else if (mimeType.startsWith("audio")) {
//				return "AUDIO: " + description;
//			} else if (mimeType.startsWith("video")) {
//				return "VIDEO: " + description;
//			} else {
//				return description;
//			}
//		}
//		return description;
//	}
	/**
	 * toString() returns a String representation of a prompt item. The internal fields are given
	 * as attribute-value pairs with the individual pairs separated from each other by
	 * tabs.
	 */
    public String toString() {
        List<Mediaitem> mis=getMediaitems();
        int misSize=mis.size();
        StringBuffer prText=new StringBuffer();
        for(int i=0;i<misSize;i++){
            Mediaitem mi=mis.get(i);
            prText.append(mi.toString());
            if(i+1<misSize){
                prText.append(", ");
            }
        }
        return  ATTPREREC + "=" + prerecdelay + "\t" + ATTRECDUR + "=" + recduration + "\t" + ATTPOSTREC + "=" + postrecdelay+ "\tRecinstruction=" + getRecinstructions() + "\tPrompttext=" + prText + "\t" + ATTITEMCODE + "=" + itemcode;
    }
    
    public void propertyChange(PropertyChangeEvent evt) {
		String propName=evt.getPropertyName();
		String hPropName=ELEMENT_NAME+"."+propName;
		propertyChangeSupport.firePropertyChange(hPropName, evt.getOldValue(), evt.getNewValue());
	}

    @ResourceKey("recording.blockedByPromptPlayback")
    public Boolean getBlocked() {
        return blocked;
    }

    public void setBlocked(Boolean blocked) {
        this.blocked = blocked;
    }
    public void setNNBlocked(boolean blocked) {
        if(blocked==DEF_BLOCKED){
            this.blocked = null;
        }else{
            this.blocked = blocked;
        }
    }
    @Transient
    @XmlTransient
    public boolean getNNBlocked() {
        if(blocked!=null){
        return blocked;
        }else{
            return DEF_BLOCKED;
        }
    }
    
     @Transient
     //@XmlTransient
    public boolean needsSilenceDetector(){
         return(finalsilence!=null && finalsilence>0);
     }
     
     @Transient
     //@XmlTransient
    public boolean needsBeep(){
    	 boolean needsBeep=Boolean.parseBoolean(getBeep());
    	 return needsBeep;
     }


     /**
      * Apply default pre-recording delay.
      * @param defaultPrerecdelay default pre-recording delay in milliseconds
      */
     public void setDefaultPrerecdelay(int defaultPrerecdelay) {
         this.defaultPrerecdelay=defaultPrerecdelay;
     }
     
     @Transient
     @XmlTransient
     public int getDefaultPrerecdelay() {
         return defaultPrerecdelay;
     }

     /**
      * Apply default post-recording delay.
      * @param defaultPostrecdelay default post-recording delay in milliseconds
      */
    public void setDefaultPostrecdelay(int defaultPostrecdelay) {
       this.defaultPostrecdelay=defaultPostrecdelay;
    }
    
    @Transient
    @XmlTransient
    public int getDefaultPostrecdelay() {
        return defaultPostrecdelay;
    }
    
    public boolean equals(Object o) {
    	if(o==this) {
    		return true;
    	}
    	if( o instanceof Recording) {
    		Recording or=(Recording)o;
    		int oId=or.getRecpromptId();
    		if(oId==getRecpromptId()) {
    			return true;
    		}
    	}
    	return false;
    }
    
    public int hashCode() {
    	return getRecpromptId();
    }

   
}
