//    IPS Java Speech Database
//    (c) Copyright 2011
//    Institute of Phonetics and Speech Processing,
//    Ludwig-Maximilians-University, Munich, Germany
//
//
//    This file is part of IPS Java Speech Database
//
//
//    IPS Java Speech Database is free software: you can redistribute it and/or modify
//    it under the terms of the GNU Lesser General Public License as published by
//    the Free Software Foundation, version 3 of the License.
//
//    IPS Java Speech Database is distributed in the hope that it will be useful,
//    but WITHOUT ANY WARRANTY; without even the implied warranty of
//    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
//    GNU Lesser General Public License for more details.
//
//    You should have received a copy of the GNU Lesser General Public License
//    along with IPS Java Speech Database.  If not, see <http://www.gnu.org/licenses/>.

package ipsk.db.speech.script;


import java.awt.datatransfer.DataFlavor;
import java.awt.datatransfer.Transferable;
import java.awt.datatransfer.UnsupportedFlavorException;
import java.beans.PropertyChangeEvent;
import java.beans.PropertyChangeListener;
import java.beans.PropertyChangeSupport;
import java.io.IOException;
import java.net.URI;
import java.util.ArrayList;
import java.util.HashSet;
import java.util.List;
import java.util.Random;
import java.util.Set;

import javax.persistence.Basic;
import javax.persistence.CascadeType;
import javax.persistence.Column;
import javax.persistence.Entity;
import javax.persistence.EnumType;
import javax.persistence.Enumerated;
import javax.persistence.FetchType;
import javax.persistence.GeneratedValue;
import javax.persistence.Id;
import javax.persistence.JoinColumn;
import javax.persistence.ManyToOne;
import javax.persistence.OneToMany;
import javax.persistence.OrderColumn;
import javax.persistence.Table;
import javax.persistence.Transient;
import javax.xml.bind.annotation.XmlTransient;
import javax.xml.bind.annotation.XmlType;

import org.w3c.dom.Attr;
import org.w3c.dom.Document;
import org.w3c.dom.Element;
import org.w3c.dom.Node;
import org.w3c.dom.NodeList;

import ipsk.beans.PreferredDisplayOrder;
import ipsk.db.speech.BasicPropertyChangeSupport;
import ipsk.persistence.ImmutibilityProvider;
import ipsk.persistence.IntegerSequenceGenerator;
import ipsk.util.PluralResourceKey;
import ipsk.util.ResourceBundleName;
import ipsk.util.ResourceKey;
import ipsk.xml.DOMConverter;
import ipsk.xml.DOMConverterException;

/**
 * Represents a section element of the recording script.
 */
@Entity
@Table(name = "section")
@ResourceBundleName("ipsk.db.speech.PropertyNames")
@ResourceKey("section")
@PluralResourceKey("sections")
@PreferredDisplayOrder("sectionId,name,groups,mode,order,*")
@XmlType(name="section",namespace="section",propOrder={"name","mode","order","promptphase","training","speakerDisplay","groups"})
//@XmlType(name="section",namespace="section")
public class Section extends BasicPropertyChangeSupport implements
		java.io.Serializable, Cloneable, Transferable,ImmutibilityProvider{

	public final static String ELEMENT_NAME = "section";
	
	public static final DataFlavor CLASS_DATA_FLAVOR=new DataFlavor(Section.class,null);
	private DataFlavor textXmlFlavor;

	// TODO: should one not use the status constants as defined in RecStatus?
	public enum PromptPhase {
	    IDLE("idle"), RECORDING("recording"), PRERECORDING("prerecording");

	    PromptPhase(String value) {
	    	this.value = value;
	    }
	    private final String value;

	    public String value() {
	    	return value; 
	    }
	    public String toString() {
	    	return value; 
	    }
	    public static PromptPhase getByValue(String value){
	    	for(PromptPhase pp:PromptPhase.values()){
	    		if(pp.value.equals(value)){
	    			return pp;
	    		}
	    	}
	    	return null;
	    }
	}
	@XmlType(namespace="section")
	public enum Order {
	    SEQUENTIAL("sequential"), RANDOM("random"), RANDOMIZED("randomized");

	    Order(String value) {
	    	this.value = value;
	    }
	    private final String value;

	    public String value() {
	    	return value; 
	    }
	    public String toString() {
	    	return value; 
	    }
	    public static Order getByValue(String value){
	    	for(Order pp:Order.values()){
	    		if(pp.value.equals(value)){
	    			return pp;
	    		}
	    	}
	    	return null;
	    }
	}
	
	public enum Mode {
	    MANUAL("manual"), AUTOPROGRESS("autoprogress"),AUTORECORDING("autorecording");

	    Mode(String value) {
	    	this.value = value;
	    }
	    private final String value;

	    public String value() {
	    	return value; 
	    }
	    public String toString() {
	    	return value; 
	    }
	    public static Mode getByValue(String value){
	    	for(Mode pp:Mode.values()){
	    		if(pp.value.equals(value)){
	    			return pp;
	    		}
	    	}
	    	return null;
	    }
	}

	public static final PromptPhase DEF_PROMPT_PHASE=PromptPhase.IDLE;
	

	private int sectionId;

	// private Recordingscript recordingscript;
	private Script script;

	private String name;

	private Boolean speakerDisplay;

	private String speakerdisplayStr;

	private Integer sectionPosition;

	private PromptPhase promptphase;

	private Mode mode;

	private Order order;

//	private Set<PromptItem> promptItemsSet = new HashSet<PromptItem>(0);
//
//	// TODO use base class for Recording, Nonrecording: PromptItem
//	// private Recording[] recordings=new Recording[0];
//	private PromptItem[] promptItems = null;
	private List<Group> groups=new ArrayList<Group>();
	
	private String[] comments=new String[0];
	
	private boolean defaultSpeakerDisplay=false;

	private Mode defaultMode=Mode.MANUAL;
	
	private Order defaultOrder=Order.SEQUENTIAL;
	
	private boolean training=false;
	
	private List<Group> shuffledPromptGroups=null;
	
	private boolean propertyChangeSupportEnabled=false;
	
	private DataFlavor[] dataFlavors;

	private transient DOMConverter domConverter;

	@Transient
	@XmlTransient
	public boolean isPropertyChangeSupportEnabled() {
		return propertyChangeSupportEnabled;
	}

	public void setPropertyChangeSupportEnabled(boolean propertyChangeSupportEnabled) {
		this.propertyChangeSupportEnabled = propertyChangeSupportEnabled;
	}

	// Constructors

	/** default constructor */
	public Section() {
		super();
//		try {
////			textXmlFlavor=new DataFlavor("text/plain");
//		} catch (ClassNotFoundException e) {
//			
//		}
//		textXmlFlavor=DataFlavor.getTextPlainUnicodeFlavor();
		textXmlFlavor=DataFlavor.stringFlavor;
		if(textXmlFlavor!=null){
			dataFlavors=new DataFlavor[]{CLASS_DATA_FLAVOR,textXmlFlavor};
		}else{
			dataFlavors=new DataFlavor[]{CLASS_DATA_FLAVOR};
		}
	}
	
//	public Section(boolean initialize) {
//		this();
//		if (initialize) {
//			// section requires at least one prompt Item
////			setPromptItems(new PromptItem[] { new Recording() });
//			
//			groups=new ArrayList<PromptUnit>();
//			groups.add(new Recording());
//		}
//	}
//		
		
//
//	/** minimal constructor */
//	public Section(int sectionId) {
//		this();
//		this.sectionId = sectionId;
//	}

//	/** full constructor */
//	public Section(int sectionId, Script script, String name,
//			String speakerdisplay, Integer sectionPosition, PromptPhase promptphase,
//			Mode mode, Order order, Set<PromptItem> recordings) {
//		this(sectionId);
//		this.script = script;
//		this.name = name;
//		this.speakerdisplayStr = speakerdisplay;
//		this.sectionPosition = sectionPosition;
//		this.promptphase = promptphase;
//		this.mode = mode;
//		this.order = order;
//		this.promptItemsSet = recordings;
//	}

	public Section(IntegerSequenceGenerator seqGen, Element e) {
		super();
		if(seqGen!=null){
			setSectionId(seqGen.getAndIncrement());
		}
		NodeList childs=e.getChildNodes();
		ArrayList<String>commentsArrList=new ArrayList<String>();
		for(int ci=0;ci<childs.getLength();ci++){
			Node n=childs.item(ci);
			if(n.getNodeType()==Node.COMMENT_NODE){
				commentsArrList.add(n.getNodeValue());
			}
		}
		comments=commentsArrList.toArray(new String[0]);
		initializeSection();
		// all attributes are optional
		Attr attr = e.getAttributeNode("name");
		if (attr != null)
			setName(attr.getValue());
		attr = e.getAttributeNode("mode");
		if (attr != null)
			setMode(Mode.getByValue(attr.getValue()));
		attr = e.getAttributeNode("order");
		if (attr != null)
			setOrder(Order.getByValue(attr.getValue()));
		attr = e.getAttributeNode("promptphase");
		if (attr != null)
			setPromptphase(PromptPhase.getByValue(attr.getValue()));

		attr = e.getAttributeNode("speakerdisplay");
		if (attr != null) {
			setSpeakerdisplayStr(attr.getValue());
		}

		ArrayList<Group> promptItemArrList=new ArrayList<Group>();
		NodeList eNodeList=e.getChildNodes();
//		int promptItemPosition=0;
		for(int i=0;i<eNodeList.getLength();i++){
		    Node n=eNodeList.item(i);
		    if(n.getNodeType()==Node.ELEMENT_NODE){
		        Element piE=(Element)n;
		        if(piE.getTagName().equals(Recording.ELEMENT_NAME)){
		            Recording recording = new Recording(seqGen,piE);
//		          recording.setPosition(promptItemPosition++);
		          //recording.setSection(this);
		          Group group = new Group(seqGen,recording,this);
//                  group.setPosition(promptItemPosition++);
		          group.setSection(this);
		          promptItemArrList.add(group);
		        }else if(piE.getTagName().equals(Nonrecording.ELEMENT_NAME)){
                    Nonrecording nonrecording = new Nonrecording(seqGen,piE);
//                    nonrecording.setPosition(promptItemPosition++);
                    //nonrecording.setSection(this);
                    Group group = new Group(seqGen,nonrecording,this);
//                    group.setPosition(promptItemPosition++);
                    group.setSection(this);
                    promptItemArrList.add(group);
                  }
		        else if(piE.getTagName().equals(Group.ELEMENT_NAME)){
                    Group group = new Group(seqGen,piE,this);
//                    group.setPosition(promptItemPosition++);
                    group.setSection(this);
                    promptItemArrList.add(group);
                  }
		    }
		}
		setGroups(promptItemArrList);
    }

	/**
	 * sets all recording section data to the empty string or false
	 * 
	 */
	private void initializeSection() {
		//setName("");
		// setSpeakerdisplay(false);
		//setMode(null);
		//setOrder(Order.SEQUENTIAL);
		//setPromptphase(IDLE);
		// promptItems = null;
	}

	public Set<List<String>> requiredMIMETypeCombinations(){
	    HashSet<List<String>> reqMIMETypes=new HashSet<List<String>>();
	    List<Group> pisList=getGroups();
	    for(Group g:pisList){
	        for(PromptItem pi:g.getPromptItems()){
	       List<String> piMimes=pi.getMIMETypes();
	       reqMIMETypes.add(piMimes);
	        }
	    }
	    return reqMIMETypes;
	}
	
	// Property accessors
	@Id
	@Column(name = "section_id", unique = true, nullable = false)
	//@SequenceGenerator(name = "ID_SEQ", sequenceName = "id_seq")
	@GeneratedValue(generator = "id_gen")
	@ResourceKey("id")
	@XmlTransient
	public int getSectionId() {
		return this.sectionId;
	}

	public void setSectionId(int sectionId) {
		int oldsectionId = this.sectionId;
		this.sectionId = sectionId;
		propertyChangeSupport.firePropertyChange("sectionId", oldsectionId,
				this.sectionId);
	}

    @ManyToOne
	@JoinColumn(name = "script_id")
	@ResourceKey("script")
	@XmlTransient
	public Script getScript() {
		return this.script;
	}

	public void setScript(Script script) {
		// Script oldScript=this.script;
		this.script = script;
		// propertyChangeSupport.firePropertyChange("script", oldScript,
		// this.script);
	}

	// @ManyToOne(cascade = {}, fetch = FetchType.LAZY)
	// @JoinColumn(name = "recording_id", unique = false, nullable = true,
	// insertable = true, updatable = true)
	// public Recordingscript getRecordingscript() {
	// return this.recordingscript;
	// }
	//
	// public void setRecordingscript(Recordingscript recordingscript) {
	// this.recordingscript = recordingscript;
	// }

	@Column(name = "name", length = 100)
	@ResourceKey("name")
	public String getName() {
		return this.name;
	}

	public void setName(String name) {
		String oldName = this.name;
		this.name = name;
		PropertyChangeEvent pce=new PropertyChangeEvent(this, "name", oldName,this.name);
		//propertyChangeSupport.firePropertyChange("name", oldName, this.name);
		propertyChangeSupport.firePropertyChange(pce);
	}

	@Column(name = "speakerdisplay", length = 10)
	@ResourceKey("speakerdisplay")
	@XmlTransient
	public String getSpeakerdisplayStr() {
		return this.speakerdisplayStr;
	}

	public void setSpeakerdisplayStr(String speakerdisplayStr) {
		// TODO fire prop change here ??
		this.speakerdisplayStr = speakerdisplayStr;
		if (speakerdisplayStr != null) {
			if (speakerdisplayStr.equalsIgnoreCase("yes")) {
				speakerDisplay = true;
			} else {
				speakerDisplay = Boolean.parseBoolean(speakerdisplayStr);
			}
		} else {
			this.speakerDisplay = false;
		}
		
	}
	
	@Transient
	//@XmlTransient
	public boolean getNNSpeakerDisplay(){
		if(speakerDisplay==null)return defaultSpeakerDisplay;
		return speakerDisplay;
	}

//	// Not used in Wikispeech JPA/SQL column is empty !!
//	// JPA uses  @OrderColumn(name="section_position") in getSections() in RecordingScript class
//	// TODO Can we remove the SQL column and change this to transient
//    @Column(name = "section_position_in_script")
//	@ResourceKey("position")
//    @XmlTransient
//	public Integer getSectionPosition() {
//		return this.sectionPosition;
//	}
//
//	public void setSectionPosition(Integer sectionPosition) {
//		// TODO fire property change ??
//		this.sectionPosition = sectionPosition;
//	}

	@Column(name = "promptphase", length = 100)
	@Enumerated(EnumType.STRING)
	@ResourceKey("promptphase")
	public PromptPhase getPromptphase() {
		return this.promptphase;
	}
	
	@Transient
//    @XmlTransient
	public PromptPhase getNNPromptphase(){
		if (this.promptphase==null){
			return DEF_PROMPT_PHASE;
		}else{
			return this.promptphase;
		}
	}

	public void setPromptphase(PromptPhase promptphase) {
		PromptPhase oldPromptphase = this.promptphase;
		this.promptphase = promptphase;
		if(propertyChangeSupportEnabled){
		propertyChangeSupport.firePropertyChange("promptphase", oldPromptphase,
				this.promptphase);
	}
	}

	@Column(name = "mode", length = 100)
	@Enumerated(EnumType.STRING)
	@ResourceKey("mode")
	public Mode getMode() {
		return this.mode;
	}

	public void setMode(Mode mode) {
		Mode oldMode = this.mode;
		this.mode = mode;
		if(propertyChangeSupportEnabled){
		propertyChangeSupport.firePropertyChange("mode", oldMode, this.mode);
	}
	}
	
	@Transient
//    @XmlTransient
	public Mode getNNMode() {
		if(mode==null)return defaultMode;
		return this.mode;
	}

	// Renamed order column to ordering (order is SQL keyword, most JPA providers cannot handle this)
	@Column(name = "ordering", length = 100)
	@Enumerated(EnumType.STRING)
	@ResourceKey("order")
	public Order getOrder() {
		return this.order;
	}

	public void setOrder(Order order) {
		Order oldOrder = this.order;
		this.order = order;
		propertyChangeSupport.firePropertyChange("order", oldOrder, this.order);
	}
	
	@Transient
//    @XmlTransient
	public Order getNNOrder() {
		if(order==null)return defaultOrder;
		return this.order;
	}

	@OneToMany(cascade={CascadeType.REMOVE,CascadeType.MERGE},fetch = FetchType.LAZY, mappedBy = "section")
	@OrderColumn(name="group_position")
	@ResourceKey("groups")
	public List<Group> getGroups(){
		return groups;
	}

	public void setGroups(List<Group> promptItems){
	    List<Group> oldPromptItems = this.groups;
	    this.groups = promptItems;
	    shuffledPromptGroups=null;
	    if(propertyChangeSupportEnabled){
	        if (this.groups != null) {
	            for (Group g : this.groups) {
	                g.addPropertyChangeListener(new PropertyChangeListener(){
	                    public void propertyChange(PropertyChangeEvent evt) {
	                        String propName=evt.getPropertyName();
	                        String hPropName="promptItems."+propName;
	                        propertyChangeSupport.firePropertyChange(hPropName, evt.getOldValue(), evt.getNewValue());
	                    }
	                });
	            }
	        }
	        propertyChangeSupport.firePropertyChange("promptItems", oldPromptItems,
	                this.groups);
	    }	
	}
	
    /**
     * Return true if section is intended for subject training.
     * @return true if a training section
     */
	@ResourceKey("training")
	@Basic
	public boolean isTraining() {
		return training;
	}
    /**
     * If set to true this section is intended for subject training.
     * The recordings of training sections will be ignored for distributions.
     * Default is false.
     * @param training true if training section
     */
	public void setTraining(boolean training) {
		this.training = training;
	}
//
//	public int compareTo(Section o) {
//		return getSectionPosition().compareTo(o.getSectionPosition());
//	}

	public Element toElement(Document d) {
		Element e = d.createElement(ELEMENT_NAME);
		for(String comm:comments){
			e.appendChild(d.createComment(comm));
		}
		String name = getName();
		if (name != null)
			e.setAttribute("name", name);
		Order order = getOrder();
		if (order != null)
			e.setAttribute("order", order.value());
		PromptPhase promptphase = getPromptphase();
		if (promptphase != null)
			e.setAttribute("promptphase", promptphase.value);
		Mode mode = getMode();
		if (mode != null)
			e.setAttribute("mode", mode.value());
		String speakerdisplay = getSpeakerdisplayStr();
		if (speakerdisplay != null)
			e.setAttribute("speakerdisplay", speakerdisplay);

			List<Group> gs=getGroups();
			for (Group g : gs) {
			    List<PromptItem> pis=g.getPromptItems();
			    if(pis.size()==1){
			        // do not persist dummy groups
			        e.appendChild(pis.get(0).toElement(d));
			    }else{
			        e.appendChild(g.toElement(d));
			    }
			}
		return e;
	}

	@Transient
	public Boolean getSpeakerDisplay() {
		return speakerDisplay;
	}

	public void setSpeakerDisplay(Boolean speakerDisplay) {
		Boolean oldSpeakerDisplay = this.speakerDisplay;
		this.speakerDisplay = speakerDisplay;
		if (speakerDisplay == null) {
			speakerdisplayStr = null;
		} else {
			speakerdisplayStr = speakerDisplay.toString();
		}
		if(propertyChangeSupportEnabled){
		propertyChangeSupport.firePropertyChange("speakerDisplay",
				oldSpeakerDisplay, this.speakerDisplay);
	}
	}
	


	public String toString() {
//		StringBuffer sb = new StringBuffer();
//		sb.append(getName() + ", " + getOrder() + ", " + getMode() + ", "
//				+ getNNPromptphase() + ", " + getSpeakerDisplay() + "\n");
//
//		PromptItem[] promptItems = getPromptItems();
//		for (int i = 0; i < promptItems.length; i++) {
//			PromptItem pi = (PromptItem) promptItems[i];
//			sb.append(pi.toString() + "\n");
//		}
//		return sb.toString();
		String name =getName();
		if(name!=null){
			return name;
		}
		return Integer.toString(getSectionId());
	}

	@Transient
	public String getInfo() {
		
		String attrs=getNNMode() + ", " + getNNOrder() + ", "
				+ getNNPromptphase() + ", " + getNNSpeakerDisplay();
		if(name==null){
			return attrs;
		}else{
			return name+", "+attrs;
		}
	}


	@Transient
	public boolean isDataFlavorSupported(DataFlavor flavor) {
		for(DataFlavor df:dataFlavors){
			if(df.equals(flavor)){
				return true;
			}
		}
		return false;
	}
	@Transient
	public void setDefaultMode(Mode defaultMode) {
		this.defaultMode=defaultMode;
	}

    @Transient
	public void setDefaultSpeakerDisplay(boolean defaultSpeakerDisplay) {
		this.defaultSpeakerDisplay=defaultSpeakerDisplay;
	}


    /**
     * Apply default pre-recording delay.
     * @param defaultPrerecdelay default pre-recording delay in milliseconds
     */
    @Transient
    public void defaultPrerecdelay(int defaultPrerecdelay) {
        List<Group> grs=getGroups();
        for(Group g:grs){
            g.defaultPrerecdelay(defaultPrerecdelay);
        }
    }

    /**
     * Apply default post-recording delay.
     * @param defaultPostrecdelay default post-recording delay in milliseconds
     */
    @Transient
    public void defaultPostrecdelay(int defaultPostrecdelay) {
        List<Group> grs=getGroups();
        for(Group g:grs){
            g.defaultPostrecdelay(defaultPostrecdelay);
        }
    }
    
    @Transient
    public void defaultAutoplay(boolean defaultAutoplay){
        List<Group> grs=getGroups();
        for(Group g:grs){
            g.defaultAutoplay(defaultAutoplay);
        }
    }

	public Object clone() throws CloneNotSupportedException{
	       
        Object c=super.clone();
        Section cs=(Section)c;
        
        List<Group> gs=groups;
        ArrayList<Group> cGs=new ArrayList<Group>();
        for(Group g:gs){
            Group cG=(Group)g.clone();
            cGs.add(cG);
        }
        cs.groups=cGs;
       // TODO shuffled items are not cloned
        cs.propertyChangeSupport=new PropertyChangeSupport(this);
       return cs;
   }
	
	@Transient
	@XmlTransient
	public Mode getDefaultMode() {
		return defaultMode;
	}
	@Transient
	@XmlTransient
	public boolean isDefaultSpeakerDisplay() {
		return defaultSpeakerDisplay;
	}
	
	
	 @Transient
	
		public void defaultVirtualViewBox(VirtualViewBox defaultVirtualViewBox) {
			for(Group g:groups) {
				g.defaultVirtualViewBox(defaultVirtualViewBox);
			}
		}

	@Transient
	public boolean recordingCodesUnique(){
	    HashSet<String> itemCodes=new HashSet<String>();

	    for(Group g:groups){
	        for(PromptItem pi:g.getPromptItems()){
	            Recording r=(Recording)pi;
	            String itemCode=r.getItemcode();
	            if(itemCodes.contains(itemCode)){
	                return false;
	            }else{
	                itemCodes.add(itemCode);
	            }
	        }
	    }
		return true;
	}
	
//    @Transient
//	 public void updatePositions(){
//            int pos=0;
//	        int pisSize=groups.size();
//	        for(int i=0;i<pisSize;i++){
//	            Group g=groups.get(i);
//	            g.setSection(this);
//	            List<PromptItem> pis=g.getPromptItems();
//	            for(int j=0;j<pis.size();j++){
//	                PromptItem pi=pis.get(j);
//	               // pi.setPosition(pos);
//	                pos++;
//	            }
//	        }
//	 }
	
	@Transient
	public void updateUpwardsRelations(){
        for(Group g:groups){
            g.setSection(this);
            g.updateUpwardsRelations();
        }
	}

	
    @Transient
    private List<PromptItem> toPromptItemList(List<Group> groups){
        List<PromptItem> pis = new ArrayList<PromptItem>();
        for(Group g:groups){
            // add a group in order
            pis.addAll(g.shuffledPromptItems());
        }
        return pis;
    }
	 
	/**
     * Shuffles the prompt units.
     */
	@Transient
    public void shuffleItems() {
        List<Group> shuffledUnits = new ArrayList<Group>(groups);
        
        Random rnd = new Random();
        for (int i = groups.size(); i > 1; i--) {
                swap(shuffledUnits, i - 1, rnd.nextInt(i));
        }
        for(Group g:shuffledUnits){
              g.shuffleItems();
        }
        shuffledPromptGroups=shuffledUnits;
    }
    
    private void swap(List<Group> items, int i, int j) {
        Group tmp = items.get(i);
        items.set(i, items.get(j));
        items.set(j, tmp);
    }
    
    /**
     * Get shuffled prompt items.
     * @return previously shuffled prompt items if section has random order otherwise sequential ordered items
     */
    @Transient
    public List<PromptItem> getShuffledPromptItems() {
        if(getNNOrder().equals(Order.RANDOM)){
            if(shuffledPromptGroups==null){
                shuffleItems();
            }
            return toPromptItemList(shuffledPromptGroups);
        }else{
            return toPromptItemList(groups);
        }
    }
    
    
    void apply(){
    	
    	// apply 
    	setSpeakerDisplay(getNNSpeakerDisplay());
    	setMode(getNNMode());
    	
    	for(Group g:getGroups()){
    		g.apply();
    	}
    	
    	// shuffle and set to randomized
    	if(getNNOrder().equals(Order.RANDOM)){
            if(shuffledPromptGroups==null){
                shuffleItems();
            }
            setGroups(shuffledPromptGroups);
//            updatePositions();
            setOrder(Order.RANDOMIZED);
    	}
        
    }
	 
	@Transient
	public Object getTransferData(DataFlavor flavor) throws UnsupportedFlavorException, IOException {
		if (!isDataFlavorSupported(flavor)) {
			throw new UnsupportedFlavorException(flavor);
		}
		if(CLASS_DATA_FLAVOR.equals(flavor)){
			return this;
		}else if(textXmlFlavor!=null && textXmlFlavor.equals(flavor)){

			try {
				
				if(domConverter==null){
					domConverter = new DOMConverter();
				}
				Document d=domConverter.newDocument();
				d.appendChild(toElement(d));
				return domConverter.writeFragmentToString(d);
			} catch (DOMConverterException e) {
				throw new IOException(e);
			}

		}else{
			throw new UnsupportedFlavorException(flavor);
		}
	}
	
	@Transient
	public DataFlavor[] getTransferDataFlavors() {
		return dataFlavors;
	}
	
	public static void main(String[] args){
		Mode m=Mode.getByValue("autoprogress");
		System.out.println(m.value());
	}

    /**
     * Returns all itemcodes used without duplicates.
     * A section may have duplicate itemcodes. To retrieve
     * itemcodes including duplicates use {@link #itemCodesList() itemCodesList}.
     * @return set of unique itemcodes
     */
    @Transient
    public Set<String> itemCodesSet() {
        Set<String> ics=new HashSet<String>();
        List<String> itemCodesList=itemCodesList();
        ics.addAll(itemCodesList);
//        for(PromptUnit pi:groups){
//            if(pi instanceof Recording){
//                Recording r=(Recording)pi;
//                String ic=r.getItemcode();
//                if(ic!=null){
//                    ics.add(ic);
//                }
//            }
//        }
        return ics;
    }
    
    /**
     * Returns all itemcodes used with duplicates.
     * @return list of itemcodes
     */
    @Transient
    public List<String> itemCodesList() {
        List<String> icl=new ArrayList<String>();
        for(Group g:groups){
            for(PromptItem gPi:g.getPromptItems()){
                if(gPi instanceof Recording){
                    Recording gr=(Recording)gPi;
                    String ic=gr.getItemcode();
                    if(ic!=null){
                        icl.add(ic);
                    }
                }
            }
        }
        return icl;
    }
    
    @Transient
    public boolean needsSilenceDetector(){
        for(PromptItem pi:promptItemsList()){
            if(pi instanceof Recording){
                Recording r=(Recording)pi;
                if(r.needsSilenceDetector()){
                    return true;
                }
            }
        }
        return false;
    }

    @Transient
    public List<PromptItem> promptItemsList() {
        ArrayList<PromptItem> pis=new ArrayList<PromptItem>();
        for(Group g:getGroups()){
            pis.addAll(g.getPromptItems());
        }
        return pis;
    }
    
    @Transient
    public boolean needsBeep(){
        for(PromptItem pi:promptItemsList()){
            if(pi instanceof Recording){
                Recording r=(Recording)pi;
                if(r.needsBeep()){
                    return true;
                }
            }
        }
        return false;
    }
    
    @Transient
    public Set<URI> resourceURIs(){
        Set<URI> usedUris=new HashSet<URI>();
        for(PromptItem pi:promptItemsList()){
            usedUris.addAll(pi.resourceURIs());
        }
        return usedUris;
    }

	/* (non-Javadoc)
	 * @see ipsk.persistence.ImmutibilityProvider#isImmutable()
	 */
	@Override
	 @Transient
	public boolean isImmutable() {
		Script script=getScript();
		return (script!=null && script.isImmutable()); 
	}
	
	@Override
	@Transient
	public boolean isRemovable() {
		return !isImmutable();
	}
}
