//    Speechrecorder
//    (c) Copyright 2012
// 	  Institute of Phonetics and Speech Processing,
//    Ludwig-Maximilians-University, Munich, Germany
//
//
//    This file is part of Speechrecorder
//
//
//    Speechrecorder is free software: you can redistribute it and/or modify
//    it under the terms of the GNU Lesser General Public License as published by
//    the Free Software Foundation, version 3 of the License.
//
//    Speechrecorder is distributed in the hope that it will be useful,
//    but WITHOUT ANY WARRANTY; without even the implied warranty of
//    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
//    GNU Lesser General Public License for more details.
//
//    You should have received a copy of the GNU Lesser General Public License
//    along with Speechrecorder.  If not, see <http://www.gnu.org/licenses/>.
 
package ipsk.apps.speechrecorder.config;




import ipsk.apps.speechrecorder.config.MixerName.ProviderId.ProviderIdType;
import ipsk.beans.dom.DOMAttributes;
import ipsk.beans.dom.DOMTextNodePropertyName;
import ipsk.db.speech.Section.PromptPhase;
import ipsk.text.ParserException;

/**
 * A JavaSound device name or regular expression match.
 * @author K.Jaensch, klausj@phonetik.uni-muenchen.de
 *
 */

@DOMAttributes({"providerId","interfaceName","regex"})
@DOMTextNodePropertyName("name")
public class MixerName {

    
    
    public static class ProviderId{
        
        public enum ProviderIdType {
//            JAVA_CLASS("java","class");
           
           JAVA_SOUND("java","javasound",false),JAVA_CLASS("java","class",true), SELECT_DEFAULT("select","default",false),SELECT_BY_NAME("select","name",true);
            
            private final String major;
            private final String minor;
            private final boolean hasIdValue;
            public boolean isHasIdValue() {
                return hasIdValue;
            }
            ProviderIdType(String major,String minor,boolean hasIdValue) {
                this.major=major;
                this.minor=minor;
                this.hasIdValue=hasIdValue;
                String val= major;
                if(minor!=null){
                    val=val.concat(":"+minor);
                }
                this.value=val;
            }
            private final String value;

            public String value() {
                
                return value; 
            }
            public String toString() {
                return value; 
            }
            public static ProviderIdType getByValue(String value){
                for(ProviderIdType pit:ProviderIdType.values()){
                    if(pit.value.equals(value)){
                        return pit;
                    }
                }
                return null;
            }
            
            public static ProviderIdType parse(String providerIdStr) throws ParserException{

                for(ProviderIdType pit:ProviderIdType.values()){
                    String major=pit.getMajor();
                    if(providerIdStr.startsWith(major+":")){
                        String minor=pit.getMinor();
                        if(minor==null){
                            // match
                            return pit;
                        }else{
                            if(pit.isHasIdValue()){
                                if(providerIdStr.startsWith(pit.prefix())){
                                    // match
                                    return pit;
                                }
                            }else{
                                if(providerIdStr.equals(pit.value())){
                                    // match
                                    return pit;
                                }
                            }
                        }
                    }
                }

                throw new ParserException("Not a valid audio interface provider ID type.");
            }
            public String prefix(){
                return this.value+":";
            }
            public String getMajor() {
                return major;
            }
            /**
             * @return the minor
             */
            public String getMinor() {
                return minor;
            }
        }
        
        private ProviderIdType idType;
        public ProviderIdType getIdType() {
            return idType;
        }

        public String getId() {
            return id;
        }

        private String id;
        
        public ProviderId(ProviderIdType idType) {
           this(idType,null);
        }
        
        public ProviderId(ProviderIdType idType, String id) {
            super();
            this.idType = idType;
            this.id = id;
        }
        
        public static ProviderId parse(String providerIdStr) throws ParserException{
           ProviderIdType idType=ProviderIdType.parse(providerIdStr);
           String id=null;
           if(idType.hasIdValue){
               id=providerIdStr.substring(idType.prefix().length());
           }else{
               if(idType.value().length()!=providerIdStr.length())
               throw new ParserException("Audio interface ID type "+idType+" must not have an ID value");
           }
           return new ProviderId(idType, id);
        }
        
        public String toString(){
           return idType.prefix()+id;
        }
    }
   
    
//    public static final String ID_JAVA_CLASS="java:class:";
    private String name;
    private String interfaceName=null;
    private String providerId=null;
    private boolean regex=false;


    public MixerName(){
        super();
    }
    
    public MixerName(String name){
        this(null,name);
    }
    public MixerName(String providerId,String name){
        this(providerId,name,false);
    }
    public MixerName(String providerId,String name,boolean regex){
        this(providerId,null,name,regex);
    }
    /**
     * @param providerId
     * @param interfaceName
     * @param name
     */
    public MixerName(String providerId, String interfaceName,
            String name) {
        this(providerId,interfaceName,name,false);
    }
    public MixerName(String providerId,String interfaceName,String name,boolean regex){
        this();
        this.providerId=providerId;
        this.interfaceName=interfaceName;
        this.name=name;
        this.regex=regex;
    }
    
    

    public boolean isRegex() {
        return regex;
    }

    public void setRegex(boolean regex) {
        this.regex = regex;
    }

    public String getName() {
        return name;
    }

    public void setName(String name) {
        this.name = name;
    }
    public String getProviderId() {
        return providerId;
    }
    
    public ProviderId providerIdObj() throws ParserException{
         
        // check legacy default JavaSound and legacy attribute "interfaceName" 
        if(this.providerId==null){
            if(interfaceName==null || "".equals(interfaceName)){
                // legacy default: javaSound
                return new ProviderId(ProviderIdType.JAVA_SOUND);
            }else{
               // by interface name
                return new ProviderId(ProviderIdType.SELECT_BY_NAME,interfaceName);
            }
        }else{
            // parse providerId attribute
            // if both attributes interfaceName and providerId are set then providerId has precedence
        
           return ProviderId.parse(this.providerId);
        }
    }
    
//    public String providerIdAsJavaClassName(){
//       return providerIdToJavaClassName(providerId);
//    }
//    public static String providerIdToJavaClassName(String providerId){
//        if(providerId!=null){
//            String trimmedIId=providerId.trim();
//            if(trimmedIId.startsWith(ProviderIdType.JAVA_CLASS.value())){
//                return providerId.substring(ProviderIdType.JAVA_CLASS.value().length()).trim();
//            }
//        }
//        return null;
//    }
//    
//    public static String javaClassnameToProviderId(String className){
//        return ProviderIdType.JAVA_CLASS.value()+className;
//    }
    
    public void setProviderId(String interfaceId) {
        this.providerId = interfaceId;
    }
    
   
    
    public boolean equals(Object o){
        if (o!=null && o instanceof MixerName){
            MixerName oRmn=(MixerName)o;
            if(oRmn.isRegex()==isRegex()){
                String oProviderId=oRmn.getProviderId();
                if(oProviderId==null){
                    if(providerId!=null){
                        return false;
                    }
                    // no provider ID
                    // check interface names
                    String oInterfaceName=oRmn.getInterfaceName();
                    if(oInterfaceName==null){
                        if(interfaceName!=null){
                            return false;
                        }
                    }else{
                        if(!oInterfaceName.equals(interfaceName)){
                            return false;
                        }
                    }
                }else{
                    if(!oProviderId.equals(providerId)){
                        return false;
                    }
                    // do not care about the 
                }
                String oRmnName=oRmn.getName();
                if(oRmnName==null){
                    if(name==null){
                        return true;
                    }
                }else{
                    if(oRmnName.equals(name)){
                        return true;
                    }
                }
            }
        }
        return false;
    }
    public String toString(){
        StringBuffer s=new StringBuffer();
        if(providerId!=null){
            s.append(providerId);
            s.append(": ");
        }
        if(interfaceName!=null){
            s.append(interfaceName);
            s.append(": ");
        }
        if(name!=null){
            s.append(name);
        }
        if(regex){
            s.append(" (regular expr)");
        }
        return s.toString();
    }

    public String getInterfaceName() {
        return interfaceName;
    }

    public void setInterfaceName(String interfaceName) {
        this.interfaceName = interfaceName;
    }

   
}
