//    Speechrecorder
//    (c) Copyright 2012
// 	  Institute of Phonetics and Speech Processing,
//    Ludwig-Maximilians-University, Munich, Germany
//
//
//    This file is part of Speechrecorder
//
//
//    Speechrecorder is free software: you can redistribute it and/or modify
//    it under the terms of the GNU Lesser General Public License as published by
//    the Free Software Foundation, version 3 of the License.
//
//    Speechrecorder is distributed in the hope that it will be useful,
//    but WITHOUT ANY WARRANTY; without even the implied warranty of
//    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
//    GNU Lesser General Public License for more details.
//
//    You should have received a copy of the GNU Lesser General Public License
//    along with Speechrecorder.  If not, see <http://www.gnu.org/licenses/>.


package ipsk.apps.speechrecorder.prompting;

import java.awt.BorderLayout;
import java.awt.Color;
import java.awt.Dimension;
import java.awt.Font;
import java.awt.FontMetrics;
import java.awt.GridLayout;
import java.awt.Insets;
import java.awt.Rectangle;
import java.awt.Toolkit;
import java.awt.event.ActionEvent;
import java.awt.event.ActionListener;
import java.awt.event.ComponentEvent;
import java.awt.event.ComponentListener;
import java.io.IOException;
import java.net.MalformedURLException;
import java.net.URL;

import javax.swing.BorderFactory;
import javax.swing.JButton;
import javax.swing.JFrame;
import javax.swing.JLabel;
import javax.swing.JPanel;
import javax.swing.JTextField;
import javax.swing.JTextPane;
import javax.swing.border.Border;
import javax.swing.text.BadLocationException;
import javax.swing.text.Style;
import javax.swing.text.StyleConstants;
import javax.swing.text.StyleContext;
import javax.swing.text.StyledDocument;

import ipsk.apps.speechrecorder.prompting.presenter.PromptPresenter;
import ipsk.apps.speechrecorder.prompting.presenter.PromptPresenterException;
import ipsk.db.speech.Mediaitem;
import ipsk.util.LocalizableMessage;
import ipsk.util.services.Description;
import ipsk.util.services.Title;
import ipsk.util.services.Vendor;
import ipsk.util.services.Version;

/**
 * PromptTextViewer displays prompt text horizontally and vertically aligned.
 * The source of the text may be a String or a URL. For a String, the font
 * size is adjusted until the entire String fits into the panel. For a URL, 
 * the page is read in as is.
 *
 * @author Chr. Draxler
 * @version 1.0 July 2016
 */

@Title("Plain text prompter")
@Description("Presents plain text. Automatically inserts line feeds.")
@Vendor("Institute of Phonetics and Speech processing, Munich")
@Version(major=1)
public class PromptTextViewer extends BasicPromptPresenter implements PromptPresenter, ComponentListener {
   
	private static final long serialVersionUID = 1L;
	public static final PromptPresenterServiceDescriptor DESCRIPTOR = new BasicPromptPresenterServiceDescriptor(PromptTextViewer.class.getName(),new LocalizableMessage("Plain text prompter"), "Institute of Phonetics and Speech processing, Munich", new ipsk.text.Version(new int[]{1,0,0}), new LocalizableMessage("Presents plain text. Automatically inserts line feeds."),PromptPlainTextViewer.getSupportedMIMETypes());
	public final static int MINFONTSIZE = 12;
	public final static int MAXFONTSIZE = 48;
	public final static int FONTDOWNSTEP = 4;
	public final static int LEFTBORDER = 5;
	public final static int RIGHTBORDER = 5;
	
	public final static int LEFT = 0;
	public final static int RIGHT = 1;
	public final static int CENTER = 2;
	public final static int TOP = 3;
	public final static int BOTTOM = 4;
	public final static int MIDDLE = 5;
    
    public final static String DEF_CHARSET="UTF-8";

    public static enum Source {TEXT, URL};

	private Color textColor = Color.black;
//	private Color idleColor = Color.lightGray;
	private Color backgroundColor = Color.white;
	private Font font;
	private String fontFamily = Font.SANS_SERIF;
//	private int fontStyle = Font.BOLD;
	private int fontSize = MAXFONTSIZE;

	private JTextPane textPane;
	private StyledDocument doc;
	private StyleContext context;
	private Style style;
	private String [] promptText = { "" };
	private String [] promptTextStyles = { "regular" };
	private PromptTextViewer.Source source = Source.TEXT;
	private Border border;
	
	public PromptTextViewer() {
		super();
		setBackground(backgroundColor);
		setForeground(textColor);
		
        context = new StyleContext();
        style = context.getStyle(StyleContext.DEFAULT_STYLE);
        StyleConstants.setAlignment(style, StyleConstants.ALIGN_CENTER);
        StyleConstants.setFontFamily(style, fontFamily);
        StyleConstants.setFontSize(style, fontSize);
        StyleConstants.setSpaceAbove(style, 4);
        StyleConstants.setSpaceBelow(style, 4);
        
		textPane = new JTextPane();
		textPane.setEditable(false);
//		textPane.setMargin(new Insets(5, 2, 5, 2));
		
		doc = textPane.getStyledDocument();
		addStylesToDocument(doc);
//		doc.setLogicalStyle(0, style);

		source = Source.TEXT;
		
		setLayout(new GridLayout(1,1));
		add(textPane);
		
		addComponentListener(this);
	}
	
    protected void addStylesToDocument(StyledDocument doc) {
        //Initialize some styles.
        Style def = StyleContext.getDefaultStyleContext().
                        getStyle(StyleContext.DEFAULT_STYLE);

        Style regular = doc.addStyle("regular", def);
        StyleConstants.setFontFamily(def, Font.SANS_SERIF);

        Style s = doc.addStyle("italic", regular);
        StyleConstants.setItalic(s, true);

        s = doc.addStyle("bold", regular);
        StyleConstants.setBold(s, true);
    }

    public void setContents(Mediaitem[] mediaitems) throws PromptPresenterException{
        setContents(mediaitems[0].getText());
    }
    

	public void setPromptText(String promptText) {
		this.promptText = new String [1];
		this.promptText[0] = promptText;
		this.promptTextStyles = new String [1];
		this.promptTextStyles[0] = "regular";
	}


//	public String getPromptText () {
//		return promptText;
//	}


	public void setContents (String text) {
		source = Source.TEXT;
		setPromptText (text);
		fitTextToPane ();
	}


	public void setContents (URL url) {
		try {
			source = Source.URL;
			border = BorderFactory.createEmptyBorder();
			textPane.setBorder(border);
			textPane.setPage(url);
		} catch (IOException e) {
			System.err.println("setContents() " + e.getMessage());
			e.printStackTrace();
		}
	}
	
	
	/**
	 * fitTextToPane() reduces the font size until the text fits the screen area for the 
	 * prompt display.
	 * 
	 * TODO: improve computation of bounding box of document.
	 */
	private void fitTextToPane() {

		boolean fits = false;
		int width = this.getWidth();
		int height = this.getHeight();
		if(width<=0 || height<=0){
		    return;
		}
		// System.out.println("Textpane size: " + width + ", " + height);
		
		// --- counter is used to prevent non-terminating loops ---
		int counter = 0;
		
		while (!fits && counter < 20) {

			counter = counter + 1;
			
			// TODO: does this work with multiple fonts?
			
			font = doc.getFont(style);
			fontSize = font.getSize();
			FontMetrics fm = textPane.getFontMetrics(font);
			
			int stringWidth = 0;
			for (int i = 0; i < promptText.length; i++) {
				stringWidth = stringWidth + fm.stringWidth(promptText[i]);
			}
			int stringHeight = fm.getHeight();
			int lines = stringWidth / width + 2;
			// System.out.println("String: " + stringWidth + ", " +
			// stringHeight);

			int docHeight = lines * stringHeight;
			int vSpace = height - docHeight;
			int newY = vSpace / 2;
			border = BorderFactory.createEmptyBorder(newY, LEFTBORDER, newY, RIGHTBORDER);
			textPane.setBorder(border);

			if (lines * stringHeight > height) {
				// System.out.println("String does not fit: " + width + ", " +
				// height + " -> " + lines);
				fontSize = fontSize - FONTDOWNSTEP;
			} else if (newY > stringHeight) {
				// System.out.println("String does not fit: " + width + ", " +
				// height + " -> " + lines);
				fontSize = fontSize + FONTDOWNSTEP;
			} else {
				fits = true;
			}
			
			if (fontSize < MINFONTSIZE) {
				fits = true;
				fontSize =  MINFONTSIZE; 
			} else if (fontSize > MAXFONTSIZE) {
				fits = true;
				fontSize = MAXFONTSIZE;
			}
			
			StyleConstants.setFontSize(style, fontSize);
			doc.setLogicalStyle(0, style);
		}
		try {
			doc.remove(0, doc.getLength());
			for (int i = 0; i < promptText.length; i++) {
				doc.insertString(doc.getLength(), promptText[i], doc.getStyle(promptTextStyles[i]));
			}
		} catch (BadLocationException ble) {
			System.err.println("Couldn't insert initial text into text pane.");
		}
	}
	
	public Dimension getPreferredSize () {
		return (new Dimension(800, 400));
	}

	public Dimension getMinimumSize() {
		return (new Dimension(500, 250));
	}
	
	/* (non-Javadoc)
	 * @see ipsk.apps.speechrecorder.prompting.presenter.PromptPresenter#getServiceDescriptor()
	 */
	@Override
	public PromptPresenterServiceDescriptor getServiceDescriptor() {
		
		return DESCRIPTOR;
	}


	
	
	/* (non-Javadoc)
	 * @see ipsk.apps.speechrecorder.prompting.presenter.PromptPresenter#loadContents()
	 */
	@Override
	public void loadContents() throws PromptPresenterException {
		// TODO Auto-generated method stub
	}


	/* (non-Javadoc)
	 * @see java.awt.event.ComponentListener#componentResized(java.awt.event.ComponentEvent)
	 */
	@Override
	public void componentResized(ComponentEvent e) {
		PromptTextViewer ptv = (PromptTextViewer) e.getSource();
		if (ptv.source.equals(Source.TEXT)) {
			fitTextToPane();
		} else {
			this.textPane.setBorder(BorderFactory.createEmptyBorder());
		}
	}


	/* (non-Javadoc)
	 * @see java.awt.event.ComponentListener#componentMoved(java.awt.event.ComponentEvent)
	 */
	@Override
	public void componentMoved(ComponentEvent e) {}


	/* (non-Javadoc)
	 * @see java.awt.event.ComponentListener#componentShown(java.awt.event.ComponentEvent)
	 */
	@Override
	public void componentShown(ComponentEvent e) {}


	/* (non-Javadoc)
	 * @see java.awt.event.ComponentListener#componentHidden(java.awt.event.ComponentEvent)
	 */
	@Override
	public void componentHidden(ComponentEvent e) {}
	
	
    public static void main (String [] args) {
    	JFrame frame = new JFrame();
    	frame.setLocationRelativeTo(null);
    	frame.setDefaultCloseOperation(JFrame.EXIT_ON_CLOSE);
    	Dimension screenSize = Toolkit.getDefaultToolkit().getScreenSize();
    	
    	String promptText = "Und ich sagte noch: يوم جيد، كيف حالك؟, aber er antwortete nicht." + 
    			" Auch nicht auf Nachfragen. Und dann kam noch das: " + 
    			"\u178F\u17BE" +
    			" und danach noch: 良い一日、お元気ですか" +
    			" und sogar: " + "יום טוב, מה שלומך";
    	
    	PromptTextViewer ptv = new PromptTextViewer();
        ptv.setPromptText(promptText);
    	
    	frame.setContentPane(ptv);

//        PromptTextViewerTest ptvt = new PromptTextViewerTest(ptv);
//        frame.setContentPane(ptvt);
    	frame.pack();
    	
//        System.out.println("PromptTextViewer: " + ptv.getWidth() + ", " + ptv.getHeight());
    	
        frame.setVisible(true);
        Rectangle r = frame.getBounds();
        frame.setLocation((screenSize.width - r.width) / 2, (screenSize.height - r.height) / 2);
        
//        ptv.setContents(ptv.getPromptText());
        
        JFrame testViewer = new JFrame();
        testViewer.setDefaultCloseOperation(JFrame.EXIT_ON_CLOSE);
        PromptTextViewerTest ptvt = new PromptTextViewerTest(ptv);
        testViewer.setContentPane(ptvt);
        testViewer.pack();
        testViewer.setVisible(true);
    }
}


class PromptTextViewerTest extends JPanel implements ActionListener {
	public static final String GO = "Go!";
	private URL url;
	
	private JTextField textField = null;
	private PromptTextViewer ptv;
	
	public PromptTextViewerTest (PromptTextViewer ptv) {

		super();
		this.ptv = ptv;
		
		JLabel label = new JLabel("Text");
		textField = new JTextField(50);
		textField.setText("http://www.speechrecorder.org");
		JPanel inputPanel = new JPanel();
		JButton goButton = new JButton(GO);
		goButton.addActionListener(this);
		inputPanel.add(label);
		inputPanel.add(textField);
		inputPanel.add(goButton);
		inputPanel.setBackground(Color.lightGray);
		add(inputPanel);
	}

	/* (non-Javadoc)
	 * @see java.awt.event.ActionListener#actionPerformed(java.awt.event.ActionEvent)
	 */
	@Override
	public void actionPerformed(ActionEvent e) {
		if (e.getActionCommand().equals(PromptTextViewerTest.GO)) {
			if (textField.getText().startsWith("http://")) {
				try {
					url = new URL(textField.getText());
					ptv.setContents(url);
				} catch (MalformedURLException e1) {
					// TODO Auto-generated catch block
					e1.printStackTrace();
				}
			}
		}
	}
}
