//    Speechrecorder
// 	  (c) Copyright 2018
// 	  Institute of Phonetics and Speech Processing,
//    Ludwig-Maximilians-University, Munich, Germany
//
//
//    This file is part of Speechrecorder
//
//
//    Speechrecorder is free software: you can redistribute it and/or modify
//    it under the terms of the GNU Lesser General Public License as published by
//    the Free Software Foundation, version 3 of the License.
//
//    Speechrecorder is distributed in the hope that it will be useful,
//    but WITHOUT ANY WARRANTY; without even the implied warranty of
//    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
//    GNU Lesser General Public License for more details.
//
//    You should have received a copy of the GNU Lesser General Public License
//    along with Speechrecorder.  If not, see <http://www.gnu.org/licenses/>.



package ipsk.apps.speechrecorder.workspace.ui;

import java.awt.Component;
import java.awt.Desktop;
import java.io.ByteArrayOutputStream;
import java.io.File;
import java.io.FileOutputStream;
import java.io.IOException;
import java.net.URI;
import java.net.URISyntaxException;
import java.net.URL;
import java.util.ArrayList;

import javax.swing.JOptionPane;
import javax.xml.bind.JAXB;

import ipsk.apps.speechrecorder.SpeechRecorderException;
import ipsk.apps.speechrecorder.config.ProjectConfiguration;
import ipsk.apps.speechrecorder.project.ProjectManager;
import ipsk.apps.speechrecorder.script.RecScriptManager;
import ipsk.apps.speechrecorder.script.RecscriptManagerException;
import ipsk.apps.speechrecorder.script.ui.MediaitemUI;
import ipsk.awt.WorkerException;
import ipsk.net.URLContext;
import ipsk.net.Utils;
import ipsk.swing.JProgressDialogPanel;
import ipsk.util.LocalizableMessage;
import ipsk.util.ProgressStatus;
import ipsk.util.zip.ZipPacker;
import ipsk.util.zip.ZipPackerWorker;

/**
 * @author klausj
 *
 */
public class ProjectExporter {
    public void exportProject(URL projectContext,ProjectConfiguration project,File zipFile,boolean exportSessionData,Component speechRecorderUI) throws IOException, URISyntaxException, SpeechRecorderException {

        // first search project main dir


        File projDir = new File(projectContext.toURI().getPath());

        FileOutputStream zipStream = new FileOutputStream(zipFile);
        ProgressStatus finalStatus=null;
        if(exportSessionData){
            // use worker for complete project directory including recordings
            ZipPackerWorker zipPackerWorker=new ZipPackerWorker();
            zipPackerWorker.setSrcDir(projDir);
            zipPackerWorker.setPackRecusive(true);
            zipPackerWorker.setOutputStream(zipStream);
            JProgressDialogPanel progressDialog=new JProgressDialogPanel(zipPackerWorker,"Export project","Exporting...");
            try {
                zipPackerWorker.open();
            } catch (WorkerException e) {
                e.printStackTrace();
                throw new SpeechRecorderException("Could not open ZIP file packer!", e);
            }
            zipPackerWorker.start();

            Object val=progressDialog.showDialog(speechRecorderUI);

            try {
                zipPackerWorker.close();
            } catch (WorkerException e) {
                e.printStackTrace();
                throw new SpeechRecorderException("Could not close ZIP file packer!", e);
            }
            finalStatus=zipPackerWorker.getProgressStatus();
//          if(!val.equals(JProgressDialogPanel.OK_OPTION)){
//              finalStatus.canceled();
//          }
            if(!finalStatus.isDone()){
                // delete (partial) file
                zipFile.delete();
            }
        }else{
            // pack synchronous (project configuration and script(s) only)
            finalStatus=new ProgressStatus();
            try{
                ZipPacker zp=new ZipPacker(zipStream);

                String dirZipEntryName=zp.packDir(projDir.getName());

                File projectConfFile=new File(projDir,project.getName()+ProjectManager.PROJECT_FILE_EXTENSION);
                
                
                String scriptUrlStr=project.getPromptConfiguration().getPromptsUrl();
                if (scriptUrlStr != null && !"".equals(scriptUrlStr)) {
                    URI promptURI = new URI(scriptUrlStr);
                    if (promptURI.isAbsolute()){
                        JOptionPane.showMessageDialog(speechRecorderUI, "Cannot export project with absolute recording script URL!");
                        return;
                    }
                    URL scriptURL=URLContext.getContextURL(projectContext,scriptUrlStr);
                    File scriptFile=Utils.fileFromDecodedURL(scriptURL);
                    zp.packFile(scriptFile, dirZipEntryName+scriptUrlStr);
                    
                    // get the DTD 
                    // create new instance of rec script manager 
                    RecScriptManager scriptExportManager=new RecScriptManager();
                    scriptExportManager.setContext(projectContext);
                    scriptExportManager.setSystemIdBase(projectContext.toExternalForm());
                    try {
                        scriptExportManager.load(scriptURL);
                        String sysId=scriptExportManager.getSystemId();
                        String sysIDBase=scriptExportManager.getSystemIdBase();
                        
                        URL dtdURL=URLContext.getContextURL(projectContext,sysIDBase+sysId);
                        File dtdFile=Utils.fileFromDecodedURL(dtdURL);
                        // TODO
                        // zip entry name maybe wrong if DTD is in a subdirectory
                        zp.packFile(dtdFile, dirZipEntryName+sysId);
                    } catch (RecscriptManagerException e) {
                        e.printStackTrace();
                        JOptionPane.showMessageDialog(speechRecorderUI, "Cannot load recording script for export!");
                        return;
                        
                    }
                    
                }
                
                File resourcesDir=new File(projDir,MediaitemUI.RESOURCE_PATH);
                if(resourcesDir.isDirectory()) {
                	zp.packDirRecursive(projDir.getName(),resourcesDir);
                }
                
                String spkUrlStr=project.getSpeakers().getSpeakersUrl();
                if (spkUrlStr != null && !"".equals(spkUrlStr)) {
                    URI spkURI = new URI(spkUrlStr);
                    if (spkURI.isAbsolute()){
                        JOptionPane.showMessageDialog(speechRecorderUI, "Cannot export project with absolute speaker database URL!");
                        return;
                    }
                 
                    ByteArrayOutputStream emptySpkFileOs=new ByteArrayOutputStream();
                    // Do not export session data: pack an _empty_ speakers database here
                    JAXB.marshal(new ArrayList<ipsk.db.speech.Speaker>(), emptySpkFileOs);
                    byte[] spkData=emptySpkFileOs.toByteArray();
                    zp.packData(spkData, dirZipEntryName+spkUrlStr);
                }
                
                
                zp.packFile(projectConfFile,dirZipEntryName+projectConfFile.getName());
                zp.close();
                finalStatus.done();
            }catch(IOException ioe){
                finalStatus.error(new LocalizableMessage(ioe.getLocalizedMessage()));
            }

        }

        if(finalStatus.isDone()){
            // success
            Object[] options=new Object[]{"OK","Open containing folder"};
            int selOpt=JOptionPane.showOptionDialog(speechRecorderUI, "Successful export to ZIP file:\n"+zipFile, "Project export",JOptionPane.DEFAULT_OPTION, JOptionPane.INFORMATION_MESSAGE, null,options,options[0]);
            if(selOpt==1){
                if(Desktop.isDesktopSupported()){
                    Desktop dt=Desktop.getDesktop();
                    File containingFolder=zipFile.getParentFile();
                    dt.open(containingFolder);
                }

            }
        }else if(finalStatus.isError()){
            JOptionPane.showMessageDialog(speechRecorderUI, "Project export failed:"+finalStatus.getMessage().localize());

        }else if(finalStatus.isCanceled()){
            JOptionPane.showMessageDialog(speechRecorderUI, "Project export canceled.");
        }

    }
}
