//    Speechrecorder
//    (c) Copyright 2012
// 	  Institute of Phonetics and Speech Processing,
//    Ludwig-Maximilians-University, Munich, Germany
//
//
//    This file is part of Speechrecorder
//
//
//    Speechrecorder is free software: you can redistribute it and/or modify
//    it under the terms of the GNU Lesser General Public License as published by
//    the Free Software Foundation, version 3 of the License.
//
//    Speechrecorder is distributed in the hope that it will be useful,
//    but WITHOUT ANY WARRANTY; without even the implied warranty of
//    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
//    GNU Lesser General Public License for more details.
//
//    You should have received a copy of the GNU Lesser General Public License
//    along with Speechrecorder.  If not, see <http://www.gnu.org/licenses/>.


package ipsk.apps.speechrecorder.prompting;

import ipsk.apps.speechrecorder.MIMETypes;
import ipsk.apps.speechrecorder.prompting.SprCompatTextViewer.TextFragment.FontStyle;
import ipsk.apps.speechrecorder.prompting.presenter.PromptPresenter;
import ipsk.apps.speechrecorder.prompting.presenter.PromptPresenterException;
import ipsk.apps.speechrecorder.prompting.presenter.UnsupportedContentException;
import ipsk.db.speech.Mediaitem;
import ipsk.swing.JRuler;
import ipsk.util.LocalizableMessage;
import ipsk.util.services.Description;
import ipsk.util.services.Title;
import ipsk.util.services.Vendor;
import ipsk.util.services.Version;

import java.awt.Color;
import java.awt.Dimension;
import java.awt.Font;
import java.awt.FontMetrics;
import java.awt.Graphics;
import java.awt.Graphics2D;
import java.awt.RenderingHints;
import java.awt.font.FontRenderContext;
import java.awt.font.LineMetrics;
import java.io.IOException;
import java.io.InputStreamReader;
import java.io.StringWriter;
import java.net.URL;
import java.nio.charset.Charset;
import java.util.ArrayList;
import java.util.List;

import javax.swing.BorderFactory;
import javax.swing.JFrame;
import javax.swing.SwingUtilities;

/**
 * Displays ProjectManager script formatted prompt text. 
 * 
 * Text alignment can be set horizontally or vertically.
 * Font family, style, and size are set from the configuration file.
 * NOTE: if a word is longer than the width of the window the display does not work
 *
 * @author Chr. Draxler, K.Jaensch
 * @version 1.0 November 2002, March 2017
 */

@Title("Compatible formatted text prompter")
@Description("Presents Speechrecorder fromatted text. Automatically inserts line feeds. Compatible to existing plain text prompts.")
@Vendor("Institute of Phonetics and Speech processing, Munich")
@Version(major=1)
public class SprCompatTextViewer extends BasicPromptPresenter implements PromptPresenter {
   
	private static final long serialVersionUID = 1L;
	public static final PromptPresenterServiceDescriptor DESCRIPTOR=new BasicPromptPresenterServiceDescriptor(SprCompatTextViewer.class.getName(),new LocalizableMessage("Plain text prompter"), "Institute of Phonetics and Speech processing, Munich", new ipsk.text.Version(new int[]{1,0,0}), new LocalizableMessage("Presents plain text. Automatically inserts line feeds."),SprCompatTextViewer.getSupportedMIMETypes());
	public final static int MINFONTSIZE = 12;
	public final static int MAXFONTSIZE = 48;
//	public final static int MAXFONTSIZE = 160;
	public final static int FONTDOWNSTEP = 4;
	
	
	public final static int LEFT = 0;
	public final static int RIGHT = 1;
	public final static int CENTER = 2;
	public final static int TOP = 3;
	public final static int BOTTOM = 4;
	public final static int MIDDLE = 5;
    
    public final static String DEF_CHARSET="UTF-8";
    
    public class PromptDoc{
        private List<Line> lines=new ArrayList<>();

        public List<Line> getLines() {
            return lines;
        }

        public PromptDoc(List<Line> lines) {
            super();
            this.lines = lines;
        }
        
        public int wordCount(){
            int words=0;
            for(Line line:lines){
                words+=line.getWords().size();
            }
            return words;
        }
        
      
    }
    
    public class Line{
        private List<Word> words=new ArrayList<>();

        public List<Word> getWords() {
            return words;
        }

        public Line(List<Word> words) {
            super();
            this.words = words;
        }
    }
    
    public class Word{
        private List<TextFragment> textFragments=new ArrayList<>();

        public List<TextFragment> getTextFragments() {
            return textFragments;
        }

        public Word(List<TextFragment> textFragments) {
            super();
            this.textFragments = textFragments;
        }
        
        public Word(TextFragment textFragment) {
            super();
            this.textFragments.add(textFragment);
        }


        /**
         * @param newFontSize
         */
        public void setFontSize(int newFontSize) {
            for(TextFragment tfr:textFragments){
                tfr.setFontSize((float)newFontSize);
            }
        }
        
    }
    
    public static class TextFragment{
    	
    	public enum FontStyle {NORMAL,ITALIC,OBLIQUE};
    	public static class FontWeight{
    		public static final int NORMAL=400;
    		public static final int BOLD=700;
    		private int weight;
			public int getWeight() {
				return weight;
			}
			public FontWeight(int weight) {
				super();
				this.weight = weight;
			}
			public boolean isNormal(){
				return this.weight==NORMAL;
			}
			public boolean isBold(){
				return this.weight==BOLD;
			}
    	}
        private String text;
       
        private String fontFamily = null;
        private FontStyle fontStyle=null;
        private Color color=null;
        private Boolean underLine=null;
        public Boolean getUnderLine() {
			return underLine;
		}
		public void setUnderLine(Boolean underLine) {
			this.underLine = underLine;
		}
		public Color getColor() {
			return color;
		}
		public void setColor(Color color) {
			this.color = color;
		}
		public FontStyle getFontStyle() {
			return fontStyle;
		}
		public void setFontStyle(FontStyle fontStyle) {
			this.fontStyle = fontStyle;
		}
		
		private FontWeight fontWeight=null;
        public FontWeight getFontWeight() {
			return fontWeight;
		}
		public void setFontWeight(FontWeight fontWeight) {
			this.fontWeight = fontWeight;
		}

		private Float fontSize=null;
        public Float getFontSize() {
            return fontSize;
        }
        public void setFontSize(Float fontSize) {
            this.fontSize = fontSize;
        }
        public String getText() {
            return text;
        }
        public void setText(String text) {
            this.text = text;
        }
        public String getFontFamily() {
            return fontFamily;
        }
        public void setFontFamily(String fontFamily) {
            this.fontFamily = fontFamily;
        }
      
        public TextFragment(String textFragmentString){
            super();
            this.text=textFragmentString;
        }
        
        public boolean isDefaultFont(){
            return (fontFamily==null && fontStyle==null && fontWeight==null && fontSize==null); 
        }
        
    }
    
  

//	private String promptText = "";
	private PromptDoc promptText;
	private Color textColor = Color.BLACK;
//	private Color idleColor = Color.lightGray;
	private Color backgroundColor = Color.WHITE;
	private int hAlign;
	private int vAlign;
	private static String DEFAULT_FONT_NAME = "sans-serif";
	private static int DEFAULT_FONT_STYLE=Font.BOLD;
	private int fontStyle;
	private int DEFAULT_FONT_SIZE = 48;
//	private int DEFAULT_FONT_SIZE = 160;
	private int fontSize;
	
	
	public SprCompatTextViewer() {
		super();
		//fontFamily = System.getProperty("prompt.font_family");
		//fontStyleName = System.getProperty("prompt.font_style").toLowerCase();
		//fontSize = Integer.parseInt(System.getProperty("prompt.font_size"));
		setBackground(backgroundColor);
		fontSize=DEFAULT_FONT_SIZE;
		if (fontSize < MINFONTSIZE) {
			fontSize = MINFONTSIZE;
		} else if (fontSize > MAXFONTSIZE) {
			fontSize = MAXFONTSIZE;
		}
		promptFont = new Font(DEFAULT_FONT_NAME, DEFAULT_FONT_STYLE,fontSize);
		setBorder(BorderFactory.createEmptyBorder(2, 2, 2, 2));
		hAlign = SprCompatTextViewer.CENTER;
		vAlign = SprCompatTextViewer.MIDDLE;
	}
	 
	
	private PromptDoc promptDocFromString(String str) {
	    String[] pWs = str.split("\\s");
	    List<Word> ws=new ArrayList<>();
	    for(String pW:pWs){
	        List<TextFragment> frgs=new ArrayList<>();
	        TextFragment tfr=new TextFragment(pW);
//	        tfr.setFontFamily(fontFamily);
//	        tfr.setFontStyleName(fontStyleName);
//	        tfr.setFontSize(fontSize);
	        frgs.add(tfr);

	        Word w=new Word(frgs);
	        ws.add(w);
	    }

	    Line l=new Line(ws);
	    List<Line> lines=new ArrayList<>();
	    lines.add(l);
	    return new PromptDoc(lines);
	}
	

	
	private Font fontByTextFragment(TextFragment textFragment){
	    Font f=promptFont;
	    if(!textFragment.isDefaultFont()){
	        
	        int defStyle=promptFont.getStyle();
	        int defSize=promptFont.getSize();
	        
	        int style=defStyle;
	        String frFamily=textFragment.getFontFamily();
	        if(frFamily!=null){
	        	f=new Font(frFamily,defStyle,defSize);
	        }
	       TextFragment.FontStyle frStyle=textFragment.getFontStyle();
	       if(frStyle!=null){
	    	   if(TextFragment.FontStyle.NORMAL.equals(frStyle)){
	    		   // filter italic style bit
	    		   style=style & ~Font.ITALIC;
	    	   }else if(TextFragment.FontStyle.ITALIC.equals(frStyle)){
	    		   style=style | Font.ITALIC;
	    	   }
	       }
	       TextFragment.FontWeight frWeight=textFragment.getFontWeight();
	       if(frWeight!=null){
	    	   if(frWeight.isNormal()){
	    		// filter bold style bit
	    		   style=style & ~Font.BOLD;
	    	   }else if(frWeight.isBold()){
	    		   style=style | Font.BOLD;
	    	   }
	       }
	       if(style!=defStyle){
	    	   f=f.deriveFont(style);
	       }
	       Float frSize=textFragment.getFontSize();
	       if(frSize!=null){
	    	   f=f.deriveFont(frSize);
	       }
	         
	    }
	   return f;
    }
	
	private int fragmentWidth(TextFragment textFrag){
		Font tFrFt=fontByTextFragment(textFrag);
		FontMetrics fM=getFontMetrics(tFrFt);
		return fM.stringWidth(textFrag.getText());
	}
	
	private int wordWidth(Word word){
        List<TextFragment> textFrags=word.getTextFragments();
        int wordLen=0;
        for(TextFragment textFrag:textFrags){
            wordLen+=fragmentWidth(textFrag);
        }
        return wordLen;
    }
	
	public void setAlignment(int h, int v) {
		hAlign = h;
		vAlign = v;
	}

	public void prepareGraphics(Graphics g) {
	    if (g instanceof Graphics2D){
//	        ((Graphics2D)g).setRenderingHint(
//	                RenderingHints.KEY_TEXT_ANTIALIASING,
//	                RenderingHints.VALUE_TEXT_ANTIALIAS_ON);
	        ((Graphics2D)g).setRenderingHint(
                  RenderingHints.KEY_TEXT_ANTIALIASING,
                  RenderingHints.VALUE_TEXT_ANTIALIAS_GASP);
	        
	    }
		int xMargin = 2;
		TextGraphicsItem[] itemList = null;

		int width = getSize().width;
		int height = getSize().height;
		//System.out.println("prepareGraphics(" + width + ", " + height + ")");
		
		int[] lineLengths;

//		String[] pW = promptText.split("\\s");
//		String[] pStrWords = promptText.split("\\s");
		List<Line> lines=promptText.getLines();
		
		int wordCount=promptText.wordCount();
		// at most each word can be displayed in a line on its own
		lineLengths = new int[wordCount];		

		if (wordCount>0) {

		    boolean textSizeOK = false;
		    int newFontSize = fontSize;
		    int lineCount = 0;
		    // decrease font size until text fits to pane 
		    for (int fs = fontSize; ((fs >= MINFONTSIZE) && (!textSizeOK)); fs = fs - FONTDOWNSTEP) {
		        int xMax = 0;
		        int yMax = 0;
		        lineCount = 0;

		        newFontSize = fs;

		        // default font
//		        promptFont = new Font(fontFamily, fontStyle, newFontSize);
		        promptFont=promptFont.deriveFont((float)newFontSize);
		        g.setFont(promptFont);
		        FontMetrics fM = getFontMetrics(promptFont);

		        int lineSpacing = (int) (fs / 10) + 1;
		        int yStep = fs + lineSpacing;

		        //System.out.println("reduce font size: " + fs + ", " + textSizeOK + ", " + yStep + ", " + lineSpacing);
		        itemList = new TextGraphicsItem[wordCount];

		        int x = xMargin;
		        int y = yStep;

		        int lineLength = xMargin;

		        int itemIdx=0;
		        for (int docLine=0;docLine<lines.size();docLine++) {
		            Line line=lines.get(docLine);
		            List<Word> pWList=line.getWords();
		            Word[] pW=pWList.toArray(new Word[pWList.size()]);

		            for (int i = 0; i < pW.length; i++) {

		                // step to the next line, if the string item does not fit
		                // does not apply to the first item to avoid an additional 
		                // empty line
		                pW[i].setFontSize(newFontSize);
		                int spacedWordLen=wordWidth(pW[i])+fM.stringWidth(" ");
		                if (i>0 && x + spacedWordLen > width - 4) {
		                    lineLengths[lineCount] = lineLength;
		                    lineCount++;
		                    y = y + yStep;
		                    x = xMargin;
		                }
		                itemList[itemIdx++] = new TextGraphicsItem(x, y, lineCount, pW[i]);

		                lineLength = x + wordWidth(pW[i]);
		                x = x + spacedWordLen;
		                if (x > xMax)
		                    xMax = x;
		                if (y > yMax)
		                    yMax = y;
		            }
		            
		            if(docLine<lines.size()-1){  
		                y = y + yStep;
		                if (y > yMax)
		                    yMax = y;
		                x = xMargin;
		            }
		            lineLengths[lineCount] = lineLength;
	                lineCount++;
		        }


		        //System.out.println("yMax, height: " + yMax + ", " + height);

		        if (yMax <= height) {
		            textSizeOK = true;
		            int tmpHAlign = hAlign;
		            if (newFontSize < fontSize) {
		                tmpHAlign = SprCompatTextViewer.LEFT;
		            }

		            lineLengths[lineCount] = lineLength;

		            for (int i = 0; i < itemList.length; i++) {
		                if (tmpHAlign == SprCompatTextViewer.LEFT) {
		                    itemList[i].xPos = itemList[i].xPos;
		                } else if (tmpHAlign == SprCompatTextViewer.RIGHT) {
		                    itemList[i].xPos =
		                            width
		                            - xMargin
		                            - lineLengths[itemList[i].line]
		                                    + itemList[i].xPos;
		                } else if (tmpHAlign == SprCompatTextViewer.CENTER) {
		                    itemList[i].xPos =
		                            ((int) ((width - lineLengths[itemList[i].line])
		                                    / 2))
		                            + itemList[i].xPos;
		                }
		                if (vAlign == SprCompatTextViewer.TOP) {
		                    itemList[i].yPos = itemList[i].yPos;
		                } else if (vAlign == SprCompatTextViewer.BOTTOM) {
		                    itemList[i].yPos =
		                            height
		                            - lineSpacing
		                            - (yMax + lineSpacing)
		                            + itemList[i].yPos;
		                } else if (vAlign == SprCompatTextViewer.MIDDLE) {
		                    itemList[i].yPos =
		                            ((int) ((height - yMax) / 2))
		                            + itemList[i].yPos;
		                }
		            }

		        }
		    }

		    Graphics2D g2d=(Graphics2D)g;
		    for (TextGraphicsItem tgi:itemList) {
		    	
		        paintTextGraphicsItem(g,tgi);
		    }
		    // paint text decorations
		    int tgiIdx=0;
		    for(int l=0;l<lineCount;l++){
		    	Integer underLineStart=null;
		    	Integer underLineY=null;
		    	String underLineBeginStr=null;
		    	Font underLineStartFont=null;
		    	Integer underLineStop=null;
		    	int tgLine;
		    	int x=xMargin;
		    	do{
		    		TextGraphicsItem tgi=itemList[tgiIdx];
		    		tgLine=tgi.getLine();
		    		if(tgLine<=l){
		    			x=tgi.getX();
		    			Word w=tgi.getWord();
		    			List<TextFragment> tFrs=w.getTextFragments();
		    			for(TextFragment tFr:tFrs){
		    				int frWidth=fragmentWidth(tFr);
		    			
		    				Boolean ul=tFr.getUnderLine();
		    				if(Boolean.TRUE.equals(ul)){
		    					if(underLineStart==null){
		    						underLineStart=x;
		    						underLineY=tgi.getY();
		    						underLineStartFont=fontByTextFragment(tFr);
		    						underLineBeginStr=tFr.getText();
		    					}

		    				}else{
		    					if(underLineStart!=null && underLineStop==null){
		    						underLineStop=x+frWidth;
		    						paintUnderLine(g2d, underLineStartFont, underLineBeginStr, underLineStart, underLineY,underLineStop-underLineStart);
		    						underLineStart=null;
		    				    	underLineY=null;
		    				    	underLineBeginStr=null;
		    				    	underLineStartFont=null;
		    				    	underLineStop=null;
		    					}
		    				}
		    				x+=frWidth;
		    			}
		    			tgiIdx++;
		    			if(tgiIdx==itemList.length){
		    				break;
		    			}
		    		}
		    		
		    	}while(tgLine<=l);
		    	if(underLineStart!=null && underLineStop==null){
					underLineStop=x;
		    		// use first text fragments font for text decorations
		    		paintUnderLine(g2d, underLineStartFont, underLineBeginStr, underLineStart, underLineY,underLineStop-underLineStart);
		    	}
		    	
		    }
		}
	}
	
	private void paintUnderLine(Graphics2D g2d,Font f,String str,int x,int y,int width){
		LineMetrics lm=f.getLineMetrics(str,g2d.getFontRenderContext());
		float ulOffset=lm.getUnderlineOffset();
		float ulThickness=lm.getUnderlineThickness();
		g2d.setColor(textColor);
		g2d.fillRect(x, y+(int)ulOffset, width, (int)ulThickness);
	}
	
	private void paintTextGraphicsItem(Graphics g,TextGraphicsItem textGraphicsitem){
	    paintWord(g,textGraphicsitem.getWord(),textGraphicsitem.getX(),textGraphicsitem.getY());
	}
	private void paintWord(Graphics g,Word w,int wx,int wy){
	    List<TextFragment> tFrs=w.getTextFragments();
	    Font wf=null;
	    int x=wx;
	    String word="";
	    for(TextFragment tFr:tFrs){
	        int frWidth=fragmentWidth(tFr);
	        Font f=fontByTextFragment(tFr);
	        g.setFont(f);
	        if(wf==null){
	        	wf=f;
	        }
	        Color tColor=textColor;
	        Color frColor=tFr.getColor();
	        if(frColor!=null){
	        	tColor=frColor;
	        }
	        g.setColor(tColor);
	        String t=tFr.getText();
	        word=word.concat(t);
	        g.drawString(t, x, wy);
	        x+=frWidth;	       
	    }
	    
	}
//	
//	private void paintWord(Graphics g,Word w){
//        
//    }

	public void paintComponent(Graphics g) {
		//System.out.println("paint PromptPlainTextViewer: " + promptText);
//		g.setColor(backgroundColor);
//		g.fillRect(1, 1, getSize().width - 2, getSize().height - 2);
	    super.paintComponent(g);
		g.setColor(textColor);
        try{
            prepareGraphics(g);
        }catch(ArrayIndexOutOfBoundsException arrException){
            // TODO the cause should be fixed !
            arrException.printStackTrace();
        }
	}
	

	public void showContents() {};
	
	public void hideContents() {};
	
//	public void loadContents(URL url) {};
	
	
	protected void setContents(PromptDoc promptDoc){
	    promptText=promptDoc;
	}

	/**
	* sets the prompt text
	* @param s text
	*/
	public void setContents(String s) {
	    promptText=promptDocFromString(s);
	}

	/**
	* sets the prompt text
	* @param s text
	* @param d prompt text description
	*/
	public void setContents(String s, String d) {
		setContents(s.replaceAll("\\s{2,}"," "));
	}
    
    /**
    * sets the prompt text
    * @param s text
    * @param d prompt text description
    * @param type type
    */
    public void setContents(String s, String d,String type) {
        setContents(s, d);
    }
	
	/**
	* sets the prompt text
	* @param u URL where the text is stored
	*/
	public void setContents(URL u) {
	    setContents(u, null);
    }

	/**
	* sets the prompt text
	* @param u URL where the text is stored
	* @param d prompt text description
	*/
	public void setContents(URL u, String d) {
        setContents(u, null, null);
    }
    public void setContents(URL u, String d, String type) {
        setContents(u, null, null,null);
    }
	/**
	* sets the prompt text
	* @param u URL where the text is stored
	* @param d prompt text description
	* @param type type
	* @param cs charset
	*/
	public void setContents(URL u, String d, String type,String cs) {
	    try {
            InputStreamReader r=null;
            if(cs!=null){
                r=new InputStreamReader(u.openStream(),cs);
            }else{
                r=new InputStreamReader(u.openStream());
            }
            StringWriter sw=new StringWriter();
            char[] cbuf=new char[2048];
            int read=0;
            while((read=r.read(cbuf))>=0){
                sw.write(cbuf, 0, read);
            }
            r.close();
            sw.close();
            setContents(sw.getBuffer().toString(),d);
        } catch (IOException e) {
            // TODO Auto-generated catch block
            e.printStackTrace();
        }
        
    }


	/**
	 * Sets font for prompts.	 
	 * @param promptFont
	 */
	public void setPromptFont(Font promptFont) {
		this.promptFont = promptFont;
		fontSize = promptFont.getSize();
		if (fontSize < MINFONTSIZE) {
			fontSize = MINFONTSIZE;
		} 
//		if (fontSize > MAXFONTSIZE) {
//			fontSize = MAXFONTSIZE;
//		}
//		promptFont = promptFont.deriveFont(fontSize);
		fontStyle = promptFont.getStyle();
//		fontFamily = promptFont.getFamily();
		//System.out.println("PromptPlainTextViewer promptFont: "+promptFont);
		//System.out.println("PromptPlainTextViewer promptFont: "+fontFamily+" "+fontSize+" "+fontStyle);
	}
	
	

	/**
	* auxiliary class for drawing text items into a JPanel. A TextGraphicsItem
	* contains the x,y coordinates of a string, the line number, and the string itself.
	*/

	class TextGraphicsItem {
		protected int xPos;
		protected int yPos;
		protected int line;
		public int getLine() {
			return line;
		}

		protected Word word;

		public Word getWord() {
            return word;
        }

        protected TextGraphicsItem(int x, int y, int l, Word s) {
			xPos = x;
			yPos = y;
			line = l;
			word = s;
			//System.out.println("tgi - " + s + ": " + l + ", " + x + ", " + y);
		}

		public int getX() {
			return xPos;
		}

		public int getY() {
			return yPos;
		}

	
	}



    public boolean getEmphasized() {
        
        return emphasized;
    }

    public void setEmphasized(boolean emphasized) {
        this.emphasized=emphasized;
        if(emphasized){
            setForeground(Color.black);
        }else{
           setForeground(Color.lightGray);
        }
    }
    
    public static String[][] getSupportedMIMETypes() {
        return getSupportedMIMETypes(MIMETypes.PLAINTEXTMIMETYPES);
    }

   
    public void loadContents()
            throws PromptPresenterException {
        if(mediaitems==null || mediaitems.length==0){
            throw new UnsupportedContentException("No media item to display!");
        }else if(mediaitems.length > 1){
            throw new UnsupportedContentException("Multiple media items not supported!");
        }
        String text=null;
        Mediaitem mi=mediaitems[0];
        URL u=applyContextToMediaitemURL(mi);
        if(u!=null){
            // external URL 
            Charset cs=null;
            String miCharset=mi.getCharSet();
            if(miCharset!=null){
                cs=Charset.forName(miCharset.trim());
            }
            InputStreamReader r=null;
            try {
               
                if(cs!=null){
                    r=new InputStreamReader(u.openStream(),cs);
                }else{
                    r=new InputStreamReader(u.openStream());
                }
                StringWriter sw=new StringWriter();
                char[] cbuf=new char[2048];
                int read=0;
                while((read=r.read(cbuf))>=0){
                    sw.write(cbuf, 0, read);
                }
               
                sw.close();
                //setContents(sw.getBuffer().toString(),d);
                text=sw.getBuffer().toString();
            } catch (IOException e) {
              throw new PromptPresenterException(e);
            }finally{
                try {
                    r.close();
                } catch (IOException e) {
                    throw new PromptPresenterException(e);
                }
            }
        }else{
            // string
            text=mi.getText();
        }
        promptText = promptDocFromString(text.replaceAll("\\s{2,}"," "));
    }

    /* (non-Javadoc)
     * @see ipsk.apps.speechrecorder.prompting.presenter.PromptPresenter#getServiceDescriptor()
     */
    public PromptPresenterServiceDescriptor getServiceDescriptor() {
       return DESCRIPTOR;
    }
    
    
    
    protected void testPrompt(){
        List<Line> p1=new ArrayList<>();
        
       
        List<Word> wsl1=new ArrayList<>();
        
        List<TextFragment> tn1s=new ArrayList<>();
        TextFragment tn11=new TextFragment("Wel");
        tn11.setColor(Color.GREEN);
        tn11.setUnderLine(true);
        tn1s.add(tn11);
        TextFragment tn12=new TextFragment("come");
//        tn12.setFontFamily("serif");
        tn12.setFontStyle(FontStyle.ITALIC);
        tn12.setColor(Color.RED);
        tn12.setUnderLine(true);
        tn1s.add(tn12);
        Word w1=new Word(tn1s);
        wsl1.add(w1);
        
        TextFragment tn2=new TextFragment("to");
//        tn1.setFontFamily("serif");
//        tn1.setFontStyleName("italic");
        tn2.setUnderLine(true);
        Word w2=new Word(tn2);
        wsl1.add(w2);
        
        TextFragment tn3=new TextFragment("the");
//        tn3.setFontFamily("serif");
        tn3.setUnderLine(true);
        Word w3=new Word(tn3);
        wsl1.add(w3);
        
        Line l1=new Line(wsl1);
        p1.add(l1);
        
        List<Word> wsl2=new ArrayList<>();
        TextFragment tn4=new TextFragment("compatible");
//        tn4.setFontFamily("serif");
//        tn4.setFontStyleName("italic");
        tn4.setUnderLine(true);
        Word w4=new Word(tn4);
        wsl2.add(w4);
        
        TextFragment tn5=new TextFragment("formatted");
//        tn5.setFontFamily("serif");
//        tn5.setFontStyleName("italic");
        tn5.setUnderLine(true);
        Word w5=new Word(tn5);
        wsl2.add(w5);
        
        Line l2=new Line(wsl2);
        p1.add(l2);
        
        List<Word> wsl3=new ArrayList<>();
        TextFragment tn6=new TextFragment("text");
//        tn6.setFontFamily("serif");
//        tn6.set
        Word w6=new Word(tn6);
        wsl3.add(w6);
        
        TextFragment tn7=new TextFragment("viewer");
//        tn7.setFontFamily("serif");
//        tn7.setFontStyleName("italic");
        Word w7=new Word(tn7);
        wsl3.add(w7);
        
        TextFragment tn8=new TextFragment("!!!!!!!!!!!!!!!!!!!!!!!");
//        tn8.setFontFamily("serif");
//        tn8.setFontStyleName("italic");
        tn8.setUnderLine(true);
        Word w8=new Word(tn8);
        wsl3.add(w8);
        
        Line l3=new Line(wsl3);
        p1.add(l3);
        
        PromptDoc pd=new PromptDoc(p1);
        setContents(pd);
    }
  
    public static void main(String[] args){
        
        Runnable r=new Runnable(){
            public void run(){
                SprCompatTextViewer vw=new SprCompatTextViewer();
                vw.setPreferredSize(new Dimension(600,400));
                JFrame w=new JFrame();
                w.setDefaultCloseOperation(JFrame.EXIT_ON_CLOSE);
                w.getContentPane().add(vw);
                w.pack();
                w.setVisible(true);
                
                
//                vw.setContents("Welcome to the plain text prompt viewer!");
                vw.testPrompt();
            }
        };
        SwingUtilities.invokeLater(r);
    }
	
}