//    Speechrecorder
//    (c) Copyright 2012
// 	  Institute of Phonetics and Speech Processing,
//    Ludwig-Maximilians-University, Munich, Germany
//
//
//    This file is part of Speechrecorder
//
//
//    Speechrecorder is free software: you can redistribute it and/or modify
//    it under the terms of the GNU Lesser General Public License as published by
//    the Free Software Foundation, version 3 of the License.
//
//    Speechrecorder is distributed in the hope that it will be useful,
//    but WITHOUT ANY WARRANTY; without even the implied warranty of
//    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
//    GNU Lesser General Public License for more details.
//
//    You should have received a copy of the GNU Lesser General Public License
//    along with Speechrecorder.  If not, see <http://www.gnu.org/licenses/>.

package ipsk.apps.speechrecorder.script.ui;

import java.awt.BorderLayout;
import java.awt.Component;
import java.awt.Container;
import java.awt.Dimension;
import java.awt.Font;
import java.awt.FontMetrics;
import java.awt.GridBagConstraints;
import java.awt.GridBagLayout;
import java.awt.Insets;
import java.awt.Toolkit;
import java.awt.datatransfer.Clipboard;
import java.awt.datatransfer.ClipboardOwner;
import java.awt.datatransfer.DataFlavor;
import java.awt.datatransfer.FlavorEvent;
import java.awt.datatransfer.FlavorListener;
import java.awt.datatransfer.Transferable;
import java.awt.datatransfer.UnsupportedFlavorException;
import java.awt.event.ActionEvent;
import java.awt.event.ActionListener;
import java.awt.event.FocusEvent;
import java.awt.event.FocusListener;
import java.awt.event.MouseAdapter;
import java.awt.event.MouseEvent;
import java.beans.PropertyChangeEvent;
import java.beans.PropertyChangeListener;
import java.io.IOException;
import java.net.URL;
import java.security.AllPermission;
import java.util.ArrayList;
import java.util.Arrays;
import java.util.HashSet;
import java.util.Hashtable;
import java.util.List;
import java.util.Set;

import javax.swing.Action;
import javax.swing.ActionMap;
import javax.swing.BorderFactory;
import javax.swing.DefaultCellEditor;
import javax.swing.InputMap;
import javax.swing.JComboBox;
import javax.swing.JLabel;
import javax.swing.JOptionPane;
import javax.swing.JPanel;
import javax.swing.JScrollPane;
import javax.swing.JSplitPane;
import javax.swing.JTable;
import javax.swing.JTextField;
import javax.swing.KeyStroke;
import javax.swing.ListSelectionModel;
import javax.swing.SwingUtilities;
import javax.swing.event.ListSelectionEvent;
import javax.swing.event.ListSelectionListener;
import javax.swing.table.TableCellEditor;
import javax.swing.table.TableCellRenderer;
import javax.swing.table.TableColumn;
import javax.swing.undo.StateEdit;
import javax.swing.undo.StateEditable;
import javax.swing.undo.UndoManager;

import ipsk.apps.speechrecorder.prompting.PromptPresenterServiceDescriptor;
import ipsk.apps.speechrecorder.script.ItemcodeGenerator;
import ipsk.apps.speechrecorder.script.PromptItemsTableModel;
import ipsk.db.speech.Group;
import ipsk.db.speech.Mediaitem;
import ipsk.db.speech.Nonrecording;
import ipsk.db.speech.PromptItem;
import ipsk.db.speech.PromptItemsList;
import ipsk.db.speech.PromptGroupsList;
import ipsk.db.speech.Recording;
import ipsk.db.speech.Section;
import ipsk.lang.DisplayBoolean;
import ipsk.persistence.IntegerSequenceGenerator;
import ipsk.swing.CopyAction;
import ipsk.swing.CutAction;
import ipsk.swing.EnumSelectionItem;
import ipsk.swing.EnumVector;
import ipsk.swing.PasteAction;
import ipsk.swing.RedoAction;
import ipsk.swing.UndoAction;
import ipsk.swing.action.EditActions;
import ipsk.swing.action.EditActionsListener;
import ipsk.swing.table.AutoFontCellRenderer;
import ipsk.swing.text.EditorKitMenu;
import ipsk.util.collections.ObservableList;


public class SectionUI extends JPanel implements ListSelectionListener,
		FlavorListener, ClipboardOwner, PropertyChangeListener, ActionListener, FocusListener, EditActionsListener, StateEditable{

	static class EnumSel {
		private String displayName;
		private Enum<?> string;

		public EnumSel(Enum<?> e,String displayName){
			this.string = e;
			this.displayName = displayName;
		}
		public EnumSel(Enum<?> e) {
			this(e,e==null?null:e.toString());
		}
		
		public Enum<?> getEnum(){
			return string;
		}

		public String toString() {
			return displayName;
		}

		public boolean equals(Object o) {
			if (o != null) {
				if (o instanceof EnumSel) {
					EnumSel mo = (EnumSel) o;
					if (string == null) {
						if (mo.getEnum() == null)
							return true;
					} else {
						if (string.equals(mo.getEnum())) {
							return true;
						}
					}
				}
			}
			return false;
		}
	}
	
	public class ItemCodeCellEditor extends DefaultCellEditor{

		private static final long serialVersionUID = 1L;
		public ItemCodeCellEditor(){
			super(new JTextField());
		}
		public Component getTableCellEditorComponent(JTable table,
				Object value,
				boolean isSelected,
				int row,
				int column){

			if(value instanceof String){
				String ic=(String)value;

				if(itemCodesInUse!=null && itemCodesInUse.contains(ic)){
					int answ=JOptionPane.showConfirmDialog(table,"Itemcode "+ic+" is already in use. There exists at least one recording file for this prompt.\nIf you rename this itemcode the recordings will not be visible anymore.\nDo you want, nevertheless, change the itemcode ?","Change item code",JOptionPane.YES_NO_OPTION);
					if(answ==JOptionPane.YES_OPTION){
					}else{
						SwingUtilities.invokeLater(new Runnable() {
							@Override
							public void run() {
								stopCellEditing();
							}
						});

					}
					
				}
			}
			return super.getTableCellEditorComponent(table, value, isSelected, row, column);
		}
	}
	
	private EnumVector<Section.Mode> modes = new EnumVector<Section.Mode>(Section.Mode.class,"(Default)");
	private static EnumVector<Section.Order> ORDERS = new EnumVector<Section.Order>(Section.Order.class,"Sequential (Default)");
	private static EnumVector<Section.PromptPhase> PROMPTPHASES=new EnumVector<Section.PromptPhase>(Section.PromptPhase.class,"Idle (Default)");
	
	private Section section;
	private JTextField nameField;

	private JComboBox<EnumSelectionItem<Section.Mode>> modeBox;
	private JComboBox<EnumSelectionItem<Section.Order>> orderBox;
	private JComboBox<EnumSelectionItem<Section.PromptPhase>> promptphasesBox;
	private JComboBox<DisplayBoolean> speakerdisplayBox;

	private JPanel promptItemsPanel;

	private PromptItemsTableModel promptItemTableModel;
	
	private AutoFontCellRenderer promptTextCellRenderer;
	
	private JTable promptItemsTable;
	
	private String[] promptFontFamilies;
	
	private String[] instructionsFontFamilies;
	/**
	 * @return the instructionsFontFamilies
	 */
	public String[] getInstructionsFontFamilies() {
		return instructionsFontFamilies;
	}

	/**
	 * @param instructionsFontFamilies the instructionsFontFamilies to set
	 */
	public void setInstructionsFontFamilies(String[] instructionsFontFamilies) {
		this.instructionsFontFamilies = instructionsFontFamilies;
		promptItemEditor.setInstructionsFontFamilies(instructionsFontFamilies);
	}

	private String[] descriptionFontFamilies;

	/**
	 * @return the descriptionFontFamilies
	 */
	public String[] getDescriptionFontFamilies() {
		return descriptionFontFamilies;
	}

	/**
	 * @param descriptionFontFamilies the descriptionFontFamilies to set
	 */
	public void setDescriptionFontFamilies(String[] descriptionFontFamilies) {
		this.descriptionFontFamilies = descriptionFontFamilies;
		promptItemEditor.setDescriptionFontFamilies(descriptionFontFamilies);
	}

	/**
	 * @return the promptFontFamilies
	 */
	public String[] getPromptFontFamilies() {
		return promptFontFamilies;
	}

	/**
	 * @param promptFontFamilies the promptFontFamilies to set
	 */
	public void setPromptFontFamilies(String[] promptFontFamilies) {
		this.promptFontFamilies = promptFontFamilies;
		promptTextCellRenderer.setPreferredFontFamilies(promptFontFamilies);
		promptItemEditor.setPromptFontFamilies(promptFontFamilies);
	}
	
	private AddRecordingAction addRecordingAction;
    private AddNonRecordingAction addNonRecordingAction;
    private GroupItemsAction groupItemsAction;
    private UngroupItemsAction ungroupItemsAction;
    private ToggleGroupRandomAction toggleGroupRandomAction;
    
	
	private CutAction cutAction;
	private CopyAction copyAction;
	private PasteAction pasteAction;
	
	private Clipboard clipboard = null;
	
	private JSplitPane splitPane;

	private Section.Mode defaultMode=Section.Mode.MANUAL;
	private int defaultPreRecording;
	private int defaultPostRecording;
	private boolean defaultPromptAutoPlay=true;
	  
	private URL projectContext=null;
	
	/**
	 * @return the projectContext
	 */
	public URL getProjectContext() {
		return projectContext;
	}

	/**
	 * @param projectContext the projectContext to set
	 */
	public void setProjectContext(URL projectContext) {
		this.projectContext = projectContext;
		promptItemEditor.setProjectContext(projectContext);
	}

	private PromptItemUI promptItemEditor;
    /**
	 * @return the promptItemEditor
	 */
	public PromptItemUI getPromptItemEditor() {
		return promptItemEditor;
	}

	private EditActions editActions;
    private EditActionsListener editActionsListener;
    
    private boolean userSelection=true;
    private List<Action> newActionsList;
    IntegerSequenceGenerator sequenceGenerator;
    private ItemcodeGenerator itemcodeGenerator;
    private ObservableList<String> itemCodesList;
    private Set<String> itemCodesInUse=null;
   
    public Set<String> getItemCodesInUse() {
		return itemCodesInUse;
	}

	public void setItemCodesInUse(Set<String> itemCodesInUse) {
		this.itemCodesInUse = itemCodesInUse;
		promptItemEditor.setItemCodesInUse(itemCodesInUse);
	}

	private UndoManager undoManager=new UndoManager();
    private UndoAction undoAction;

    private RedoAction redoAction;
  
	public SectionUI(URL projectContext, IntegerSequenceGenerator sequenceGenerator, ObservableList<String> itemCodesSetProvider,ItemcodeGenerator itemcodeGenerator,List<PromptPresenterServiceDescriptor> availablePromptPresenters2) {
		super(new BorderLayout());
		this.projectContext=projectContext;
		this.sequenceGenerator=sequenceGenerator;
		this.itemCodesList=itemCodesSetProvider;
		this.itemcodeGenerator=itemcodeGenerator;
//		this.availablePromptPresenters=availablePromptPresenters2;
		SecurityManager security = System.getSecurityManager();
		
		if (security != null) {
			try {
//				deprecated should check for AllPermissions
//				security.checkSystemClipboardAccess();
//				security.checkPermission(new AllPermission());
				// gone with JDK11
				//security.checkSystemClipboardAccess();
				security.checkPermission(new AllPermission());
			} catch (SecurityException se) {
				System.err.println("WARNING: System clipboard not accessible.");
				clipboard = new Clipboard("Script Clipboard");
			}
		}
		if (clipboard == null) {
			clipboard = Toolkit.getDefaultToolkit().getSystemClipboard();
		}
		clipboard.addFlavorListener(this);
		
		JPanel sectionPanel=new JPanel(new GridBagLayout());
		GridBagConstraints c = new GridBagConstraints();
		c.fill = GridBagConstraints.HORIZONTAL;

		c.insets = new Insets(2, 2, 2, 2);
		c.anchor = GridBagConstraints.PAGE_START;

		c.weightx = 0;
		c.gridx = 0;
		c.gridy = 0;
		JLabel nameLabel = new JLabel("Name (Id):");
		sectionPanel.add(nameLabel, c);
		nameField = new JTextField();
        EditorKitMenu nameFieldEkm=new EditorKitMenu(nameField);
        nameFieldEkm.setPopupMenuActiv(true);
        nameField.addActionListener(this);
		c.gridx++;
		sectionPanel.add(nameField, c);

		c.gridx = 0;
		c.gridy++;
		sectionPanel.add(new JLabel("Mode:"), c);
		modeBox = new JComboBox<EnumSelectionItem<Section.Mode>>(modes);
		c.gridx++;
		sectionPanel.add(modeBox, c);
		
		c.gridx = 0;
		c.gridy++;
		sectionPanel.add(new JLabel("Order:"), c);
		orderBox = new JComboBox<EnumSelectionItem<Section.Order>>(ORDERS);
		c.gridx++;
		sectionPanel.add(orderBox, c);
		
		c.gridx = 0;
		c.gridy++;
		sectionPanel.add(new JLabel("Promptphase:"), c);
		promptphasesBox = new JComboBox<EnumSelectionItem<Section.PromptPhase>>(PROMPTPHASES);
		c.gridx++;
		sectionPanel.add(promptphasesBox, c);
		
		c.gridx = 0;
		c.gridy++;
		sectionPanel.add(new JLabel("Speakerdisplay:"), c);
		speakerdisplayBox = new JComboBox<DisplayBoolean>(DisplayBoolean.getDefinedvalues());
		c.gridx++;
		sectionPanel.add(speakerdisplayBox, c);
		
		promptItemTableModel = new PromptItemsTableModel();
		
		promptItemsTable = new JTable(promptItemTableModel);
		TableColumn txtTableCol=promptItemsTable.getColumnModel().getColumn(PromptItemsTableModel.COL_PROMPT);
		TableCellRenderer txtTcr=txtTableCol.getCellRenderer();
		if(txtTcr==null){
		    txtTcr=promptItemsTable.getDefaultRenderer(promptItemsTable.getColumnClass(PromptItemsTableModel.COL_PROMPT));
		    
		}
		promptTextCellRenderer=new AutoFontCellRenderer(txtTcr);
        txtTableCol.setCellRenderer(promptTextCellRenderer);
        
        TableColumn itemCodeTableCol=promptItemsTable.getColumnModel().getColumn(PromptItemsTableModel.COL_FILE);
        ItemCodeCellEditor icEditor=new ItemCodeCellEditor();
        itemCodeTableCol.setCellEditor(icEditor);

        Font currFont=getFont();
        if(currFont!=null){
        	FontMetrics fontMetrics=getFontMetrics(currFont);
        	if(fontMetrics!=null){
        		TableColumn groupCol=promptItemsTable.getColumnModel().getColumn(PromptItemsTableModel.COL_GROUP);
        		String maxGroupVal="99";
        		int maxGroupWidth=fontMetrics.stringWidth(maxGroupVal);
        		groupCol.setPreferredWidth(maxGroupWidth);
        	}
        }
		//promptItemsTable.setAutoResizeMode(JTable.AUTO_RESIZE_LAST_COLUMN);
		MouseAdapter listMouseListener = new MouseAdapter() {
	    	public void mouseClicked(MouseEvent e) {
	        	if (e.getClickCount() == 2) {
	        		
	        		if(isEnabled()){
	        			editSelectedPromptItem();
	        		}
	        	}
	    	}
	    };
	    promptItemsTable.addMouseListener(listMouseListener); 

		ListSelectionModel promptItemsSelModel=promptItemsTable.getSelectionModel();
		promptItemsSelModel.setSelectionMode(
				ListSelectionModel.MULTIPLE_INTERVAL_SELECTION);
		promptItemsSelModel.addListSelectionListener(this);
		promptItemsTable.setDragEnabled(true);
		c.gridwidth = 2;
		c.gridx = 0;
		c.gridy++;
		c.weightx = 1;
		c.weighty = 1;
		c.fill = GridBagConstraints.BOTH;
		JScrollPane scrollPane = new JScrollPane(promptItemsTable);
//		System.out.println(promptItemsTable.getPreferredScrollableViewportSize()+" "+promptItemsTable.getPreferredSize());
		promptItemsTable.setPreferredScrollableViewportSize(new Dimension(100,100));

		promptItemsPanel = new JPanel(new GridBagLayout());
		GridBagConstraints sc = new GridBagConstraints();
		sc.gridx = 0;
		sc.gridy = 0;
		sc.weightx = 2.0;
		sc.weighty = 2.0;
		sc.fill = GridBagConstraints.BOTH;
		promptItemsPanel.setBorder(BorderFactory
				.createTitledBorder("Prompt items"));
		
		promptItemsPanel.add(scrollPane, sc);

		//add(promptItemsPanel, c);
		sectionPanel.add(scrollPane,c);
		
		promptItemEditor=new PromptItemUI(projectContext,availablePromptPresenters2);
//		add(promptItemEditor, c);
		sectionPanel.setMinimumSize(new Dimension(0,0));
        promptItemEditor.setMinimumSize(new Dimension(0,0));
		splitPane =new JSplitPane(JSplitPane.HORIZONTAL_SPLIT,true,sectionPanel,promptItemEditor);
		
		add(splitPane,BorderLayout.CENTER);
	    
	        final ActionListener al = this;
	        addRecordingAction = new AddRecordingAction() {
	            public void actionPerformed(ActionEvent ae) {
	                al.actionPerformed(ae);
	            }
	        };
	        addRecordingAction.setEnabled(false);
	       
	        addNonRecordingAction = new AddNonRecordingAction() {
	            public void actionPerformed(ActionEvent ae) {
	                al.actionPerformed(ae);
	            }
	        };
	        addNonRecordingAction.setEnabled(false);
	        
	        groupItemsAction=new GroupItemsAction() {
				
				@Override
				public void actionPerformed(ActionEvent ae) {
					 al.actionPerformed(ae);
				}
			};
			groupItemsAction.setEnabled(false);

			ungroupItemsAction=new UngroupItemsAction() {

				@Override
				public void actionPerformed(ActionEvent ae) {
					al.actionPerformed(ae);
				}
			};
			ungroupItemsAction.setEnabled(false);
	     
			toggleGroupRandomAction=new ToggleGroupRandomAction() {
                
                @Override
                public void actionPerformed(ActionEvent e) {
                    al.actionPerformed(e);
                }
            };
            toggleGroupRandomAction.setEnabled(false);
			
	     
        cutAction = new CutAction() {
            public void actionPerformed(ActionEvent ae) {
                al.actionPerformed(ae);
            }
        };
        cutAction.setEnabled(false);
      
       
        copyAction = new CopyAction() {
            public void actionPerformed(ActionEvent ae) {
                al.actionPerformed(ae);
            }
        };

        copyAction.setEnabled(false);

        pasteAction = new PasteAction() {
            public void actionPerformed(ActionEvent ae) {
                al.actionPerformed(ae);
            }
        };
        pasteAction.setEnabled(false);
        
        undoAction = new ipsk.swing.UndoAction() {
            public void actionPerformed(ActionEvent e) {
                al.actionPerformed(e);
            }
        };
        //iMap.put(UndoAction.ACCELERATOR_VAL, UndoAction.NAME);
        //aMap.put(UndoAction.NAME, undoAction);
        redoAction = new ipsk.swing.RedoAction() {
            public void actionPerformed(ActionEvent e) {
                al.actionPerformed(e);
            }
        };
        //iMap.put(RedoAction.ACCELERATOR_VAL, RedoAction.NAME);
        //aMap.put(RedoAction.NAME, redoAction);
        undoAction.setEnabled(false);
        redoAction.setEnabled(false);
        editActions=new EditActions(cutAction,copyAction,pasteAction,undoAction,redoAction);
		updateEditActions();
		InputMap imap = promptItemsTable.getInputMap();
		imap.put(KeyStroke.getKeyStroke("ctrl X"), cutAction
				.getValue(Action.ACTION_COMMAND_KEY));
		imap.put(KeyStroke.getKeyStroke("ctrl C"), copyAction
				.getValue(Action.NAME));
		imap.put(KeyStroke.getKeyStroke("ctrl V"), pasteAction
				.getValue(Action.NAME));
		ActionMap map = promptItemsTable.getActionMap();
		map.put(cutAction.getValue(Action.ACTION_COMMAND_KEY), cutAction);
		map.put(copyAction.getValue(Action.NAME), copyAction);
		map.put(pasteAction.getValue(Action.NAME), pasteAction);
		
		
		promptItemsTable.addFocusListener(this);
		newActionsList=new ArrayList<Action>();
		newActionsList.add(addRecordingAction);
		newActionsList.add(addNonRecordingAction);
		promptItemEditor.setEditActionListener(this);
	}

	public UngroupItemsAction getUngroupItemsAction() {
		return ungroupItemsAction;
	}

	/**
	 * @return the groupItemsAction
	 */
	public GroupItemsAction getGroupItemsAction() {
		return groupItemsAction;
	}

	private void setTablePromptUnits(List<Group> tablePromptUnits) {

		promptItemTableModel.setPromptUnits(tablePromptUnits);
		if(tablePromptUnits.size()>0){
			// set first row selected by default
			promptItemsTable.setRowSelectionInterval(0, 0);
		}else{
			promptItemsTable.clearSelection();
			promptItemEditor.applyValues();
			promptItemEditor.setPromptItem(null);
		}

	}
	
	private List<Group> promptUnitsCopy(List<Group> promptUnits){
		List<Group> copyPus=new ArrayList<Group>();

		for(Group pu:promptUnits){
		    Group g=(Group)pu;
		    Group gc;
		    try {
		        gc = (Group) g.clone();
		        copyPus.add(gc);
		    } catch (CloneNotSupportedException e) {
		        // TODO Auto-generated catch block
		        e.printStackTrace();
		    }
		}

		return copyPus;
	}
	
	private void setPromptItemsCopy(){
		List<Group> copyPus=new ArrayList<Group>();
		if(section!=null){
			copyPus=promptUnitsCopy(section.getGroups());
		}else{
			copyPus=new ArrayList<Group>();
		}
		setTablePromptUnits(copyPus);
	}
    
	public void setSection(Section section) {
	    if(this.section!=null){
	        this.section.removePropertyChangeListener(this);
	        applyValues();
	    }
		this.section = section;
		userSelection=false;
		boolean enabled=(this.section!=null);
		
		if(enabled){
		    
		    nameField.setText(section.getName());
		    modeBox.setSelectedItem(modes.getItem(this.section.getMode()));
		    orderBox.setSelectedItem(ORDERS.getItem(this.section.getOrder()));
		    promptphasesBox.setSelectedItem(PROMPTPHASES.getItem(this.section.getPromptphase()));
		    speakerdisplayBox.setSelectedItem(new DisplayBoolean(this.section.getSpeakerDisplay()));
		    this.section.addPropertyChangeListener(this);
		}
		
		undoManager.discardAllEdits();
		setEnabled(enabled);
		setPromptItemsCopy();
		updateEditActions();
		userSelection=true;
	}
	
	

	/**
	 * @param args
	 */
	public static void main(String[] args) {
		new SectionUI((URL)null,null,null,new ItemcodeGenerator(),null);
	}
	
	private void editSelectedPromptItem(){
	
		ListSelectionModel selModel = promptItemsTable.getSelectionModel();
		int sel = selModel.getMinSelectionIndex();
		
	}
	
	
	private void stopCellEditing(){
	    TableCellEditor cellEditor=promptItemsTable.getCellEditor();
        if(cellEditor!=null){
            cellEditor.stopCellEditing();
        }
	}
	
	@SuppressWarnings("unchecked")
    protected void applyValues(){
	    if(section!=null){
	        promptItemEditor.applyValues();
	        String nameFieldText = nameField.getText();
	        if (!(section.getName() == null && nameFieldText.equals(""))) {
	            section.setName(nameFieldText);
	        }
	        // TODO still type unsafe, getSelectedItem seems not to support generics
	        section.setMode(((EnumSelectionItem<Section.Mode>)modeBox.getSelectedItem()).getEnumVal());
	        section.setOrder(((EnumSelectionItem<Section.Order>)orderBox.getSelectedItem()).getEnumVal());
	        section.setPromptphase(((EnumSelectionItem<Section.PromptPhase>)promptphasesBox.getSelectedItem()).getEnumVal());
	        section.setSpeakerDisplay(((DisplayBoolean)speakerdisplayBox.getSelectedItem()).getValue());
	        stopCellEditing();
	        List<Group> tablePis=promptItemTableModel.getPromptUnits();
	        section.setGroups(tablePis);
	    }
	}
    
    
	public void actionPerformed(ActionEvent e) {
		
		Object src = e.getSource();
			String cmd = e.getActionCommand();
			if (copyAction.getActionCommand().equals(cmd)) {
			   selectedPromptUnitsToClipboard();
			} else if (pasteAction.getActionCommand().equals(cmd)) {
				DataFlavor[] adf=clipboard.getAvailableDataFlavors();
				try {
					for(DataFlavor df:adf){
						if(PromptItem.CLASS_DATA_FLAVOR.equals(df)){
							insert((PromptItem) (clipboard.getContents(this)
									.getTransferData(PromptItem.CLASS_DATA_FLAVOR)));
							break;
						}else if(PromptItemsList.CLASS_DATA_FLAVOR.equals(df)){
							PromptItemsList pisDf=((PromptItemsList) (clipboard.getContents(this)
									.getTransferData(PromptItemsList.CLASS_DATA_FLAVOR)));
							if(pisDf!=null){
								insert(pisDf);
								break;
							}
						}else if(PromptGroupsList.CLASS_DATA_FLAVOR.equals(df)){
							PromptGroupsList pusDf=((PromptGroupsList) (clipboard.getContents(this)
									.getTransferData(PromptGroupsList.CLASS_DATA_FLAVOR)));
							if(pusDf!=null){
								insertPromptGroups(pusDf);
								break;
							}
						}
					}
				} catch (UnsupportedFlavorException e1) {
					// ignore
					e1.printStackTrace();
				} catch (IOException e1) {
					//ignore
					e1.printStackTrace();
				}
			} else if (cutAction.getActionCommand().equals(cmd)) {
				selectedPromptUnitsToClipboard();
				removeSelectedPromptItems();
			}else if(undoAction.getActionCommand().equals(cmd)){
                undoManager.undo();
            }else if(redoAction.getActionCommand().equals(cmd)){
                undoManager.redo();
            }else if (addRecordingAction.getActionCommand().equals(cmd)) {
			    Recording r=new Recording();
			    Mediaitem mi=new Mediaitem();
			    if(sequenceGenerator!=null){
                	r.setRecpromptId(sequenceGenerator.getAndIncrement());
                	mi.setMediaitemId(sequenceGenerator.getAndIncrement());
                }
			    List<Mediaitem> mis=new ArrayList<Mediaitem>();
			    mis.add(mi);
			    r.setMediaitems(mis);
                insert(r);
            }else if (addNonRecordingAction.getActionCommand().equals(cmd)) {
                Nonrecording nr=new Nonrecording();
                Mediaitem mi=new Mediaitem();
                if(sequenceGenerator!=null){
                	nr.setRecpromptId(sequenceGenerator.getAndIncrement());
                	mi.setMediaitemId(sequenceGenerator.getAndIncrement());
                }
                List<Mediaitem> mis=new ArrayList<Mediaitem>();
                mis.add(mi);
                nr.setMediaitems(mis);
                insert(nr);
            }else if (groupItemsAction.getActionCommand().equals(cmd)) {
               groupSelectedItems();
            }else if (ungroupItemsAction.getActionCommand().equals(cmd)) {
                ungroupSelectedItems();
            }else if (toggleGroupRandomAction.getActionCommand().equals(cmd)) {
                toggleGroupRandom();
             }else if(src==nameField){
               section.setName(nameField.getText());
            }
	}



	public void valueChanged(ListSelectionEvent e) {
	    if(!e.getValueIsAdjusting()){
	        validateSelectionModel();   // clear inconsistent selection models (Bug ID 0060) 
	        updateEditActions();
	        ListSelectionModel selModel=promptItemsTable.getSelectionModel();
	        int selIndex=selModel.getMinSelectionIndex();
	        
	        if(selIndex>=0){
	            List<PromptItem> selPis=new ArrayList<PromptItem>();
	            int maxSelIdx=selModel.getMaxSelectionIndex();
	            List<PromptItem> promptItems=promptItemTableModel.promptItems();
	            int pisSize=promptItems.size();
	            if(maxSelIdx!=-1){
	                for(int si=selIndex;si<=maxSelIdx;si++){
	                    if(selModel.isSelectedIndex(si)){
	                    	
	                    	if(si>=pisSize){
	                    		// if the last row is deleted and the row is immediately selected JTable selects a non existing row!
	                    		// Bug ID 0060
	                    		
	                    	}else{
	                    		PromptItem pi=promptItems.get(si);
	                    		selPis.add(pi);
	                    	}
	                    }
	                }
	                promptItemEditor.applyValues();
	                
	                
	                int selPisSize=selPis.size();
	                if(selPisSize==0){
	                    promptItemEditor.setPromptItem(null);
	                }else if(selPisSize==1){
	                    promptItemEditor.setPromptItem(selPis.get(0));
	                }else{
	                    promptItemEditor.setPromptItems(selPis);
	                }
	            }
	        }else{
	        	promptItemEditor.applyValues();
	            promptItemEditor.setPromptItem(null);
	        }
	    }
	}

	public void propertyChange(PropertyChangeEvent evt) {
	    Object src=evt.getSource();
	    
        String propertyName=evt.getPropertyName();
        if(src instanceof Section){
            if("name".equals(propertyName)){
                nameField.setText((String)evt.getNewValue());
            }
        }
	    
	}

	private boolean removePromptItem(List<Group> units,int selIdx){
	    
	    int cgIdx=0;
	    boolean removed=false;
	    // remove from group
	    for(Group g:units){
	        int grIdx=selIdx-cgIdx;
	        
	        List<PromptItem> gpis=g.getPromptItems();
	        int grSize=gpis.size();
	        if(grIdx>=0 && grIdx<grSize){
	            gpis.remove(grIdx);
	            removed=true;
//	            g.updatePositions();
	            break;
	        }
	        cgIdx+=grSize;
	    }

	    return removed;
	}
	
	private List<Group> removeEmptyGroups(List<Group> units){
	    List<Group> res=new ArrayList<Group>();
//	    int newPos=0;
	    for(Group g:units){

	        List<PromptItem> gpis=g.getPromptItems();
	        if(gpis!=null){
	            if(gpis.size()>0){
//	                g.setPosition(newPos++);
	                res.add(g);
	            }else{
	                // empty group, do not add
	                g.setSection(null);
//	                g.setPosition(null);
	            }
	        }

	    }
	    return res;
	}
	
	public void removeSelectedPromptItems() {
		
	    StateEdit removeStateEdit=new StateEdit(this,"Remove prompt items");
	    stopCellEditing();
		List<Group> newPromptUnits=promptUnitsCopy(promptItemTableModel.getPromptUnits());
		List<PromptItem> pis=promptItemTableModel.promptItems();
		int[] selRowsOrg=promptItemsTable.getSelectedRows();
		if(selRowsOrg.length>0){
			int[] selRowsAsc=Arrays.copyOf(selRowsOrg, selRowsOrg.length);
			// doc of getSelectedRow() does not mention if the list is ordered, so we order it here
			Arrays.sort(selRowsAsc);
			int lowestSelRow=Integer.MAX_VALUE;
			// remove items in reverse order
			for(int i=selRowsAsc.length-1;i>=0;i--){
				int selRow=selRowsAsc[i];
				if(selRow>=pis.size()){
					// JTable sometimes selects not existing rows
					continue;
				}
				if(selRow<lowestSelRow){
					lowestSelRow=selRow;
				}
				
				PromptItem itemToRemove=pis.get(selRow);
				
				boolean removed=removePromptItem(newPromptUnits, selRow);
				//itemToRemove.setSection(null);
				itemToRemove.setGroup(null);
//				itemToRemove.setPosition(null);
				if(removed && itemToRemove instanceof Recording){
					Recording removedRec=(Recording)itemToRemove;
					String removedItemcode=removedRec.getItemcode();
					itemCodesList.remove(removedItemcode);
					promptItemTableModel.setPromptUnits(newPromptUnits,selRow,selRow);
				}
			}
		
			newPromptUnits=removeEmptyGroups(newPromptUnits);
			promptItemTableModel.setPromptUnits(newPromptUnits);
//			section.updatePositions();
			
			if(newPromptUnits.size()==0){
				promptItemsTable.clearSelection();
			}else{
				int newSelRow=lowestSelRow-1;
				if(newSelRow<0)newSelRow=0;
				promptItemsTable.setRowSelectionInterval(newSelRow, newSelRow);
			}
			promptItemsTable.doLayout();
		}
		removeStateEdit.end();
        undoManager.addEdit(removeStateEdit);
        updateEditActions();
	}

	
	public PromptItemsList getSelectedPromptItems() {
		int[] selRowIdcs=promptItemsTable.getSelectedRows();
		PromptItemsList selPis=new PromptItemsList();
		for(int selRowIdx:selRowIdcs){
			PromptItem selPi=promptItemTableModel.promptItems().get(selRowIdx);
			selPis.add(selPi);
		}
		return selPis;
	}
	
	private boolean isIn(int[] arr,int i){
		for(int arrI:arr){
			if(arrI==i){
				return true;
			}
		}
		return false;
	}
	
	public PromptGroupsList selectedPromptUnits() {
		int[] selRowIdcs=promptItemsTable.getSelectedRows();
//		List<PromptUnit> selPus=new ArrayList<PromptUnit>(); 
		PromptGroupsList selPus=new PromptGroupsList();
		List<Group> pus=promptItemTableModel.getPromptUnits();
		int i=0;
		for(Group g:pus){
		    List<PromptItem> gPis=g.getPromptItems();
		    List<PromptItem> selGrPis=new ArrayList<PromptItem>();
		    boolean completeGroupSelected=true;
		    for(PromptItem gPi:gPis){
		        if(isIn(selRowIdcs,i)){
		            // prompt item of group selected
		            selGrPis.add(gPi);
		        }else{
		            completeGroupSelected=false;
		        }
		        i++;
		    }
		    if(completeGroupSelected){
		        selPus.add(g);
		    }else{
		        // split to dummy groups
		        for(PromptItem selGrPi:selGrPis){
		            Group sg=new Group();
		            sg.setSection(section);
		            sg.getPromptItems().add(selGrPi);
		            selPus.add(sg);
		        }
		    }
		}
		
		return selPus;
	}
	
	private void selectedPromptUnitsToClipboard(){
	    PromptGroupsList pul=selectedPromptUnits();
	    clipboard.setContents(pul, this);  
	}

	/**
	 * Inserts a prompt item into a list of groups.
	 * @param promptGroups prompt group list in which the prompt item will be inserted
	 * @param insertIndex index position at which to insert. the index refers to the list of prompt items of all groups
	 * @param promptItem prompt item to insert
	 */
	private void insert(List<Group> promptGroups,int insertIndex,PromptItem promptItem){

		int prUSize=promptGroups.size();
		boolean insertedInGroup=false;
//		promptItem.setSection(section);
		
		if(prUSize==0){
		    Group g=new Group();
		    g.setSection(section);
		    g.getPromptItems().add(promptItem);
			promptGroups.add(g);
		}else{
			int currIndex=0;
			Integer promptUnitInsertPosition=null;
			
			for(int pInd=0;pInd<promptGroups.size();pInd++){
				Group g=promptGroups.get(pInd);
				
					List<PromptItem> pis=g.getPromptItems();
					int pisSize=pis.size();
					if(insertIndex>=currIndex && insertIndex<=currIndex+pisSize){
					    int groupInsertPos=insertIndex-currIndex;
					    if(pisSize==1){
					        // insert in own group after current item
					        promptUnitInsertPosition=pInd+1;
					    }else{
//					        promptItem.setPosition(groupInsertPos);
					        promptItem.setGroup(g);
					        pis.add(groupInsertPos, promptItem);
					        insertedInGroup=true;
					    }
					    break;
					}else{
						currIndex+=pisSize;
					}
				
			}
			if(!insertedInGroup && promptUnitInsertPosition==null && currIndex==insertIndex){
				// append at the end
				promptUnitInsertPosition=promptGroups.size();
			}
			if(promptUnitInsertPosition!=null){
//			    promptItem.setPosition(promptUnitInsertPosition);
			    Group g=new Group();
	            g.setSection(section);
	            promptItem.setGroup(g);
	            g.getPromptItems().add(promptItem);
				promptGroups.add(promptUnitInsertPosition,g);
			}
		}
	}
	
	
	public void insert(PromptItem promptItem) {
		// Change of table requires the current values to be applied
		// It is not sufficient to let the prompt item editor aplly to the prompt item only
		// Fixes: Bug ID 0101
		applyValues();
		
	    StateEdit insertStateEdit=new StateEdit(this,"Insert prompt item");
	    stopCellEditing();
	    if(promptItem instanceof Recording){
	        Recording recording=(Recording)promptItem;
	        String itemcode=recording.getItemcode();
	        
	        if(itemcodeGenerator.getConfig().isActive() && (itemcode==null || "".equals(itemcode) || (itemCodesList.contains(itemcode)))){
	            HashSet<String> ics=new HashSet<String>(itemCodesList);
//	            itemcodeGenerator.next();
	            // TODO
	            itemcodeGenerator.toNext(ics);
	            String icGen=itemcodeGenerator.getItemCode();
	            recording.setItemcode(icGen);
	        }
	        
	        itemCodesList.add(recording.getItemcode());
	    }
	    List<Group> newPus=promptUnitsCopy(promptItemTableModel.getPromptUnits());
		ListSelectionModel selModel = promptItemsTable.getSelectionModel();
		
		int insertIndex = 0;
		if (!selModel.isSelectionEmpty()) {
			insertIndex = promptItemsTable.getSelectedRow()+1;
		}
//		newPus.add(insertIndex, promptItem);
		insert(newPus,insertIndex,promptItem);
		
		promptItemTableModel.setPromptUnits(newPus);
//		promptItemsTable.setRowSelectionInterval(insertIndex, insertIndex);
		insertStateEdit.end();
        undoManager.addEdit(insertStateEdit);
        promptItemsTable.setRowSelectionInterval(insertIndex, insertIndex);
        updateEditActions();
	}
	
	/**
	 * @param promptItemsList
	 */
	public void insert(List<PromptItem> promptItemsList) {
		 StateEdit insertStateEdit=new StateEdit(this,"Insert prompt items");
		 stopCellEditing();
		 // set new itemcodes if necessary
		 for(PromptItem promptItem:promptItemsList){
			 if(promptItem instanceof Recording){
				 Recording recording=(Recording)promptItem;
				 String itemcode=recording.getItemcode();

				 if(itemcodeGenerator.getConfig().isActive() && (itemcode==null || "".equals(itemcode) || (itemCodesList.contains(itemcode)))){
					 HashSet<String> ics=new HashSet<String>(itemCodesList);
					 //		            itemcodeGenerator.next();
					 // TODO
					 itemcodeGenerator.toNext(ics);
					 String icGen=itemcodeGenerator.getItemCode();
					 recording.setItemcode(icGen);
				 }

				 itemCodesList.add(recording.getItemcode());
			 }
		 }
		 
		 	// determine insert position
			ListSelectionModel selModel = promptItemsTable.getSelectionModel();
			
			int startInsertIndex=0;
			
			
			if (!selModel.isSelectionEmpty()) {
//				startInsertIndex = promptItemsTable.getSelectedRow()+1;
				startInsertIndex = selModel.getMaxSelectionIndex()+1;
			}
			
			// insert prompt items
			List<Group> newPus=promptUnitsCopy(promptItemTableModel.getPromptUnits());
			int insertIndex=startInsertIndex;
			for(PromptItem promptItem:promptItemsList){
				//insert into list
//				newPus.add(insertIndex, promptItem);
				insert(newPus,insertIndex,promptItem);
				insertIndex++;
			}
			int endInsertIndex=insertIndex-1;
			// apply list to table model
			promptItemTableModel.setPromptUnits(newPus);
			// edit end
			insertStateEdit.end();
	        undoManager.addEdit(insertStateEdit);
	        promptItemsTable.setRowSelectionInterval(startInsertIndex, endInsertIndex);
	        updateEditActions();
	}
	
	private void applyItemCode(Recording recording){
		 String itemcode=recording.getItemcode();

		 if(itemcodeGenerator.getConfig().isActive() && (itemcode==null || "".equals(itemcode) || (itemCodesList.contains(itemcode)))){
			 HashSet<String> ics=new HashSet<String>(itemCodesList);
			 itemcodeGenerator.toNext(ics);
			 String icGen=itemcodeGenerator.getItemCode();
			 recording.setItemcode(icGen);
		 }
		 itemCodesList.add(recording.getItemcode());
	}
	
	public void insertPromptGroups(List<Group> promptGroupsList) {
		 StateEdit insertStateEdit=new StateEdit(this,"Insert prompt units");
		 stopCellEditing();
		 int insertPisSize=0;
		 // set new itemcodes if necessary, calculate insert prompt size
		 for(Group g:promptGroupsList){
		     g.setSection(section);
			 
				 for(PromptItem gPi:g.getPromptItems()){
					 if(gPi instanceof Recording){
						 Recording recording=(Recording)gPi;
						 applyItemCode(recording);
					 }
					 insertPisSize++;
				 }
			 
		 }
		 
		 	// determine insert position
		  
			ListSelectionModel selModel = promptItemsTable.getSelectionModel();
			
			int startInsertIndex=0;
			
			
			if (!selModel.isSelectionEmpty()) {
//				startInsertIndex = promptItemsTable.getSelectedRow()+1;
				startInsertIndex = selModel.getMaxSelectionIndex()+1;
			}
			
			// insert prompt items
			List<Group> newPgs=promptUnitsCopy(promptItemTableModel.getPromptUnits());
			int insertIndex=startInsertIndex;
			int currIndex=0;
			List<Group> currPgs=promptItemTableModel.getPromptUnits();
			int pgsSize=currPgs.size();
			boolean inserted=false;
			for(int pgi=0;pgi<pgsSize;pgi++){
				Group g=currPgs.get(pgi);
				
					List<PromptItem> pis=g.getPromptItems();
					int pisSize=pis.size();
					if(insertIndex>=currIndex && insertIndex<=currIndex+pisSize){
						// add group after this group !!
						// do not fragment the existing group
						insertIndex=currIndex+pisSize;
						newPgs.addAll(pgi+1,promptGroupsList);
						inserted=true;
						break;
					}else{
						currIndex+=pisSize;
					}
				
			}
			if(!inserted){
				// append
				newPgs.addAll(promptGroupsList);
				insertIndex=currIndex;
			}
			int endInsertIndex=insertIndex+insertPisSize-1;
			// apply list to table model
			promptItemTableModel.setPromptUnits(newPgs);
			// update all transient positions in this section
//			section.updatePositions();
			// edit end
			insertStateEdit.end();
	        undoManager.addEdit(insertStateEdit);
	        
	        setRowSelectionIntervalSave(insertIndex, endInsertIndex);
	      
	        updateEditActions();
	}
	
	/*
	 * Workaround for inconsistent list selection model of JTable caused by mouse clicks on deleted rows.
	 *  
	 * @param startInsertIndex
	 * @param endInsertIndex
	 */
	private void setRowSelectionIntervalSave(int startInsertIndex,int endInsertIndex){
	    int rc=promptItemsTable.getRowCount();
	    if(startInsertIndex<0 || endInsertIndex <0){
	        return;
	    }
	    
        if(startInsertIndex>=rc){
            startInsertIndex=rc-1;
        }
        if(endInsertIndex>=rc){
            endInsertIndex=rc-1;
        }
        promptItemsTable.setRowSelectionInterval(startInsertIndex, endInsertIndex);
     
	}
	
	private void validateSelectionModel(){
	    int rc=promptItemsTable.getRowCount();
	    ListSelectionModel listSelectionModel=promptItemsTable.getSelectionModel();
	    if(listSelectionModel!=null && ! listSelectionModel.isSelectionEmpty()){
	    
	        int minSelIdx=listSelectionModel.getMinSelectionIndex();
	        int maxSelIdx=listSelectionModel.getMaxSelectionIndex();
	        if(minSelIdx>=rc || maxSelIdx>=rc){
	            System.err.println("Selection out of range: row count: "+rc+",min sel Idx: "+minSelIdx+",max sel Idx: "+maxSelIdx+",adjusting:"+listSelectionModel.getValueIsAdjusting());
	            System.err.println("Clearing invalid list selection");
	            listSelectionModel.clearSelection();
	        }
        }
	}
	
	private boolean isContinousSelection(){
		ListSelectionModel selModel = promptItemsTable.getSelectionModel();
		if(selModel.isSelectionEmpty()){
			return false;
		}else{
			for(int i=selModel.getMinSelectionIndex();i<=selModel.getMaxSelectionIndex();i++){
				if(!selModel.isSelectedIndex(i)){
					return false;
				}
			}
			return true;
		}		
	}
	
	private boolean isSelectionGroupable(){
	    
		ListSelectionModel selModel = promptItemsTable.getSelectionModel();
		if(selModel.isSelectionEmpty()){
			return false;
		}else{
			int tidx=0;
			int continousInSelCnt=0;
			boolean selComplete=false;
			List<Group> pus=promptItemTableModel.getPromptUnits();
			for(Group g:pus){
				  List<PromptItem> pis=g.getPromptItems();
				  boolean realGroup=pis.size()>1;
					for(PromptItem pi:pis){
					    if(realGroup){
					        if(selModel.isSelectedIndex(tidx)){
						        // item already in real group
						        return false;
						    }else{
						        tidx++;
						    }
						}else{
						    if(selModel.isSelectedIndex(tidx)){
		                        if(selComplete){
		                            // this is a multi-selection
		                            // not supported
		                            return false;
		                        }
		                        // mark current prompt item is in selection
		                        continousInSelCnt++;
		                    }else{
		                        if(continousInSelCnt>0){
		                            // mark selection end
		                            //inSel=false;
		                            selComplete=true;
		                        }
		                    }
		                    tidx++;
						}
					}
				
			}
			return continousInSelCnt>1;
		}	
	}

	/**
	 * Checks if a
	 * @return
	 */
	private boolean isSelectionGroupsOnly(){

	    ListSelectionModel selModel = promptItemsTable.getSelectionModel();
	    boolean atLeastOneGroupSelected=false;
	    if(!selModel.isSelectionEmpty()){
	        int tidx=0;
	        List<Group> pus=promptItemTableModel.getPromptUnits();
	        for(Group g:pus){
	            List<PromptItem> pis=g.getPromptItems();
	            boolean realGroup=pis.size()>1;
	            boolean groupSelected=selModel.isSelectedIndex(tidx);
	            if(!realGroup && groupSelected){
	                return false;
	            }
	            for(PromptItem pi:pis){
	                boolean piSel=selModel.isSelectedIndex(tidx);
	                if(piSel!=groupSelected){
	                    return false;
	                }
	                tidx++;
	            }
	            if(groupSelected){
	                atLeastOneGroupSelected=true;
	            }
	        }
	    }
	    return atLeastOneGroupSelected;
	}

	private void groupSelectedItems(){
		// check again
		if(isSelectionGroupable()){
			StateEdit groupItsStateEdit=new StateEdit(this,"Group prompt items");
			stopCellEditing();

			ListSelectionModel selModel = promptItemsTable.getSelectionModel();

			// save selected indizes in this aryray
			List<Integer> selPis=new ArrayList<>();

			List<Group> newGs=new ArrayList<Group>();
			List<Group> gs=promptItemTableModel.getPromptUnits();
			// prompt item index
			int piIdx=0;
			// placeholder for new group
			Group newGroup=null;
			for(Group g:gs){
				List<PromptItem> pis=g.getPromptItems();
				int pisSize=pis.size();
				boolean realGroup=pisSize>1;
				if(realGroup){
					// persist pending new group first
					if(newGroup!=null){
						newGs.add(newGroup);
						newGroup=null;
					}
					newGs.add(g);
				}else{
					PromptItem pi=pis.get(0);
					if(selModel.isSelectedIndex(piIdx)){
						// save selected
						selPis.add(piIdx);
						// create the new group
						if(newGroup==null){
							newGroup=new Group();
							if(sequenceGenerator!=null){
								newGroup.setGroupId(sequenceGenerator.getAndIncrement());
							}
						}
						// add selected prompt item to new group
						// ( the dummy group of this item is dropped )
						newGroup.getPromptItems().add(pi);
					}else{
						// persist pending new group first
						if(newGroup!=null){
							newGs.add(newGroup);
							newGroup=null;
						}
						// pass through
						newGs.add(g);
					}
				}
				piIdx+=pisSize;

			}
			// persist maybe pending new group
			if(newGroup!=null){
				newGs.add(newGroup);
				newGroup=null;
			}
			// apply list to table model
			promptItemTableModel.setPromptUnits(newGs);
			// edit end
			groupItsStateEdit.end();
			undoManager.addEdit(groupItsStateEdit);
			updateEditActions();

			// apply selection again
			// the structure of the section may have changed (group inserts),
			// but not the order of the promptitems in the section.
			// So the selection should be the same as before the grouping action

			selModel.clearSelection();
			for(Integer selIdx:selPis) {
				selModel.addSelectionInterval(selIdx, selIdx);
			}
		}
	}
	
	private List<Group> selectedGroups(){
	    ListSelectionModel selModel = promptItemsTable.getSelectionModel();
       
        List<Group> pus=promptItemTableModel.getPromptUnits();
        
        List<Group> selGroups=new ArrayList<>();
        int tidx=0;
        for(Group g:pus){
            List<PromptItem> pis=g.getPromptItems();
          
            boolean realGroup=pis.size()>1;
            boolean groupSelected=selModel.isSelectedIndex(tidx);
            if(realGroup){

                for(PromptItem pi:pis){
                    boolean piSel=selModel.isSelectedIndex(tidx);
                    if(piSel!=groupSelected){
                        // group must be completely selected/unselected
                        // abandon
                        return null;
                    }
                    tidx++;
                }
                if(groupSelected){
                    selGroups.add(g);
                }
            }else{
                if(groupSelected){
                    // separate prompt items must not be selected
                    // abandon
                    return null; 
                }else{
                    tidx++;
                }
            }
        }
        return selGroups;
	}
	
	private void ungroupSelectedItems(){
	    // check again

	    stopCellEditing();

	    ListSelectionModel selModel = promptItemsTable.getSelectionModel();
	    List<Group> newPus=new ArrayList<Group>();
	    List<Group> pus=promptItemTableModel.getPromptUnits();
	    int tidx=0;
	    for(Group g:pus){
	        
	            boolean groupSelected=selModel.isSelectedIndex(tidx);
	           
	            List<PromptItem> pis=g.getPromptItems();

	            for(PromptItem pi:pis){
	                boolean piSel=selModel.isSelectedIndex(tidx);
	                if(piSel!=groupSelected){
	                    // group must be completely selected/unselected
	                    // abandon
	                    return;
	                }
	                tidx++;
	            }
	            if(groupSelected){
	                // add ungrouped
	                for(PromptItem pi:pis){
	                    Group sg=new Group();
	                    sg.setSection(section);
	                    sg.getPromptItems().add(pi);
	                    newPus.add(sg);
	                }
	                
	            }else{
	                // pass group through
	                newPus.add(g);
	            }

	       
	    }

	    StateEdit ungroupItsStateEdit=new StateEdit(this,"Ungroup prompt items");
	    // apply list to table model
	    promptItemTableModel.setPromptUnits(newPus);
	    // edit end
	    ungroupItsStateEdit.end();
	    undoManager.addEdit(ungroupItsStateEdit);
	    updateEditActions();

	}


	private void toggleGroupRandom(){
	    List<Group> selGrps=selectedGroups();
	    if(selGrps!=null){
	        Object selrandomObj=toggleGroupRandomAction.getValue(Action.SELECTED_KEY);
	        if(selrandomObj instanceof Boolean){
	            boolean random=(Boolean)selrandomObj;
	            for(Group selGr:selGrps){
	                if(random){
	                    selGr.setOrder(Group.Order.RANDOM);
	                }else{
	                    selGr.setOrder(Group.Order.SEQUENTIAL);
	                }
	            }
	        }
	       
	    }
	}

	public ToggleGroupRandomAction getToggleGroupRandomAction() {
        return toggleGroupRandomAction;
    }

	public void lostOwnership(Clipboard clipboard, Transferable contents) {
		// System.out.println("lostOwnerShip");
	}

	public void flavorsChanged(FlavorEvent e) {
	    userSelection=false;
		updateEditActions();
		userSelection=true;
	}
	
	private void updateEditActions() {
		boolean editorEnabled=super.isEnabled();
		// check if clipboard contains flavors to accept 
		boolean clipBoardAvail = false;
		try {
			// accept prompt item or list of prompt items
			clipBoardAvail = clipboard.isDataFlavorAvailable(PromptItem.CLASS_DATA_FLAVOR)||
					clipboard.isDataFlavorAvailable(PromptItemsList.CLASS_DATA_FLAVOR)||
					clipboard.isDataFlavorAvailable(PromptGroupsList.CLASS_DATA_FLAVOR);
		} catch (IllegalStateException ise) {
			// Accessed by another application
		}
		// enable actions accordingly
		pasteAction.setEnabled(editorEnabled && clipBoardAvail);
		ListSelectionModel selModel = promptItemsTable.getSelectionModel();
		if (selModel.isSelectionEmpty()) {
			copyAction.setEnabled(false);
			cutAction.setEnabled(false);
		} else {
			copyAction.setEnabled(editorEnabled);
			cutAction.setEnabled(editorEnabled);
			if(editActionsListener!=null && userSelection){
			    editActionsListener.providesEditActions(this, editActions);
			}
		}
		boolean groupable=isSelectionGroupable();
		groupItemsAction.setEnabled(groupable);
		List<Group> selGrps=selectedGroups();
		boolean selectionGroupsOnly=isSelectionGroupsOnly();
		ungroupItemsAction.setEnabled(selectionGroupsOnly);
		boolean toggleRandomGroupEnabled=(selGrps!=null && selGrps.size()==1);
		toggleGroupRandomAction.setEnabled(toggleRandomGroupEnabled);
		if(toggleRandomGroupEnabled){
		    Group selGrp=selGrps.get(0);
		    Group.Order grOrder=selGrp.getNNOrder();
		    boolean grIsRandom=Group.Order.RANDOM.equals(grOrder);
		    toggleGroupRandomAction.putValue(Action.SELECTED_KEY,grIsRandom);
		}
		
		addRecordingAction.setEnabled(editorEnabled);
		addNonRecordingAction.setEnabled(editorEnabled);
		// update undo/redo manager
		undoAction.update(undoManager);
        redoAction.update(undoManager);
	
	}
	   
    private void setDeepEnabled(Container c,boolean b){
        for(Component cc:c.getComponents()){
            if(cc instanceof Container){
                setDeepEnabled((Container)cc,b);
            }
            cc.setEnabled(b);
        }
    }
    
	public void setEnabled(boolean b){
		super.setEnabled(b);
		setDeepEnabled(this, b);
	}

	public Action getCutAction(){
	    return cutAction;
	}

    public EditActionsListener getEditActionListener() {
        return editActionsListener;
    }

    public void setEditActionListener(EditActionsListener editActionsListener) {
        this.editActionsListener = editActionsListener;
    }

    public void focusGained(FocusEvent e) {
        updateEditActions();
    }

    public void focusLost(FocusEvent e) {
        // no action
    }
    
    public List<Action> getNewActionsList(){
        return newActionsList;
    }

	public void providesEditActions(Object src, EditActions editActions) {
		if(editActionsListener!=null){
			editActionsListener.providesEditActions(src, editActions);
		}
	}
	public void restoreState(Hashtable<?, ?> state) {
	    List<Group> pus=promptItemTableModel.getPromptUnits();

	    for(Group g: pus){

	        for(PromptItem pi:g.getPromptItems()){
	            if(pi instanceof Recording){
	                Recording r=(Recording)pi;
	                String ic=r.getItemcode();
	                itemCodesList.remove(ic);
	            }

	        }

	    }
	    List<Group> tablepromptItems=(List<Group>)state.get("promptUnits");
	    for(Group g: tablepromptItems){

	        for(PromptItem pi:g.getPromptItems()){
	            if(pi instanceof Recording){
	                Recording r=(Recording)pi;
	                String ic=r.getItemcode();
	                itemCodesList.add(ic);
	            }
	        }

	    }
	    setTablePromptUnits(tablepromptItems);
	    Integer selRow=(Integer)state.get("_promptItems.selected_row");
	    if(selRow==null || selRow==-1){
	        promptItemsTable.clearSelection();
	    }else{
	        promptItemsTable.setRowSelectionInterval(selRow,selRow);
	    }
	}

   

    public void storeState(Hashtable<Object, Object> state) {
        state.put("promptUnits",promptItemTableModel.getPromptUnits());
        state.put("_promptItems.selected_row", promptItemsTable.getSelectedRow());
    }

    public void setSelectedPromptItem(PromptItem pi) {
        if(pi!=null){
            List<PromptItem> pis=promptItemTableModel.promptItems();
            for(int i=0;i<pis.size();i++){
                PromptItem tPi=pis.get(i);
               
                 // TODO the IDs are currently not set !!
                if(tPi.getRecpromptId()==pi.getRecpromptId()){
                    promptItemsTable.setRowSelectionInterval(i,i);
                    break;
                }
            }
        
        }
    }

    public Section.Mode getDefaultMode() {
        return defaultMode;
    }

    public void setDefaultMode(Section.Mode defaultMode) {
        this.defaultMode = defaultMode;
        if(defaultMode!=null){
            modes.getItem(null).setDisplayName(defaultMode.toString()+" (Default)");
        }
    }

    public int getDefaultPreRecording() {
        return defaultPreRecording;
    }

    public void setDefaultPreRecording(int defaultPreRecording) {
        this.defaultPreRecording = defaultPreRecording;
        if(promptItemEditor!=null){
            promptItemEditor.setDefaultPreRecording(defaultPreRecording);
        }
    }

    public int getDefaultPostRecording() {
        return defaultPostRecording;
    }

    public void setDefaultPostRecording(int defaultPostRecording) {
        this.defaultPostRecording = defaultPostRecording;
        if(promptItemEditor!=null){
            promptItemEditor.setDefaultPostRecording(defaultPostRecording);
        }
    }

    public boolean isDefaultPromptAutoPlay() {
        return defaultPromptAutoPlay;
    }

    public void setDefaultPromptAutoPlay(boolean defaultPromptAutoPlay) {
        this.defaultPromptAutoPlay = defaultPromptAutoPlay;
        if(promptItemEditor!=null){
            promptItemEditor.setDefaultPromptAutoPlay(defaultPromptAutoPlay);
        }
    }

  
}
