//    Speechrecorder
// 	  (c) Copyright 2017
// 	  Institute of Phonetics and Speech Processing,
//    Ludwig-Maximilians-University, Munich, Germany
//
//
//    This file is part of Speechrecorder
//
//
//    Speechrecorder is free software: you can redistribute it and/or modify
//    it under the terms of the GNU Lesser General Public License as published by
//    the Free Software Foundation, version 3 of the License.
//
//    Speechrecorder is distributed in the hope that it will be useful,
//    but WITHOUT ANY WARRANTY; without even the implied warranty of
//    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
//    GNU Lesser General Public License for more details.
//
//    You should have received a copy of the GNU Lesser General Public License
//    along with Speechrecorder.  If not, see <http://www.gnu.org/licenses/>.



package ipsk.apps.speechrecorder.storage;

import java.io.File;
import java.io.IOException;
import java.io.UnsupportedEncodingException;
import java.net.MalformedURLException;
import java.net.URI;
import java.net.URISyntaxException;
import java.net.URL;
import java.net.URLEncoder;
import java.nio.file.DirectoryStream;
import java.nio.file.Files;
import java.nio.file.Path;
import java.text.DecimalFormat;
import java.text.NumberFormat;
import java.text.ParseException;
import java.util.Enumeration;
import java.util.HashSet;
import java.util.Iterator;
import java.util.List;
import java.util.Set;

import javax.sound.sampled.AudioFileFormat;

import ipsk.apps.speechrecorder.MetaData;
import ipsk.apps.speechrecorder.storage.net.HTTPStorageProtocol;
import ipsk.db.speech.Project;
import ipsk.db.speech.PromptItem;
import ipsk.db.speech.Recording;
import ipsk.db.speech.RecordingFile;
import ipsk.db.speech.Script;
import ipsk.db.speech.Session;
import ipsk.db.speech.Speaker;
import ipsk.net.Utils;
import ipsk.text.NaturalNumberFormat;

/**
 * @author klausj
 *
 */
public class SessionStorageManager {
	
	protected static final int DEF_SESSION_ID_FORMAT_DIGIT_COUNT = 4;
	public static String DEF_SESSION_ID_FORMAT="0000";
	public static String DEF_SPEAKER_ID_FORMAT="0000";
    public static String DEF_RECVERSION_FORMAT="_00";
	    
	protected URL storageURL = null;
	private boolean useSessionID = true;
	private boolean useScriptID = true;

	protected DecimalFormat speakerIDFormat;
	protected NumberFormat sessionIDFormat;
	protected DecimalFormat scriptIDFormat;
	
	protected int numLines = 1;
	
	protected AudioFileFormat.Type uploadType = null;
    protected DecimalFormat recVersionFormat;
	
	protected boolean createSpeakerDir = true;
	protected boolean createScriptDir = true;
	protected MetaData metadata = new MetaData();
	protected String scriptID;
	
    protected boolean overwrite;

	
	/**
	 * 
	 */
	public SessionStorageManager() {
		super();
		sessionIDFormat=new NaturalNumberFormat(DEF_SESSION_ID_FORMAT_DIGIT_COUNT);
		speakerIDFormat = new DecimalFormat(DEF_SESSION_ID_FORMAT);
		recVersionFormat=new DecimalFormat(DEF_RECVERSION_FORMAT);
		uploadType = AudioFileFormat.Type.WAVE;
	}
	
	/**
	 * @return the target URL for uploads.
	 */
	public URL getStorageURL() {
		return storageURL;
	}
	
	/**
	 * defines the target URL for uploads.
	 * That means the protocol, server name and possibly a 
	 * application specific path, e.g. 
	 * <code>http://OurAppServer/DemoApplication/RecordingData/StoreServlet.xml</code></br>
	 * If the protocol of the given URL is <i>file</i>, the storageManager
	 * switches to local mode. Otherwise it switches to remote mode.
	 * 
	 * @param target any valid URL
	 */
	public void setStorageURL(URL storageURL) {
		this.storageURL = storageURL;
	}
	
	/**
	 * @param i number of lines
	 */
	public void setNumLines(int i) {
		numLines = i;
	}
	
	 /**
     * @param overwrite the overwrite to set
     */
    public void setOverwrite(boolean overwrite) {
        this.overwrite = overwrite;
    }

	/**
	 * @return upload audio type
	 */
	public AudioFileFormat.Type getUploadType() {
		return uploadType;
	}

	
	/**
	 * @return speaker ID format
	 */
	public DecimalFormat getSpeakerIDFormat() {
		return speakerIDFormat;
	}
	
	/**
	 * adds (or extends) a query string to a URL. 
	 * If the givem URL already has a query string attached, the new part is added. 
	 * The method adds one <i>name/value</i> pair to the URL. It does not verify if the
	 * parameter <code>name</code> already exists in the query.
	 * @param  the URL to be extended
	 * @param the variable name for the query string without initial delimiter
	 * @param the variable value for the query string without initial delimiter
	 * @return the new URL
	 * @throws MalformedURLException
	 * @throws UnsupportedEncodingException 
	 */
	protected URL addQueryToURL(URL baseURL, String name, String value) throws MalformedURLException, UnsupportedEncodingException
	{
		String filePart = baseURL.getFile();
        // key and value strings must be URL encoded
        String urlEncName=URLEncoder.encode(name,"UTF-8");
        String urlEncValue=URLEncoder.encode(value,"UTF-8");
        
		if (baseURL.getQuery() == null)
			filePart += "?" + urlEncName + "=" + urlEncValue;
		else
			filePart += "&" + urlEncName + "=" + urlEncValue;
		URL url = new URL(baseURL.getProtocol(), baseURL.getHost(), baseURL.getPort(), filePart);
		return url;
	}
	
	/**
	 * appends a path fragment on a URL.
	 * @param baseURL
	 * @param dir
	 * @return the new URL
	 * @throws MalformedURLException
	 * @throws StorageManagerException 
	 */
	protected URL addDirToURL(URL baseURL, String dir) throws StorageManagerException {
		String resPath;
		URI resUri;
		URL url;
		try {
			resUri=baseURL.toURI();
			String resUriPath=resUri.getPath();
			if(resUriPath==null){
				throw new StorageManagerException("Could not get path from URI: "+resUri.toString());
			}
			resPath = resUriPath.concat("/" + dir);
			resUri=new URI(resUri.getScheme(),resUri.getUserInfo(),resUri.getHost(),resUri.getPort(),resPath,resUri.getQuery(),resUri.getFragment());
			url=resUri.toURL();
		} catch (URISyntaxException e) {
			throw new StorageManagerException(e.getMessage());
		} catch (MalformedURLException e) {
			throw new StorageManagerException(e.getMessage());
		}

//		URL url = new URL(baseURL.getProtocol(), baseURL.getHost(), baseURL.getPort(), resURL);
		
		return url;
	}
	
	/**
	 * Creates a base URL for this session, which is used to build
	 * several individual URLs for the upload.
	 * @return the base URL
	 * @throws StorageManagerException 
	 */
	public URL getSessionURL(int sessionID) throws StorageManagerException {

		String sessionDir = "";
		if (createSpeakerDir) {
			String tmpDir =
				sessionDir.concat("/" + sessionIDFormat.format(sessionID));
			sessionDir = tmpDir;
		}
		if (createScriptDir && scriptID !=null) {
			
			String tmpDir =
				sessionDir.concat("/" + scriptIDFormat.format(scriptID));
			sessionDir = tmpDir;
		}
		URL sessionURL = storageURL;
//		try
//		{
		sessionURL = addDirToURL(storageURL, sessionDir);
//		}
//		catch (StorageManagerException ex)
//		{
//			return null;
//		}
		return sessionURL;
	}
	
	public String sessionIDString(int sessionID) {
		return sessionIDFormat.format(sessionID);
	}
	
	public File sessionDirForSessionID(int sessionID) throws StorageManagerException{
	      URL sessURL=getSessionURL(sessionID);
	      try {
            File sessDir= Utils.fileFromDecodedURL(sessURL);
            return sessDir;
        } catch (UnsupportedEncodingException e) {
            throw new StorageManagerException(e);
        }
	      
	}
	
	/**
     * Computes a file name fragment for this session.
     * The name starts always with the given speaker code. 
     * If <b>useSpeakerID</b> is set, it is followed by the speaker ID.  
     * If <b>useSessionID</b> is set, the the session ID is added.
     * @param sessionID session ID  
     * @param speakerCode speaker code
     * @return a file name fragment for this session
     */
    protected String getSessionNamePart(int sessionID,String speakerCode) {
        String sessionFileName = "";
        if(speakerCode!=null){
            sessionFileName=sessionFileName.concat(speakerCode);
        }
        if (useSessionID) {
            sessionFileName =
                sessionFileName.concat(sessionIDString(sessionID));
        }
        if (useScriptID) {
            if (scriptID !=null){
            sessionFileName =
                sessionFileName.concat(scriptID);
            }
        }
        return sessionFileName;
    }
	
	/**
     * Computes a root file name, subject to several options.
     * The root file name consists at least of the speaker code, plus the prompt code.
     * @see #getSessionNamePart(String speakerCode)
     * @param promptCode
     * @param speakerCode
     * @return a root file name
     */
    public String getRootFileName(int sessionID,String promptCode, String speakerCode, int recVersion) {
        String rootFileName = getSessionNamePart(sessionID,speakerCode);
        rootFileName = rootFileName.concat(promptCode);
        if(!overwrite){
            rootFileName = rootFileName.concat("_"+recVersionFormat.format(new Integer(recVersion)));
        }
        return rootFileName;
    }
	
	/**
	 * Generates dynamic audio filename for the actual item based on the storage URL.
	 * In online mode, when <code>isUseAsCache()</code> is <b>true</b>, the following
	 * parameters are appended on the <code>storageURL</code> as query string: </br>
	 * <code>promptCode, speakerCode, extension, session, line</code> </br>
	 * In offline mode, prompt, speaker code etc. are appended on the storage URL as
	 * directory fragments to build a valid and individual filename.
	 * @param sessionID session ID
	 * @param speakerCode speaker code
	 * @param promptCode the item code
     * @param recVersion of the recording
	 * @return array of URLs, one entry for each audio line
	 * @throws StorageManagerException
	 */
	public URL[] generateAudioFileURLs(int sessionID,String speakerCode,String promptCode,int recVersion) throws StorageManagerException
	{
		String extension = uploadType.getExtension();
		URL[] audioFileURLs = new URL[numLines];
		URL sessionURL = getSessionURL(sessionID);
		URL newURL;
		String lineNr;
		String audioFileName;

		//try
		{
			for (int line = 0; line < numLines; line++)	// for all audio-channels
			{		
				
					newURL = sessionURL;
					
					audioFileName = getRootFileName(sessionID,promptCode, speakerCode,recVersion);
					
					if (numLines == 1) 
						lineNr = "";
					else
						lineNr = "_" + line;
					
					audioFileName += lineNr + "." + extension;
					newURL = addDirToURL(newURL, audioFileName);					
				
				audioFileURLs[line] = newURL;
			}
		}
//		catch (Exception ex)
//		{
//			String msg = "Exception: " + ex.getLocalizedMessage();
//			throw new StorageManagerException(msg, ex);
//		}		
		return audioFileURLs;
	}
	
	/**
	 * Checks if a particular version of an item is already recorded.
	 * @param promptCode code of the item
     * @param recVersion number
	 * @return true if a recording exists
	 * @throws StorageManagerException 
	 * @throws URISyntaxException 
	 */
	public boolean isRecorded(int sessionID,String speakerCode,String promptCode,int recVersion) throws StorageManagerException 
	{
		boolean recorded = true;


		URL[] urls = generateAudioFileURLs(sessionID,speakerCode,promptCode,recVersion); 

		for (int i = 0; i < numLines; i++) {
			String path;
			try {
				path = urls[i].toURI().getPath();
			} catch (URISyntaxException e) {
				e.printStackTrace();
				throw new StorageManagerException(e);
			}
			File f = new File(path);
			if (!f.exists()) {
				recorded=false;
				break;
			}
		}

		return recorded;
	}
	
    public boolean sessionHasRecordingFiles(Session sess) throws StorageManagerException{
        
        Script s=sess.getScript();
        List<PromptItem> pis=s.promptItemsList();
        for(PromptItem pi:pis){
            if(pi instanceof Recording){
                Recording r=(Recording)pi;
                int recVers=0;
                boolean recordingFound=isRecorded(sess.getSessionId(), sess.getCode(), r.getItemcode(), recVers);
                if(recordingFound){
                    return true;
                }
            }
        }
        return false;
    }
    
    public int getRecordedVersions(int sessionId,String sessionCode,String itemcode) throws StorageManagerException {
    	int testRecversion=0;
    	while(isRecorded(sessionId, sessionCode, itemcode, testRecversion)){
    		testRecversion++;
    	}     
    	return testRecversion;
    }
    
    private boolean updateRecFile(Session sess,Recording r,int recVers) throws StorageManagerException {
    	URL[] urls = generateAudioFileURLs(sess.getSessionId(), sess.getCode(), r.getItemcode(), recVers); 
    	boolean recorded=false;
    	for (int i = 0; i < numLines; i++) {
    		URL url=urls[i];
    		String path;
    		try {
    			path = url.toURI().getPath();
    		} catch (URISyntaxException e) {
    			e.printStackTrace();
    			throw new StorageManagerException(e);
    		}
    		File f = new File(path);

    		if (f.exists()) {
    			recorded=true;
    			RecordingFile rf=new RecordingFile();
    			rf.setSession(sess);
    			rf.setRecording(r);
    			rf.setSignalFile(url.toExternalForm());
    			// TODO audio format ??

    			sess.getRecordingFiles().add(rf);
    		}

    	}
    	return recorded;
    }
    
    public void updateRecFiles(Session sess,boolean recentVersionsOnly) throws StorageManagerException {

    	Script s=sess.getScript();
    	// revert lazy recording file settings
    	sess.setRecordingFiles(new HashSet<RecordingFile>());
    	sess.markHasRecordings(null);
    	List<PromptItem> pis=s.promptItemsList();
    	for(PromptItem pi:pis){
    		if(pi instanceof Recording){
    			Recording r=(Recording)pi;
    			if(overwrite) {
    				updateRecFile(sess, r,0);
    			}else {
    				if(recentVersionsOnly) {

    					int recVersions=getRecordedVersions(sess.getSessionId(), sess.getCode(), r.getItemcode());
    					if(recVersions>0) {
    						updateRecFile(sess, r, recVersions-1);
    					}
    				}else {
    					boolean recorded=false;
    					int recVers=0;
    					do {    				
    						recorded=updateRecFile(sess, r, recVers);
    						recVers++;
    					}while(recorded);
    				}
    			}
    		}
    	}
    }


    
    public Set<Session> sessionsLazy(Project prj,Script script) throws StorageManagerException{
        Set<Session> sSet=new HashSet<Session>();
//        Set<String> localSessionIds=null;
        URL storeUrl=getStorageURL();
        if("file".equalsIgnoreCase(storeUrl.getProtocol())){
//            localSessionIds=new HashSet<String>();

            File storeDir=null;
            try {
                storeDir = Utils.fileFromDecodedURL(storeUrl);
            } catch (UnsupportedEncodingException e1) {
                e1.printStackTrace();
                throw new StorageManagerException(e1);
            }
            if(storeDir.exists()){
                DirectoryStream.Filter<Path> sessionDirFilter = new DirectoryStream.Filter<Path>() {
                    public boolean accept(Path file) throws IOException {
                        if(file.toFile().isDirectory()){
                            String fn=(file.getName(file.getNameCount()-1)).toString();
                            try {
                                Number sessionNr=sessionIDFormat.parse(fn);
                                if(sessionNr!=null){
                                    return true;
                                }
                            } catch (ParseException e) {
                                return false;
                            }
                        }
                        return false;
                    }
                };
                Path storePath = storeDir.toPath();
                DirectoryStream<Path> storeDirStream =null;
                try {
                    storeDirStream = Files.newDirectoryStream(storePath, sessionDirFilter);
                    Iterator<Path> sessionPathsIt=storeDirStream.iterator();
                    while(sessionPathsIt.hasNext()){
                        Path sessionPath=sessionPathsIt.next();

                        Path sessionPathLast=sessionPath.getName(sessionPath.getNameCount()-1);
                        if(sessionPathLast!=null){
                            String sessionNm=sessionPathLast.toString();
                            if(sessionNm!=null){
                                Number sessionNr=sessionIDFormat.parse(sessionNm);
                                Session s=new Session();
                                s.setProject(prj);
                                s.setStorageDirectoryURL(sessionPath.toUri().toURL().toString());
                                s.setSessionId(sessionNr.intValue());
                                s.setScript(script);
                                sSet.add(s);
                            }
                        }

                    }

                } catch (IOException e) {
                    e.printStackTrace();
                    throw new StorageManagerException(e);
                } catch (ParseException e) {
                    e.printStackTrace();
                    throw new StorageManagerException(e);
                }finally{
                    if(storeDirStream!=null){
                        try {
                            storeDirStream.close();
                        } catch (IOException e) {
                            e.printStackTrace();
                            throw new StorageManagerException(e);
                        }
                    }
                }
            }
        } 
        return sSet; 
    }
    

	/**
	 * @return session ID format
	 */
	public NumberFormat getSessionIDFormat() {
		return sessionIDFormat;
	}



	/**
	 * @param format  session ID format
	 */
	public void setSessionIDFormat(NumberFormat format) {
		sessionIDFormat = format;
	}
    
	/**
	 * @return Returns the metadata.
	 */
	public MetaData getMetadata()
	{
		return metadata;
	}
	/**
	 * @param metadata The metadata to set.
	 */
	public void setMetadata(MetaData metadata)
	{
		this.metadata = metadata;
	}
	
	public boolean isUseScriptID() {
		return useScriptID;
	}

	
	public boolean isUseSpeakerID() {
		return useSessionID;
	}

	
	public void setUseScriptID(boolean b) {
		useScriptID = b;
	}

	
	public void setUseSpeakerID(boolean b) {
		useSessionID = b;
	}




}
