//    Speechrecorder
//    (c) Copyright 2009-2011
// 	  Institute of Phonetics and Speech Processing,
//    Ludwig-Maximilians-University, Munich, Germany
//
//
//    This file is part of Speechrecorder
//
//
//    Speechrecorder is free software: you can redistribute it and/or modify
//    it under the terms of the GNU Lesser General Public License as published by
//    the Free Software Foundation, version 3 of the License.
//
//    Speechrecorder is distributed in the hope that it will be useful,
//    but WITHOUT ANY WARRANTY; without even the implied warranty of
//    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
//    GNU Lesser General Public License for more details.
//
//    You should have received a copy of the GNU Lesser General Public License
//    along with Speechrecorder.  If not, see <http://www.gnu.org/licenses/>.

/*
 * Date  : Jan 15, 2008
 * Author: K.Jaensch, klausj@phonetik.uni-muenchen.de
 */

package ipsk.apps.speechrecorder.workspace;

import ipsk.apps.speechrecorder.PluginLoadingException;
import ipsk.apps.speechrecorder.SpeechRecorder;
import ipsk.apps.speechrecorder.SpeechRecorderException;
import ipsk.apps.speechrecorder.config.ConfigHelper;
import ipsk.apps.speechrecorder.config.ProjectConfiguration;
import ipsk.apps.speechrecorder.config.PromptConfiguration;
import ipsk.apps.speechrecorder.config.RecordingConfiguration;
import ipsk.apps.speechrecorder.config.SpeakersConfiguration;
import ipsk.apps.speechrecorder.config.WorkspaceProject;
import ipsk.apps.speechrecorder.db.export.emu.EmuExportWorker;
import ipsk.apps.speechrecorder.db.export.emu.EmuExporter;
import ipsk.apps.speechrecorder.project.ProjectManager;
import ipsk.apps.speechrecorder.project.ProjectManagerException;
import ipsk.apps.speechrecorder.script.RecscriptHandler;
import ipsk.apps.speechrecorder.script.RecscriptHandlerException;
import ipsk.apps.speechrecorder.script.RecscriptManagerException;
import ipsk.apps.speechrecorder.storage.StorageManagerException;
import ipsk.audio.AudioControllerException;
import ipsk.awt.WorkerException;
import ipsk.beans.DOMCodec;
import ipsk.beans.DOMCodecException;
import ipsk.db.speech.Script;
import ipsk.io.FileUtils;
import ipsk.net.URLContext;
import ipsk.net.Utils;
import ipsk.swing.JProgressDialogPanel;
import ipsk.util.ProgressStatus;
import ipsk.util.zip.ZipPackerWorker;
import ipsk.xml.DOMConverter;
import ipsk.xml.DOMConverterException;

import java.beans.PropertyChangeListener;
import java.beans.PropertyChangeSupport;
import java.io.File;
import java.io.FileInputStream;
import java.io.FileOutputStream;
import java.io.IOException;
import java.io.OutputStreamWriter;
import java.net.MalformedURLException;
import java.net.URI;
import java.net.URISyntaxException;
import java.net.URL;
import java.nio.charset.Charset;
import java.nio.file.Path;
import java.util.ArrayList;
import java.util.HashSet;
import java.util.List;

import javax.swing.AbstractListModel;
import javax.swing.JFrame;
import javax.swing.JOptionPane;
import javax.xml.parsers.ParserConfigurationException;

import org.w3c.dom.Document;

/**
 * Workspace manager.
 * 
 * @author K.Jaensch, klausj@phonetik.uni-muenchen.de
 * 
 */

public class WorkspaceManager extends AbstractListModel<String> {

	private static final long serialVersionUID = 1L;

    public static final String PROJECT_CONFIG_PACKAGE = "ipsk.apps.speechrecorder.config.ProjectConfiguration";

    public static final int COL_NAME=0;
    //private ProjectManager sessionManager;
    private File workspaceDir = null;
    private SpeechRecorder speechRecorder;
    private List<WorkspaceProject> workspaceProjects=new ArrayList<WorkspaceProject>();
    private PropertyChangeSupport propertyChangeSupport;
    private HashSet<String> lockedProjects=new HashSet<String>();
    
    
    public WorkspaceManager(File workspaceDir,SpeechRecorder speechRecorder) {
        super();
    	this.speechRecorder=speechRecorder;
        this.workspaceDir = workspaceDir;
        propertyChangeSupport=new PropertyChangeSupport(this);
    }

    public void lock(String projectName){
        lockedProjects.add(projectName);
    }
    
    public void unlock(String projectName){
        lockedProjects.remove(projectName);
    }
    
    public boolean locked(String projectName){
        return(lockedProjects.contains(projectName));
    }
    
    public boolean hasLock() {
    	return !lockedProjects.isEmpty();
    }
    
    public boolean locked(WorkspaceProject workspaceProject){
        if(workspaceProject!=null){
        	ProjectConfiguration prjCfg=workspaceProject.getConfiguration();
        	if(prjCfg!=null) {
        		String projectName=prjCfg.getName();
        		 return(locked(projectName));
        	}  
        }
        return false;
    }
    
//    private File projectConfigFile(String projectName) throws WorkspaceException{
//        File projectDir=new File(workspaceDir,projectName);
//        File projectConfigFile=null;
//        File[] projectFiles = projectDir.listFiles();
//        if(projectFiles!=null){
//
//            for (int j = 0; j < projectFiles.length; j++) {
//
//                File projectFile = projectFiles[j];
//                if (projectFile.isFile() && ! projectFile.isHidden()) {
//                    if (projectFile.getName().endsWith(
//                            SpeechRecorder.PROJECT_FILE_EXTENSION)) {
//                        if(projectConfigFile!=null){
//                            throw new WorkspaceException("Multiple project file candidates in "+projectDir+": "+projectConfigFile.getName()+" ,"+projectFile.getName());
//                        }
//                        projectConfigFile=projectFile;
//                        // do not break here to continue search for multiple (ambigious config) files
//                    }
//                }
//            }
//        }
//        return projectConfigFile;
//    }
    
    public WorkspaceProject[] scanWorkspace() throws WorkspaceException {
    	fireIntervalRemoved(this, 0, workspaceProjects.size());
    	DOMConverter domConverter;
    	
    	domConverter = new DOMConverter();
    	Package configBasePack;
    	try {
    		configBasePack = Class.forName(PROJECT_CONFIG_PACKAGE).getPackage();
    	} catch (ClassNotFoundException e1) {
    		e1.printStackTrace();
    		throw new WorkspaceException("Package " + PROJECT_CONFIG_PACKAGE,
    				e1);
    	}

    	DOMCodec domCodec;
    	try {
    		domCodec = new DOMCodec(configBasePack);
    	} catch (DOMCodecException e1) {
    		e1.printStackTrace();
    		throw new WorkspaceException("Could not create DOM codec", e1);
    	}
    	List<WorkspaceProject> newWorkspaceProjectsList = new ArrayList<WorkspaceProject>();
    	if (workspaceDir.exists()) {
    		File[] projectDirs = workspaceDir.listFiles();
    		if(projectDirs!=null){
    			for (int i = 0; i < projectDirs.length; i++) {
    				File projectDir=projectDirs[i];
    				if (projectDir!=null && projectDir.isDirectory() && ! projectDir.isHidden()) {
    					File projectConfigFile=null;
    					String projNameByDirName=projectDir.getName();
    					String projName=projNameByDirName;
    					String standardProjectFilename=projNameByDirName+SpeechRecorder.PROJECT_FILE_EXTENSION;
    					File standardProjectFile=new File(projectDir,standardProjectFilename);
    					if(standardProjectFile.exists()){
    						projectConfigFile=standardProjectFile;
    					}
    					if(projectConfigFile==null){
    						File[] projectFiles = projectDir.listFiles();
    						if(projectFiles!=null){

    							for (int j = 0; j < projectFiles.length; j++) {

    								File projectFile = projectFiles[j];
    								if (projectFile.isFile() && ! projectFile.isHidden()) {
    									if (projectFile.getName().endsWith(
    											SpeechRecorder.PROJECT_FILE_EXTENSION)) {
    										if(projectConfigFile!=null){
    											throw new WorkspaceException("Multiple project file candidates in "+projNameByDirName+": "+projectConfigFile.getName()+" ,"+projectFile.getName());
    										}
    										projectConfigFile=projectFile;
    										// do not break here to continue search for multiple (ambigious config) files
    									}
    								}
    							}
    						}
    					}
    					if(projectConfigFile!=null){
    						ProjectConfiguration p = null;
    						try {
    							Document d = domConverter
    									.readXML(new FileInputStream(
    											projectConfigFile));
    							p = (ProjectConfiguration) domCodec
    									.readDocument(d);
    							projName=p.getName();
    						} catch (Exception e) {
//    							e.printStackTrace();
//    							JOptionPane.showMessageDialog(null,
//    									"Cannot create project from "
//    											+ projectConfigFile.getPath()
//    											+ " .\n"
//    											+ e.getLocalizedMessage(),
//    											"Warning !",
//    											JOptionPane.WARNING_MESSAGE);
//    							continue;
    						    // new policy: fail silently here and list the project
    						}
    						WorkspaceProject wp = new WorkspaceProject(projName,p,projectConfigFile);
    						newWorkspaceProjectsList.add(wp);
    					}
    				}
    			}
    		}
    	}
    	List<WorkspaceProject> oldWorkspaceProjects=workspaceProjects;
    	workspaceProjects=newWorkspaceProjectsList;
    	propertyChangeSupport.firePropertyChange("workspaceProjects",oldWorkspaceProjects, workspaceProjects);
//    	fireTableDataChanged();
    	fireIntervalAdded(this, 0, workspaceProjects.size());
    	return newWorkspaceProjectsList.toArray(new WorkspaceProject[0]);
    }
    
    
    
    public String getColumnName(int colIndex){
        if(colIndex==COL_NAME){
            return "Name";
        }else return null;
    }
    
    public Class<?> getColumnClass(int colIndex){
        if(colIndex==COL_NAME){
            return String.class;
        }else return null;
    }

//    /* (non-Javadoc)
//     * @see javax.swing.table.TableModel#getColumnCount()
//     */
//    @Override
//    public int getColumnCount() {
//       
//        return 1;
//    }
//
//    /* (non-Javadoc)
//     * @see javax.swing.table.TableModel#getRowCount()
//     */
//    @Override
//    public int getRowCount() {
//        
//        return workspaceProjects.size();
//    }

//    /* (non-Javadoc)
//     * @see javax.swing.table.TableModel#getValueAt(int, int)
//     */
//    @Override
//    public Object getValueAt(int rowIndex, int colIndex) {
//        if(colIndex==0){
//            WorkspaceProject proj=workspaceProjects.get(rowIndex);
//            ProjectConfiguration pCfg=proj.getConfiguration();
//            return pCfg.getName();
//        }else{
//            return null;
//        }
//    }

    public File getWorkspaceDir() {
        return workspaceDir;
    }

    public List<WorkspaceProject> getWorkspaceProjects() {
        return workspaceProjects;
    }
    
    public WorkspaceProject projectByName(String name){
        for(WorkspaceProject wp:workspaceProjects){
            ProjectConfiguration pc=wp.getConfiguration();
            if(pc!=null){
                String wpName=pc.getName();
                if(wpName.equals(name)){
                    return wp;
                }
            }
        }
        return null;
    }
    
    public void openProject(String projectName) throws WorkspaceException {
    	try {
			this.speechRecorder.openProject(projectName);
		} catch (SpeechRecorderException e) {
			throw new WorkspaceException(e);
		}
    }
    
    public void exportToEmuDB(URL projectContext,ProjectConfiguration prCfg,Path outDir) throws WorkspaceException {
    	EmuExportWorker emuExporter=new EmuExportWorker();
    	// TODO listener and GUI
    	emuExporter.setProjectContext(projectContext);
    	try {
			emuExporter.config(prCfg);
//			emuExporter.exportProjectAsEmuDB(outDir);
			emuExporter.setExportBasepath(outDir);
			JProgressDialogPanel progressDialog=new JProgressDialogPanel(emuExporter,"Export project to EmuDB","Exporting...");
			emuExporter.open();
			emuExporter.start();
			JFrame parent=null;
			Object val=progressDialog.showDialog(parent);
			ProgressStatus finalStatus=emuExporter.getProgressStatus();
			emuExporter.close();
	        if(val.equals(JProgressDialogPanel.CANCEL_OPTION)){
	            finalStatus.canceled();
	        }
	        if(!finalStatus.isDone()){
	            // delete (partial) emu Dir
	           
	            if(finalStatus.isError()){
	                JOptionPane.showMessageDialog(null,finalStatus.getMessage(), "EMU export error", JOptionPane.ERROR_MESSAGE);
	            }
	        }
			
		} catch ( ProjectManagerException | WorkerException e) {
			
			e.printStackTrace();
			JOptionPane.showMessageDialog(null,e.getMessage(), "EMU export ", JOptionPane.ERROR_MESSAGE);
		}
    }
    
    public void renameProject(String oldName,String newName) throws WorkspaceException{
        WorkspaceProject wp=projectByName(oldName);
        
        ProjectConfiguration wpPc=wp.getConfiguration();
        
        if(locked(wp)){
            throw new WorkspaceException("Project "+wpPc.getName()+" is in use.");
        }

        File wsPDir=new File(workspaceDir,oldName);
       
        URI oldPrjDirURI=wsPDir.toURI();
        URL oldPrjDirURL;
        try {
            oldPrjDirURL = oldPrjDirURI.toURL();
        } catch (MalformedURLException e2) {
            e2.printStackTrace();
            throw new WorkspaceException(e2);
        }
        URLContext oldPrjCtx=new URLContext(oldPrjDirURL);
        
        // new project directory,URI and URL
        File newWsPDir=new File(workspaceDir,newName);
        if(newWsPDir.exists()){
            throw new WorkspaceException("Could not rename project: Path "+newWsPDir+" already exists!");
        }
        
        // rename project directory
        boolean renamed=wsPDir.renameTo(newWsPDir);
        if(!renamed){
            throw new WorkspaceException("Could not rename folder "+wsPDir+" to "+newWsPDir);
        }
      
        // Important note: the directory must exist in order to get a "directory" URL whose path is ending with a slash.
        URL dirURL=null;
        
        URI dirURI=newWsPDir.toURI();
        try {
            dirURL = dirURI.toURL();
        } catch (MalformedURLException e1) {
        	throw new WorkspaceException("Could not convert to URL: "+dirURL+": "+e1.getMessage());
        }
        // set new name in project config
        wpPc.setName(newName);
        
        //check recsURL for absolute URL pointing inside project dir, fixes the recordings URL part of  BUG ID0057)
        RecordingConfiguration recCfg=wpPc.getRecordingConfiguration();
        String recsUrlStr=recCfg.getUrl();
        String newRecsUrlStr;
        try {
            newRecsUrlStr = oldPrjCtx.renameContextSpec(dirURL, recsUrlStr);
        } catch (MalformedURLException e2) {
            e2.printStackTrace();
            throw new WorkspaceException(e2);
        }
        recCfg.setUrl(newRecsUrlStr);
            
        File oldProjectFile=new File(newWsPDir,oldName+SpeechRecorder.PROJECT_FILE_EXTENSION);  
        File newProjectFile=new File(newWsPDir,newName+SpeechRecorder.PROJECT_FILE_EXTENSION);
        
        ConfigHelper ch=new ConfigHelper();
        
        try {
            ch.writeConfig(wpPc, newProjectFile);
        } catch (Exception e1) {
            e1.printStackTrace();
            throw new WorkspaceException(e1);
            
        }
        
        // rename old project config file as backup file
        File projectOldBkpFile=FileUtils.moveToBackup(oldProjectFile, ".bak");
        

        // should already work ...
        scanWorkspace();
        
        // Rename speaker db and script file as well
        
        // Three cases to consider
        // relative file spec -> OK
        // absolute URL outside project dir -> OK
        // absolute URL inside project dir -> rename
        
        // then rename standard file names beginning with project name
        
        // speaker db file:
        
        WorkspaceProject newWsp=projectByName(newName);
        ProjectConfiguration pc=newWsp.getConfiguration();
        try {

            SpeakersConfiguration spksCfg=pc.getSpeakers();
            String spksUrl=spksCfg.getSpeakersUrl();
            if (spksUrl != null && !spksUrl.equals("")) {
                
                // check for absolute URLs pointing into project folder
                String newSpksSpec=oldPrjCtx.renameContextSpec(dirURL, spksUrl);
                
                if(!spksUrl.equals(newSpksSpec)){
                    spksCfg.setSpeakersUrl(newSpksSpec);
                    ch.writeConfig(pc, newProjectFile);
                }
                URL speakerURL = URLContext.getContextURL(dirURL,newSpksSpec);
                File spksFile=Utils.fileFromDecodedURL(speakerURL);
                if(spksFile!=null){
                    // check default location in top level
                    File spksParent=spksFile.getParentFile();
                   
                    if(newWsPDir.equals(spksParent)){
                        String oldSpksFileName=spksFile.getName();
                        if(oldSpksFileName.startsWith(oldName)){
                            // the filename starts with project name (likely the default speakers db file)
                            // try to rename
                            String newSpksFn=oldSpksFileName.replaceFirst("^"+oldName, newName);
                            File newSpksFile=new File(spksParent,newSpksFn);
                            if(!newSpksFile.exists()){
                                if(spksFile.renameTo(newSpksFile)){
                                    spksCfg.setSpeakersUrl(newSpksFn);
                                    ch.writeConfig(pc, newProjectFile);
                                }
                            }
                        }
                    }
                }
            }
            
            // prompt script file
            
            PromptConfiguration prCfg=pc.getPromptConfiguration();
            String prUrl=prCfg.getPromptsUrl();
            if (prUrl != null && !prUrl.equals("")) {
                
                // check for absolute URLs pointing into project folder, Fixes the prompt script part of  BUG ID0057
                String newPromptSpec=oldPrjCtx.renameContextSpec(dirURL, prUrl);
                
                if(!prUrl.equals(newPromptSpec)){
                    prCfg.setPromptsUrl(newPromptSpec);
                    ch.writeConfig(pc, newProjectFile);
                }
                
                // check default location in top level
                URL promptsURL = URLContext.getContextURL(dirURL,newPromptSpec);
                File promptsFile=Utils.fileFromDecodedURL(promptsURL);
                if(promptsFile!=null){
                    
                    // check the script for absolute prompt source URLs referencing into project directory (Fixes the prompt items part of  BUG ID0057)
                    RecscriptHandler scriptHandler=new RecscriptHandler();
                 
                    String contextUrlStr=Utils.createAsciiURLExtFormFromFile(newWsPDir);
                    scriptHandler.setSystemIdBase(contextUrlStr);
                    try {
                        scriptHandler.renameContext(promptsFile, oldPrjDirURL, dirURL);
                    } catch (RecscriptHandlerException e) {
                        e.printStackTrace();
                        throw new WorkspaceException("Could not rename recording script context: "+e.getLocalizedMessage(),e);
                    }
                    
                    File promptsParent=promptsFile.getParentFile();
                    if(newWsPDir.equals(promptsParent)){
                        String oldPromptsFileName=promptsFile.getName();
                        if(oldPromptsFileName.startsWith(oldName)){
                            // the filename starts with project name (likely the default script file )
                            // try to rename
                            String newPromptsFn=oldPromptsFileName.replaceFirst("^"+oldName, newName);
                            File newPromptsFile=new File(promptsParent,newPromptsFn);
                            if(!newPromptsFile.exists()){
                                if(promptsFile.renameTo(newPromptsFile)){
                                    prCfg.setPromptsUrl(newPromptsFn);
                                    ch.writeConfig(pc, newProjectFile);
                                }
                            }
                        }
                    }
                }
                
            
                
            }
            
            
            
        } catch (MalformedURLException e) {
            throw new WorkspaceException(e);
        } catch (DOMCodecException e) {
        	throw new WorkspaceException(e);
        } catch (DOMConverterException e) {
        	throw new WorkspaceException(e);
        } catch (IOException e) {
        	throw new WorkspaceException(e);
        } catch (URISyntaxException e) {
            throw new WorkspaceException(e);
        }
        if(projectOldBkpFile!=null){
            projectOldBkpFile.delete();
        }
        
        
    }

    public void addPropertyChangeListener(PropertyChangeListener listener) {
        propertyChangeSupport.addPropertyChangeListener(listener);
    }

    public void addPropertyChangeListener(String propertyName,
            PropertyChangeListener listener) {
        propertyChangeSupport.addPropertyChangeListener(propertyName, listener);
    }

    public void removePropertyChangeListener(PropertyChangeListener listener) {
        propertyChangeSupport.removePropertyChangeListener(listener);
    }

    public void removePropertyChangeListener(String propertyName,
            PropertyChangeListener listener) {
        propertyChangeSupport.removePropertyChangeListener(propertyName,
                listener);
    }

    /* (non-Javadoc)
     * @see javax.swing.ListModel#getElementAt(int)
     */
    @Override
    public String getElementAt(int rowIndex) {
    	WorkspaceProject proj=workspaceProjects.get(rowIndex);
    	return proj.getName();
    }

	/* (non-Javadoc)
	 * @see javax.swing.ListModel#getSize()
	 */
	@Override
	public int getSize() {
		return workspaceProjects.size();
	}

	public void closeActiveProject() throws WorkspaceException {
		try {
			speechRecorder.closeActiveProject();
		} catch (SpeechRecorderException e) {
			throw new WorkspaceException(e);
		}
	}

	public void duplicateProject(String currentName, String newName) throws WorkspaceException {
		  WorkspaceProject wp=projectByName(currentName);
	        
	        ProjectConfiguration wpPc=wp.getConfiguration();
	        
	        // TODO check if project and script do not contain absolute references
	        
	        if(locked(wp)){
	            throw new WorkspaceException("Project "+wpPc.getName()+" is in use.");
	        }

	        File wsPDir=new File(workspaceDir,currentName);
	       
	        URI oldPrjDirURI=wsPDir.toURI();
	        URL oldPrjDirURL;
	        try {
	            oldPrjDirURL = oldPrjDirURI.toURL();
	        } catch (MalformedURLException e2) {
	            e2.printStackTrace();
	            throw new WorkspaceException(e2);
	        }
	        URLContext oldPrjCtx=new URLContext(oldPrjDirURL);
	        
	        // new project directory,URI and URL
	        File newWsPDir=new File(workspaceDir,newName);
	        if(newWsPDir.exists()){
	            throw new WorkspaceException("Could not duplicate project: Path "+newWsPDir+" already exists!");
	        }
	        
	        // create project directory
	        try {
	        	boolean created=newWsPDir.createNewFile();
	        	if(!created){
	        		throw new WorkspaceException("Could not create folder "+newWsPDir);
	        	}
	        }catch(IOException ioe) {
	        	throw new WorkspaceException("Could not create folder "+newWsPDir);
	        }

	        // Important note: the directory must exist in order to get a "directory" URL whose path is ending with a slash.
	        URL dirURL=null;
	        
	        URI dirURI=newWsPDir.toURI();
	        try {
	            dirURL = dirURI.toURL();
	        } catch (MalformedURLException e1) {
	        	throw new WorkspaceException("Could not convert to URL: "+dirURL+": "+e1.getMessage());
	        }
	        // set new name in project config
	        wpPc.setName(newName);
	        
	        //check recsURL for absolute URL pointing inside project dir, fixes the recordings URL part of  BUG ID0057)
	        RecordingConfiguration recCfg=wpPc.getRecordingConfiguration();
	        String recsUrlStr=recCfg.getUrl();
	        String newRecsUrlStr;
	        try {
	            newRecsUrlStr = oldPrjCtx.renameContextSpec(dirURL, recsUrlStr);
	        } catch (MalformedURLException e2) {
	            e2.printStackTrace();
	            throw new WorkspaceException(e2);
	        }
	        recCfg.setUrl(newRecsUrlStr);
	            
	        File oldProjectFile=new File(newWsPDir,currentName+SpeechRecorder.PROJECT_FILE_EXTENSION);  
	        File newProjectFile=new File(newWsPDir,newName+SpeechRecorder.PROJECT_FILE_EXTENSION);
	        
	        ConfigHelper ch=new ConfigHelper();
	        
	        try {
	            ch.writeConfig(wpPc, newProjectFile);
	        } catch (Exception e1) {
	            e1.printStackTrace();
	            throw new WorkspaceException(e1);
	            
	        }
	        
	        // should already work ...
	        scanWorkspace();
	        
	        //TODO copy script and/or speaker db 
	}

//	/**
//	 * @param selProj
//	 * @throws WorkspaceException 
//	 */
//	public void openProject(WorkspaceProject selProj) throws WorkspaceException {
//		try {
//			sessionManager.openProject(selProj.getConfiguration().getName());
//		} catch (Exception e) {
//			
//			e.printStackTrace();
//			throw new WorkspaceException(e);
//		} 
//	}



}
