//    Speechrecorder
// 	  (c) Copyright 2017
// 	  Institute of Phonetics and Speech Processing,
//    Ludwig-Maximilians-University, Munich, Germany
//
//
//    This file is part of Speechrecorder
//
//
//    Speechrecorder is free software: you can redistribute it and/or modify
//    it under the terms of the GNU Lesser General Public License as published by
//    the Free Software Foundation, version 3 of the License.
//
//    Speechrecorder is distributed in the hope that it will be useful,
//    but WITHOUT ANY WARRANTY; without even the implied warranty of
//    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
//    GNU Lesser General Public License for more details.
//
//    You should have received a copy of the GNU Lesser General Public License
//    along with Speechrecorder.  If not, see <http://www.gnu.org/licenses/>.

package ipsk.apps.speechrecorder.project;

import java.io.File;
import java.io.FileNotFoundException;
import java.io.FileOutputStream;
import java.io.IOException;
import java.io.InputStream;
import java.io.OutputStreamWriter;
import java.io.StringReader;
import java.io.StringWriter;
import java.net.MalformedURLException;
import java.net.URI;
import java.net.URISyntaxException;
import java.net.URL;
import java.nio.charset.Charset;
import java.nio.file.Files;
import java.util.ArrayList;
import java.util.List;
import java.util.Set;

import javax.xml.bind.JAXB;
import javax.xml.parsers.ParserConfigurationException;

import org.w3c.dom.Document;

import ipsk.apps.speechrecorder.SpeakerManager;
import ipsk.apps.speechrecorder.config.ItemcodeGeneratorConfiguration;
import ipsk.apps.speechrecorder.config.ProjectConfiguration;
import ipsk.apps.speechrecorder.config.PromptConfiguration;
import ipsk.apps.speechrecorder.config.RecordingConfiguration;
import ipsk.apps.speechrecorder.db.Speaker;
import ipsk.apps.speechrecorder.script.RecScriptManager;
import ipsk.apps.speechrecorder.script.RecscriptHandler;
import ipsk.apps.speechrecorder.script.RecscriptManagerException;
import ipsk.apps.speechrecorder.storage.SessionStorageManager;
import ipsk.apps.speechrecorder.storage.StorageManagerException;
import ipsk.beans.DOMCodec;
import ipsk.beans.DOMCodecException;
import ipsk.db.speech.Project;
import ipsk.db.speech.Script;
import ipsk.io.FileUtils;
import ipsk.io.StreamCopy;
import ipsk.net.URLContext;
import ipsk.xml.DOMConverter;
import ipsk.xml.DOMConverterException;

/**
 * @author klausj
 *
 */
public class ProjectManager {

	// version of project configuration
	// increase value if project configuration schema changes
	// the schema is only defined by Java code classes (there is no DTD or XML
	// Schema yet)
	public final static String PROJECT_VERSION = "4.0.0";

	public static final String PROJECT_FILE_EXTENSION = "_project.prj";
	public static final String SPEAKER_FILE_SUFFIX = "_speakers.xml";
	public static final String REC_SCRIPT_FILE_EXTENSION = "_script.xml";
	public static final String DEFAULT_RESOURCE_PATH = "resources";

	public static final String REC_SCRIPT_EXAMPLE = "ExampleRecScript_3.xml";

	protected Project projectDb;
	protected ProjectConfiguration project;
	protected URL projectContext = null;
	protected URL projectURL;

	protected DOMCodec domCodec;
	protected DOMConverter domConverter;

	protected SpeakerManager speakerManager;
	protected String speakerFileName;
	protected URL speakerURL;

	protected SessionStorageManager projectStorageManager;

	public SessionStorageManager getProjectStorageManager() {
		return projectStorageManager;
	}

	protected RecScriptManager recScriptManager;
	protected URL promptFile;
	protected URL recBaseURL;
	protected int numLines;

	/**
	 * @throws ClassNotFoundException
	 * @throws DOMCodecException
	 * 
	 */
	public ProjectManager() throws ProjectManagerException {
		super();
		Package configBasePack;
		try {
			configBasePack = Class.forName("ipsk.apps.speechrecorder.config.ProjectConfiguration").getPackage();
			domCodec = new DOMCodec(configBasePack);
		} catch (ClassNotFoundException | DOMCodecException e) {
			throw new ProjectManagerException(e);
		}
		domConverter = new DOMConverter();
		speakerManager = new SpeakerManager();
		recScriptManager = new RecScriptManager();
		projectStorageManager = new SessionStorageManager();
	}

	/**
	 * 
	 */
	public void init() {

	}

	/**
	 * @return project context URL
	 */
	public URL getProjectContext() {
		return projectContext;
	}

	/**
	 * @param context the project context (directory) URL
	 */
	public void setProjectContext(URL context) {
		projectContext = context;
	}

	/**
	 * Sets a new configuration. Does not reconfigure the application.
	 * 
	 * @param newProject
	 */
	protected void setConfiguration(ProjectConfiguration newProject) {
		project = newProject;
	}

	/**
	 * Returns the project configuration (bean).
	 * 
	 * @return the project configuration
	 */
	public ProjectConfiguration getConfiguration() {
		return project;
	}

	public void config(ProjectConfiguration cfgProject) throws ProjectManagerException {
		try {
			commonConfig(cfgProject);
		} catch (MalformedURLException e1) {

			e1.printStackTrace();
			throw new ProjectManagerException(e1);
		}
		PromptConfiguration pc = cfgProject.getPromptConfiguration();
		String promptFileName = pc.getPromptsUrl();

		// set instance variables to the values given as arguments
		if (promptFileName != null && !promptFileName.equals("")) {

			try {
				promptFile = URLContext.getContextURL(projectContext, promptFileName);
				recScriptManager.load(promptFile);
			} catch (MalformedURLException | RecscriptManagerException e) {
				e.printStackTrace();
				throw new ProjectManagerException(e);
			}
		} else {
			throw new ProjectManagerException("Prompt file not configured");
		}
		String newSpeakerFilename = project.getSpeakers().getSpeakersUrl();
		if (speakerFileName == null || !speakerFileName.equals(newSpeakerFilename)) {
			// speakers URL has changed
			speakerFileName = project.getSpeakers().getSpeakersUrl();

			if (speakerFileName != null && !speakerFileName.equals("")) {
				try {
					speakerURL = URLContext.getContextURL(projectContext, speakerFileName);
				} catch (MalformedURLException e) {

					e.printStackTrace();
					throw new ProjectManagerException(e);
				}
			} else {
				throw new ProjectManagerException("Speaker database not configured");
			}

			// storageManager.setSpeakerCode(speakerManager.getSpeaker().getCode());
			// currSpeaker=null;
		}
		// Speaker database is closed when project is closed
		// therefore we always have to (re)load speaker db
		speakerManager.loadURL(speakerURL);

	}

	protected void commonConfig(ProjectConfiguration cfgProject) throws MalformedURLException {
		projectDb = new Project();
		projectDb.setName(cfgProject.getName());
		project = cfgProject;
		recScriptManager.setContext(projectContext);
		RecordingConfiguration recCfg = project.getRecordingConfiguration();
		String recDirName = recCfg.getUrl();
		recBaseURL = URLContext.getContextURL(projectContext, recDirName);
		projectStorageManager.setStorageURL(recBaseURL);
		numLines = 1;
		projectStorageManager.setNumLines(numLines);
		boolean overwrite = recCfg.getOverwrite();
		projectStorageManager.setOverwrite(overwrite);

		// Commented out: At this point the configuration is already parsed and
		// Speechrecorder should run this project without problems.
		//
//			// if the project configuration could be parsed and converted
//			// it is OK if the version is higher than the running instance
//			String projVerStr=project.getVersion();
//			try {
//	            Version version=Version.parseString(projVerStr);
//	            Version prjVersion=Version.parseString(PROJECT_VERSION);
//	            if(version.compareTo(prjVersion)>0){
//	                errMsg="Unable to load project!\nVersion of Speechrecorder ("+PROJECT_VERSION+") is less than project version ("+version+").\nPlease update Speechrecorder to load this project.";
//	                throw new SpeechRecorderException(errMsg);
//	            }
//	        } catch (ParserException e1) {
//	            errMsg="Could not parse project version: "+projVerStr;
//	            throw new SpeechRecorderException(errMsg);
//	        }

	}

	/**
	 * Returns the project configuration URL.
	 * 
	 * @return the URL where the configuration is stored
	 */
	public URL getProjectURL() {
		return projectURL;
	}

	/**
	 * Sets new project configuration URL.
	 * 
	 * @param url the URL where the configuration is stored
	 */
	public void setProjectURL(URL url) {
		projectURL = url;
	}

	public File getProjectDir() throws MalformedURLException, URISyntaxException {
		ProjectConfiguration pc = getConfiguration();
		if (pc != null) {
			String dirStr = pc.getDirectory();
			URL projectDirURL = new URL(projectContext, dirStr);
			return new File(projectDirURL.toURI().getPath());
		} else {

			return null;
		}
	}

	public Project getProjectDb() {
		return projectDb;
	}

	public void createEmptySpeakerDatabase(File speakerDbFile, boolean overwrite) {

		if (overwrite || !speakerDbFile.exists()) {
			JAXB.marshal(new ArrayList<ipsk.db.speech.Speaker>(), speakerDbFile);
		}
	}

	public void rebuildDb() throws StorageManagerException {
		Script script = recScriptManager.getScript();
		projectDb.getScripts().clear();
		projectDb.getScripts().add(script);
		script.getSessions().clear();
		// build lazy in mem DB
//	     Set<ipsk.db.speech.Session> sesss=new SessionsSetProxy(storageManager, projectDb,script);
		Set<ipsk.db.speech.Session> sesss = projectStorageManager.sessionsLazy(projectDb, script);
		projectDb.setSessions(sesss);
		List<Speaker> speakersList = speakerManager.getSpeakersList();
		for (ipsk.apps.speechrecorder.db.Speaker spk : speakersList) {
			spk.getSessions().clear();
		}
		for (ipsk.db.speech.Session s : sesss) {
			s.setScript(script);
			script.getSessions().add(s);
			s.getSpeakers().clear();
			// create speaker/session relationship and set speaker code to session
			for (ipsk.apps.speechrecorder.db.Speaker spk : speakersList) {
				if (spk.getPersonId().equals(s.getSessionId())) {
					s.setCode(spk.getCode());
					s.getSpeakers().add(spk);
					spk.getSessions().add(s);
				}
			}
			// load recording files lazy
			boolean hasRecs = projectStorageManager.sessionHasRecordingFiles(s);
			s.setRecordingFiles(null);
			s.markHasRecordings(hasRecs);

		}
		speakerManager.setSessions(sesss);

	}

	/**
	 * Save current project configuration.
	 * 
	 * @throws ProjectManagerException
	 * 
	 * @throws DOMCodecException
	 * @throws DOMConverterException
	 * @throws IOException
	 * @throws URISyntaxException
	 */
	public void saveProject() throws ProjectManagerException {
		// save project file
		String fPath;
		try {
			fPath = getProjectURL().toURI().getPath();
		} catch (URISyntaxException e) {
			throw new ProjectManagerException(e);
		}
		File f = new File(fPath);

// 		File backupF=new File(fPath+".bak");
// 		f.renameTo(backupF);
		ProjectConfiguration pc = getConfiguration();
// 		String pcVersion=pc.getVersion();
// 		if(pcVersion==null || PROJECT_VERSION.equals(pcVersion)){

		// always set to the version which created the configuration
		pc.setVersion(PROJECT_VERSION);
// 		}
		Document d;
		try {
			d = domCodec.createDocument(getConfiguration());
		} catch (DOMCodecException e) {
			throw new ProjectManagerException(e);
		}
		// backup
		FileUtils.moveToBackup(f, ".bak");
		FileOutputStream fos;
		try {
			fos = new FileOutputStream(f);
			OutputStreamWriter ow = new OutputStreamWriter(fos, Charset.forName("UTF-8"));
			domConverter.writeXML(d, ow);
			ow.close();
		} catch (DOMConverterException | IOException e) {
			throw new ProjectManagerException(e);
		}

	}

	/**
	 * Save current project configuration.
	 * 
	 * @throws ProjectManagerException
	 */
	public void saveScript() throws ProjectManagerException {

		RecscriptHandler recScriptHandler = new RecscriptHandler();
		recScriptHandler.setValidating(true);

		Script script = recScriptManager.getScript();

		// Write DTD file to project workspace if it does not exist

		File dtdFile;
		try {
			dtdFile = new File(getProjectDir(), RecscriptHandler.REC_SCRIPT_DTD);
		} catch (MalformedURLException | URISyntaxException e) {
			throw new ProjectManagerException(e);
		}
		if (!dtdFile.exists()) {
			InputStream is = RecscriptHandler.class.getResourceAsStream(RecscriptHandler.REC_SCRIPT_DTD);
			FileOutputStream fos;
			try {
				fos = new FileOutputStream(dtdFile);
				StreamCopy.copy(is, fos);
			} catch (IOException e) {
				throw new ProjectManagerException(e);
			}
		}

		// Validate

		StringWriter stringWriter = new StringWriter();
		try {
			recScriptHandler.writeXML(script, stringWriter);
			StringReader stringReader = new StringReader(stringWriter.toString());
			recScriptHandler.readScriptFromXML(stringReader, projectContext.toExternalForm());
		} catch (DOMConverterException | ParserConfigurationException e) {
			throw new ProjectManagerException(e);
		}

		// Backup old file
		String promptFileName = project.getPromptConfiguration().getPromptsUrl();

		// set instance variables to the values given as arguments
		if (promptFileName != null && !promptFileName.equals("")) {
			URL promptFile;
			try {
				promptFile = URLContext.getContextURL(projectContext, promptFileName);
			} catch (MalformedURLException e) {
				throw new ProjectManagerException(e);
			}
			String protocol = promptFile.getProtocol();
			if ("file".equalsIgnoreCase(protocol)) {
				String fPath;
				try {
					fPath = promptFile.toURI().getPath();
				} catch (URISyntaxException e) {
					throw new ProjectManagerException(e);
				}
				File f = new File(fPath);
				// File backupFile=new File(promptFile.toURI().getPath()+".bak");
				// f.renameTo(backupFile);
				FileUtils.moveToBackup(f, ".bak");
				// Finally write the script file
				recScriptHandler = new RecscriptHandler();
				recScriptHandler.setValidating(true);
				try {
					recScriptHandler.writeXML(script,
							new OutputStreamWriter(new FileOutputStream(f), Charset.forName("UTF-8")));
				} catch (FileNotFoundException | DOMConverterException | ParserConfigurationException e) {
					throw new ProjectManagerException(e);
				}
				recScriptManager.setScriptSaved(true);

			} else {
				throw new ProjectManagerException(
						"Cannot save script to URL: " + promptFile + ", protocol " + protocol + "not supported.");
			}
		} else {
			throw new ProjectManagerException("Cannot save script to empty URL!");
		}
	}

	public void createNewProject(NewProjectConfiguration newProjectConfig) throws ProjectManagerException {
		
		ProjectConfiguration newProject = newProjectConfig.getProjectConfiguration();
		
		// TODO clean up, should use the fields of this project manager
		URL projectDirURL;
		File projectDir;
		try {
			projectDirURL = new URL(projectContext, newProject.getDirectory());
			projectDir = new File(projectDirURL.toURI().getPath());
			
		} catch (URISyntaxException | IOException e) {
			throw new ProjectManagerException(e);
		}

		URI pdURI = projectDir.toURI();
		String projectDirURIstr = pdURI.toASCIIString();
		// setProjectContext(projectDirURLascii);
		// Handler.setProjectDir(projectDir);
		String projectFilename = newProject.getName() + PROJECT_FILE_EXTENSION;
		String recScriptFileName = newProject.getName() + REC_SCRIPT_FILE_EXTENSION;
		String speakersFilename = newProject.getName() + SPEAKER_FILE_SUFFIX;
		File projectFile = new File(projectDir, projectFilename);
		File recScriptFile = new File(projectDir, recScriptFileName);
		File speakersFile = new File(projectDir, speakersFilename);

		// copy recording script DTD

		InputStream is = RecscriptHandler.class.getResourceAsStream(RecscriptHandler.REC_SCRIPT_DTD);
		FileOutputStream fos;
		try {
			fos = new FileOutputStream(new File(projectDir, RecscriptHandler.REC_SCRIPT_DTD));
			StreamCopy.copy(is, fos);
		} catch (IOException e) {
			throw new ProjectManagerException(e);
		}

		if (newProjectConfig.isUseExampleScript()) {
			// copy example recording script
			is = RecscriptHandler.class.getResourceAsStream(ProjectManager.REC_SCRIPT_EXAMPLE);
			try {
				fos = new FileOutputStream(recScriptFile);
				StreamCopy.copy(is, fos);
			} catch (IOException e) {
				throw new ProjectManagerException(e);
			}

			// set item code generator to match example script itemcodes

			ItemcodeGeneratorConfiguration icCfg = newProject.getPromptConfiguration()
					.getItemcodeGeneratorConfiguration();
			icCfg.setGeneratorName("Demo script itemcode generator");

			// demo script has itemcodes from demo_000 ...
			icCfg.setPrefix("demo_");
			icCfg.setFixedDecimalPlaces(3);

			// ... to demo_063
			// each unit starts a new block of the items
			// So new items added by the user should start at demo_070
			icCfg.setCounterStart(70);

			icCfg.setActive(true);

		} else {
			Script newScript = new Script();
			newScript.setPropertyChangeSupportEnabled(true);
			recScriptManager.setScript(newScript);

		}

		createEmptySpeakerDatabase(speakersFile, true);

		URI recScriptURI = recScriptFile.toURI();
		URI projectDirURI = projectDir.toURI();
		URI recScriptRelURI = projectDirURI.relativize(recScriptURI);
		newProject.getPromptConfiguration().setPromptsUrl(recScriptRelURI.toString());
		URI speakersFileURI = speakersFile.toURI();
		URI speakersFileRelURI = projectDirURI.relativize(speakersFileURI);
		newProject.getSpeakers().setSpeakersUrl(speakersFileRelURI.toString());

		// TODO
		newProject.getRecordingConfiguration().setUrl("RECS/");
		setConfiguration(newProject);
		try {
			setProjectURL(projectFile.toURI().toURL());
		} catch (MalformedURLException e) {
			throw new ProjectManagerException(e);
		}

		String systemIdBase = projectContext.toExternalForm();
		recScriptManager.setSystemIdBase(systemIdBase);

		try {
			commonConfig(newProject);
		} catch (MalformedURLException e) {
			throw new ProjectManagerException(e);
		}
		
		saveProject();
		if (!newProjectConfig.isUseExampleScript()) {
			saveScript();
		}
		// configure(newProject);
	}

}
