//    Speechrecorder
// 	  (c) Copyright 2017
// 	  Institute of Phonetics and Speech Processing,
//    Ludwig-Maximilians-University, Munich, Germany
//
//
//    This file is part of Speechrecorder
//
//
//    Speechrecorder is free software: you can redistribute it and/or modify
//    it under the terms of the GNU Lesser General Public License as published by
//    the Free Software Foundation, version 3 of the License.
//
//    Speechrecorder is distributed in the hope that it will be useful,
//    but WITHOUT ANY WARRANTY; without even the implied warranty of
//    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
//    GNU Lesser General Public License for more details.
//
//    You should have received a copy of the GNU Lesser General Public License
//    along with Speechrecorder.  If not, see <http://www.gnu.org/licenses/>.



package ipsk.apps.speechrecorder.project;

import java.io.File;
import java.io.IOException;
import java.net.MalformedURLException;
import java.net.URISyntaxException;
import java.net.URL;
import java.util.ArrayList;
import java.util.HashMap;
import java.util.List;
import java.util.Set;
import java.util.UUID;

import javax.sound.sampled.AudioFormat;
import javax.sound.sampled.UnsupportedAudioFileException;
import javax.swing.JFileChooser;
import javax.xml.bind.JAXB;
import javax.xml.bind.JAXBContext;
import javax.xml.bind.JAXBException;
import javax.xml.bind.Marshaller;

import ips.annot.model.db.Bundle;
import ips.annot.model.db.Session;
import ips.annot.model.emu.EmuBundleAnnotationPersistor;
import ips.annot.model.emu.EmuDB;
import ips.annot.model.emu.webapp.EMUwebAppConfig;
import ipsk.apps.speechrecorder.SpeakerManager;
import ipsk.apps.speechrecorder.SpeechRecorderException;
import ipsk.apps.speechrecorder.config.ProjectConfiguration;
import ipsk.apps.speechrecorder.config.PromptConfiguration;
import ipsk.apps.speechrecorder.config.RecordingConfiguration;
import ipsk.apps.speechrecorder.db.Speaker;
import ipsk.apps.speechrecorder.project.WorkspaceProjectManager.ProjectSessionIterator;
import ipsk.apps.speechrecorder.script.RecScriptManager;
import ipsk.apps.speechrecorder.script.RecscriptManagerException;
import ipsk.apps.speechrecorder.storage.ActiveSessionStorageManager;
import ipsk.apps.speechrecorder.storage.SessionStorageManager;
import ipsk.apps.speechrecorder.storage.StorageManagerException;
import ipsk.audio.AudioSourceException;
import ipsk.audio.ConvenienceFileAudioSource;
import ipsk.beans.DOMCodec;
import ipsk.beans.DOMCodecException;
import ipsk.db.speech.Mediaitem;
import ipsk.db.speech.Project;
import ipsk.db.speech.Recording;
import ipsk.db.speech.RecordingFile;
import ipsk.db.speech.Script;
import ipsk.io.StreamCopy;
import ipsk.net.URLContext;
import ipsk.text.EncodeException;

/**
 * @author klausj
 *
 */
public class ProjectManager {

    // version of project configuration
    // increase value if project configuration schema changes
    // the schema is only defined by Java code classes (there is no DTD or XML Schema yet)
    public final static String PROJECT_VERSION="4.0.0";
    
    public static final String PROJECT_FILE_EXTENSION = "_project.prj";
	public static final String SPEAKER_FILE_SUFFIX = "_speakers.xml";
	public static final String REC_SCRIPT_FILE_EXTENSION = "_script.xml";
	public static final String DEFAULT_RESOURCE_PATH="resources";
 
	protected Project projectDb;
	protected ProjectConfiguration project;
	protected URL projectContext = null;
	
	protected DOMCodec domCodec;
	
	protected SpeakerManager speakerManager;
	protected String speakerFileName;
	protected URL speakerURL;
	
	protected SessionStorageManager projectStorageManager;
	
	public SessionStorageManager getProjectStorageManager() {
        return projectStorageManager;
    }

    protected RecScriptManager recScriptManager;
	protected URL promptFile;
	protected URL recBaseURL;
	protected int numLines;
	
	/**
	 * @throws ClassNotFoundException 
	 * @throws DOMCodecException 
	 * 
	 */
	public ProjectManager() throws ProjectManagerException {
		super();
		  Package configBasePack;
		try {
			configBasePack = Class.forName("ipsk.apps.speechrecorder.config.ProjectConfiguration").getPackage();
			domCodec = new DOMCodec(configBasePack);
		} catch (ClassNotFoundException | DOMCodecException e) {
			throw new ProjectManagerException(e);
		}
		
	    speakerManager = new SpeakerManager();
	    recScriptManager=new RecScriptManager();
	    projectStorageManager=new SessionStorageManager();
	}
	
	

	/**
	 * 
	 */
	public void init() {
		
		
	}
	
	/**
	 * @return project context URL
	 */
	public URL getProjectContext() {
		return projectContext;
	}

	/**
	 * @param context the project context (directory) URL
	 */
	public void setProjectContext(URL context) {
		projectContext = context;
	}
	
	/**
	 * Sets a new configuration. Does not reconfigure the application.
	 * 
	 * @param newProject
	 */
	protected void setConfiguration(ProjectConfiguration newProject) {
		project = newProject;
	}

	/**
	 * Returns the project configuration (bean).
	 * 
	 * @return the project configuration
	 */
	public ProjectConfiguration getConfiguration() {
		return project;
	}


	public void config(ProjectConfiguration cfgProject) throws ProjectManagerException {
			try {
				commonConfig(cfgProject);
			} catch (MalformedURLException e1) {
				
				e1.printStackTrace();
				throw new ProjectManagerException(e1);
			}
			PromptConfiguration pc=cfgProject.getPromptConfiguration();
			String promptFileName = pc.getPromptsUrl();

			// set instance variables to the values given as arguments
			if (promptFileName != null && !promptFileName.equals("")) {
			    
				try {
					promptFile = URLContext.getContextURL(projectContext,
							promptFileName);
					recScriptManager.load(promptFile);
				} catch (MalformedURLException | RecscriptManagerException e) {
					e.printStackTrace();
					throw new ProjectManagerException(e);
				}
			} else {
				throw new ProjectManagerException("Prompt file not configured");
			}
		 String newSpeakerFilename = project.getSpeakers().getSpeakersUrl();
			if (speakerFileName == null
					|| !speakerFileName.equals(newSpeakerFilename)) {
				// speakers URL has changed
				speakerFileName = project.getSpeakers().getSpeakersUrl();

				if (speakerFileName != null && !speakerFileName.equals("")) {
					try {
						speakerURL = URLContext.getContextURL(projectContext,
								speakerFileName);
					} catch (MalformedURLException e) {
						
						e.printStackTrace();
						throw new ProjectManagerException(e);
					}
				} else {
					throw new ProjectManagerException("Speaker database not configured");
				}
				
				// storageManager.setSpeakerCode(speakerManager.getSpeaker().getCode());
				// currSpeaker=null;
			}
			// Speaker database is closed when project is closed
			// therefore we always have to (re)load speaker db
			speakerManager.loadURL(speakerURL);
			
	}
	
	protected void commonConfig(ProjectConfiguration cfgProject) throws MalformedURLException {
		 projectDb = new Project();
		 projectDb.setName(cfgProject.getName());
		 project = cfgProject;
		 recScriptManager.setContext(projectContext);
		 RecordingConfiguration recCfg = project.getRecordingConfiguration();
		 String recDirName = recCfg.getUrl();
		 recBaseURL = URLContext.getContextURL(projectContext, recDirName);
		 projectStorageManager.setStorageURL(recBaseURL);
		 numLines = 1;
		 projectStorageManager.setNumLines(numLines);
		 boolean overwrite = recCfg.getOverwrite();
		 projectStorageManager.setOverwrite(overwrite);
		 
		// Commented out: At this point the configuration is already parsed and
	        // Speechrecorder should run this project without problems. 
	        //
//			// if the project configuration could be parsed and converted
//			// it is OK if the version is higher than the running instance
//			String projVerStr=project.getVersion();
//			try {
//	            Version version=Version.parseString(projVerStr);
//	            Version prjVersion=Version.parseString(PROJECT_VERSION);
//	            if(version.compareTo(prjVersion)>0){
//	                errMsg="Unable to load project!\nVersion of Speechrecorder ("+PROJECT_VERSION+") is less than project version ("+version+").\nPlease update Speechrecorder to load this project.";
//	                throw new SpeechRecorderException(errMsg);
//	            }
//	        } catch (ParserException e1) {
//	            errMsg="Could not parse project version: "+projVerStr;
//	            throw new SpeechRecorderException(errMsg);
//	        }
		 
	}
	
	public Project getProjectDb() {
        return projectDb;
    }
	
	public void createEmptySpeakerDatabase(File speakerDbFile,boolean overwrite) {
		
		if(overwrite || !speakerDbFile.exists()) {
			JAXB.marshal(new ArrayList<ipsk.db.speech.Speaker>(), speakerDbFile);
		}
	}



    public void rebuildDb() throws StorageManagerException{
		   Script script=recScriptManager.getScript();
	       projectDb.getScripts().clear();
	       projectDb.getScripts().add(script);
	       script.getSessions().clear();
	       // build lazy in mem DB
//	     Set<ipsk.db.speech.Session> sesss=new SessionsSetProxy(storageManager, projectDb,script);
	       Set<ipsk.db.speech.Session> sesss=projectStorageManager.sessionsLazy(projectDb,script);
	       projectDb.setSessions(sesss);
	       List<Speaker> speakersList=speakerManager.getSpeakersList();
	       for(ipsk.apps.speechrecorder.db.Speaker spk:speakersList){
	           spk.getSessions().clear();
	       }
	       for(ipsk.db.speech.Session s:sesss){
	           s.setScript(script);
	           script.getSessions().add(s);
	           s.getSpeakers().clear();
	           // create speaker/session relationship and set speaker code to session
	           for(ipsk.apps.speechrecorder.db.Speaker spk:speakersList){
	               if(spk.getPersonId().equals(s.getSessionId())){
	                   s.setCode(spk.getCode());
	                   s.getSpeakers().add(spk);     
	                   spk.getSessions().add(s);
	               }
	           }
	           // load recording files lazy
	           boolean hasRecs=projectStorageManager.sessionHasRecordingFiles(s);
	           s.setRecordingFiles(null);
	           s.markHasRecordings(hasRecs);
	      
	       }
	       speakerManager.setSessions(sesss);

		}
	  

	
}
