//    Speechrecorder
//    (c) Copyright 2009-2011
//    Institute of Phonetics and Speech Processing,
//    Ludwig-Maximilians-University, Munich, Germany
//
//
//    This file is part of Speechrecorder
//
//
//    Speechrecorder is free software: you can redistribute it and/or modify
//    it under the terms of the GNU Lesser General Public License as published by
//    the Free Software Foundation, version 3 of the License.
//
//    Speechrecorder is distributed in the hope that it will be useful,
//    but WITHOUT ANY WARRANTY; without even the implied warranty of
//    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
//    GNU Lesser General Public License for more details.
//
//    You should have received a copy of the GNU Lesser General Public License
//    along with Speechrecorder.  If not, see <http://www.gnu.org/licenses/>.

package ipsk.apps.speechrecorder.script;

import java.io.BufferedInputStream;
import java.io.ByteArrayInputStream;
import java.io.File;
import java.io.FileInputStream;
import java.io.FileNotFoundException;
import java.io.FileOutputStream;
import java.io.IOException;
import java.io.InputStream;
import java.io.OutputStream;
import java.io.OutputStreamWriter;
import java.io.Reader;
import java.io.StringReader;
import java.io.StringWriter;
import java.io.Writer;
import java.net.MalformedURLException;
import java.net.URI;
import java.net.URISyntaxException;
import java.net.URL;
import java.net.URLConnection;
import java.nio.charset.Charset;
import java.nio.file.Files;
import java.nio.file.Path;
import java.util.List;
import java.util.regex.Matcher;
import java.util.regex.Pattern;

import ipsk.apps.speechrecorder.SpeechRecorder;
import ipsk.db.speech.Mediaitem;
import ipsk.db.speech.PromptItem;
import ipsk.db.speech.Script;
import ipsk.db.speech.Section;
import ipsk.io.StreamCopy;
import ipsk.net.URLContext;
import ipsk.persistence.IntegerSequenceGenerator;
import ipsk.xml.DOMConverter;
import ipsk.xml.DOMConverterException;

import javax.xml.parsers.DocumentBuilder;
import javax.xml.parsers.DocumentBuilderFactory;
import javax.xml.parsers.ParserConfigurationException;

import org.w3c.dom.DOMImplementation;
import org.w3c.dom.Document;
import org.w3c.dom.DocumentType;
import org.w3c.dom.Element;
import org.xml.sax.EntityResolver;
import org.xml.sax.InputSource;
import org.xml.sax.SAXException;


public class RecscriptHandler extends DOMConverter{
	public final static String REC_SCRIPT_DTD_PREFIX = "SpeechRecPrompts";
	public final static String REC_SCRIPT_DTD_EXTENSION = "dtd";
	public final static int REC_SCRIPT_DTD_VERSION=4;
    public final static String REC_SCRIPT_DTD_1 = REC_SCRIPT_DTD_PREFIX+"."+REC_SCRIPT_DTD_EXTENSION;
    public final static String REC_SCRIPT_DTD_2 = REC_SCRIPT_DTD_PREFIX+"_2."+REC_SCRIPT_DTD_EXTENSION;
    public final static String REC_SCRIPT_DTD_3 = REC_SCRIPT_DTD_PREFIX+"_3."+REC_SCRIPT_DTD_EXTENSION;
    public final static String REC_SCRIPT_DTD_4 = REC_SCRIPT_DTD_PREFIX+"_4."+REC_SCRIPT_DTD_EXTENSION;
    public final static String REC_SCRIPT_DTD = REC_SCRIPT_DTD_4;
    public static final String NAME="script";
    private DocumentBuilderFactory docFactory;
    private String systemIdBase = null;
    private String systemId=REC_SCRIPT_DTD;
    
    private IntegerSequenceGenerator seqGen=null;
    
    public IntegerSequenceGenerator getSeqGen() {
		return seqGen;
	}

	public void setSeqGen(IntegerSequenceGenerator seqGen) {
		this.seqGen = seqGen;
	}

	public String getSystemId() {
		return systemId;
	}

	public void setSystemId(String systemId) {
		this.systemId = systemId;
	}

	public RecscriptHandler(){
        super();
        docFactory=DocumentBuilderFactory.newInstance();
    }
    
    public String getSystemIdBase() {
		return systemIdBase;
	}
	public void setSystemIdBase(String string) {
		systemIdBase = string;
	}
    
    public Document newRecScriptDocument() throws ParserConfigurationException{
    
        DocumentBuilder docBuilder=docFactory.newDocumentBuilder();
        DOMImplementation domImpl=docBuilder.getDOMImplementation();
        DocumentType docType=domImpl.createDocumentType(NAME, null,systemId);
        return domImpl.createDocument(null, NAME, docType);
    }
    
    /*
     * (non-Javadoc)
     * @see ipsk.xml.DOMConverter#writeXML(org.w3c.dom.Document, java.io.OutputStream)
     */
    @Deprecated
    public void writeXML(Document doc,OutputStream out) throws DOMConverterException{
        super.writeXML(doc,null,systemId, out);
        
    }
    
   
    public void writeXML(Script s,OutputStream out) throws DOMConverterException, ParserConfigurationException{
        writeXML(s,new OutputStreamWriter(out,Charset.forName("UTF-8")));
    }
    
    public void writeXML(Document d,Writer out) throws DOMConverterException{
        super.writeXML(d,null,systemId, out);
    }
    
    public void writeXML(Script s,Writer out) throws DOMConverterException, ParserConfigurationException{
        if(s==null)throw new DOMConverterException("No script object to convert (null) !");
        Document d=newRecScriptDocument();
        //d.appendChild(s.toElement(d));
        s.insertIntoElement(d,d.getDocumentElement());
        super.writeXML(d,null,systemId, out);
    }
    
    public void writeXML(Script s,String systemID,Writer out) throws DOMConverterException, ParserConfigurationException{
        if(s==null)throw new DOMConverterException("No script object to convert (null) !");
        Document d=newRecScriptDocument();
        //d.appendChild(s.toElement(d));
        s.insertIntoElement(d,d.getDocumentElement());
        String sID=systemID;
        if(sID==null){
            sID=this.systemId;
        }
        super.writeXML(d,null,sID, out);
    }
    
    public Script scriptFromDocument(Document d) throws DOMConverterException{
    	return scriptFromDocument(d,true);
    }
    
    public Script scriptFromDocument(Document d,boolean propertyChangeSupport) throws DOMConverterException{
    	Element rootE=d.getDocumentElement();
    	Script s=new Script(rootE);
    	if(propertyChangeSupport){
    		s.setPropertyChangeSupportEnabled(true);
    	}
    	return s;
    }
    
    public Script readScriptFromXML(InputSource is) throws DOMConverterException{
        Document d=readXML(is);
      return scriptFromDocument(d,true);
    }
    
    public Script readScriptFromXML(InputStream is,String systemId) throws DOMConverterException{
        Document d=readXML(is,systemId);
        return scriptFromDocument(d);
    }
    
    public Script readScriptFromXML(Reader isr,String systemId) throws DOMConverterException{
        InputSource is=new InputSource(isr);
        if(systemId!=null){
            is.setSystemId(systemId);
        }
        Document d=readXML(is);
        return scriptFromDocument(d);
    }
    
    public Document toDOM(Script script) throws ParserConfigurationException{
		DocumentBuilderFactory docFactory=DocumentBuilderFactory.newInstance();
		DocumentBuilder docBuilder=docFactory.newDocumentBuilder();
		
		Document d=docBuilder.newDocument();
		
		d.appendChild(script.toElement(d));
	
		
		return d;
	}
    
//    public Script readScriptFromXMLwithoutDTD(InputSource is) throws DOMConverterException, IOException, ParserConfigurationException, SAXException{
//    	return readScriptFromXMLwithoutDTD(is, null);
//    }
    
    public Script readScriptFromXMLwithoutDTD(InputSource is) throws DOMConverterException, IOException, ParserConfigurationException, SAXException{
		
       
        setValidating(true);
//        dbf.setFeature("http://xml.org/sax/features/namespaces", false);
//        dbf.setFeature("http://xml.org/sax/features/validation", false);
//        dbf.setFeature("http://apache.org/xml/features/nonvalidating/load-dtd-grammar", false);
//        dbf.setFeature("http://apache.org/xml/features/nonvalidating/load-external-dtd", false);

        //DocumentBuilder db=dbf.newDocumentBuilder();
        EntityResolver er=new EntityResolver()
        {
            public InputSource resolveEntity(String publicId, String systemId) throws SAXException, IOException
            {
            	
            	if(systemId.matches(".*"+REC_SCRIPT_DTD_PREFIX+"_[0-9]*."+REC_SCRIPT_DTD_EXTENSION+"$")  || systemId.matches(".*"+REC_SCRIPT_DTD_PREFIX+"."+REC_SCRIPT_DTD_EXTENSION+"$")){
                	// matches one of the DTD versions (pattern matches existing and future versions)
                	// Inject latest DTD as class resource
                    InputStream dtdStream = RecscriptHandler.class
                            .getResourceAsStream(REC_SCRIPT_DTD);
                    BufferedInputStream bis=new BufferedInputStream(dtdStream);
                    return new InputSource(bis);
                }else{
                    return null;
                }
            }
        };
        Document d=readXML(is,er);
        Element rootE=d.getDocumentElement();
        Script s=new Script(rootE);
        return s;
    }
    
    
    
    public String scriptValid(Script script) throws ParserConfigurationException, IOException{
    	try {
			validateScript(script);
		} catch (DOMConverterException e) {
			return e.getMessage();
		}  catch (SAXException e) {
			return e.getMessage();
		}
    	return null;
    }
    
  
    
    public void validateScript(Script script) throws DOMConverterException, ParserConfigurationException, IOException, SAXException{
    	StringWriter sw=new StringWriter();
		Document d = toDOM(script);
		writeXML(d, null, RecscriptHandler.REC_SCRIPT_DTD, sw);
    	String ss=sw.toString();
    	StringReader sr=new StringReader(ss);
    	InputSource sis=new InputSource(sr);
    	readScriptFromXMLwithoutDTD(sis);
    	
    }
    
    public Script insertScriptElementsFromXML(Script s,InputSource is) throws DOMConverterException{
        Document d=readXML(is);
        Element rootE=d.getDocumentElement();
        s.insertElement(seqGen,rootE);
        return s;
    }
    public Script readScriptFromXML(Reader is) throws DOMConverterException{
        Document d=readXML(is);
        Element rootE=d.getDocumentElement();
        Script s=new Script(seqGen,rootE);
        s.setPropertyChangeSupportEnabled(true);
        return s;
    }
    
    private File getRequiredDTDFile() throws MalformedURLException {
	    if(systemIdBase!=null){
	        URL systemIdBaseURL=new URL(systemIdBase);
	        if(systemIdBaseURL.getProtocol().equalsIgnoreCase("file")){
	            String systemIdBaseDirname;
				try {
					systemIdBaseDirname = systemIdBaseURL.toURI().getPath();
				} catch (URISyntaxException e) {
					throw new MalformedURLException(e.getMessage());
				}
	            File systemBaseDir=new File(systemIdBaseDirname);

	            File dtdFile=new File(systemBaseDir,REC_SCRIPT_DTD);
	            return dtdFile;
	        }
	    }
	    return null;
	}

	public boolean isNewVersionOfDTDFileRequired() throws MalformedURLException{
	    File requiredDtdFile=getRequiredDTDFile();
	    return (requiredDtdFile!=null && !requiredDtdFile.exists());
	}

	public void createDTDFileIfRequired() throws IOException{
	    File dtdFile=getRequiredDTDFile();
	    if(dtdFile !=null && !dtdFile.exists()){
	        // copy recording script DTD
	        InputStream is = RecscriptHandler.class.getResourceAsStream(REC_SCRIPT_DTD);
	        FileOutputStream fos = new FileOutputStream(dtdFile);
	        StreamCopy.copy(is, fos);
	    }
	}
	
	
	private Integer parseSystemIDVersion(String systemId){
	    String originalDtdPattern=REC_SCRIPT_DTD_PREFIX+"."+REC_SCRIPT_DTD_EXTENSION;
	    String versionedDtdsPattern=REC_SCRIPT_DTD_PREFIX+"_([0-9]{1,2})."+REC_SCRIPT_DTD_EXTENSION;
	    if(systemId==null){
	        return null;
	    }else{
	        if(systemId.matches(originalDtdPattern)){
	            return 1;
	        }else{
	            Pattern p=Pattern.compile(versionedDtdsPattern);
	            Matcher m=p.matcher(systemId);
	            if(m.matches()){
	                String dtdVersString=m.group(1);
	                return Integer.parseInt(dtdVersString);
	            }else{
	                return null;
	            }
	        }
	    }
	}
	
	/**
     * Reads recording and prompting instructions from an XML-Document.
     * After that, a vector with the prompt file contents can be obtained via
     * <code>getRecSections()</code>.
     * 
     * @param recScriptURL URL
     * @return the script
     * @throws ParserConfigurationException
     * @throws IOException
     * @throws SAXException
     * @throws DOMConverterException 
     * @throws IOException 
     */
    public Script readRecScriptXMLFile(URL recScriptURL) throws RecscriptHandlerException {
        return readRecScriptXMLFile(recScriptURL, false);
    }
    
	/**
	 * Reads recording and prompting instructions from an XML-Document.
	 * After that, a vector with the prompt file contents can be obtained via
	 * <code>getRecSections()</code>.
	 * 
	 * @param recScriptURL URL
	 * @param force if true forces loading script even if its has a newer DTD schema version
	 * @return the script
	 * @throws ParserConfigurationException
	 * @throws IOException
	 * @throws SAXException
	 * @throws DOMConverterException 
	 * @throws IOException 
	 */
    public Script readRecScriptXMLFile(URL recScriptURL,boolean force) throws RecscriptHandlerException {

        Document doc=null;
        String recScriptURLString=recScriptURL.toExternalForm();
        DOMConverter domConverter=new DOMConverter();
        domConverter.setValidating(true);

        if (systemIdBase == null) {				// Pfadpraefix fuer DTD relativ zum Projekt
            try {
                doc=domConverter.readXML(recScriptURLString);
            } catch (DOMConverterException e) {
                throw new RecscriptHandlerException("Could not parse recording script "+recScriptURLString+": "+e.getLocalizedMessage(),e);
            }
        }
        else {
            URLConnection recScriptUrlConn;
            try {
                recScriptUrlConn = recScriptURL.openConnection();
            } catch (IOException e) {
                throw new RecscriptHandlerException("Could not open recording script URL "+recScriptURLString+": "+e.getLocalizedMessage(),e);
            }
            recScriptUrlConn.setUseCaches(false);
            recScriptUrlConn.setRequestProperty("Accept","application/xml");
            InputStream recScriptIs;
            try {
                recScriptIs = recScriptUrlConn.getInputStream();
                doc=domConverter.readXML(recScriptIs, systemIdBase);       
            } catch (IOException e) {
                throw new RecscriptHandlerException("Could not read recording script URL "+recScriptURLString+": "+e.getLocalizedMessage(),e);
            } catch (DOMConverterException e) {
                throw new RecscriptHandlerException("Could not parse recording script "+recScriptURLString+": "+e.getLocalizedMessage(),e);
            }
        }

        DocumentType documentType=doc.getDoctype();
        String systemId=documentType.getSystemId();
        Integer dtdVers=parseSystemIDVersion(systemId);
        if(!force && dtdVers >REC_SCRIPT_DTD_VERSION){
            throw new RecScriptSchemaVersionNewerException(REC_SCRIPT_DTD_VERSION, dtdVers);
        }
        this.systemId=systemId;
        return new Script(seqGen,doc.getDocumentElement());

    }


	/**
	 * reads recording and prompting instructions from an XML-Document.
	 * After that, a vector with the prompt file contents can be obtained via
	 * <code>getRecSections()</code>.
	 * 
	 * @param recScriptURL URL
	 * @return the script
	 * @throws ParserConfigurationException
	 * @throws IOException
	 * @throws SAXException
	 * @throws DOMConverterException 
	 * @throws IOException 
	 */
	public Script readRecScriptXMLFileWithoutDTD(URL recScriptURL) throws DOMConverterException, IOException, ParserConfigurationException, SAXException {
		
		InputStream iStream=recScriptURL.openStream();
		InputSource iSrc=new InputSource(iStream);
		return readScriptFromXMLwithoutDTD(iSrc);
	}
	
	public void renameContext(File recScriptFile,URL oldCtxURL,URL newCtxURL) throws RecscriptHandlerException{
	    URI recScriptUri=recScriptFile.toURI();
	    URL recScriptURL;
	    try {
	        recScriptURL = recScriptUri.toURL();
	    } catch (MalformedURLException e1) {
	        throw new RecscriptHandlerException(e1);
	    }
	    Script script=readRecScriptXMLFile(recScriptURL);

	    Script modScript=null;
	    try {
	        modScript=(Script) script.clone();

	    } catch (CloneNotSupportedException e) {
	        e.printStackTrace();
	        return;
	    }
	    boolean modified=renameContext(modScript, oldCtxURL, newCtxURL);
	    if(modified){
	        FileOutputStream fos=null;
	        try {
	            fos = new FileOutputStream(recScriptFile);
	            writeXML(modScript,fos);
	        } catch (FileNotFoundException e) {
	            throw new RecscriptHandlerException("Could not find file: "+e.getLocalizedMessage(), e);
	        } catch (DOMConverterException e) {
	            throw new RecscriptHandlerException("Could not convert script to XML file: "+e.getLocalizedMessage(), e);
	        } catch (ParserConfigurationException e) {
	            throw new RecscriptHandlerException("Could not configure XML parser for script file: "+e.getLocalizedMessage(), e);
	        }finally{
	            if(fos!=null){
	                try {
	                    fos.close();
	                } catch (IOException e) {

	                }
	            }
	        }
	    }
	}

	public boolean renameContext(Script script,URL oldCtxURL,URL newCtxURL) throws RecscriptHandlerException{
	    boolean changed=false;
	    URLContext oldCtx=new URLContext(oldCtxURL); 

	    List<PromptItem> pis=script.promptItemsList();
	    for(PromptItem pi:pis){
	        List<Mediaitem> mis=pi.getMediaitems();
	        for(Mediaitem mi:mis){
	            String srcStr=mi.getSrcStr();
	            if(srcStr!=null){
	                String newScrStr;
                    try {
                        newScrStr = oldCtx.renameContextSpec(newCtxURL, srcStr);
                    } catch (MalformedURLException e) {
                       throw new RecscriptHandlerException(e);
                    }
	                if(!srcStr.equals(newScrStr)){
	                    mi.setSrcStr(newScrStr);
	                    changed=true;
	                }
	            }
	        }
	    }

	    return changed;
	}
	
	
	public void duplicate(URL orgCtxURL,String orgScriptSpec,URL newCtxURL) throws RecscriptHandlerException {
			URL orgScriptUrl;
			try {
				orgScriptUrl = URLContext.getContextURLStrict(orgCtxURL, orgScriptSpec);
			} catch (MalformedURLException e) {
				 throw new RecscriptHandlerException(e);
			}
		   Script orgScript=readRecScriptXMLFile(orgScriptUrl);

		    Script newScript=null;
		    try {
		        newScript=(Script) orgScript.clone();
		    } catch (CloneNotSupportedException e) {
		       throw new RecscriptHandlerException(e);
		    }
		    
		    URLContext orgCtx=new URLContext(orgCtxURL);
		    
		    // copy resources and modify resource URLs if necessary  
		    
		    List<PromptItem> pis=newScript.promptItemsList();
		    for(PromptItem pi:pis){
		        List<Mediaitem> mis=pi.getMediaitems();
		        for(Mediaitem mi:mis){
		            String orgSrcStr=mi.getSrcStr();
		            if(orgSrcStr!=null){
		            	
		            	try {
							URL orgSrcUrl=URLContext.getContextURLStrict(orgScriptUrl, orgSrcStr);
						} catch (MalformedURLException e) {
							throw new RecscriptHandlerException(e);
						}
		            	
		                String newScrStr;
		                
	                    try {
	                        newScrStr = orgCtx.renameContextSpec(newCtxURL, orgSrcStr);
	                    } catch (MalformedURLException e) {
	                       throw new RecscriptHandlerException(e);
	                    }
	                    
	                    // TODO  not complete!!
		            }
		        }
		    }
		    
		   
	}
	
    
    public static void main(String[] args){
        // Test: create empty script
        
        try {
            RecscriptHandler h=new RecscriptHandler();
            if (args.length==2){
                FileInputStream fis=new FileInputStream(args[1]);
                Script script=h.readScriptFromXML(fis,args[0]);
                System.out.println(script);
                h.writeXML(script,new OutputStreamWriter(System.out,Charset.forName("UTF-8")));
            }else{
            
            
            h.writeXML(h.newRecScriptDocument(),new OutputStreamWriter(System.out,Charset.forName("UTF-8")));
            }
        } catch (DOMConverterException e) {
            // TODO Auto-generated catch block
            e.printStackTrace();
        } catch (ParserConfigurationException e) {
            // TODO Auto-generated catch block
            e.printStackTrace();
        } catch (FileNotFoundException e) {
            // TODO Auto-generated catch block
            e.printStackTrace();
        } 
    }
    
}
