//    Speechrecorder
// 	  (c) Copyright 2014
// 	  Institute of Phonetics and Speech Processing,
//    Ludwig-Maximilians-University, Munich, Germany
//
//
//    This file is part of Speechrecorder
//
//
//    Speechrecorder is free software: you can redistribute it and/or modify
//    it under the terms of the GNU Lesser General Public License as published by
//    the Free Software Foundation, version 3 of the License.
//
//    Speechrecorder is distributed in the hope that it will be useful,
//    but WITHOUT ANY WARRANTY; without even the implied warranty of
//    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
//    GNU Lesser General Public License for more details.
//
//    You should have received a copy of the GNU Lesser General Public License
//    along with Speechrecorder.  If not, see <http://www.gnu.org/licenses/>.



package ipsk.apps.speechrecorder.workspace.ui;

import ipsk.apps.speechrecorder.config.ProjectConfiguration;
import ipsk.apps.speechrecorder.config.WorkspaceProject;
import ipsk.apps.speechrecorder.project.ProjectManagerException;
import ipsk.apps.speechrecorder.ui.ExportProjectUIDialog;
import ipsk.apps.speechrecorder.workspace.WorkspaceException;
import ipsk.apps.speechrecorder.workspace.WorkspaceManager;
import ipsk.net.Utils;

import java.awt.Component;
import java.awt.Cursor;
import java.awt.GridBagConstraints;
import java.awt.GridBagLayout;
import java.awt.Insets;
import java.awt.Point;
import java.awt.event.ActionEvent;
import java.awt.event.ActionListener;
import java.beans.PropertyChangeEvent;
import java.beans.PropertyChangeListener;
import java.io.File;
import java.net.MalformedURLException;
import java.net.URISyntaxException;
import java.net.URL;
import java.util.ArrayList;

import javax.swing.JList;
import javax.swing.JMenu;
import javax.swing.JMenuItem;
import javax.swing.JOptionPane;
import javax.swing.JPanel;
import javax.swing.JPopupMenu;
import javax.swing.JScrollPane;
import javax.swing.ListSelectionModel;
import javax.swing.event.ListDataEvent;
import javax.swing.event.ListDataListener;
import javax.swing.event.ListSelectionEvent;
import javax.swing.event.ListSelectionListener;
import javax.swing.filechooser.FileSystemView;

/**
 * @author klausj
 *
 */
public class WorkspaceProjectsPanel extends JPanel implements ActionListener, ListSelectionListener, ListDataListener, PropertyChangeListener{

	private static final long serialVersionUID = 1L;
    private WorkspaceManager workspaceManager;
    private JScrollPane projectsScrollPane;
    private JList<String> projectsList;
    private JMenuItem openMi;
    private JMenuItem closeMi;
    private JMenu exportMenu;
    private JMenuItem exportToArchMi;
    private JMenuItem exportToEmuDBMi;
    private JMenuItem deleteMi;
    private JMenuItem renameMi;
    private JMenuItem duplicateMi;
    
    private ListSelectionModel listSelectionModel;
    
    /**
     * 
     */
    public WorkspaceProjectsPanel(WorkspaceManager workspaceManager) {
       super(new GridBagLayout());
      
       this.workspaceManager=workspaceManager;
       this.workspaceManager.addListDataListener(this);
       this.workspaceManager.addPropertyChangeListener(this);
       GridBagConstraints c = new GridBagConstraints();
       c.insets=new Insets(2,2,2,2);
       c.gridx=0;
       c.gridy=0;
       c.fill=GridBagConstraints.BOTH;
       c.weightx=1.0;
       c.weighty=1.0;
       projectsList=new JList<String>(workspaceManager);
       projectsList.setSelectionMode(ListSelectionModel.SINGLE_SELECTION);
       listSelectionModel = projectsList.getSelectionModel();
       listSelectionModel.addListSelectionListener(this);
       projectsScrollPane=new JScrollPane(projectsList);
       add(projectsScrollPane,c);
       
//       c.weightx=0;
//       c.weighty=0;
//       c.fill=GridBagConstraints.NONE;
       
       openMi=new JMenuItem("Open");
       openMi.addActionListener(this);
       
       closeMi=new JMenuItem("Close");
       closeMi.addActionListener(this);
       
       
       exportMenu=new JMenu("Export");
       
       exportToArchMi=new JMenuItem("Export to SpeechRecorder project Zip archive...");
       exportToArchMi.addActionListener(this);

       exportMenu.add(exportToArchMi);
       
       exportToEmuDBMi=new JMenuItem("Export as EMU-DB...");
       exportToEmuDBMi.addActionListener(this);
       
       exportMenu.add(exportToEmuDBMi);
       
       deleteMi=new JMenuItem("Delete...");
       deleteMi.addActionListener(this);
//       c.gridx=0;
//       c.gridy++;
//       add(deleteButton,c);
       
       renameMi=new JMenuItem("Rename...");
       renameMi.addActionListener(this);
       
       duplicateMi=new JMenuItem("Duplicate configuration and script...");
       duplicateMi.addActionListener(this);
       
   	JPopupMenu jPopupMenu = new JPopupMenu() {
	  
		private static final long serialVersionUID = 1L;

	    @Override
	    public void show(Component invoker, int x, int y) {
	        int row = projectsList.locationToIndex(new Point(x, y));
	        if (row != -1) {
	            projectsList.setSelectedIndex(row);
	        }
	        super.show(invoker, x, y);
	    }
	};
	jPopupMenu.add(openMi);
	jPopupMenu.add(closeMi);
	jPopupMenu.add(exportMenu);
	jPopupMenu.add(deleteMi);
	jPopupMenu.add(renameMi);
	jPopupMenu.add(duplicateMi);
	projectsList.setComponentPopupMenu(jPopupMenu);
     updateUIDependencies();
    }

    /* (non-Javadoc)
     * @see java.awt.event.ActionListener#actionPerformed(java.awt.event.ActionEvent)
     */
    @Override
    public void actionPerformed(ActionEvent ae) {
        Object src=ae.getSource();
        WorkspaceProject selProj=null;
        int selRow=projectsList.getSelectedIndex();
        if(selRow>=0){
            selProj=workspaceManager.getWorkspaceProjects().get(selRow);
        }
        if(src==openMi || src==exportToEmuDBMi || src==exportToArchMi ||  src==deleteMi || src==renameMi|| src==duplicateMi){
            if(selProj!=null) {
                if(workspaceManager.locked(selProj)){
                	String msg;
//                	if(src==openMi){
//                		msg="Project already open.";
//                	}else{
                		msg="Please close project first.";
//                	}
                    JOptionPane.showMessageDialog(this,msg,"Project", JOptionPane.INFORMATION_MESSAGE);
                    return;
                }
            }
        }
        if(src==openMi){
            if(selProj!=null){
            	try {
					workspaceManager.openProject(selProj.getName());
				} catch (WorkspaceException e) {
					e.printStackTrace();
					JOptionPane.showMessageDialog(this, "Could not open project: \n"+e.getLocalizedMessage(),"Project open error",JOptionPane.ERROR_MESSAGE);
				}
            }
            
        }else if(src==closeMi){
            if(selProj!=null){
            	try {
					workspaceManager.closeActiveProject();
				} catch (WorkspaceException e) {
					e.printStackTrace();
					JOptionPane.showMessageDialog(this, "Could not close project: \n"+e.getLocalizedMessage(),"Project close error",JOptionPane.ERROR_MESSAGE);
				}
            }
            
        }else if(src==exportToArchMi){
            if(selProj!=null){
                ProjectConfiguration selPrjCfg=selProj.getConfiguration();
                File projectDir=new File(workspaceManager.getWorkspaceDir(),selPrjCfg.getName());
              
                URL projectContext;
                try {
                    projectContext = Utils.createAsciiURLFromFile(projectDir);
                    exportProjectAsZipArchive(projectContext, selPrjCfg);
                } catch (MalformedURLException | URISyntaxException e) {
                    e.printStackTrace();
                    JOptionPane.showMessageDialog(this,"Could not export project:\n"+e.getLocalizedMessage(), "Export project error!", JOptionPane.ERROR_MESSAGE);
                    
                }
            }
            
        }else
        if(src==exportToEmuDBMi){
            if(selProj!=null){
            	ProjectConfiguration selPrjCfg=selProj.getConfiguration();
                File projectDir=new File(workspaceManager.getWorkspaceDir(),selPrjCfg.getName());
              
                ProjectEmuDBExportDialog peed=new ProjectEmuDBExportDialog(selProj, projectDir);
                Object res=peed.showDialog(this);
                if(res.equals(JOptionPane.OK_OPTION)){
                    try {
                        URL projectContext=Utils.createAsciiURLFromFile(projectDir);
                        workspaceManager.exportToEmuDB(projectContext,selPrjCfg, peed.getBaseDir());
                    } catch (WorkspaceException | URISyntaxException | MalformedURLException e) {
                        JOptionPane.showMessageDialog(this,"Could not export project to EmuDB:\n"+e.getLocalizedMessage(), "emuDB export project error!", JOptionPane.ERROR_MESSAGE);
                    }
                }
                
                
            }
            
        }else 
        if(src==deleteMi){
            if(selProj!=null){
           
                File projectDir=new File(workspaceManager.getWorkspaceDir(),selProj.getConfiguration().getName());
                ProjectDeleteDialog projdelDlg=new ProjectDeleteDialog(selProj,projectDir);
                
                projdelDlg.showDialog(this);
                try {
                    workspaceManager.scanWorkspace();
                } catch (WorkspaceException e) {
                	JOptionPane.showMessageDialog(this,"Workspace scan error:\n"+e.getLocalizedMessage(), "Workspace scan error!", JOptionPane.ERROR_MESSAGE);
                }
            }
            
        }else if(src==renameMi){
            if(selProj!=null){
                String currentName=selProj.getConfiguration().getName();
                ProjectRenameDialog.RenameModel rnM=new ProjectRenameDialog.RenameModel(currentName);
                Object res=ProjectRenameDialog.showDialog(this,new ArrayList<String>(),rnM);
                if(res.equals(JOptionPane.OK_OPTION) && rnM.changed()){
                    try {
                        workspaceManager.renameProject(currentName, rnM.getNewName());
                    } catch (WorkspaceException e) {
                        JOptionPane.showMessageDialog(this,"Could not rename project:\n"+e.getLocalizedMessage(), "Rename project error!", JOptionPane.ERROR_MESSAGE);
                    }
                }
                
            }
            
        }else if(src==duplicateMi){
            if(selProj!=null){
                String currentName=selProj.getConfiguration().getName();
                ProjectDuplicateDialog.DuplicateModel dpM=new ProjectDuplicateDialog.DuplicateModel(currentName);
                Object res=ProjectDuplicateDialog.showDialog(this,new ArrayList<String>(),dpM);
                if(res.equals(JOptionPane.OK_OPTION) && dpM.changed()){
                    try {
                        workspaceManager.duplicateProject(currentName, dpM.getNewName());
                    } catch (WorkspaceException | ProjectManagerException e) {
                        JOptionPane.showMessageDialog(this,"Could not duplicate project:\n"+e.getLocalizedMessage(), "Duplicate project error!", JOptionPane.ERROR_MESSAGE);
                    }
                } 
            }
        } 
        updateUIDependencies();
    }
    
    
    public void exportProjectAsZipArchive(URL projectContext,ProjectConfiguration prCfg){
    	
    	// TODO duplicate code: SpeechRecorderUI.doExportProject()
    	String suggestedExportBaseDirName=FileSystemView.getFileSystemView().getDefaultDirectory().getPath();
    	File suggestedExportBaseDir=new File(suggestedExportBaseDirName);
        File suggestedExportFile=new File(suggestedExportBaseDir,prCfg.getName()+".zip");
        ExportProjectUIDialog epd=new ExportProjectUIDialog();
        epd.setExportFile(suggestedExportFile);
        Object returnVal=epd.showDialog(this);

        if (!returnVal.equals(ExportProjectUIDialog.CANCEL_OPTION)) {

            File file = epd.getExportFile();
            // TODO localize !
            if (!file.exists()
                    || JOptionPane.showConfirmDialog(this, file.getName()
                            + " exists. Do you want to overwrite ?",
                            "Overwrite file ?", JOptionPane.YES_NO_OPTION) == JOptionPane.YES_OPTION) {

                setCursor(new Cursor(Cursor.WAIT_CURSOR));
                ProjectExporter pe=new ProjectExporter();
                try{
                    pe.exportProject(projectContext, prCfg,file, epd.isExportSessionData(),this);
                } catch (Exception e) {
                    JOptionPane.showMessageDialog(this,
                            e.getLocalizedMessage(), "Project export",
                            JOptionPane.ERROR_MESSAGE);
                    e.printStackTrace();
                } finally {
                    setCursor(null);
                }
            }
        }
    }


    /* (non-Javadoc)
     * @see javax.swing.event.ListSelectionListener#valueChanged(javax.swing.event.ListSelectionEvent)
     */
    @Override
    public void valueChanged(ListSelectionEvent lse) {
        updateUIDependencies();
    }

    /**
     * 
     */
    private void updateUIDependencies() {
       //ListSelectionModel lsm=projectsList.getSelectionModel();
//       deleteButton.setEnabled(lsm!=null && !lsm.isSelectionEmpty());
    	
    	 WorkspaceProject selProj=null;
    	 int selRow=projectsList.getSelectedIndex();
         if(selRow>=0){
             selProj=workspaceManager.getWorkspaceProjects().get(selRow);
             boolean hasLock=workspaceManager.hasLock();
             boolean selPrjLocked=workspaceManager.locked(selProj);
             openMi.setEnabled(!hasLock && !selPrjLocked);
             closeMi.setEnabled(selPrjLocked);
             exportMenu.setEnabled(!selPrjLocked);
             exportToArchMi.setEnabled(!selPrjLocked);
             exportToEmuDBMi.setEnabled(!selPrjLocked);
             renameMi.setEnabled(!selPrjLocked);
             deleteMi.setEnabled(!selPrjLocked);
             duplicateMi.setEnabled(!selPrjLocked);
             
         }
           
    }

	@Override
	public void contentsChanged(ListDataEvent arg0) {
		updateUIDependencies();
	}

	@Override
	public void intervalAdded(ListDataEvent arg0) {
		updateUIDependencies();
	}

	@Override
	public void intervalRemoved(ListDataEvent arg0) {
		updateUIDependencies();
	}

	@Override
	public void propertyChange(PropertyChangeEvent arg0) {
		updateUIDependencies();
	}

   

}
