//    Speechrecorder
//    (c) Copyright 2018-2020
// 	  Institute of Phonetics and Speech Processing,
//    Ludwig-Maximilians-University, Munich, Germany
//
//
//    This file is part of Speechrecorder
//
//
//    Speechrecorder is free software: you can redistribute it and/or modify
//    it under the terms of the GNU Lesser General Public License as published by
//    the Free Software Foundation, version 3 of the License.
//
//    Speechrecorder is distributed in the hope that it will be useful,
//    but WITHOUT ANY WARRANTY; without even the implied warranty of
//    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
//    GNU Lesser General Public License for more details.
//
//    You should have received a copy of the GNU Lesser General Public License
//    along with Speechrecorder.  If not, see <http://www.gnu.org/licenses/>.

package ipsk.apps.speechrecorder.prompting;

import java.awt.Color;
import java.awt.Dimension;
import java.awt.Font;
import java.awt.GridBagConstraints;
import java.awt.GridBagLayout;

import javax.swing.JFrame;
import javax.swing.JTextPane;
import javax.swing.SwingUtilities;
import javax.swing.text.Document;

import ipsk.apps.speechrecorder.MIMETypes;
import ipsk.apps.speechrecorder.prompting.presenter.PromptPresenter;
import ipsk.apps.speechrecorder.prompting.presenter.PromptPresenterException;
import ipsk.apps.speechrecorder.prompting.presenter.UnsupportedContentException;
import ipsk.apps.speechrecorder.script.ui.prompt.styled.PromptDocEditorKit;
import ipsk.apps.speechrecorder.script.ui.prompt.styled.PromptStyleContext;
import ipsk.apps.speechrecorder.script.ui.prompt.styled.PromptStyledDocument;
import ipsk.db.speech.script.VirtualViewBox;
import ipsk.db.speech.script.prompt.Mediaitem;
import ipsk.db.speech.script.prompt.doc.PromptDoc;
import ipsk.text.ParserException;
import ipsk.util.LocalizableMessage;
import ipsk.util.services.Description;
import ipsk.util.services.Title;
import ipsk.util.services.Vendor;
import ipsk.util.services.Version;

/**
 * Displays SpeechRecorder script formatted prompt text.
 * 
 */

@Title("Formatted text prompter")
@Description("Presents Speechrecorder formatted text.")
@Vendor("Institute of Phonetics and Speech processing, Munich")
@Version(major = 0,minor=0,subminor=1)
public class FormattedPromptTextViewer extends BasicPromptPresenter implements PromptPresenter {

	private static final long serialVersionUID = 1L;

	public static final PromptPresenterServiceDescriptor DESCRIPTOR = new BasicPromptPresenterServiceDescriptor(
			FormattedPromptTextViewer.class.getName(), new LocalizableMessage("Formatted text prompter"),
			"Institute of Phonetics and Speech processing, Munich", new ipsk.text.Version(new int[] { 0, 0, 1 }),
			new LocalizableMessage("Presents formatted prompt text."),
			FormattedPromptTextViewer.getSupportedMIMETypes());
	
	public final static float VIRTUAL_HEIGHT=600;
	
	public final static String DEF_CHARSET = "UTF-8";

//	private JAntiAliasedEditorPane textPane;
	private JTextPane textPane;

	private Color backgroundColor = Color.WHITE;
	private static String DEFAULT_FONT_NAME = "sans-serif";
	private static int DEFAULT_FONT_STYLE = Font.BOLD;
	private int DEFAULT_FONT_SIZE = 48;
	private int fontSize;
	
	private boolean updatingDoc=false;
	private Integer renderedHeight=null;

	public FormattedPromptTextViewer() {
	    super();

	    setBackground(backgroundColor);
	    
	    // Overwriting paint or paintComponent does not work!!
//	    textPane=new JAntiAliasedEditorPane();
	    textPane=new JTextPane();
	    // Enable to debug vertical centering issues
	    //textPane.setBackground(Color.YELLOW);
	    textPane.setEditable(false);
	    PromptDocEditorKit promptDocKit=new PromptDocEditorKit(true);
        textPane.setEditorKitForContentType("text/x-prompt",promptDocKit);
	    //setLayout(new CenterMiddleLayout(false));
	    setLayout(new GridBagLayout());
	    GridBagConstraints gbc=new GridBagConstraints();
	    gbc.fill=GridBagConstraints.HORIZONTAL;
	    gbc.weightx=2.0;
	    add(textPane,gbc);
	    fontSize = DEFAULT_FONT_SIZE;

	    promptFont = new Font(DEFAULT_FONT_NAME, DEFAULT_FONT_STYLE, fontSize);
	}

	/**
	 * Sets font for prompts.
	 * 
	 * @param promptFont prompt font
	 */
	public void setPromptFont(Font promptFont) {
		this.promptFont = promptFont;
	
	}

	public boolean getEmphasized() {

		return emphasized;
	}

	public void setEmphasized(boolean emphasized) {
		this.emphasized = emphasized;
		if (emphasized) {
			setForeground(Color.black);
		} else {
			setForeground(Color.lightGray);
		}
	}

	public static String[][] getSupportedMIMETypes() {
	    
		return getSupportedMIMETypes(new String[] {MIMETypes.FORMATTEDPROMPTTEXTMIMETYPE});
	}

	public void loadContents() throws PromptPresenterException {
	    if (mediaitems == null || mediaitems.length == 0) {
	        throw new UnsupportedContentException("No media item to display!");
	    } else if (mediaitems.length > 1) {
	        throw new UnsupportedContentException("Multiple media items not supported!");
	    }
	    Mediaitem mi = mediaitems[0];
	    PromptDoc prompt=mi.getPromptDoc();
	    
	    if(prompt!=null) {
	    	//TODO
	    	// default style from Speechrecorder project default prompt font 

	    	// scale to virtual height

	    	PromptStyledDocument doc;
	    	PromptStyleContext psc=null;
	    	//Style style=prompt.getStyle();
	    	
	    	VirtualViewBox vvb=mi.getDefaultVirtualViewBox();
	    	if(vvb==null) {
	    		psc=new PromptStyleContext(promptFont);
	    	}else {
	    		psc=scaledPromptStyleContext(vvb.getHeight());
	    	}
	    	
	    	try {
	    		
	    		//System.out.println("PSC: "+psc);
	    		doc = new PromptStyledDocument(psc, prompt);
	    	} catch (ParserException e) {
	    		throw new PromptPresenterException(e);
	    	}

	    	updatingDoc=true;
	    	String cCt=textPane.getContentType();
	    	if(!"text/x-prompt".equals(cCt)) {
	    		textPane.setContentType("text/x-prompt");
	    	}
	    	Document cDoc=textPane.getDocument();
	    	if(cDoc==null || !cDoc.equals(doc)) {
	    		textPane.setDocument(doc);


	    	}
	    	int h=getHeight();
	    	if(h>0) {
	    		renderedHeight=h;
	    	}
	    	updatingDoc=false;
	    	//System.out.println(System.identityHashCode(this)+" Pref size: "+textPane.getPreferredSize());

	    }else {
	    	renderedHeight=null;
	    }
	}
	

	public void doLayout(){
		//System.out.println(System.identityHashCode(this)+" Rendered height: "+renderedHeight+ " height: "+getHeight());
		boolean heightChanged=(renderedHeight==null || renderedHeight!=getHeight());
		
		if(heightChanged && !updatingDoc){
			SwingUtilities.invokeLater(new Runnable() {
				
				@Override
				public void run() {
					try {
						loadContents();
					} catch (PromptPresenterException e) {
						// TODO Auto-generated catch block
						e.printStackTrace();
					}
				}
			});
			
		}
	    super.doLayout();
	}
	
	private PromptStyleContext scaledPromptStyleContext(int virtualHeight){
		Font resFnt=promptFont;
	    
	    int h=getHeight();
	    if(h>0) {
	    	int promptFontSize=promptFont.getSize();
	    	float scaledPromptFontSize=promptFontSize*h/virtualHeight;
	    	Font scaledPromptFont=promptFont.deriveFont(scaledPromptFontSize);
	    	resFnt=scaledPromptFont;
	    	//System.out.println(System.identityHashCode(this)+" Font size: def: "+promptFontSize+" scld: "+scaledPromptFontSize+" h:"+h);
	    }
        return new PromptStyleContext(resFnt);
	}

	/*
	 * (non-Javadoc)
	 * 
	 * @see ipsk.apps.speechrecorder.prompting.presenter.PromptPresenter#
	 * getServiceDescriptor()
	 */
	public PromptPresenterServiceDescriptor getServiceDescriptor() {
		return DESCRIPTOR;
	}

	
	// Override to allow arbitrary down scaling
	public Dimension getMinimumSize(){
	    return new Dimension(0,0);
	}


	public static void main(String[] args) {

		Runnable r = new Runnable() {
			public void run() {
				FormattedPromptTextViewer vw = new FormattedPromptTextViewer();
				vw.setPreferredSize(new Dimension(600, 400));
				JFrame w = new JFrame();
				w.setDefaultCloseOperation(JFrame.EXIT_ON_CLOSE);
				w.getContentPane().add(vw);
				w.pack();
				w.setVisible(true);

				// vw.setContents("Welcome to the plain text prompt viewer!");
				//vw.testPrompt();
			}
		};
		SwingUtilities.invokeLater(r);
	}

}