//    Speechrecorder
//    (c) Copyright 2002-2020
//    Institute of Phonetics and Speech Processing,
//    Ludwig-Maximilians-University, Munich, Germany
//
//
//    This file is part of Speechrecorder
//
//
//    Speechrecorder is free software: you can redistribute it and/or modify
//    it under the terms of the GNU Lesser General Public License as published by
//    the Free Software Foundation, version 3 of the License.
//
//    Speechrecorder is distributed in the hope that it will be useful,
//    but WITHOUT ANY WARRANTY; without even the implied warranty of
//    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
//    GNU Lesser General Public License for more details.
//
//    You should have received a copy of the GNU Lesser General Public License
//    along with Speechrecorder.  If not, see <http://www.gnu.org/licenses/>.

//
//  PromptViewer.java
//  JSpeechRecorder
//
//  Created by Christoph Draxler on Thu Dec 05 2002.
//

package ipsk.apps.speechrecorder.prompting;

import ipsk.apps.speechrecorder.DialogTargetProvider;
import ipsk.apps.speechrecorder.MIMETypes;
import ipsk.apps.speechrecorder.prompting.event.PromptViewerClosedEvent;
import ipsk.apps.speechrecorder.prompting.event.PromptViewerEvent;
import ipsk.apps.speechrecorder.prompting.event.PromptViewerOpenedEvent;
import ipsk.apps.speechrecorder.prompting.event.PromptViewerPresenterClosedEvent;
import ipsk.apps.speechrecorder.prompting.event.PromptViewerStartedEvent;
import ipsk.apps.speechrecorder.prompting.event.PromptViewerStoppedEvent;
import ipsk.apps.speechrecorder.prompting.presenter.PromptPresenter;
import ipsk.apps.speechrecorder.prompting.presenter.PromptPresenterException;
import ipsk.apps.speechrecorder.prompting.presenter.PromptPresenterListener;
import ipsk.apps.speechrecorder.prompting.presenter.PromptPresenterPluginException;
import ipsk.apps.speechrecorder.prompting.presenter.event.PromptPresenterClosedEvent;
import ipsk.apps.speechrecorder.prompting.presenter.event.PromptPresenterEvent;
import ipsk.apps.speechrecorder.prompting.presenter.event.PromptPresenterOpenedEvent;
import ipsk.apps.speechrecorder.prompting.presenter.event.PromptPresenterStartEvent;
import ipsk.apps.speechrecorder.prompting.presenter.event.PromptPresenterStopEvent;
import ipsk.db.speech.script.Nonrecording;
import ipsk.db.speech.script.PromptItem;
import ipsk.db.speech.script.Reccomment;
import ipsk.db.speech.script.Recinstructions;
import ipsk.db.speech.script.Recording;
import ipsk.db.speech.script.prompt.Mediaitem;
import ipsk.db.speech.script.prompt.Presenter;

import java.awt.BorderLayout;
import java.awt.Color;
import java.awt.Component;
import java.awt.ComponentOrientation;
import java.awt.Font;
import java.beans.PropertyChangeEvent;
import java.beans.PropertyChangeListener;
import java.net.URL;
import java.text.Bidi;
import java.util.ArrayList;
import java.util.Arrays;
import java.util.HashSet;
import java.util.List;
import java.util.Set;
import java.util.Vector;

import javax.sound.sampled.Mixer;
import javax.swing.BorderFactory;
import javax.swing.JLabel;
import javax.swing.JOptionPane;
import javax.swing.JPanel;

/**
 * PromptViewer displays the recording script prompt on the experimenter and the
 * speaker display. The display changes according to the prompt item's type of
 * media:
 * <ul>
 * <li>prompt texts are displayed as is</li>
 * <li>image items are scaled to fit the display size</li>
 * <li>audio items are shown with a clickable audio button</li>
 * <li>video items are displayed with a controller to start, stop and pause the
 * display</li>
 * </ul>
 * PromptViewer also displays descriptive text associated with a prompt item.
 * 
 * Some display attributes change according to the recording status: the
 * instruction text is printed in black only during the recording phase,
 * otherwise it is gray.
 * 
 * @author draxler
 * 
 */
public class PromptViewer extends JPanel implements PropertyChangeListener, PromptPresenterListener,DialogTargetProvider {

    /**
     * 
     */
    private static final long serialVersionUID = -6099442015723739386L;
    private static final boolean DEBUG = false;

    public static enum Status {
        CLOSED, OPEN, RUNNING, STOPPED, PRESENTER_CLOSED
    }
 
    private List<PromptPresenterServiceDescriptor> promptPresenterServiceDescriptors;
    private List<PromptPresenter> plugins;
    private final String EMPTY_STRING = "";

    private Font promptFont;

    private Font instructionsFont = new Font("sans-serif", Font.BOLD, 48);

    private Font descriptionFont = new Font("sans-serif", Font.PLAIN, 14);

    private JLabel promptInstructions;

    private PromptPresenter promptPresenter;

    private JLabel promptDescription;

    private URL context;

    private ipsk.db.speech.script.PromptItem promptItem = null;

    private Integer recIndex;

    private boolean isSilent = false;

    private boolean showComments = true;
    private boolean instructionNumbering = true;

    private Vector<PromptViewerListener> listeners;

    private boolean showPrompt;

    private StartPromptPlaybackAction startPromptPlaybackAction;
    private StopPromptPlaybackAction stopPromptPlaybackAction;

    private Mixer promptMixer;

    private Status status;
    private int audioChannelOffset = 0;
    
    private DialogTargetProvider dialogTargetProvider;

    /**
     * Displays the individual prompt items according to their MIME type.
     * PromptViewer is notified of status changes from the prompt presenter.
     * 
     * @param promptPresenterServiceDescriptors
     *            list of available prompt presenter descriptors
     * @param startPromptPlaybackAction
     *            action to start prompt playback
     * @param stopPromptPlaybackAction
     *            action to stop prompt playback
     */
    public PromptViewer(List<PromptPresenterServiceDescriptor> promptPresenterServiceDescriptors,
            StartPromptPlaybackAction startPromptPlaybackAction, StopPromptPlaybackAction stopPromptPlaybackAction) {
        super(new BorderLayout());
        this.promptPresenterServiceDescriptors = promptPresenterServiceDescriptors;
        this.startPromptPlaybackAction = startPromptPlaybackAction;
        this.stopPromptPlaybackAction = stopPromptPlaybackAction;
        this.dialogTargetProvider=this;
        status = Status.CLOSED;
        promptInstructions = new JLabel(EMPTY_STRING);
        promptInstructions.setForeground(Color.darkGray);
        promptInstructions.setBackground(Color.lightGray);
        promptInstructions.setFont(instructionsFont);

        promptDescription = new JLabel(EMPTY_STRING);
        promptDescription.setFont(descriptionFont);

        setBorder(BorderFactory.createEmptyBorder(2, 2, 2, 2));

        listeners = new Vector<PromptViewerListener>();
        plugins = new ArrayList<PromptPresenter>();

    }

    /**
     * Returns combinations of supported MIME types
     * 
     * @return combinations of supported MIME types
     */
    public Set<List<String>> getSupportedMIMETypes() {
        Set<List<String>> s = new HashSet<List<String>>();
        String[][] ppMtcArrs = null;
        for (PromptPresenter pp : plugins) {
            PromptPresenterServiceDescriptor ppsd = pp.getServiceDescriptor();
            ppMtcArrs = ppsd.getSupportedMIMETypes();
            for (String[] ppMtcArr : ppMtcArrs) {
                List<String> ppMtc = Arrays.asList(ppMtcArr);
                s.add(ppMtc);
            }
        }
        return s;

    }

    public void setPromptMixer(Mixer mixer) throws PromptPresenterPluginException {
        promptMixer = mixer;
        for (PromptPresenter pp : plugins) {
            if (pp != null && pp instanceof MediaPromptPresenter) {
                ((MediaPromptPresenter) pp).setAudioMixer(promptMixer);
            }
        }
    }

    public void setPromptAudioChannelOffset(int audioChannelOffset) throws PromptPresenterPluginException {
        this.audioChannelOffset = audioChannelOffset;
        for (PromptPresenter pp : plugins) {
            if (pp != null && pp instanceof MediaPromptPresenter2) {
                ((MediaPromptPresenter2) pp).setAudioChannelOffset(audioChannelOffset);
            }
        }
    }

    /**
     * Sets the switch for audio output. If true, then audio sound is muted.
     * This switch is necessary to have only one source of audio output, e.g.
     * the speaker display, but not the experimenter display.
     * 
     * @param silence
     *            true to mute audio output
     */
    public void setSilent(boolean silence) {
        isSilent = silence;
    }

    /**
     * Returns the audio mute switch for the current display
     * 
     * @return true if the audio is muted
     */
    public boolean getSilent() {
        return isSilent;
    }

    /**
     * Returns true if the comments text is to be displayed,
     * false otherwise
     * 
     * @return true to show comments text
     */
    public boolean getShowComments() {
        return showComments;
    }

    /**
     * Sets the switch for displaying comments text to true or
     * false
     * 
     * @param comments true to show comments
     */
    public void setShowComments(boolean comments) {
        showComments = comments;
    }

    /**
     * checks whether instructions text from the current prompt item is the
     * empty string or null. If yes, then the instructions are not displayed,
     * otherwise the text is displayed.
     * 
     * @param instructions
     *            text of the instructions field of the current prompt item
     */
    public void displayInstructions(Recinstructions instructions) {

        this.remove(promptInstructions);
        String instructionsText = null;

        if (instructions != null) {
            instructionsText = instructions.getRecinstructions();
            instructionsText.trim();
            if (instructionNumbering) {
                instructionsText = recIndex + ": " + instructionsText;
            }
        }
        if (instructionsText == null || instructionsText.equals("")) {
        } else {
            promptInstructions.setText(instructionsText);
            promptInstructions.setHorizontalAlignment(JLabel.LEADING);
            Bidi bidi=new Bidi(instructionsText,Bidi.DIRECTION_DEFAULT_LEFT_TO_RIGHT);
            if(bidi.baseIsLeftToRight()){
            	promptInstructions.setComponentOrientation(ComponentOrientation.LEFT_TO_RIGHT);
            }else{
            	promptInstructions.setComponentOrientation(ComponentOrientation.RIGHT_TO_LEFT);
            }
            this.add(promptInstructions, BorderLayout.NORTH);
        }
    }

    /**
     * displayComments() checks whether comments should be displayed at all. If
     * yes, then it checks whether the comments text from the current prompt
     * item is the empty string or null. If yes, then the comments are not
     * displayed, otherwise the comments text is displayed.
     * 
     * @param comments
     *            text of the comment field of the current prompt item
     */
    public void displayComments(Reccomment comments) {
        this.remove(promptDescription);
        String commentsText = null;
        if (comments != null) {
            commentsText = comments.getReccomment();
            commentsText.trim();
        }
        if (!getShowComments()) {
        } else {
            if (commentsText == null || commentsText.equals("")) {
            } else {
                promptDescription.setText(commentsText);
                this.add(promptDescription, BorderLayout.SOUTH);
            }
        }
    }

    public void init() {
        promptInstructions.setText(EMPTY_STRING);
        displayInstructions(null);
        displayComments(null);
        setPromptPresenterEnabled(false);
        setShowPrompt(false);
    }

    private boolean supportsMIMEtypeCombination(PromptPresenterServiceDescriptor ppsd, String[] mimeTypeCombination) {
        // workaround for false legacy MIME types
        String serviceImplClassname = ppsd.getServiceImplementationClassname();
        if (PromptPlainTextViewer.class.getName().equals(serviceImplClassname)) {
            if (mimeTypeCombination != null && mimeTypeCombination.length == 1) {
                for (String falseMt : MIMETypes.FALSE_PLAINTEXTMIMETYPES) {
                    if (falseMt.equals(mimeTypeCombination[0]))
                        return true;
                }

            }
        }
        
        // TODO use CollectionUtils
        String[][] suppMimes = ppsd.getSupportedMIMETypes();
        if (suppMimes != null && mimeTypeCombination != null) {

            for (String[] suppMimeCombination : suppMimes) {
                boolean combinationMatch = true;
                if (suppMimeCombination.length != mimeTypeCombination.length) {
                    combinationMatch = false;

                } else {
                    Vector<String> suppMimeCombinationVector = new Vector<String>(Arrays.asList(suppMimeCombination));

                    for (String mimeType : mimeTypeCombination) {
                        boolean match = false;
                        int availsuppMimes = suppMimeCombinationVector.size();
                        for (int i = 0; i < availsuppMimes; i++) {
                            String am = suppMimeCombinationVector.get(i);
                            if (am.equals(mimeType)) {
                                // OK match
                                match = true;
                                suppMimeCombinationVector.remove(i);
                                break;
                            }
                        }
                        if (!match) {
                            combinationMatch = false;
                            break;
                        }
                    }
                }
                if (combinationMatch) {
                    return true;
                }
            }
        }
        return false;
    }

    public void prepare() throws PromptPresenterException {
        Recording recItem = null;
        Mediaitem[] mediaitemsArr = null;
        String[] mimeTypes = null;

        if (promptItem != null) {
            List<Mediaitem> miList = promptItem.getMediaitems();
            mediaitemsArr = miList.toArray(new Mediaitem[0]);
            mimeTypes = new String[miList.size()];
            for (int i = 0; i < mimeTypes.length; i++) {
                mimeTypes[i] = miList.get(i).getNNMimetype();
            }
            if (promptItem != null && !Arrays.asList(promptItem.getPropertyChangeListeners()).contains(this)) {
                promptItem.addPropertyChangeListener(this);
                for (Mediaitem mi : miList) {
                    mi.addPropertyChangeListener(this);
                }
            }

            if (promptItem instanceof Recording) {
                recItem = (Recording) promptItem;
                // //TODO
                // Mediaitem mi = recItem.getMediaitems().get(0);
                // mimeType = mi.getNNMimetype();
                // charSet=mi.getCharSet();
                // src = mi.getSrc();
                //
                // volume=mi.getNormalizedVolume();
                promptInstructions.setForeground(Color.black);
                Recinstructions instr = recItem.getRecinstructions();
                displayInstructions(instr);
                Reccomment rc = recItem.getReccomment();
                displayComments(rc);
            } else if (promptItem instanceof Nonrecording) {
                // nonrecItem = (Nonrecording) promptItem;
                // //TODO
                // Mediaitem mi = nonrecItem.getMediaitems().get(0);
                // mimeType = mi.getNNMimetype();
                // //charSet=nonrecItem.getNNCharSet();
                // src = mi.getSrc();
                //// if (src == null) {
                // text = mi.getText();
                //// }
                // volume=mi.getNormalizedVolume();
                displayInstructions(null);
                displayComments(null);
            }
        }
        if (DEBUG)
            System.out.println("Prepare");
        setShowPrompt(false);
        promptPresenter = null;
        if (promptItem != null) {
            Presenter presenter = promptItem.getPresenter();
            if (presenter != null) {
                // prompt requests a particular presenter plugin
                String type = presenter.getType();
                if (Presenter.TYPE_JAVA_CLASS.equalsIgnoreCase(type)) {
                    String presenterClassname = presenter.getClassname();
                    if (presenterClassname != null) {
                        boolean found = false;
                        // Check already loaded plugins
                        for (PromptPresenter pp : plugins) {
                            Class<?> ppClass = pp.getClass();
                            if (presenterClassname.equals(ppClass.getName())) {
                                promptPresenter = pp;
                                found = true;
                                break;
                            }
                        }
                        if (!found) {
                            Class<?> pClass = null;
                            Object presenterObj = null;
                            try {
                                pClass = Class.forName(presenterClassname);
                                presenterObj = pClass.newInstance();
                            } catch (ClassNotFoundException e) {
                                e.printStackTrace();
                            } catch (InstantiationException e) {
                                e.printStackTrace();
                            } catch (IllegalAccessException e) {
                                e.printStackTrace();
                            }
                            if (presenterObj != null && presenterObj instanceof PromptPresenter) {
                                promptPresenter = (PromptPresenter) presenterObj;
                                plugins.add(promptPresenter);
                            }
                        }
                    }
                }
            } else {
                // Get default prompt presenter for this MIME combination
                
                // Bug fix ID0065:
                for (PromptPresenterServiceDescriptor ppsd : promptPresenterServiceDescriptors) {
                    if (supportsMIMEtypeCombination(ppsd, mimeTypes)) {
                        String ppcn = ppsd.getServiceImplementationClassname();
                        // Lookup list of already loaded plugins
                        for (PromptPresenter pp : plugins) {
                            String lpcn = pp.getServiceDescriptor().getServiceImplementationClassname();
                            if (lpcn != null && lpcn.equals(ppcn)) {
                                promptPresenter = pp;
                                break;
                            }
                        }
                        // Default presenter descriptor found
                        break;
                    }

                }

            }
            if (promptPresenter == null && promptPresenterServiceDescriptors != null) {
                // prompt presenter not found in cache

                // lookup presenter service descriptors
                for (PromptPresenterServiceDescriptor ppsd : promptPresenterServiceDescriptors) {
                    if (supportsMIMEtypeCombination(ppsd, mimeTypes)) {
                        String ppcn = ppsd.getServiceImplementationClassname();
                        Class<?> ppc = null;
                        Object ppObj = null;
                        try {
                            // load class
                            ppc = Class.forName(ppcn);
                        } catch (ClassNotFoundException e) {
                            // fail silently and try next
                            if (DEBUG)
                                e.printStackTrace();
                            continue;
                        }
                        try {
                            ppObj = ppc.newInstance();
                        } catch (InstantiationException e) {
                            // fail silently and try next
                            if (DEBUG)
                                e.printStackTrace();
                            continue;
                        } catch (IllegalAccessException e) {
                            // fail silently and try next
                            if (DEBUG)
                                e.printStackTrace();
                            continue;
                        }
                        if (ppObj != null && ppObj instanceof PromptPresenter) {
                            // found presenter
                            promptPresenter = (PromptPresenter) ppObj;
                            plugins.add(promptPresenter);
                            break;
                        }

                    }
                }
            }

            if (promptPresenter == null) {
                // no presenter found
                if (mimeTypes.length == 1) {
                    throw new PromptPresenterException(
                            "Could not load presenter plugin for MIME type \"" + mimeTypes[0] + "\"");
                } else {
                    StringBuffer mimetypeCombinationSb = new StringBuffer();
                    int mimeTypesCount = mimeTypes.length;
                    for (int i = 0; i < mimeTypesCount; i++) {
                        mimetypeCombinationSb.append(mimeTypes[i]);
                        if (i + 1 < mimeTypesCount) {
                            mimetypeCombinationSb.append(',');
                        }
                    }
                    throw new PromptPresenterException("Could not load presenter plugin for MIME type combination \""
                            + mimetypeCombinationSb + "\"");
                }
            }
            setPromptEmphased(false);

            if (recItem != null) {

            }
            if (promptPresenter instanceof MediaPromptPresenter) {

                // ((MediaPromptPresenter)promptPresenter).setVolume((float)volumeInPercent/(float)100.0);
                MediaPromptPresenter mpp = ((MediaPromptPresenter) promptPresenter);
                mpp.setStartControlAction(startPromptPlaybackAction);
                mpp.setStopControlAction(stopPromptPlaybackAction);
                // mpp.setVolume(volume);
                mpp.setSilent(isSilent);
                mpp.setAudioMixer(promptMixer);
                if (promptPresenter instanceof MediaPromptPresenter2) {
                    MediaPromptPresenter2 mpp2 = (MediaPromptPresenter2) mpp;
                    mpp2.setAudioChannelOffset(audioChannelOffset);
                }
            }

            if (promptPresenter != null) {
                promptPresenter.setContextURL(context);
                promptPresenter.setPromptFont(promptFont);
                promptPresenter.setContents(mediaitemsArr);
                promptPresenter.loadContents();
            }
        }
    }

    public boolean isShowPrompt() {
        return showPrompt;
    }

    public void setShowPrompt(boolean showPrompt) {
    	if (showPrompt) {
    		if (!this.showPrompt) {
    			if (promptItem instanceof Recording) {
    				Recording recItem = (Recording) promptItem;
    				displayInstructions(recItem.getRecinstructions());
    				displayComments(recItem.getReccomment());
    			}

    			if (promptPresenter!=null && !isAncestorOf((Component) promptPresenter)) {
    				add((Component) promptPresenter, BorderLayout.CENTER);
    			}
    			setPromptPresenterEnabled(true);
    		}
    	} else {
    		if (promptPresenter != null) {
    			remove((Component) promptPresenter);
    		}
    	}
    	this.showPrompt = showPrompt;
    	if (DEBUG)
    		System.out.println(this.hashCode() + " show: " + this.showPrompt);
    	revalidate();
    	repaint();
    }

    public void open() throws PromptViewerException {
        if (Status.CLOSED.equals(status) || Status.PRESENTER_CLOSED.equals(status)) {
            if (DEBUG)
                System.out.println(this.hashCode() + " opening...");
            if (promptPresenter instanceof MediaPromptPresenter) {
                MediaPromptPresenter mpp = (MediaPromptPresenter) promptPresenter;
                mpp.removePromptPresenterListener(this);
                mpp.addPromptPresenterListener(this);
                try {
                    mpp.open();

                } catch (PromptPresenterException e) {
                    e.printStackTrace();
                        JOptionPane.showMessageDialog(
                        		dialogTargetProvider.getDialogTarget(), 
                        		e.getLocalizedMessage(),
                                "Prompt presenter error",
                            JOptionPane.ERROR_MESSAGE);
                    throw new PromptViewerException(e);
                }
            } else {
                // generate open event for non multimedia (immediately
                // displaying) prompters to start
                // recording

                status = Status.OPEN;
                if (DEBUG)
                    System.out.println(this.hashCode() + " open");
                updateListeners(new PromptViewerOpenedEvent(this));
            }
        }
    }

    public void start() {
        if (!Status.RUNNING.equals(status) && !Status.CLOSED.equals(status)
                && !Status.PRESENTER_CLOSED.equals(status)) {
            if (promptPresenter instanceof MediaPromptPresenter) {
                MediaPromptPresenter mpp = (MediaPromptPresenter) promptPresenter;
                try {
                    mpp.start();
                } catch (PromptPresenterException e) {
                    e.printStackTrace();
                        JOptionPane.showMessageDialog(
                        		dialogTargetProvider.getDialogTarget(),
                        		e.getLocalizedMessage(),
                                "Prompt presenter error",
                            JOptionPane.ERROR_MESSAGE);
                    return;
                }
            } else {
                // generate directly the final close event for non multimedia
                // prompters to start
                // recording
                status = Status.PRESENTER_CLOSED;
                updateListeners(new PromptViewerPresenterClosedEvent(this));
            }
        }
    }

    public void stop() {
        if (Status.RUNNING.equals(status)) {
            promptInstructions.setForeground(Color.lightGray);
            if (promptItem instanceof Recording) {
                Recording recItem = (Recording) promptItem;
                displayInstructions(recItem.getRecinstructions());
                displayComments(recItem.getReccomment());
            }
            if (promptPresenter instanceof MediaPromptPresenter) {
                try {
                    ((MediaPromptPresenter) promptPresenter).stop();
                } catch (PromptPresenterException e) {
                    e.printStackTrace();
                JOptionPane.showMessageDialog(
                		dialogTargetProvider.getDialogTarget(),
                		e.getLocalizedMessage(),
                        "Prompt presenter error", JOptionPane.ERROR_MESSAGE);
                }
            } else {
                status = Status.STOPPED;
            }
        }
    }

    private void _close() {
        status = Status.CLOSED;
        promptInstructions.setText(EMPTY_STRING);
        displayComments(null);
        if (promptPresenter != null) {
            remove((Component) promptPresenter);
        }
        if (DEBUG)
            System.out.println("Prompt viewer close. showPrompt aus");
        setShowPrompt(false);
        revalidate();
        repaint();
        updateListeners(new PromptViewerClosedEvent(this));
    }

    public void close() {

        if (!Status.CLOSED.equals(status)) {
            if (promptPresenter instanceof MediaPromptPresenter && !status.equals(Status.PRESENTER_CLOSED)) {
                try {
                    ((MediaPromptPresenter) promptPresenter).close();
                } catch (PromptPresenterException e) {
                    e.printStackTrace();
                    JOptionPane.showMessageDialog(
                    		dialogTargetProvider.getDialogTarget(),
                            e.getLocalizedMessage(), "Prompt presenter error",
                            JOptionPane.ERROR_MESSAGE);
                }
            } else {
                _close();
            }
        }
    }

    public void setInstructionsEmphased(boolean b) {
        if (b) {
            promptInstructions.setForeground(Color.black);
        } else {
            promptInstructions.setForeground(Color.lightGray);
        }
    }

    public void setPromptEmphased(boolean b) {
        if (promptPresenter != null)
            promptPresenter.setEmphasized(b);
    }

    public void setPromptPresenterEnabled(boolean b) {
        if (promptPresenter != null) {
            ((Component) promptPresenter).setEnabled(b);
        }
    }

    public void setPromptFont(Font font) {
        promptFont = font;
        if (promptPresenter != null) {
            promptPresenter.setPromptFont(promptFont);
        }
    }

    public Font getDescriptionFont() {
        return descriptionFont;
    }

    public Font getInstructionsFont() {
        return instructionsFont;
    }

    public void setDescriptionFont(Font font) {
        descriptionFont = font;
        promptDescription.setFont(descriptionFont);
    }

    public void setInstructionsFont(Font font) {
        instructionsFont = font;
        promptInstructions.setFont(instructionsFont);
    }

    public boolean isInstructionNumbering() {
        return instructionNumbering;
    }

    public void setInstructionNumbering(boolean instructionNumbering) {
        this.instructionNumbering = instructionNumbering;
    }

    /**
     * Get the context URL for prompt sources.
     * 
     * @return context URL
     */
    public URL getContext() {
        return context;
    }

    /**
     * Set the context URL for prompt sources. This is usually the workspace
     * project directory.
     * 
     * @param url
     *            context
     */
    public void setContext(URL url) {
        context = url;
    }

    protected synchronized void updateListeners(PromptViewerEvent event) {
        for (PromptViewerListener ppl : listeners) {
            ppl.update(event);
        }
    }

    public void addPromptViewerListener(PromptViewerListener listener) {
        if (listener != null && !listeners.contains(listener)) {
            listeners.addElement(listener);
        }
    }

    public void removePromptViewerListener(PromptViewerListener listener) {
        if (listener != null) {
            listeners.removeElement(listener);
        }
    }

    public void propertyChange(PropertyChangeEvent evt) {
        boolean showingBefore = isShowPrompt();
        if (evt.getPropertyName().startsWith("recording")) {
            if (DEBUG)
                System.out.println("PromptViewer: " + evt.getPropertyName());
        }
        try {
            prepare();
            if (showingBefore) {
                setShowPrompt(true);
            }
        } catch (PromptPresenterException e) {
            // exception occurs on every property, do not handle it here
        }
    }

    public ipsk.db.speech.script.PromptItem getPromptItem() {
        return promptItem;
    }

    public void setPromptItem(ipsk.db.speech.script.PromptItem promptItem) {
        PromptItem oldPromptItem = this.promptItem;
        if (oldPromptItem != null) {
            List<Mediaitem> miList = oldPromptItem.getMediaitems();
            for (Mediaitem mi : miList) {
                mi.removePropertyChangeListener(this);
            }
            oldPromptItem.removePropertyChangeListener(this);
        }
        this.promptItem = promptItem;

    }

    public int getRecIndex() {
        return recIndex;
    }

    public void setRecIndex(Integer recIndex) {
        this.recIndex = recIndex;
    }

    public void update(PromptPresenterEvent promptPresenterEvent) {
        if (promptPresenterEvent instanceof PromptPresenterOpenedEvent) {
            if (Status.PRESENTER_CLOSED.equals(status) || Status.CLOSED.equals(status)) {
                status = Status.OPEN;
                if (DEBUG)
                    System.out.println(this.hashCode() + " open");
                updateListeners(new PromptViewerOpenedEvent(this));
            }
        } else if (promptPresenterEvent instanceof PromptPresenterStartEvent) {
            if (!Status.RUNNING.equals(status)) {
                status = Status.RUNNING;
                updateListeners(new PromptViewerStartedEvent(this));
            }
        } else if (promptPresenterEvent instanceof PromptPresenterStopEvent) {
            if (Status.RUNNING.equals(status)) {
                status = Status.STOPPED;
                updateListeners(new PromptViewerStoppedEvent(this));
            }
        } else if (promptPresenterEvent instanceof PromptPresenterClosedEvent) {
            if (!Status.PRESENTER_CLOSED.equals(status)) {
                status = Status.PRESENTER_CLOSED;
                updateListeners(new PromptViewerPresenterClosedEvent(this));
            }
        }
    }

    public void closeMediaPresenter() {
        if (promptPresenter instanceof MediaPromptPresenter && !Status.PRESENTER_CLOSED.equals(status)) {
            try {
                ((MediaPromptPresenter) promptPresenter).close();
            } catch (PromptPresenterException e) {
                e.printStackTrace();
                    JOptionPane.showMessageDialog(
                    		dialogTargetProvider.getDialogTarget(),
                    		e.getLocalizedMessage(),
                            "Prompt presenter error",
                        JOptionPane.ERROR_MESSAGE);
            }
        }
    }

    public boolean isClosed() {
        return (Status.CLOSED.equals(status));
    }

    public Status getStatus() {
        return status;
    }

	public void setDialogTargetProvider(DialogTargetProvider dialogTargetProvider) {
		this.dialogTargetProvider=dialogTargetProvider;
	}



	/* (non-Javadoc)
	 * @see ipsk.apps.speechrecorder.DialogTargetProvider#getDialogTarget()
	 */
	@Override
	public Component getDialogTarget() {
		return this;
	}

 
  

}
