//    Speechrecorder
// 	  (c) Copyright 2016
// 	  Institute of Phonetics and Speech Processing,
//    Ludwig-Maximilians-University, Munich, Germany
//
//
//    This file is part of Speechrecorder
//
//
//    Speechrecorder is free software: you can redistribute it and/or modify
//    it under the terms of the GNU Lesser General Public License as published by
//    the Free Software Foundation, version 3 of the License.
//
//    Speechrecorder is distributed in the hope that it will be useful,
//    but WITHOUT ANY WARRANTY; without even the implied warranty of
//    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
//    GNU Lesser General Public License for more details.
//
//    You should have received a copy of the GNU Lesser General Public License
//    along with Speechrecorder.  If not, see <http://www.gnu.org/licenses/>.

package ipsk.apps.speechrecorder.annotation.auto;

import java.io.File;
import java.io.IOException;
import java.net.URISyntaxException;
import java.net.URL;
import java.nio.file.Files;
import java.nio.file.Path;
import java.util.ArrayList;
import java.util.HashMap;
import java.util.List;
import java.util.Locale;
import java.util.Set;
import java.util.UUID;

import javax.sound.sampled.AudioFormat;
import javax.sound.sampled.UnsupportedAudioFileException;
import javax.xml.bind.JAXBContext;
import javax.xml.bind.JAXBException;
import javax.xml.bind.Marshaller;

import ips.annot.BundleAnnotationPersistor;
import ips.annot.autoannotator.AutoAnnotator;
import ips.annot.autoannotator.AutoAnnotator.AnnotationRequest;
import ips.annot.io.BundleAnnotationFilePersistor;
import ips.annot.model.PredefinedLevelDefinition;
import ips.annot.model.db.Bundle;
import ips.annot.model.db.LevelDefinition;
import ips.annot.model.db.Session;
import ips.annot.model.emu.EmuBundleAnnotationPersistor;
import ips.annot.model.emu.EmuDB;
import ips.annot.model.emu.webapp.EMUwebAppConfig;
import ipsk.apps.speechrecorder.SpeechRecorderException;
import ipsk.apps.speechrecorder.annotation.auto.AutoAnnotationManager;
import ipsk.apps.speechrecorder.annotation.auto.impl.PromptAutoAnnotator;
import ipsk.apps.speechrecorder.annotation.auto.impl.TemplateAutoAnnotator;
import ipsk.apps.speechrecorder.config.ProjectConfiguration;
import ipsk.apps.speechrecorder.project.ActiveProjectManager;
import ipsk.apps.speechrecorder.project.ProjectManager;
import ipsk.apps.speechrecorder.project.ProjectManagerException;
import ipsk.apps.speechrecorder.storage.SessionStorageManager;
import ipsk.apps.speechrecorder.storage.StorageManagerException;
import ipsk.audio.AudioSourceException;
import ipsk.audio.ConvenienceFileAudioSource;
import ipsk.db.speech.LocalizedText;
import ipsk.db.speech.Project;
import ipsk.db.speech.RecordingFile;
import ipsk.db.speech.script.Recording;
import ipsk.io.StreamCopy;
import ipsk.text.EncodeException;
import ipsk.text.ParserException;
import ipsk.util.LocalizableMessage;
import ipsk.util.ProgressUpdate;
import ipsk.util.Task;

/**
 * @author klausj
 *
 */
public class ProjectAutoAnnotator implements Task{

    private ActiveProjectManager projectManager;
    private volatile boolean cancelRequest=false; 

    public ProjectAutoAnnotator(ActiveProjectManager projectManager) {
        super();
        this.projectManager = projectManager;
    }

    public void autoAnnotateProject(boolean overwrite, ProgressUpdate progressUpdate)
            throws IOException, URISyntaxException, SpeechRecorderException, ProjectManagerException {

        if (progressUpdate != null) {
            progressUpdate.getProgressStatus().setIndeterminate(true);
        }

        AutoAnnotationManager autoAnnotationManager = new AutoAnnotationManager();

        try {
            if (progressUpdate != null) {
                progressUpdate.getProgressStatus()
                        .setMessage(new LocalizableMessage("Calculate approximate duration..."));
                progressUpdate.fireProgressEvent();
            }
            projectManager.rebuildDb();

            Project prjDb = projectManager.getProjectDb();
            SessionStorageManager projectStorageManager = projectManager.getProjectStorageManager();

            int recFileCnt = 0;
            Set<ipsk.db.speech.Session> sesss = prjDb.getSessions();
            for (ipsk.db.speech.Session sess : sesss) {
                projectStorageManager.updateRecFiles(sess, true);
                Set<RecordingFile> rfs = sess.getRecordingFiles();
                if (rfs != null) {
                    recFileCnt += rfs.size();
                }
            }

            if (progressUpdate != null) {
                progressUpdate.getProgressStatus().setLength(recFileCnt);
                progressUpdate.getProgressStatus().setIndeterminate(false);
                progressUpdate.getProgressStatus().setMessage(new LocalizableMessage("Auto annotating..."));
                progressUpdate.fireProgressEvent();
            }

            List<BundleAnnotationPersistor> bundleAnnotationPersistorList = projectManager
                    .getBundleAnnotationPersistorList();

            int annotRecFileCnt = 0;

            for (ipsk.db.speech.Session sess : sesss) {

                Session annotSession = new Session();
                projectStorageManager.updateRecFiles(sess, true);
                int sessionId = sess.getSessionId();
                String sessIdStr = projectStorageManager.sessionIDString(sessionId);

                if (progressUpdate != null) {
                    progressUpdate.getProgressStatus()
                            .setMessage(new LocalizableMessage("Auto annotating session \"" + sessIdStr + "\"..."));
                    progressUpdate.fireProgressEvent();
                }

                File sessDir = projectStorageManager.sessionDirForSessionID(sessionId);

                annotSession.setName(sessIdStr);

                autoAnnotationManager.setAnnotationSession(annotSession);
                Set<RecordingFile> rfs = sess.getRecordingFiles();
                if (rfs != null) {
                    for (RecordingFile rf : rfs) {
                        Recording r = rf.getRecording();

                        String ic = r.getItemcode();
                        int recVers = 0;
                        Integer recVersion = rf.getVersion();
                        if (recVersion != null) {
                            recVers = recVersion;
                        }
                        String bundleRootFn = projectStorageManager.getRootFileName(sessionId, ic, sess.getCode(),
                                recVers);

                        String sigUrlsStr = rf.getSignalFile();
                        URL sigUrl = new URL(sigUrlsStr);
                        File recFile = new File(sigUrl.toURI().getPath());

                        // Check if auto annotation is required for this item (bundle) 
                        boolean annotRequired = false;

                        if(cancelRequest){
                            if (progressUpdate != null) {
                                progressUpdate.getProgressStatus().canceled();
                                progressUpdate.fireProgressEvent();
                            }
                            return;
                        }
                        
                        if (overwrite) {
                            
                            annotRequired = true;
                        } else {
                            // Check all requested persistors
                            for (BundleAnnotationPersistor bap : bundleAnnotationPersistorList) {
                                if (bap instanceof BundleAnnotationFilePersistor) {
                                    BundleAnnotationFilePersistor bafp = (BundleAnnotationFilePersistor) bap;
                                    File annoFile = new File(sessDir, bundleRootFn + bafp.getPreferredFilenameSuffix()
                                            + "." + bafp.getPreferredFileExtension());
                                    if (!annoFile.exists()) {
                                        // at least this requested annotation file is required
                                        annotRequired = true;
                                        break;
                                    }
                                }
                            }
                        }

                        if (annotRequired) {
                            Bundle annotBundle = null;

                            if (annotBundle == null) {
                                annotBundle = new Bundle();
                            }
                            Session s = annotBundle.getSession();
                            if (s == null) {
                                annotBundle.setSession(annotSession);
                            }
                            String bNm = annotBundle.getName();
                            if (bNm == null) {
                                annotBundle.setName(bundleRootFn);
                            }
                            String annotates = annotBundle.getAnnotates();
                            if (annotates == null) {
                                annotBundle.setAnnotates(recFile.getName());
                            }
                            List<String> sigPathes = annotBundle.getSignalpaths();
                            if (sigPathes == null) {
                                sigPathes = new ArrayList<String>();
                            }
                            if (sigPathes.size() == 0) {
                                sigPathes.add(recFile.getAbsolutePath());
                                annotBundle.setSignalpaths(sigPathes);
                            }
                            Float sr = annotBundle.getSampleRate();
                            Long frLen = annotBundle.getFrameLength();
                            if (sr == null || frLen == null) {
                                try {
                                    ConvenienceFileAudioSource cfas = new ConvenienceFileAudioSource(recFile);
                                    AudioFormat af = cfas.getFormat();
                                    long fl = cfas.getFrameLength();
                                    annotBundle.setSampleRate(af.getSampleRate());
                                    annotBundle.setFrameLength(fl);
                                } catch (AudioSourceException e) {

                                    e.printStackTrace();
                                    // OK could not retrieve the sample rate and
                                    // frame length
                                }

                            }

                            for (AutoAnnotator aa : projectManager.getEnabledAutoAnnotators()) {
                                if (!aa.isBundleSupported(annotBundle)) {
                                    continue;
                                }
                                AnnotationRequest ar = new AnnotationRequest(annotBundle);
                                aa.setAnnotationRequest(ar);
                                if (aa instanceof PromptAutoAnnotator) {
                                    PromptAutoAnnotator paa = (PromptAutoAnnotator) aa;
                                    paa.open();
                                    paa.setPromptText(r.getDescription());
                                    paa.call();
                                    paa.close();
                                } else if (aa instanceof TemplateAutoAnnotator) {
                                    TemplateAutoAnnotator taa = (TemplateAutoAnnotator) aa;
                                    taa.open();
                                    String templText = null;
                                    LocalizedText lt = r.annotationTemplateLocalizedText();
                                    if (lt != null) {
                                        templText = lt.getText();
                                        Locale templLocale = lt.getLocale();
                                        if (templLocale == null) {
                                            templLocale = java.util.Locale.getDefault();
                                        }
                                        annotBundle.setLocale(templLocale);
                                    }
                                    taa.setTemplateText(templText);
                                    taa.call();
                                    taa.close();

                                } else {
                                    aa.open();
                                    aa.call();
                                    aa.close();
                                }
                            }
                            for (BundleAnnotationPersistor bap : bundleAnnotationPersistorList) {
                                if (bap instanceof BundleAnnotationFilePersistor) {
                                    BundleAnnotationFilePersistor bafp = (BundleAnnotationFilePersistor) bap;
                                    File annoFile = new File(sessDir, bundleRootFn + bafp.getPreferredFilenameSuffix()
                                            + "." + bafp.getPreferredFileExtension());
                                    if (!annoFile.exists() || overwrite) {
                                        bafp.setFile(annoFile);
                                        if (bap.isWritable(annotBundle)) {

                                            try {
                                                bap.write(annotBundle);
                                            } catch (EncodeException ee) {
                                                ee.printStackTrace();
                                                // Continue with other file
                                                // writers
                                                // TODO Log this error
                                            }
                                        } else {
                                            // Log this
                                            // Only happens when tried to write
                                            // TextRid file without at least one
                                            // level

                                        }
                                    }
                                }
                            }
                        }
                        annotRecFileCnt++;
                        if (progressUpdate != null) {
                            progressUpdate.getProgressStatus().setProgress(annotRecFileCnt);
                            progressUpdate.fireProgressEvent();
                        }
                    }

                }
            }
        } catch (Exception e) {
            // TODO Auto-generated catch block
            e.printStackTrace();
        }

    }

    /* (non-Javadoc)
     * @see ipsk.util.Task#cancel()
     */
    @Override
    public void cancel() {
       cancelRequest=true;
        
    }

}
