//    Speechrecorder
//    (c) Copyright 2012-2020
// 	  Institute of Phonetics and Speech Processing,
//    Ludwig-Maximilians-University, Munich, Germany
//
//
//    This file is part of Speechrecorder
//
//
//    Speechrecorder is free software: you can redistribute it and/or modify
//    it under the terms of the GNU Lesser General Public License as published by
//    the Free Software Foundation, version 3 of the License.
//
//    Speechrecorder is distributed in the hope that it will be useful,
//    but WITHOUT ANY WARRANTY; without even the implied warranty of
//    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
//    GNU Lesser General Public License for more details.
//
//    You should have received a copy of the GNU Lesser General Public License
//    along with Speechrecorder.  If not, see <http://www.gnu.org/licenses/>.

package ipsk.apps.speechrecorder.session.progress;

import java.net.URL;
import java.util.EventListener;
import java.util.List;
import java.util.Vector;
import java.util.logging.Logger;

import javax.swing.DefaultListSelectionModel;
import javax.swing.JOptionPane;
import javax.swing.ListSelectionModel;
import javax.swing.event.ListSelectionListener;
import javax.swing.table.AbstractTableModel;

import ipsk.apps.speechrecorder.RecLogger;
import ipsk.apps.speechrecorder.UIResources;
import ipsk.apps.speechrecorder.session.action.SetIndexAction;
import ipsk.apps.speechrecorder.session.action.SetIndexEvent;
import ipsk.apps.speechrecorder.storage.ActiveSessionStorageManager;
import ipsk.apps.speechrecorder.storage.StorageManagerException;
import ipsk.db.speech.script.Group;
import ipsk.db.speech.script.Nonrecording;
import ipsk.db.speech.script.PromptItem;
import ipsk.db.speech.script.Recording;
import ipsk.db.speech.script.Script;
import ipsk.db.speech.script.Section;

/**
 * Loads the recording script and manages the progress through
 * the recording. It keeps track of the items recorded within the current
 * sessionid, and it knows which item to record next.
 * 
 * RecScriptManager is the model for the ProgressViewer. ProgressViewer displays
 * the recording script in a table. The column names are defined in the resource
 * file of the GUI. Each table row represents a recording item.
 * 
 * @author Christoph Draxler
 * @version 2.0 Feb. 2004
 * 
 *  
 */

public class ProgressManager extends AbstractTableModel implements ListSelectionModel{
    

	public static final int AUTOMATIC = 0;
	public static final int MANUAL = 1;
	public static final int SEQUENTIAL = 2;
	public static final int RANDOM = 3;
	
	public static final int COL_ITEM = 0;
	public static final int COL_URL = 1;
	public static final int COL_TEXT = 2;
	public static final int COL_REC = 3;
	public static final int COL_UPLOAD = 4;
	public static final int COLUMNS = 4;

	// arbitrary identifier required to set a custom cell renderer
	public static final String RECORDED_COL_ID="progress.table.col.recorded";
	
	public static final int ERROR_MSG_MAX_ITEMS=20;
	
	private String [] tableHead;
	private Logger logger;

	private UIResources uiString = null;
	
	private Script script;
	private boolean scriptSaved=true;
	
	private Integer recIndex;
	private int [] recCounter;
	private boolean [] recProcessed;

	//recResources stores all objects pre-fetched from URLs
	private URL context=null;

	private String systemIdBase = null;
	private SetIndexAction setIndexAction;
	public SetIndexAction getSetIndexAction() {
		return setIndexAction;
	}

	public void setSetIndexAction(SetIndexAction setIndexAction) {
		this.setIndexAction = setIndexAction;
	}

	private ActiveSessionStorageManager storageManager;

    private DefaultListSelectionModel selModel;
    
    private boolean progresToNextUnrecorded=false;
   
    private Vector<ProgressManagerListener> listeners=new Vector<ProgressManagerListener>();
	
	/**
	 * Loads the recording script and organizes the sequence of
	 * recordings. A recording script can be either a text file (the use of text
	 * files is deprecated) or an XML file defined in a DTD or XML-Schema.
	 * Furthermore, RecScriptManager pre-fetches all resources referred to via
	 * URLs so that they can be displayed in the prompt window without delay.
	 * 
	 * The sequence of recordings can be either automatic mode in sequence or in
	 * random order, or it can be manual. For selecting prompts in manual mode
	 * the RecTransporter or the ProgressViewer are used.
	 * 
	 * RecScriptManager is implemented as a singleton because there can be only
	 * a single such manager for a given recording sessionid.
	 *  
	 */

	public ProgressManager() {
		super();
		logger = Logger.getLogger("ipsk.apps.speechrecorder");

	
		
		// get description of table columns from the GUI property file
		uiString = UIResources.getInstance();
		tableHead = new String[COLUMNS];
		tableHead[COL_ITEM] = uiString.getString("ItemNo");
		tableHead[COL_URL] = uiString.getString("ItemFile");
		tableHead[COL_TEXT] = uiString.getString("ItemPrompt");
		tableHead[COL_REC] = uiString.getString("ItemRecStatus");
		
		// TODO Not implemented yet !
		//tableHead[COL_UPLOAD] = uiString.getString("ItemRecSaved");
        
        selModel=new DefaultListSelectionModel();
        selModel.setSelectionMode(ListSelectionModel.SINGLE_SELECTION);
      
        
        initialize();
	}


	/**
	 * Initializes the RecScriptManager, i.e. set the current recording index
	 * and the maximum index to 0 and creates a new recording sections Vector 
	 */
	public void initialize() {
		recIndex = null;
		selModel.clearSelection();
	}
	

	
	/**
	 * Get currently used script.
	 * @return script
	 */
	public Script getScript() {
		return script;
	}
	
	
	/**
	 * Set the script to use.
	 * @param script
	 */
	public void setScript(Script script){
		this.script = script;
		
		initialize();
        resetItemMarkers();
        try {
			updateItemMarkers();
		} catch (ProgressManagerException e) {
			e.printStackTrace();
		}
	}
	
	
    public boolean needsSilenceDetector(){
        if(script!=null){
            return script.needsSilenceDetector();
        }
        return false;
    }
    
    public boolean needsBeep(){
        if(script!=null){
            return script.needsBeep();
        }
        return false;
    }

	/**
	 * checks which items have been recorded already and sets their
	 * recording counter and processing status accordingly.
	 * @throws ProgressManagerException progress manager error 
	 *
	 */
	public void updateItemMarkers () throws ProgressManagerException{
		try {
			checkRecordedItems();
			for (int i = 0; i < numberOfPromptItems(); i++) {
				if (recCounter[i] > 0) {
					recProcessed[i] = true;
				} else {
					recProcessed[i] = false;
				}
			}
		} catch (ProgressManagerException e) {
			e.printStackTrace();
			throw new ProgressManagerException(e);
		}finally{
			fireTableDataChanged();
		}
		
		
	}

	/**
	 * resets the recording index to 0 and marks all items as
	 * not recorded and not processed.
	 */
	public void resetItemMarkers() {
//		recIndex = null;
		int maxIndex=numberOfPromptItems();
		recCounter = new int[maxIndex];
		recProcessed = new boolean[maxIndex];
	}

	public void setStorageManager(ActiveSessionStorageManager sm) {
		storageManager = sm;
	}



	/**
	 * returns the recording section corresponding to the given
	 * recording item index
	 * 
	 * @return Section
	 */
	public Section getCurrentRecSection() {
	    Integer recIndex=getRecIndex();
	    Section section=null;
	    if(recIndex!=null && script !=null){
	    	section=script.sectionForItemIndex(recIndex);
	    }
	    return section;
	}
	
	public Section.PromptPhase currentPromptPhase(){
		Section cs=getCurrentRecSection();
		if(cs!=null){
			return cs.getNNPromptphase();
		}
		return null;
	}
	
	/**
	 * returns the information string of the current recording
	 * section.
	 * 
	 * @return String
	 */
	public String getRecSectionInfo() {
        Section section=getCurrentRecSection();
        if(section!=null){
            return section.getInfo();
        }else{
		return null;
        }
	}
	
	/**
	 * returns the prompt item corresponding to the given recording index
	 * @param promptIndex
	 * @return PromptItem
	 */
	public ipsk.db.speech.script.PromptItem getPromptItem(int promptIndex) {
        ipsk.db.speech.script.PromptItem promptItem = null;
		int index = promptIndex;
		if(script!=null){
		List<Section> sections=script.getSections();
		if(sections!=null){
		for (int i = 0; i < sections.size(); i++) {
			
		    Section s=sections.get(i);
		    List<PromptItem> pis=s.getShuffledPromptItems();
		    int pisSize=pis.size();
			if (index >= pisSize) {
				index = index - pisSize;
			} else {
				promptItem = pis.get(index);
				break;
			}
		}
		}
		}
		return promptItem;
	}
	
	/**
	 * returns the prompt item for the current recording item index
	 * @return PromptItem
	 */
	public ipsk.db.speech.script.PromptItem getCurrentPromptItem() {
	    Integer recIndex=getRecIndex();
	    if(recIndex==null){
	        return null;
	    }else{
	        return getPromptItem(recIndex);
	    }
	}
	
	public boolean currentPromptBlocking(){
		ipsk.db.speech.script.PromptItem pi=getCurrentPromptItem();
		if(pi instanceof Recording){
			return ((Recording)pi).getNNBlocked();
		}
		return false;
	}
	
	/**
	 * checks whether all items have been recorded or not
	 * @throws StorageManagerException 
	 */
	private void checkRecordedItems() throws ProgressManagerException {

		for (int i = 0; i < numberOfPromptItems(); i++) {
			ipsk.db.speech.script.PromptItem pi=getPromptItem(i);
            if (pi instanceof Recording){
			String itemCode = ((Recording)pi).getItemcode();
			if (itemCode != null) {
                try {
					recCounter[i] = storageManager.getRecordedVersions(itemCode);
				} catch (StorageManagerException e) {
					throw new ProgressManagerException(e);
				}
                
			}
            }
		}
	}


	/**
	 * returns the number of recordings for the prompt item with
	 * a given index
	 * 
	 * @param index
	 * @return the number of times the item was recorded
	 */
	public int getRecCounter(int index) {
	    
		if (index >= recCounter.length) return 0;
		return recCounter[index];
	}

	/**
	 * increments the recording counter for a given item
	 * 
	 * Discussion: should the fact that the count of recordings of this item has
	 * increased be reported to the ProgressViewer? How?
	 * 
	 * @param index
	 */
	public void incrementRecCounter(int index) {
		recCounter[index]++;
		//notify listeners that data has changed
		fireTableCellUpdated(index, COL_REC);
	}

	/**
	 * sets the processing flag of the current recording
	 * item to true.
	 * 
	 * @param index
	 */
	public void markItemAsProcessed(int index) {
		recProcessed[index] = true;
		fireTableCellUpdated(index, COL_UPLOAD);
	}

	/**
	 * adds the current prompt item data (LBR, CCD) to the recording logger
	 */
	public void setLogEntries() {
		RecLogger recLog = RecLogger.getInstance();
		PromptItem pi = (PromptItem) getPromptItem(recIndex);
		if (pi instanceof Recording){
			Recording r=(Recording)pi;
		recLog.setLogEntry("LBR: ", r.getMediaitems().get(0).getText());
		recLog.setLogEntry("CCD: ", r.getItemcode());
		}
	}

	/**
	 * decrements the current recording index by 1 if the
	 * original index was greater than 0; otherwise the index is set to the
	 * maximum recording index.
	 */
	public void decrementIndex(){
		if (recIndex > 0) {
			setRecIndex(recIndex-1);
		} else {
			setRecIndex(numberOfPromptItems() - 1);
		}
	}

	/**
	 * increments the current recording index by 1 if the
	 * original index was less than the maximum recording index; otherwise the
	 * index is set to 0.
	 */
	public void incrementIndex(){
		if (recIndex == numberOfPromptItems() - 1) {
			setRecIndex(0);
		} else {
			setRecIndex(recIndex+1);
		}
       
	}

	/**
	 * returns the index of the current recording.
	 * 
	 * @return index of current recording
	 */
	public Integer getRecIndex() {
		return recIndex;
	}

	/**
	 * Sets the recording index to a given value.
	 * @param index recording index
	 *  
	 */
	public void setRecIndex(Integer index){

	    if (index!=null && (index <0 || index >= numberOfPromptItems())) {
	        throw new IllegalArgumentException();
	    }
	    boolean changed;
	    if(recIndex==null){
	        changed=(index!=null);
	    }else{
	        if(index==null){
	            changed=true;
	        }else{
	            changed=(index.intValue() != recIndex.intValue());
	        }
	    }
	    recIndex=index;
	    if(changed){
	        if(recIndex!=null){
	            selModel.setSelectionInterval(recIndex, recIndex);
	        }else{
	            selModel.clearSelection();
	        }
	        fireSessionManagerUpdate(new SessionPositionChangedEvent(this,recIndex));
	    }
	    //        if(recIndex==null|| index != recIndex.intValue()){
	    //		recIndex = new Integer(index);
	    //        selModel.setSelectionInterval(recIndex, recIndex);
	    //        fireRecscriptManagerUpdate(new RecScriptPositionChangedEvent(this));
	    //        }
	}

	/**
	 * Returns the number of items in the recording script.
	 * 
	 * @return number of prompt items
	 */
	public int numberOfPromptItems() {
		if (script==null){
			return 0;
		}else{
			return script.numberOfPromptItems();
				}
			}

	/**
	 * advanceToNextRecording() selects the next non-recorded recording item. 
	 * If no free item can be found a dialog is displayed to inform the speaker 
	 * that the last item has been reached.
	 */
	public void advanceToNextRecording(){
		int tmpIndex = getRecIndex();
		int maxIndex = numberOfPromptItems();

		boolean itemAlreadyRecorded = true;
		while (itemAlreadyRecorded && tmpIndex < maxIndex) {
            ipsk.db.speech.script.PromptItem pi = getPromptItem(tmpIndex);
            if (pi instanceof Recording) {
                logger.info("recCounter[" + tmpIndex + "] = "
                        + recCounter[tmpIndex]);
                if (recCounter[tmpIndex] == 0) {
                    itemAlreadyRecorded = false;
                    break;
                }
            }

            tmpIndex++;
            if (tmpIndex == maxIndex) {
                // ask user whether searching for an unrecorded item
                // should start at the first item of the list and search up to
                // the current index
                int response = JOptionPane
                        .showConfirmDialog(null,
                                "No unrecorded items found. Continue search at first item?");
                if (response == JOptionPane.YES_OPTION) {
                    tmpIndex = 0;
                    
                    // the following line sets the index to 0, but doesn't continue search
                    //maxIndex = tmpIndex;
                } else {
                    tmpIndex = getRecIndex();
                    break;
                }
            }

        }
		setRecIndex(tmpIndex);
	}


	/**
	 * Returns true if all items have been recorded at least
	 * once, false otherwise. The RecScriptManager keeps track of all recordings
	 * performed in the current sessionid by incrementing a counter for every
	 * prompt item after it has been recorded.
	 * @return true if all recordings are done
	 */
	public boolean allRecordingsDone() {
		boolean allRecordingsDone = true;
		for (int i = 0; i < numberOfPromptItems(); i++) {
            ipsk.db.speech.script.PromptItem pi = getPromptItem(i);
            if (pi instanceof Recording) {
                if (getRecCounter(i) < 1) {
                    allRecordingsDone = false;
                    break;
                }
            }
        }
		return allRecordingsDone;
	}

	// overwriting AbstracTableModel methods with data from the promptList

	/**
	 * Returns the number of prompt items in the current recording script
	 * 
	 * @see javax.swing.table.TableModel#getRowCount()
	 */
	public int getRowCount() {
		return numberOfPromptItems();
	}

	/**
	 * Returns the number of items to be displayed in a prompt
	 * item. Currently, five items are displayed:
	 * <ol>
	 * <li>sequence number</li>
	 * <li>prompt item URL</li>
	 * <li>prompt item text or description</li>
	 * <li>recording indicator</li>
	 * <li>upload indicator</li>
	 * </ol>
	 * The display of the URL is truncated, recording and upload indicators are
	 * check boxes or integer values.
	 * 
	 * @see javax.swing.table.TableModel#getColumnCount()
	 */
	public int getColumnCount() {
		return COLUMNS;
	}

	/**
	 * Returns the appropriate item for the given row and column
	 * coordinates. The columns are given as
	 * <ol>
	 * <li>sequence number</li>
	 * <li>recording URL</li>
	 * <li>prompt item text or description</li>
	 * <li>recording indicator</li>
	 * <li>upload indicator</li>
	 * </ol>
	 * 
	 * @see javax.swing.table.TableModel#getValueAt(int, int)
	 */
	public Object getValueAt(int row, int column) {
		//logger.info("getValueAt(): " + row + ", " + column);
		if (numberOfPromptItems() == 0)
			return null;
        
		ipsk.db.speech.script.PromptItem pi =  getPromptItem(row);
		//Mediaitem mi=pi.getMediaitems().get(0);
        String itemCode="";
       
        if(pi instanceof Recording){
            Recording recItem=(Recording)pi; 
            itemCode=recItem.getItemcode();
        }
		if (column == 0) {
			return row;
		} else if (column == 1) {
			return itemCode;
		} else if (column == 2) {
			String descr=pi.getDescription();
			String descrOneLine=descr.replaceAll("\n+", " ");
			return descrOneLine;
		} else if (column == 3) {
            if(pi instanceof Nonrecording || row >=recCounter.length){
              return null; 
            }else {
				return Boolean.valueOf(recCounter[row] > 0);
        }
		} else if (column == 4) {
			return Boolean.valueOf(recProcessed[row]);
		} else {
			return null;
		}
	}

	/**
	 * Returns true if the cell at a given row and column index
	 * is editable, false otherwise
	 * 
	 * @param row
	 *            row index
	 * @param column
	 *            column index
	 * @return boolean true if cell is editable
	 */
	public boolean isCellEditable(int row, int column) {
		return false;
	}

	/**
	 * Returns the name of the column specified by the column
	 * index
	 * 
	 * @param col
	 *            column index
	 * @return String column name
	 */
	public String getColumnName(int col) {
		return tableHead[col];
	}

	/**
	 * Returns the Class of the column specified by the column index
	 * 
	 * @param col
	 *            column index
	 * @return Class of column
	 */
	public Class<?> getColumnClass(int col) {
		//logger.info("getColumnClass(): " + col);
		//return getValueAt(0, col).getClass();
		if(col==0){
		    return Integer.class;
		}else if(col==1){
		   return String.class;
		}else if(col==2){
           return String.class;
        }else if(col==3){
           return Boolean.class;
        }else if(col==4){
           return Boolean.class;
        }else{
            // Error!
            return String.class;
        }
	}

	public String getSystemIdBase() {
		return systemIdBase;
	}

	
	public void setSystemIdBase(String string) {
		systemIdBase = string;
	}

	/**
	 * @return Returns the script id attribute.
	 */
	public String getScriptID() {
		return script.getName();
	}

	/**
	 * Set the URL context (usually the project directory in the workspace). 
	 * @param context
	 */
	public void setContext(URL context) {
		this.context=context;
		
	}
	/** 
	 * Get the workspace project context.
	 * @return the URL workspace context
	 */
	public URL getContext() {
		return context;
	}

	/**
	 * @return logger
	 */
	public Logger getLogger() {
		return logger;
	}

    /**
     * resets the manager to the initial state
     */
    public void doClose(){
		setScript(null);
		resetItemMarkers();
		initialize();
    }

    /**
     * @param l
     * @see javax.swing.DefaultListSelectionModel#addListSelectionListener(javax.swing.event.ListSelectionListener)
     */
    public void addListSelectionListener(ListSelectionListener l) {
        selModel.addListSelectionListener(l);
    }

    /**
     * @param index0
     * @param index1
     * @see javax.swing.DefaultListSelectionModel#addSelectionInterval(int, int)
     */
    public void addSelectionInterval(int index0, int index1) {
    	// not expected to be called (multi selections disabled)
    	// selModel.addSelectionInterval(index0, index1);
    }

    /**
     * 
     * @see javax.swing.DefaultListSelectionModel#clearSelection()
     */
    public void clearSelection() {
        selModel.clearSelection();
    }

    /**
     * @see javax.swing.DefaultListSelectionModel#getAnchorSelectionIndex()
     */
    public int getAnchorSelectionIndex() {
        return selModel.getAnchorSelectionIndex();
    }

    /**
     * @see javax.swing.DefaultListSelectionModel#getLeadSelectionIndex()
     */
    public int getLeadSelectionIndex() {
        return selModel.getLeadSelectionIndex();
    }

    /**
     * @param <T>
     * @param listenerType
     * @return event listeners
     * @see javax.swing.DefaultListSelectionModel#getListeners(java.lang.Class)
     */
    public <T extends EventListener> T[] getListeners(Class<T> listenerType) {
        return selModel.getListeners(listenerType);
    }

    /**
     * @return selection listeners
     * @see javax.swing.DefaultListSelectionModel#getListSelectionListeners()
     */
    public ListSelectionListener[] getListSelectionListeners() {
        return selModel.getListSelectionListeners();
    }

    /**
     * @return max selection index
     * @see javax.swing.DefaultListSelectionModel#getMaxSelectionIndex()
     */
    public int getMaxSelectionIndex() {
        return selModel.getMaxSelectionIndex();
    }

   
    public int getMinSelectionIndex() {
        return selModel.getMinSelectionIndex();
    }

    /**
     * @return selection mode
     * @see javax.swing.DefaultListSelectionModel#getSelectionMode()
     */
    public int getSelectionMode() {
        return selModel.getSelectionMode();
    }

    /**
     * @return is value adjusting
     * @see javax.swing.DefaultListSelectionModel#getValueIsAdjusting()
     */
    public boolean getValueIsAdjusting() {
        return selModel.getValueIsAdjusting();
    }

    /**
     * @param index
     * @param length
     * @param before
     * @see javax.swing.DefaultListSelectionModel#insertIndexInterval(int, int, boolean)
     */
    public void insertIndexInterval(int index, int length, boolean before) {
        selModel.insertIndexInterval(index, length, before);
    }

    
    public boolean isLeadAnchorNotificationEnabled() {
        return selModel.isLeadAnchorNotificationEnabled();
    }

    
    public boolean isSelectedIndex(int index) {
        return selModel.isSelectedIndex(index);
    }

    
    public boolean isSelectionEmpty() {
        return selModel.isSelectionEmpty();
    }

    /**
     * @param leadIndex
     * @see javax.swing.DefaultListSelectionModel#moveLeadSelectionIndex(int)
     */
    public void moveLeadSelectionIndex(int leadIndex) {
        selModel.moveLeadSelectionIndex(leadIndex);
    }

    /**
     * @param index0
     * @param index1
     * @see javax.swing.DefaultListSelectionModel#removeIndexInterval(int, int)
     */
    public void removeIndexInterval(int index0, int index1) {
        selModel.removeIndexInterval(index0, index1);
    }

    /**
     * @param l
     * @see javax.swing.DefaultListSelectionModel#removeListSelectionListener(javax.swing.event.ListSelectionListener)
     */
    public void removeListSelectionListener(ListSelectionListener l) {
        selModel.removeListSelectionListener(l);
    }

    /**
     * @param index0
     * @param index1
     * @see javax.swing.DefaultListSelectionModel#removeSelectionInterval(int, int)
     */
    public void removeSelectionInterval(int index0, int index1) {
        selModel.removeSelectionInterval(index0, index1);
    }

    /**
     * @param anchorIndex
     * @see javax.swing.DefaultListSelectionModel#setAnchorSelectionIndex(int)
     */
    public void setAnchorSelectionIndex(int anchorIndex) {
        selModel.setAnchorSelectionIndex(anchorIndex);
    }

    /**
     * @param flag
     * @see javax.swing.DefaultListSelectionModel#setLeadAnchorNotificationEnabled(boolean)
     */
    public void setLeadAnchorNotificationEnabled(boolean flag) {
        selModel.setLeadAnchorNotificationEnabled(flag);
    }

    /**
     * @param leadIndex
     * @see javax.swing.DefaultListSelectionModel#setLeadSelectionIndex(int)
     */
    public void setLeadSelectionIndex(int leadIndex) {
        selModel.setLeadSelectionIndex(leadIndex);
    }

    /**
     * @param index0
     * @param index1
     * @see javax.swing.DefaultListSelectionModel#setSelectionInterval(int, int)
     */
    public void setSelectionInterval(int index0, int index1) {
        // user events from the progress viewer table
        if(setIndexAction!=null && setIndexAction.isEnabled() && index0==index1) {
        	setIndexAction.actionPerformed(new SetIndexEvent(this, index0));
        }
 
    }

    /**
     * @param selectionMode
     * @see javax.swing.DefaultListSelectionModel#setSelectionMode(int)
     */
    public void setSelectionMode(int selectionMode) {
        selModel.setSelectionMode(selectionMode);
    }

    /**
     * @param isAdjusting
     * @see javax.swing.DefaultListSelectionModel#setValueIsAdjusting(boolean)
     */
    public void setValueIsAdjusting(boolean isAdjusting) {
        selModel.setValueIsAdjusting(isAdjusting);
    }

	public PromptItem setCurrentPromptItem(PromptItem promptItem){
	    // try to find the item
	    if(script==null){
//	        throw new IllegalArgumentException();
	    	return null;
	    }
	    int newIndex=0;
	    List<Section> sections=script.getSections();
	    if(sections!=null){
	        for(Section s :sections){
	            List<Group> pis=s.getGroups();
	            for(Group g:pis){
	                for(PromptItem pi:g.getPromptItems()){
	                if(promptItem==pi){
	                    setRecIndex(newIndex);
	                    return pi;
	                }
	                newIndex++;
	                }
	            }
	        }
	    }
//	    throw new IllegalArgumentException();
	    return null;
	}
	
	
	  /**
     * Add listener.
     * 
     * @param acl
     *            new listener
     */
    public synchronized void addSessionManagerListener(ProgressManagerListener acl) {
        if (acl != null && !listeners.contains(acl)) {
            listeners.addElement(acl);
        }
    }

    /**
     * Remove listener.
     * 
     * @param acl
     *            listener to remove
     */
    public synchronized void removeSessionManagerListener(ProgressManagerListener acl) {
        if (acl != null) {
            listeners.removeElement(acl);
        }
    }

    protected synchronized void fireSessionManagerUpdate(ProgressManagerEvent event){
       
    	for( ProgressManagerListener listener:listeners){
            listener.update(event);
        }
    }

	public boolean isScriptSaved() {
		return scriptSaved;
	}

    public boolean isProgresToNextUnrecorded() {
        return progresToNextUnrecorded;
    }

    public void setProgresToNextUnrecorded(boolean progresToNextUnrecorded) {
        this.progresToNextUnrecorded = progresToNextUnrecorded;
    }

}