//    IPS Java Utils
//    (c) Copyright 2009-2011
//    Institute of Phonetics and Speech Processing,
//    Ludwig-Maximilians-University, Munich, Germany
//
//
//    This file is part of IPS Java Utils
//
//
//    IPS Java Utils is free software: you can redistribute it and/or modify
//    it under the terms of the GNU Lesser General Public License as published by
//    the Free Software Foundation, version 3 of the License.
//
//    IPS Java Utils is distributed in the hope that it will be useful,
//    but WITHOUT ANY WARRANTY; without even the implied warranty of
//    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
//    GNU Lesser General Public License for more details.
//
//    You should have received a copy of the GNU Lesser General Public License
//    along with IPS Java Utils.  If not, see <http://www.gnu.org/licenses/>.

package ipsk.io;

import java.io.File;
import java.io.FileInputStream;
import java.io.FileOutputStream;
import java.io.IOException;
import java.io.InputStream;
import java.io.InputStreamReader;
import java.io.OutputStream;
import java.io.OutputStreamWriter;
import java.io.Reader;
import java.io.Writer;
import java.nio.channels.FileChannel;
import java.nio.channels.FileLock;
import java.nio.charset.Charset;
import java.nio.file.Files;
import java.nio.file.Path;

/**
 * Utility methods to copy streams and files.
 * Note: This class was created before Java got its own convenience classes and methods to copy data. See @see java.nio.file.Files .
 * @author K.Jaensch, klausj@phonetik.uni-muenchen.de
 *
 */

public class StreamCopy {

    public static int DEF_BUF_SIZE=8192;
    
    public static Charset DEF_CHARSET=Charset.forName("UTF-8");
    
    /**
     * Copies data from src file to dst file.
     * The directory for the dst file must already exists. 
     * @param src source file
     * @param dst destination file
     * @throws IOException I/O exception
     */
    public static void copy(File src,File dst) throws IOException{
        copy(src,dst,false);
    }
    
    /**
     * Copies data from src file to dst file.
     * 
     * @param src source file
     * @param dst destination file
     * @param createParentDirectories if true create all necessary (parent) directories to create the destination file 
     * @throws IOException I/O exception
     */
    public static void copy(File src,File dst,boolean createParentDirectories) throws IOException{
        if(createParentDirectories){
            File parentDir=dst.getParentFile();
            //if(parentDir.isDirectory()){
                parentDir.mkdirs();
            //}
        }
        FileInputStream fis=new FileInputStream(src);
        FileOutputStream fos=new FileOutputStream(dst);
        copy(fis,fos);
    }
    
    /**
     * Copies data from src file path to dst file path.
     * 
     * @param src source file path
     * @param dst destination file path
     * @param createParentDirectories if true create all necessary (parent) directories to create dst file 
     * @throws IOException I/O exception
     */
    public static void copy(Path src,Path dst,boolean createParentDirectories) throws IOException{
        if(createParentDirectories){
            Path parentDir=dst.getParent();
            Files.createDirectories(parentDir);
        }
        FileInputStream fis=new FileInputStream(src.toFile());
        FileOutputStream fos=new FileOutputStream(dst.toFile());
        copy(fis,fos);
    }
    
//    /**
//     * Copies data from src file path to dst file path.
//     * 
//     * @param src
//     * @param dst
//     * @param createParentDirectories If true create all necessary (parent) directories to create dst file 
//     * @throws IOException
//     */
//    public static void copy(Path src,Path dst,boolean createParentDirectories,boolean locked) throws IOException{
//    	if(createParentDirectories){
//    		Path parentDir=dst.getParent();
//    		Files.createDirectories(parentDir);
//    	}
//    	FileInputStream fis=new FileInputStream(src.toFile());
//    	FileOutputStream fos=new FileOutputStream(dst.toFile());
//    	copy(fis,fos);
//    }
    
    /**
     * Copies data from source input stream to dst file.
     * @param src input stream
     * @param dst destination file 
     * @param createParentDirectories if true create all necessary (parent) directories to create destination file 
     * @throws IOException I/O exception
     */
    public static void copy(InputStream src,File dst,boolean createParentDirectories) throws IOException{
       copy(src,dst,createParentDirectories,false);
    }
    
    /**
     * Copies data from source input stream to destination file.
     * @param src input stream
     * @param dst destination file 
     * @param createParentDirectories if true create all necessary (parent) directories to create destination file 
     * @param locked lock destination file channel if set to true 
     * @throws IOException I/O exception
     */
    public static void copy(InputStream src,File dst,boolean createParentDirectories,boolean locked) throws IOException{
    	if(createParentDirectories){
            File parentDir=dst.getParentFile();
            parentDir.mkdirs();
        }
    	
    	if(locked) {
    		// Use try with resources for all resources
    		try(FileOutputStream fos=new FileOutputStream(dst);
    			FileChannel fileChannel=fos.getChannel();
    			FileLock fileLock=fileChannel.lock();
    			){
    				copy(src,fos,false);
    				fos.flush(); // Flush before the lock is released by auto close
    		}
    	}else{
    		FileOutputStream fos=new FileOutputStream(dst);
    		copy(src,fos);
    	}
    }
    
    /**
     * Copies data from source input stream to destination file path.
     * @param src input stream
     * @param dstPath destination file path
     * @param createParentDirectories If true create all necessary (parent) directories to create dst file path
     * @param locked lock destination file channel if set to true 
     * @throws IOException I/O exception
     */
    public static void copy(InputStream src,Path dstPath,boolean createParentDirectories,boolean locked) throws IOException{
    	if(createParentDirectories){
    		Path parentDirPath=dstPath.getParent();
    		Files.createDirectories(parentDirPath);
    	}

    	if(locked) {
    		// Use try with resources for all resources
    		try(
    				FileOutputStream fos=new FileOutputStream(dstPath.toFile());
    				FileChannel fileChannel=fos.getChannel();
    				FileLock fileLock=fileChannel.lock();
    				){
    			copy(src,fos,false);
    			fos.flush(); // Flush before the lock is released by auto close
    		}

    	}else{
    		FileOutputStream fos=new FileOutputStream(dstPath.toFile());
    		copy(src,fos);
    	}
    }
    
    /**
    * Reads data from input stream and appends it to the output stream.
    * Only the input stream is closed at end of input stream data.
    * @param is input stream
    * @param os output stream
    * @throws IOException I/O exception
    */ 
    public static void append(InputStream is,OutputStream os) throws IOException{
    	int read=0;
        byte[] buf=new byte[DEF_BUF_SIZE];
        try{
        while((read=is.read(buf))!=-1){
            if(read>0){
            os.write(buf,0,read);
            }
        }
        }catch(IOException ioe){
            throw ioe;
        }finally{
            is.close();
        }
    }
    
    /**
     * Reads data from source file and writes it to output stream.
     * Output stream is closed at end of input stream data of the file.
     * @param src source file
     * @param os output stream
     * @throws IOException I/O exception
     */
    public static void copy(File src,OutputStream os) throws IOException{
       FileInputStream fis=new FileInputStream(src);
       copy(fis,os,DEF_BUF_SIZE);
    }
    
    /**
     * Reads data from input stream and writes it to output stream.
     * Input and output stream are closed at end of input stream data.
     * @param is input stream
     * @param os output stream
     * @throws IOException I/O exception
     */
    public static void copy(InputStream is,OutputStream os) throws IOException{
       copy(is,os,DEF_BUF_SIZE);
    }
    
    /**
     * Reads data from input stream and writes it to output stream.
     * Input stream is closed at end of input stream data.
     * @param is input stream
     * @param os output stream
     * @param closeOutStream if true close output stream at end of copy 
     * @throws IOException I/O exception
     */
    public static void copy(InputStream is,OutputStream os,boolean closeOutStream) throws IOException{
        copy(is,os,closeOutStream,DEF_BUF_SIZE);
    }
    
    /**
      * Reads data from input stream and writes it to output stream.
     * Input stream is closed at end of input stream data.
     * @param is input stream
     * @param os output stream
     * @param bufSize use given buffer size in bytes
     * @throws IOException I/O exception
     */
    public static void copy(InputStream is,OutputStream os,int bufSize) throws IOException{
        copy(is,os,true,bufSize);
    }
    
    /**
     * Reads data from input stream and writes it to output stream.
     * Input stream is closed at end of input stream data.
     * @param is input stream
     * @param os output stream
     * @param closeOutStream if true close output stream at end of copy
     * @param bufSize use given buffer size in bytes
     * @throws IOException I/O exception
     */
    public static void copy(InputStream is,OutputStream os,boolean closeOutStream,int bufSize) throws IOException{
        int read=0;
        byte[] buf=new byte[bufSize];
        try{
        while((read=is.read(buf))!=-1){
            if(read>0){
            os.write(buf,0,read);
            }
        }
        }catch(IOException ioe){
            throw ioe;
        }finally{
            try{
                if(closeOutStream){
                    os.close();
                }
            }catch(IOException ioe){
                throw ioe;
            }finally{
                is.close();
            }
        }
        
    }
    
    /**
     * Copy characters from.
     * Input reader and output writer are closed after the copy process.
     * @param ir input reader
     * @param ow output writer
     * @throws IOException I/O exception
     */
    public static void copyChars(Reader ir,Writer ow) throws IOException{
    	int read=0;
    	char[] buf=new char[DEF_BUF_SIZE];
    	try{
    		while((read=ir.read(buf))!=-1){
    			if(read>0){
    				ow.write(buf,0,read);
    			}
    		}
    	}catch(IOException ioe){
    		throw ioe;
    	}finally{
    		try{
    			ow.close();
    		}catch(IOException ioe){
    			throw ioe;
    		}finally{
    			ir.close();
    		}
    	}
    }
    
    /**
     * Write string to file in UTF-8 encoding.
     * @param text source text
     * @param file destination file
     * @throws IOException I/O exception
     */
    public static void writeUTF8TextFile(String text,File file) throws IOException{
            writeTextFile(text, file, DEF_CHARSET);
     }
    
    /**
     * Write string to file.
     * @param text source text
     * @param file destination file
     * @param charset character set for encoding
     * @throws IOException I/O exception
     */
    public static void writeTextFile(String text,File file,Charset charset) throws IOException{
       FileOutputStream fos=new FileOutputStream(file);
       OutputStreamWriter osw=new OutputStreamWriter(fos,charset);
       osw.write(text);
       osw.close();
    }
    
    /**
     * Read text from input stream.
     * @param textStream text input stream
     * @param charset character set of input stream encoding
     * @return the text
     * @throws IOException I/O exception
     */
    public static String readTextStream(InputStream textStream,Charset charset) throws IOException{
        
        InputStreamReader isr=new InputStreamReader(textStream, charset);
        StringBuffer sb=new StringBuffer();
        int read=0;
        char[] buf=new char[DEF_BUF_SIZE];
        try{
            while((read=isr.read(buf))!=-1){
                if(read>0){
                    sb.append(buf,0,read);
                }
            }
        }catch(IOException ioe){
            throw ioe;
        }finally{
            try{
                isr.close();
            }catch(IOException ioe){
                throw ioe;
            }
        }
        return sb.toString();
    }
    
    /**
     * Read from text file.
     * @param textFile source text file
     * @param charset character set of input file encoding
     * @return the text
     * @throws IOException I/O exception
     */
    public static String readTextFile(File textFile,Charset charset) throws IOException{
        FileInputStream fis=new FileInputStream(textFile);
        return readTextStream(fis, charset);
    }
    
    /**
     * Stream input stream to System.out 
     * @param is input stream
     * @throws IOException I/O exception
     */
    public static void toSystemOut(InputStream is) throws IOException{
        copy(is,System.out,false);
    }
    /**
     * Stream input stream to System.err
     * @param is input stream
     * @throws IOException I/O exception
     */
    public static void toSystemErr(InputStream is) throws IOException{
        copy(is,System.err,false);
    }

}
