package ips.security.jaas;

import java.security.Principal;
import java.sql.Connection;
import java.sql.DriverManager;
import java.sql.PreparedStatement;
import java.sql.ResultSet;
import java.sql.SQLException;
import java.util.ArrayList;
import java.util.List;
import java.util.Map;
import java.util.Set;

import javax.security.auth.Subject;
import javax.security.auth.callback.Callback;
import javax.security.auth.callback.CallbackHandler;
import javax.security.auth.callback.NameCallback;
import javax.security.auth.callback.PasswordCallback;
import javax.security.auth.callback.UnsupportedCallbackException;
import javax.security.auth.login.AccountNotFoundException;
import javax.security.auth.login.FailedLoginException;
import javax.security.auth.login.LoginException;
import javax.security.auth.spi.LoginModule;

import org.jasypt.util.password.rfc2307.RFC2307SSHAPasswordEncryptor;

/**
 * Authenticates to SQL server via JDBC. Role principals are also added.
 * Configuration options: connectionURL: JDBC connection URL baseDN: bas
 * Distinguished name of base node accountsDN: DN part where accounts can be
 * found rolesDN: DN part where roles (groupOfNames) can be found
 * 
 * example JAAS config file:
 * 
 * JAASLDAPLoginTest { ips.net.auth.jaas.JDBCLoginModule Sufficient
 * dbURL=""
 * baseDN="dc=phonetik,dc=uni-muenchen,dc=de" accountsDN="ou=People"
 * rolesDN="ou=roles,ou=webapp" debug=true; };
 * 
 * @author klausj
 * 
 */
public class JDBCLoginModule implements LoginModule {
    private boolean debug = false;

    public final static String DB_CONN_URL = "dbURL";
    public final static String DB_USER = "dbUser";
    public final static String DB_PASSWORD = "dbPassword";
    public final static String ACCOUNT_TABLENAME_PARAM_NAME = "userTable";
    public final static String ROLES_TABLENAME_PARAM_NAME = "userRoleTable";
    public final static String USERNAME_COL_PARAM_NAME = "userNameCol";
    public final static String ACCOUNT_DISABLED_COL_PARAM_NAME = "accountDisabledCol";
    public final static String USERNAME_CASE_INSENSITIVE_COL_PARAM_NAME = "userNameCaseInsensitivCol";
    public final static String ROLENAME_COL_PARAM_NAME = "roleNameCol";
    public final static String CRED_COL_PARAM_NAME = "userCredCol";
//    public final static String CRED_TYPE_PARAM_NAME="userCredType"; // rfc2307 or strong
//    public final static String RFC2307_PAASWORD_COL_PARAM_NAME = "rf2307PasswordColumn";

    //private static final String JDBC_DRIVER_CLASSNAME_PARAM = null;
    // initial state
    private Subject subject;
    private CallbackHandler callbackHandler;
    //private Map<String,?> sharedState;
    private Map<String,?> options;

    // configurable option

    // the authentication status
    private boolean succeeded = false;
    private boolean commitSucceeded = false;

    // username and password
    private String username;
    private char[] password;

    //private String tableName;
    //private String usernameColumn;

    // principals for user and role
    private String principalName=null;
    private Principal userPrincipal;
    private List<RolePrincipal> rolePrincipalList = new ArrayList<RolePrincipal>();

    public void initialize(Subject subject, CallbackHandler callbackHandler,
            Map<String,?> sharedState, Map<String,?> options) {

        this.subject = subject;
        this.callbackHandler = callbackHandler;
        //this.sharedState = sharedState;
        this.options = options;

        // initialize any configured options
        debug = "true".equalsIgnoreCase((String) options.get("debug"));
        if (debug) {
            System.out.println(getClass().getName() + " Initialized");
        }
        // String
        // jdbcDricerCn=((String)(options.get(JDBC_DRIVER_CLASSNAME_PARAM))).trim();
        // Class.forName(jdbcDricerCn);

    }

    /**
     * Authenticate the user by prompting for a user name and password.
     * 
     * <p>
     * 
     * @return true in all cases since this <code>LoginModule</code> should not
     *         be ignored.
     * 
     * @exception FailedLoginException
     *                if the authentication fails.
     *                <p>
     * 
     * @exception LoginException
     *                if this <code>LoginModule</code> is unable to perform the
     *                authentication.
     */
    public boolean login() throws LoginException {

        // prompt for a user name and password
        if (callbackHandler == null) {
            throw new LoginException("No CallbackHandler available.");
        }

        Callback[] callbacks = new Callback[2];
        callbacks[0] = new NameCallback("Login: ");
        callbacks[1] = new PasswordCallback("Password: ", false);

        try {
            callbackHandler.handle(callbacks);
            username = ((NameCallback) callbacks[0]).getName();
            char[] tmpPassword = ((PasswordCallback) callbacks[1])
                    .getPassword();
            if (tmpPassword == null) {
                // treat a NULL password as an empty password
                tmpPassword = new char[0];
            }
            password = new char[tmpPassword.length];
            System.arraycopy(tmpPassword, 0, password, 0, tmpPassword.length);
            ((PasswordCallback) callbacks[1]).clearPassword();

        } catch (java.io.IOException ioe) {
            throw new LoginException(ioe.toString());
        } catch (UnsupportedCallbackException uce) {
            throw new LoginException("Error: Unsupported callback:"
                    + uce.getCallback());
        }

        // print debugging information
        if (debug) {
            System.out.println("[JDBCLoginModule] "
                    + "user entered user name: " + username);
            System.out
                    .println("[JDBCLoginModule] " + "user entered a password");

        }
        String tableName = ((String) options.get(ACCOUNT_TABLENAME_PARAM_NAME))
                .trim();
        String userNameColumn = ((String) options.get(USERNAME_COL_PARAM_NAME))
                .trim();
        String accountDisabledColumn = null;
        Object accountDisabledColumnOptObj=options.get(ACCOUNT_DISABLED_COL_PARAM_NAME);
        if (debug) {
        	System.out.println("[JDBCLoginModule] " + ACCOUNT_DISABLED_COL_PARAM_NAME+": "+accountDisabledColumnOptObj);
        }
        if(accountDisabledColumnOptObj instanceof String) {
        	String accountDisabledColumnOptStr=(String)accountDisabledColumnOptObj;
        	String accountDisabledColumnOptStrTr=accountDisabledColumnOptStr.trim();
        	if(!accountDisabledColumnOptStrTr.isBlank()) {
        		accountDisabledColumn=accountDisabledColumnOptStrTr;
        	}
        }
        
        String userNameCaseInsensitiveColumn = null;
        Object userNameCaseInsensitiveColumnOptObj=options.get(USERNAME_CASE_INSENSITIVE_COL_PARAM_NAME);
        if (debug) {
        	System.out.println("[JDBCLoginModule] " + USERNAME_CASE_INSENSITIVE_COL_PARAM_NAME+": "+userNameCaseInsensitiveColumnOptObj);
        }
        if(userNameCaseInsensitiveColumnOptObj instanceof String) {
        	String userNameCaseInsensitiveColumnOptStr=(String)userNameCaseInsensitiveColumnOptObj;
        	String userNameCaseInsensitiveColumnOptStrTr=userNameCaseInsensitiveColumnOptStr.trim();
        	if(!userNameCaseInsensitiveColumnOptStrTr.isBlank()) {
        		userNameCaseInsensitiveColumn=userNameCaseInsensitiveColumnOptStrTr;
        	}
        }
//        String rfc2307PasswordColumn = ((String) options
//                .get(RFC2307_PAASWORD_COL_PARAM_NAME)).trim();
        String passwordColumn = ((String) options
                .get(CRED_COL_PARAM_NAME)).trim();
        String rolesTableName = ((String) options
                .get(ROLES_TABLENAME_PARAM_NAME)).trim();
        String rolesColumn = ((String) options.get(ROLENAME_COL_PARAM_NAME))
                .trim();

        // String connectionURL="ldaps://ldap.phonetik.uni-muenchen.de:636";
        String dbConnectionURL = (String) options.get(DB_CONN_URL);
        String dbUser = ((String) options.get(DB_USER)).trim();
        String dbPasswd = ((String) options.get(DB_PASSWORD)).trim();

        if (debug) {
            System.out.println("[JDBCLoginModule] try to connect to "
                    + dbConnectionURL + " as user " + dbUser);
        }
        LoginException loginException=null;
        try {

            Connection conn = DriverManager.getConnection(dbConnectionURL,
                    dbUser, dbPasswd);
            if (debug) {
                System.out.println("[JDBCLoginModule] JDBC connection: "+conn);
            }
            String query = "SELECT " + userNameColumn;
            if(accountDisabledColumn!=null) {
            	query=query.concat(","+accountDisabledColumn);
            }
            if(userNameCaseInsensitiveColumn!=null) {
            	query=query.concat(","+userNameCaseInsensitiveColumn);
            }
            query=query.concat(","+passwordColumn + " FROM " + tableName + " WHERE ");
            if(userNameCaseInsensitiveColumn==null) {
            	query=query.concat(userNameColumn + "=?");
            }else {
            	if (debug) {
                    System.out.println("[JDBCLoginModule] Case-insensitive column: "+userNameCaseInsensitiveColumn);
                }
            	query=query.concat("(("+userNameColumn + "=? ) OR ("+userNameCaseInsensitiveColumn+"=TRUE AND lower("+userNameColumn+") = lower(?)))" );
            }
            
            PreparedStatement pst = conn.prepareStatement(query);
            pst.setString(1, username);
            if(userNameCaseInsensitiveColumn!=null) {
            	pst.setString(2, username);
            }
            ResultSet rs = pst.executeQuery();
          
            if(rs.next()){
            	
            	String rfc2307Password = rs.getString(passwordColumn);
            	RFC2307SSHAPasswordEncryptor pwe = new RFC2307SSHAPasswordEncryptor();

            	String passwordStr = String.valueOf(password);
            	boolean disabled=false;
            	if(accountDisabledColumn!=null) {
            		disabled=rs.getBoolean(accountDisabledColumn);
            	}

            	succeeded = !disabled && pwe.checkPassword(passwordStr, rfc2307Password);
            	
            	principalName=rs.getString(userNameColumn);
            	
            	if (succeeded) {
            		if (debug) {
                		System.out
                		.println("[JDBCLoginModule] user "+username+" authenticated as "+principalName+" .");
                	}
            		// get roles
            		String rquery = "SELECT " + userNameColumn + "," + rolesColumn
            				+ " FROM " + rolesTableName + " WHERE "
            				+ userNameColumn + "=?";
            		PreparedStatement rpst = conn.prepareStatement(rquery);
            		rpst.setString(1, principalName);
            		ResultSet rrs = rpst.executeQuery();

            		while (rrs.next()) {
            			
            			String role = rrs.getString(rolesColumn);
            			RolePrincipal rp = new RolePrincipal(role);
            			rolePrincipalList.add(rp);
            			if (debug) {
                    		System.out
                    		.println("[JDBCLoginModule] added role "+rp+" to user "+principalName);
                    	}
            		}
            	}else{
            		loginException=new FailedLoginException("Password could not be verified.");
            		if (debug) {
                		System.out
                		.println("[JDBCLoginModule] password could not be verified for user "+username);
                	}
            	}
            	passwordStr = null;
            }else{
            	loginException=new AccountNotFoundException(); 
            	if (debug) {
            		System.out
            		.println("[JDBCLoginModule] user "+username+" not found in database.");
            	}
            }
            conn.close();

            if (debug) {
            	System.out
            	.println("[JDBCLoginModule] finished password check for user "+username);
            }

        } catch (SQLException e) {

        	e.printStackTrace();
        	if (debug) {
        		System.out
        		.println("[JDBCLoginModule] could not connect to database.");
        	}
        	loginException= new FailedLoginException("Could not connect to database");
        }
        
        if(loginException!=null) {
        	throw loginException;
        }
        
        return true;

    }

    /**
     * <p>
     * This method is called if the LoginContext's overall authentication
     * succeeded (the relevant REQUIRED, REQUISITE, SUFFICIENT and OPTIONAL
     * LoginModules succeeded).
     * 
     * <p>
     * If this LoginModule's own authentication attempt succeeded (checked by
     * retrieving the private state saved by the <code>login</code> method),
     * then this method associates a <code>SamplePrincipal</code> with the
     * <code>Subject</code> located in the <code>LoginModule</code>. If this
     * LoginModule's own authentication attempted failed, then this method
     * removes any state that was originally saved.
     * 
     * <p>
     * 
     * @exception LoginException
     *                if the commit fails.
     * 
     * @return true if this LoginModule's own login and commit attempts
     *         succeeded, or false otherwise.
     */
    public boolean commit() throws LoginException {
        if (succeeded == false) {
            return false;
        } else {
            // add user principal (authenticated identity)
            // to the Subject

            Set<Principal> principalSet = subject.getPrincipals();

            userPrincipal = new JDBCPrincipal(principalName);
            if (!principalSet.contains(userPrincipal)) {
                principalSet.add(userPrincipal);
                if (debug) {
                    System.out.println("[JDBCLoginModule] added Principal "
                            + userPrincipal.getName() + " to Subject");
                }
            }

            for (RolePrincipal rp : rolePrincipalList) {

                if (!principalSet.contains(rp)) {
                    principalSet.add(rp);

                    if (debug) {
                        System.out
                                .println("[JDBCLoginModule] added RolePrincipal "
                                        + rp.getName() + " to Subject");
                    }
                }
            }

            // in any case, clean out state
            username = null;
            principalName=null;
            for (int i = 0; i < password.length; i++)
                password[i] = ' ';
            password = null;

            commitSucceeded = true;
            return true;
        }
    }

    /**
     * <p>
     * This method is called if the LoginContext's overall authentication
     * failed. (the relevant REQUIRED, REQUISITE, SUFFICIENT and OPTIONAL
     * LoginModules did not succeed).
     * 
     * <p>
     * If this LoginModule's own authentication attempt succeeded (checked by
     * retrieving the private state saved by the <code>login</code> and
     * <code>commit</code> methods), then this method cleans up any state that
     * was originally saved.
     * 
     * <p>
     * 
     * @exception LoginException
     *                if the abort fails.
     * 
     * @return false if this LoginModule's own login and/or commit attempts
     *         failed, and true otherwise.
     */
    public boolean abort() throws LoginException {
        if (succeeded == false) {
            return false;
        } else if (succeeded == true && commitSucceeded == false) {
            // login succeeded but overall authentication failed
            succeeded = false;
            username = null;
            principalName=null;
            if (password != null) {
                for (int i = 0; i < password.length; i++)
                    password[i] = ' ';
                password = null;
            }
            userPrincipal = null;
            rolePrincipalList.clear();
        } else {
            // overall authentication succeeded and commit succeeded,
            // but someone else's commit failed
            logout();
        }
        return true;
    }

    /**
     * Logout the user.
     * 
     * <p>
     * This method removes the <code>SamplePrincipal</code> that was added by
     * the <code>commit</code> method.
     * 
     * <p>
     * 
     * @exception LoginException
     *                if the logout fails.
     * 
     * @return true in all cases since this <code>LoginModule</code> should not
     *         be ignored.
     */
    public boolean logout() throws LoginException {

        subject.getPrincipals().clear();
        succeeded = false;
        succeeded = commitSucceeded;
        username = null;
        if (password != null) {
            for (int i = 0; i < password.length; i++)
                password[i] = ' ';
            password = null;
        }
        userPrincipal = null;
        rolePrincipalList.clear();
        return true;
    }
}
