package ips.security.jaas;

import java.security.Principal;
import java.time.Instant;
import java.util.Date;
import java.util.Hashtable;
import java.util.Map;
import java.util.Set;

import javax.naming.Context;
import javax.naming.NamingEnumeration;
import javax.naming.NamingException;
import javax.naming.directory.Attribute;
import javax.naming.directory.Attributes;
import javax.naming.directory.BasicAttribute;
import javax.naming.directory.BasicAttributes;
import javax.naming.directory.DirContext;
import javax.naming.directory.InitialDirContext;
import javax.naming.directory.SearchResult;
import javax.security.auth.Subject;
import javax.security.auth.callback.Callback;
import javax.security.auth.callback.CallbackHandler;
import javax.security.auth.callback.NameCallback;
import javax.security.auth.callback.PasswordCallback;
import javax.security.auth.callback.UnsupportedCallbackException;
import javax.security.auth.login.AccountExpiredException;
import javax.security.auth.login.CredentialExpiredException;
import javax.security.auth.login.FailedLoginException;
import javax.security.auth.login.LoginException;
import javax.security.auth.spi.LoginModule;


/**
 * Authenticates to LDAP server and stores some inetOrgPerson (RFC 2798) attributes to the user principal.
 * Role principals are also added.
 * Configuration options:
 * connectionURL: LDAP connection URL
 * baseDN: bas Distinguished name of base node
 * accountsDN: DN part where accounts can be found
 * rolesDN: DN part where roles (groupOfNames) can be found
 * 
 * example JAAS config file:
 * 
 * JAASLDAPLoginTest {
 *  ips.net.auth.jaas.LDAPLoginModule  Sufficient
 *   connectionURL="ldaps://ldap.phonetik.uni-muenchen.de:636"
 *   baseDN="dc=phonetik,dc=uni-muenchen,dc=de"
 *   accountsDN="ou=People"
 *   rolesDN="ou=roles,ou=webapp"
 *   defaultRole="USER"
 *   debug=true;
 * };
 *   
 * @author klausj
 *
 */
public class LDAPLoginModule implements LoginModule {
	private boolean debug = false;

	// initial state
	private Subject subject;
	private CallbackHandler callbackHandler;
	private Map<String,?> sharedState;
	private Map<String,?> options;

	// configurable option

	// the authentication status
	private boolean succeeded = false;
	private boolean commitSucceeded = false;

	// username and password
	private String username;
	private char[] password;

	private String baseDN;
	private String userDn;

	// principals for user and role 
	private InetOrgPersonPrincipal userPrincipal;
	private RolePrincipal rolePrincipal;

	private DirContext context=null;


	public void initialize(Subject subject, CallbackHandler callbackHandler,
			Map<String,?> sharedState, Map<String,?> options) {

		this.subject = subject;
		this.callbackHandler = callbackHandler;
		this.sharedState = sharedState;
		this.options = options;


		// initialize any configured options
		debug = "true".equalsIgnoreCase((String)options.get("debug"));
		if(debug){
			String implVersion=getClass().getPackage().getImplementationVersion();
			System.out.println("[LDAPLoginModule] LDAP login module version: "+implVersion);
			System.out.println("[LDAPLoginModule] "+getClass().getName()+" Initialized. Object: "+((Object)this).toString());
		}
	}

	/**
	 * Authenticate the user by prompting for a user name and password.
	 *
	 * <p>
	 *
	 * @return true in all cases since this <code>LoginModule</code>
	 *      should not be ignored.
	 *
	 * @exception FailedLoginException if the authentication fails. <p>
	 *
	 * @exception LoginException if this <code>LoginModule</code>
	 *      is unable to perform the authentication.
	 */
	public boolean login() throws LoginException {

		// prompt for a user name and password
		if (callbackHandler == null){
			throw new LoginException("No CallbackHandler available.");
		}

		Callback[] callbacks = new Callback[2];
		callbacks[0] = new NameCallback("Login: ");
		callbacks[1] = new PasswordCallback("Password: ", false);

		try {
			callbackHandler.handle(callbacks);
			username = ((NameCallback)callbacks[0]).getName();
			char[] tmpPassword = ((PasswordCallback)callbacks[1]).getPassword();
			if (tmpPassword == null) {
				// treat a NULL password as an empty password
				tmpPassword = new char[0];
			}
			password = new char[tmpPassword.length];
			System.arraycopy(tmpPassword, 0,
					password, 0, tmpPassword.length);
			((PasswordCallback)callbacks[1]).clearPassword();

		} catch (java.io.IOException ioe) {
			throw new LoginException(ioe.toString());
		} catch (UnsupportedCallbackException uce) {
			throw new LoginException("Error: Unsupported callback:" + uce.getCallback());
		}

		// print debugging information
		if (debug) {
			System.out.println("[LDAPLoginModule] " +
					"user entered user name: " +
					username);
			System.out.println("[LDAPLoginModule] " +
					"user entered a password");

		}
		baseDN=((String)options.get("baseDN")).trim();
		String accountsDN=((String)options.get("accountsDN")).trim();
		StringBuffer userDNBuffer=new StringBuffer("uid=");
		userDNBuffer.append(username);
		userDNBuffer.append(',');
		if (debug) {
			System.out.println("[LDAPLoginModule] accountsDN: "+accountsDN);
		}
		if(accountsDN!=null && ! "".equals(accountsDN)){
			userDNBuffer.append(accountsDN);
			if(!accountsDN.endsWith(",")){
				userDNBuffer.append(',');
			}
		}
		userDNBuffer.append(baseDN);

		// verify the username/password
		userDn=userDNBuffer.toString();


		//    String connectionURL="ldaps://ldap.phonetik.uni-muenchen.de:636";
		String connectionURL=(String)options.get("connectionURL");
		if (debug) {
			System.out.println("[LDAPLoginModule] try to connect to "+connectionURL+" as user "+userDn); 
		}
		Hashtable<Object, Object> env = new Hashtable<Object, Object>();
		// Provider (Implementierung setzen)
		env.put(Context.PROVIDER_URL, 
				connectionURL);
		env.put(Context.INITIAL_CONTEXT_FACTORY, "com.sun.jndi.ldap.LdapCtxFactory");
		env.put(Context.SECURITY_PRINCIPAL,userDn);
		env.put(Context.SECURITY_CREDENTIALS,password);
		// Kontext setzen (Directory-Kontext)
		try {
			context = new InitialDirContext(env);
			// if we could bind to LDAP user is successfully authenticated
			if (debug) {
				System.out.println("[LDAPLoginModule] successfully bound to LDAP.");
			}
			
			boolean expired=false;
			try {
				Attributes attrs=context.getAttributes(userDn);
				Attribute inetOrgPersonClassAttr=attrs.get("objectClass");
				NamingEnumeration<?> objClassesEnum=inetOrgPersonClassAttr.getAll();
				while(objClassesEnum.hasMore()){
					Object objClassObj=objClassesEnum.next();

					if("shadowAccount".equals(objClassObj)){
						if(debug) {
							System.out.println("[LDAPLoginModule] User account has object class shadowAccount.");
						}
						Attribute expireAttr=attrs.get("shadowExpire");
						if(expireAttr!=null) {
							Object expireObj=expireAttr.get();
							// rfc2252 INTEGER string
							if(expireObj instanceof String) {
								String expireStr=(String)expireObj;
								if(debug) {
									System.out.println("[LDAPLoginModule] User account shadowExpire: "+expireStr);
								}
								try {
									long shadowExpire=Long.valueOf(expireStr);
									if(shadowExpire>=0) {
										long expireEpochSeconds=shadowExpire*60*60*24;
										Instant now=Instant.now();
										Instant expire=Instant.ofEpochSecond(expireEpochSeconds);
										if(now.isAfter(expire)) {
											expired=true;
											if(debug) {
												System.out.println("[LDAPLoginModule] User account expired.");
											}
										}else {
											if(debug) {
												System.out.println("[LDAPLoginModule] User account not expired.");
											}
										}
									}
								}catch(NumberFormatException nfe) {
									// proceed (??)
									System.err.println("[LDAPLoginModule] Could not parse shadowExpire value "+expireStr);
								}
							}else {
								System.err.println("[LDAPLoginModule] Unexpected type of shadowExpire attribute (not a string).");
							}
							
						}
					}
					
				}

			} catch (NamingException e) {
				System.err.println(e.getMessage());
				e.printStackTrace();
				// no principals added
			}
			
			if(expired) {
				throw new AccountExpiredException();
			}
			succeeded=true;
			
			return succeeded;
		} catch (NamingException e) {
			if (debug) {
				System.out.println("[LDAPLoginModule] Could not bind to LDAP.");
			}
			throw new FailedLoginException("Could not bind to LDAP");
		}

	}

	/**
	 * <p> This method is called if the LoginContext's
	 * overall authentication succeeded
	 * (the relevant REQUIRED, REQUISITE, SUFFICIENT and OPTIONAL LoginModules
	 * succeeded).
	 *
	 * <p> If this LoginModule's own authentication attempt
	 * succeeded (checked by retrieving the private state saved by the
	 * <code>login</code> method), then this method associates a
	 * <code>SamplePrincipal</code>
	 * with the <code>Subject</code> located in the
	 * <code>LoginModule</code>.  If this LoginModule's own
	 * authentication attempted failed, then this method removes
	 * any state that was originally saved.
	 *
	 * <p>
	 *
	 * @exception LoginException if the commit fails.
	 *
	 * @return true if this LoginModule's own login and commit
	 *      attempts succeeded, or false otherwise.
	 */
	public boolean commit() throws LoginException {
		if (succeeded == false) {
			return false;
		} else {
			// add user principal (authenticated identity)
			// to the Subject

			Set<Principal> principalSet=subject.getPrincipals();
			try {
				Attributes attrs=context.getAttributes(userDn);
				Attribute inetOrgPersonClassAttr=attrs.get("objectClass");
				NamingEnumeration<?> objClassesEnum=inetOrgPersonClassAttr.getAll();
				while(objClassesEnum.hasMore()){
					Object objClassObj=objClassesEnum.next();

					if("inetOrgPerson".equals(objClassObj)){
						userPrincipal = new InetOrgPersonPrincipal(userDn,attrs);
						if (!principalSet.contains(userPrincipal)){
							principalSet.add(userPrincipal);
							if (debug) {
								System.out.println("[LDAPLoginModule] added InetOrgPrincipal "+userPrincipal.getName()+" to Subject");
							}
						}
						if(debug){
							System.out.println("[LDAPLoginModule] Displayname: "+userPrincipal.getAttrDisplayname());
							System.out.println("[LDAPLoginModule] Given name: "+userPrincipal.getAttrGivenname());
							System.out.println("[LDAPLoginModule] Surname: "+userPrincipal.getAttrSurname());
							System.out.println("[LDAPLoginModule] E-Mail: "+userPrincipal.getAttrMail());
						}
					}
					// TODO accept posixAccounts?
				}



			} catch (NamingException e) {
				System.err.println(e.getMessage());
				e.printStackTrace();
				// no principals added
			}

			StringBuffer rolesNodeDNBuffer=new StringBuffer("");
			String rolesDN=((String)options.get("rolesDN")).trim();

			if (debug) {
				System.out.println("[LDAPLoginModule] rolesDN option: "+rolesDN);
			}
			if(rolesDN!=null && ! "".equals(rolesDN)){
				rolesNodeDNBuffer.append(rolesDN);
				if(!rolesDN.endsWith(",")){
					rolesNodeDNBuffer.append(',');
				}
			}
			rolesNodeDNBuffer.append(baseDN);
			String rolesNodeDN=rolesNodeDNBuffer.toString();
			Attribute objClassAttr=new BasicAttribute("objectClass","groupOfNames");
			Attribute memberAttr=new BasicAttribute("member",userDn);
			Attributes attrs=new BasicAttributes();
			attrs.put(objClassAttr);
			attrs.put(memberAttr);
			try {
				NamingEnumeration<SearchResult> rolesSearchRes=context.search(rolesNodeDN, attrs);
				while(rolesSearchRes.hasMore()){
					SearchResult sr=rolesSearchRes.next();
					//System.out.println("Sr: "+sr.getName());
					Attributes srAttrs=sr.getAttributes();
					Attribute cnattr=srAttrs.get("cn");
					Object roleObj=cnattr.get();
					if(roleObj instanceof String){
						String role=(String)roleObj;
						rolePrincipal= new RolePrincipal(role);
						if (!principalSet.contains(rolePrincipal)){
							principalSet.add(rolePrincipal);
							if (debug) {
								System.out.println("[LDAPLoginModule] added RolePrincipal "+rolePrincipal.getName()+" to Subject");
							}
						}

					}
					//sr.getName()
				}
			} catch (NamingException e) {
				System.err.println(e.getMessage());
				e.printStackTrace();
				// no principals added
			}

			String defaultRolesOpt=((String)options.get("defaultRoles"));
			
			if(defaultRolesOpt!=null){
				String[] defaultRoleNms=defaultRolesOpt.trim().split("\\s*,\\s*");
				for(String defRoleNm:defaultRoleNms){
					RolePrincipal rolePrincipal= new RolePrincipal(defRoleNm);
					if (!principalSet.contains(rolePrincipal)){
						principalSet.add(rolePrincipal);
						if (debug) {
							System.out.println("[LDAPLoginModule] added default RolePrincipal "+rolePrincipal.getName()+" to Subject");
						}
					}
				}

			}

			// in any case, clean out state
			username = null;
			for (int i = 0; i < password.length; i++)
				password[i] = ' ';
			password = null;

			commitSucceeded = true;
			closeContext("commit");
			return true;
		}
	}

	/**
	 * <p> This method is called if the LoginContext's
	 * overall authentication failed.
	 * (the relevant REQUIRED, REQUISITE, SUFFICIENT and OPTIONAL LoginModules
	 * did not succeed).
	 *
	 * <p> If this LoginModule's own authentication attempt
	 * succeeded (checked by retrieving the private state saved by the
	 * <code>login</code> and <code>commit</code> methods),
	 * then this method cleans up any state that was originally saved.
	 *
	 * <p>
	 *
	 * @exception LoginException if the abort fails.
	 *
	 * @return false if this LoginModule's own login and/or commit attempts
	 *      failed, and true otherwise.
	 */
	public boolean abort() throws LoginException {
		
		if (succeeded == false) {
			return false;
		} else if (succeeded == true && commitSucceeded == false) {
			// login succeeded but overall authentication failed
			succeeded = false;
			username = null;
			if (password != null) {
				for (int i = 0; i < password.length; i++)
					password[i] = ' ';
				password = null;
			}
			userPrincipal = null;
		} else {
			// overall authentication succeeded and commit succeeded,
			// but someone else's commit failed
			logout();
		}
		closeContext("abort");
		if (debug) {
			System.out.println("[LDAPLoginModule] login aborted.");
		}
		return true;
	}

	/**
	 * Logout the user.
	 *
	 * <p> This method removes the <code>SamplePrincipal</code>
	 * that was added by the <code>commit</code> method.
	 *
	 * <p>
	 *
	 * @exception LoginException if the logout fails.
	 *
	 * @return true in all cases since this <code>LoginModule</code>
	 *          should not be ignored.
	 */
	public boolean logout() throws LoginException {

		subject.getPrincipals().remove(userPrincipal);
		// TODO remove all roles ??
		subject.getPrincipals().remove(rolePrincipal);
		succeeded = false;
		succeeded = commitSucceeded;
		username = null;
		if (password != null) {
			for (int i = 0; i < password.length; i++)
				password[i] = ' ';
			password = null;
		}
		userPrincipal = null;
		// TODO set of roles
		rolePrincipal = null;
		// context should not be open here
		closeContext("logout");
		return true;
	}
	
	private void closeContext(String methodName) {
		try {
			if(context!=null) {
				context.close();
				if(debug) {
					System.out.println("[LDAPLoginModule] closed LDAP context.");
				}
			}
		} catch (NamingException e) {
			System.err.println("[LDAPLoginModule] Could not close LDAP context on "+methodName+" :");
			e.printStackTrace(System.err);
		}
		
	}
	
}




