//    IPS Java Speech Database
//    (c) Copyright 2011
//    Institute of Phonetics and Speech Processing,
//    Ludwig-Maximilians-University, Munich, Germany
//
//
//    This file is part of IPS Java Speech Database
//
//
//    IPS Java Speech Database is free software: you can redistribute it and/or modify
//    it under the terms of the GNU Lesser General Public License as published by
//    the Free Software Foundation, version 3 of the License.
//
//    IPS Java Speech Database is distributed in the hope that it will be useful,
//    but WITHOUT ANY WARRANTY; without even the implied warranty of
//    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
//    GNU Lesser General Public License for more details.
//
//    You should have received a copy of the GNU Lesser General Public License
//    along with IPS Java Speech Database.  If not, see <http://www.gnu.org/licenses/>.

package ipsk.db.speech;


import java.util.ArrayList;
import java.util.Collection;
import java.util.Date;
import java.util.HashSet;
import java.util.List;
import java.util.Set;

import javax.persistence.CascadeType;
import javax.persistence.CollectionTable;
import javax.persistence.Column;
import javax.persistence.ElementCollection;
import javax.persistence.Entity;
import javax.persistence.EnumType;
import javax.persistence.Enumerated;
import javax.persistence.FetchType;
import javax.persistence.Id;
import javax.persistence.JoinColumn;
import javax.persistence.JoinTable;
import javax.persistence.ManyToMany;
import javax.persistence.ManyToOne;
import javax.persistence.OneToMany;
import javax.persistence.OneToOne;
import javax.persistence.OrderColumn;
import javax.persistence.Table;
import javax.persistence.Temporal;
import javax.persistence.TemporalType;
import javax.xml.bind.annotation.XmlID;
import javax.xml.bind.annotation.XmlRootElement;
import javax.xml.bind.annotation.XmlTransient;
import javax.xml.bind.annotation.XmlType;

import ipsk.beans.PreferredDisplayOrder;
import ipsk.beans.dom.DOMAttributes;
import ipsk.beans.dom.DOMCollectionElement;
import ipsk.beans.dom.DOMElements;
import ipsk.beans.validation.Input;
import ipsk.db.speech.account.InvitationRequest;
import ipsk.db.speech.project.AutoGainControlConfig;
import ipsk.db.speech.project.LocalizedDefaultPurposeText;
import ipsk.db.speech.project.LocalizedInformedConsentText;
import ipsk.db.speech.project.MediaCaptureFormat;
import ipsk.db.speech.project.MediaStorageFormat;
import ipsk.db.speech.script.Script;
import ipsk.util.EnumResourceKeys;
import ipsk.util.MemberResourceKey;
import ipsk.util.PluralResourceKey;
import ipsk.util.ResourceBundleName;
import ipsk.util.ResourceKey;
import ipsk.util.TooltipResourceKey;


/**
 * Project
 */

@Entity
@Table(name = "project", schema = "public")
@ResourceBundleName("ipsk.db.speech.PropertyNames")
@ResourceKey("project")
@PluralResourceKey("projects")
//@PreferredDisplayOrder("name,description,hosts,organisations,dialectRegions,audioDevices,sessionFinishedMessage,*")
@PreferredDisplayOrder("name,uuid,description,registered,registeredByAccount,defaultResearchPurpose,defaultResearchPurposeLanguageISO3code,defaultResearchPurposeAdditionalLanguages,defaultInformedConsentText,defaultInformedConsentTextLanguageISO3code,informedConsentTextAdditionalLanguages,enableBroadConsent,organisations,dialectRegions,scriptSelectionMode,recordingDeviceWakeLock,audioDevices,chunkedRecording,clientAudioStorageType,ownedForms,ownedScripts,scripts,showSessionCompleteMessage,allowPublicPseudonymizedAudioExport,*")
@DOMAttributes({"name"})
@DOMElements({"description","audioFormat","sessionCode","organisations"})
@XmlType(name="project",propOrder={"name","uuid", "registered","description" , "defaultResearchPurpose", "defaultResearchPurposeLanguageISO3code","defaultResearchPurposeAdditionalLanguages","defaultInformedConsentText","defaultInformedConsentTextLanguageISO3code","informedConsentTextAdditionalLanguages","enableBroadConsent","informedConsentPaperForm", "scriptSelectionMode", "recordingDeviceWakeLock", "mediaCaptureFormat", "audioFormat", "audioDevices","autoGainControlConfigs","allowEchoCancellation","mediaStorageFormat","chunkedRecording","clientAudioStorageType","speakerFormConfiguration","showSessionCompleteMessage","exportFormatVersion","allowPublicPseudonymizedAudioExport","speakerWindowShowStopRecordAction" })
@XmlRootElement
public class Project implements java.io.Serializable {

	@EnumResourceKeys(memberResourceKeys = {
			@MemberResourceKey(name = "LEAST_USAGE", key = "script.selection.mode.least_usage"),
			@MemberResourceKey(name = "MANUAL", key = "script.selection.mode.manual"),
			}
	)
	public enum ScriptSelectionMode {
	     LEAST_USAGE("least_usage"),MANUAL("manual");

	    ScriptSelectionMode(String value) {
	    	this.value = value;
	    }
	    private final String value;

	    public String value() {
	    	return value; 
	    }
	    public String toString() {
	    	return value; 
	    }
	    public static ScriptSelectionMode getByValue(String value){
	    	for(ScriptSelectionMode scrSel:ScriptSelectionMode.values()){
	    		if(scrSel.value.equals(value)){
	    			return scrSel;
	    		}
	    	}
	    	return null;
	    }
	}
	
	@EnumResourceKeys(memberResourceKeys = {
			@MemberResourceKey(name = "MEM_ENTIRE", key = "storage.type.mem_entire"),
			@MemberResourceKey(name = "MEM_CHUNKED", key = "storage.type.mem_chunked"),
			@MemberResourceKey(name = "DB_CHUNKED", key = "storage.type.db_chunked"),
			@MemberResourceKey(name = "NET_CHUNKED", key = "storage.type.network_chunked"),
			@MemberResourceKey(name = "MEM_ENTIRE_AUTO_NET_CHUNKED", key = "storage.type.mem_entire_auto_network_chunked"),
			@MemberResourceKey(name = "MEM_CHUNKED_AUTO_NET_CHUNKED", key = "storage.type.mem_chunked_auto_network_chunked"),
			}
	)
	public enum AudioStorageType {
		MEM_ENTIRE("Memory entire"),MEM_CHUNKED("Memory chunked"),DB_CHUNKED("Database chunked"),NET_CHUNKED("Network chunked"),MEM_ENTIRE_AUTO_NET_CHUNKED("Memory entire auto network chunked"),MEM_CHUNKED_AUTO_NET_CHUNKED("Memory chunked auto network chunked");

		AudioStorageType(String value) {
	    	this.value = value;
	    }
	    private final String value;

	    public String value() {
	    	return value; 
	    }
	    public String toString() {
	    	return value; 
	    }
	    public static AudioStorageType getByValue(String value){
	    	for(AudioStorageType scrSel:AudioStorageType.values()){
	    		if(scrSel.value.equals(value)){
	    			return scrSel;
	    		}
	    	}
	    	return null;
	    }
	}
	
	@EnumResourceKeys(memberResourceKeys = {
			@MemberResourceKey(name = "PLAIN_TEXT_UTF8", key = "text.plain.utf8"),
			@MemberResourceKey(name = "HTML5_SUBSET", key = "text.html.5.subset"),
			}
	)
	public enum TextFormat {
	     PLAIN_TEXT_UTF8("plain_text_utf8"),HTML5_SUBSET("html5_subset");

	    TextFormat(String value) {
	    	this.value = value;
	    }
	    private final String value;

	    public String value() {
	    	return value; 
	    }
	    public String toString() {
	    	return value; 
	    }
	    public static TextFormat getByValue(String value){
	    	for(TextFormat tf:TextFormat.values()){
	    		if(tf.value.equals(value)){
	    			return tf;
	    		}
	    	}
	    	return null;
	    }
	}
	
	private String name;
	
	private String uuid;
	
	private Date registered=new Date();
	
	private Account registeredByAccount;
	
	private String contextPath;

	private String description;

	private AudioFormat audioFormat;
	
	
	private MediaCaptureFormat mediaCaptureFormat;

	@OneToOne(mappedBy="project",cascade = CascadeType.REMOVE)
	@ResourceKey("media.capture.format")
	@TooltipResourceKey("media.capture.format.tooltip")
	public MediaCaptureFormat getMediaCaptureFormat() {
		return mediaCaptureFormat;
	}

	public void setMediaCaptureFormat(MediaCaptureFormat audioCaptureFormat) {
		this.mediaCaptureFormat = audioCaptureFormat;
	}
	
	private MediaStorageFormat mediaStorageFormat;
	
	@OneToOne(mappedBy="project",cascade = CascadeType.REMOVE)
	@ResourceKey("media.storage.format")
	@TooltipResourceKey("media.storage.format.tooltip")
	public MediaStorageFormat getMediaStorageFormat() {
		return mediaStorageFormat;
	}

	public void setMediaStorageFormat(MediaStorageFormat audioStorageFormat) {
		this.mediaStorageFormat = audioStorageFormat;
	}

	private List<AutoGainControlConfig> autoGainControlConfigs=new ArrayList<AutoGainControlConfig>();

	@ResourceKey("configs.auto_gain_control")
	@TooltipResourceKey("configs.auto_gain_control.tooltip")
	@OneToMany(mappedBy="project",cascade = CascadeType.REMOVE)
	public List<AutoGainControlConfig> getAutoGainControlConfigs() {
		return autoGainControlConfigs;
	}

	public void setAutoGainControlConfigs(List<AutoGainControlConfig> autoGainControlConfigs) {
		this.autoGainControlConfigs = autoGainControlConfigs;
	}
	
	private Boolean allowEchoCancellation=null;
	
	/**
	 * Returns true if echo cancellation is allowed.
	 * @return true if echo cancellation is allowed.
	 */
	@ResourceKey("recording.allowEchoCancellation")
	//@TooltipResourceKey("recording.allowEchoCancellation.tooltip")
	public Boolean getAllowEchoCancellation() {
		return allowEchoCancellation;
	}

	/**
	 * Returns true if echo cancellation is allowed.
	 * @param allowEchoCancellation true if echo cancellation is allowed.
	 */
	public void setAllowEchoCancellation(Boolean allowEchoCancellation) {
		this.allowEchoCancellation = allowEchoCancellation;
	}

	private AudioStorageType clientAudioStorageType=null;
	
	@Column(name = "client_audio_storage_type", length = 32)
	@Enumerated(EnumType.STRING)
	@ResourceKey("client.audio.storage.type")
	@TooltipResourceKey("client.audio.storage.type.tooltip")
	public AudioStorageType getClientAudioStorageType() {
		return clientAudioStorageType;
	}

	public void setClientAudioStorageType(AudioStorageType clientAudioStorageType) {
		this.clientAudioStorageType = clientAudioStorageType;
	}

	private String sessionCode;
	
	private ScriptSelectionMode scriptSelectionMode;

	private String defaultResearchPurpose;
	private String defaultResearchPurposeLanguageISO3code;
	
	private List<LocalizedDefaultPurposeText> defaultResearchPurposeAdditionalLanguages=new ArrayList<>();
	
	private String defaultInformedConsentText;
	
//	private TextFormat informedConsentTextFormat=null;
	
	//@Column(name = "informed_consent_text_format", length = 32)
	//@Enumerated(EnumType.STRING)
	//@ResourceKey("consent.informed.text.format")
	//@TooltipResourceKey("consent.informed.text.format.tooltip")
//	public TextFormat getInformedConsentTextFormat() {
//		return informedConsentTextFormat;
//	}
//
//	public void setInformedConsentTextFormat(TextFormat informedConsentTextFormat) {
//		this.informedConsentTextFormat = informedConsentTextFormat;
//	}

	private String defaultInformedConsentTextLanguageISO3code;
	
	private List<LocalizedInformedConsentText> informedConsentTextAdditionalLanguages=new ArrayList<>();
	
	private boolean enableBroadConsent=false;
	
	private Integer exportFormatVersion=1;

	private Set<Account> adminAccounts=new HashSet<Account>(0);
	
	private Set<Account> accounts=new HashSet<Account>(0);

	private Set<Session> sessions = new HashSet<Session>(0);
	
	private Set<InformedConsent> informedConsents = new HashSet<InformedConsent>(0);
	
	private Set<DialectRegion> dialectRegions = new HashSet<DialectRegion>(0);
	
	private List<SpeechRecorderClient> allowedSpeechRecorderClients=new ArrayList<SpeechRecorderClient>();
	
	private List<AudioDevice> audioDevices = new ArrayList<AudioDevice>(0);
	
	private Set<Organisation> organisations = new HashSet<Organisation>(0);
	
	private Set<InvitationRequest> invitationRequests = new HashSet<>();
	
	private Set<Script> ownedScripts=new HashSet<Script>();
	
	private Set<FormConfiguration> ownedForms=new HashSet<FormConfiguration>();
	
	private Set<Speaker> speakers=new HashSet<Speaker>();

	private Set<Script> scripts = new HashSet<Script>(0);
	
	private boolean speakerWindowShowStopRecordAction=true;
	
	private boolean showSessionCompleteMessage=true;
	
	/**
	 * Returns true if the recording client should show a message dialog when the session is completed. Default is true.
	 * @return True if the recording client should show a message dialog when the session is completed.
	 */
	@ResourceKey("session.finishedMessage.show")
	@TooltipResourceKey("session.finishedMessage.show.tooltip")
	// Note: columnDefinition is for PostgreSQL, might not work on other DBs
	@Column(name = "show_session_complete_message",columnDefinition = "BOOLEAN NOT NULL DEFAULT TRUE",nullable = false)
	public boolean isShowSessionCompleteMessage() {
		return showSessionCompleteMessage;
	}

	public void setShowSessionCompleteMessage(boolean showSessionCompleteMessage) {
		this.showSessionCompleteMessage = showSessionCompleteMessage;
	}
	
	//private Set<TypedPropertyDescriptor> immediateAnnotations=new HashSet<TypedPropertyDescriptor>(0);
	
//	private Set<FormConfiguration> formConfigurations=new HashSet<FormConfiguration>();
//	@ManyToMany
//	@ResourceKey("form.configurations")
//	public Set<FormConfiguration> getFormConfigurations() {
//		return formConfigurations;
//	}
//
//	public void setFormConfigurations(Set<FormConfiguration> formConfigurations) {
//		this.formConfigurations = formConfigurations;
//	}
	

	private Boolean recordingDeviceWakeLock=null;
	
	/**
	 * Returns true if the recording client should try to keep the recording device awake through the recordings.
	 * @return True if the recording client should try to keep the recording device awake, false if not. Null for the application default. 
	 */
	@ResourceKey("recording.device.wake.lock")
	@TooltipResourceKey("recording.device.wake.lock.tooltip")
	public Boolean getRecordingDeviceWakeLock() {
		return recordingDeviceWakeLock;
	}

	/**
	 * Set to true if the recording client should try to keep the recording device awake, false if not.
	 * Default value is null.
	 * @param wakeLock True if the recording client should try to keep the recording device awake, false if not. Null for the application default. 
	 */
	public void setRecordingDeviceWakeLock(Boolean wakeLock) {
		this.recordingDeviceWakeLock = wakeLock;
	}
	
	private Boolean chunkedRecording=null;
	
	/**
	 * Returns true if the recordings should be transferred in chunks.
	 * @return true if the recordings should be transferred in chunks, null for the application default setting.
	 */
	@ResourceKey("recording.chunked")
	@TooltipResourceKey("recording.chunked.tooltip")
	public Boolean getChunkedRecording() {
		return chunkedRecording;
	}

	/**
	 * Set to true if the recording client should transfer recordings in chunks. To false if each recording should be transferred as one complete file. Set to null for application default.
	 * @param chunkedRecording true if the recording client should transfer recordings in chunks
	 */
	public void setChunkedRecording(Boolean chunkedRecording) {
		this.chunkedRecording = chunkedRecording;
	}

	private boolean informedConsentPaperForm=false;
	@ResourceKey("consent.informed.paperform")
	//@TooltipResourceKey("consent.informed.paperform.tooltip")
	public boolean isInformedConsentPaperForm() {
		return informedConsentPaperForm;
	}

	public void setInformedConsentPaperForm(boolean informedConsentPaperForm) {
		this.informedConsentPaperForm = informedConsentPaperForm;
	}

	
	private FormConfiguration speakerFormConfiguration;
	@ManyToOne
	@ResourceKey("speaker.form.configuration")
	public FormConfiguration getSpeakerFormConfiguration() {
		return speakerFormConfiguration;
	}

	public void setSpeakerFormConfiguration(
			FormConfiguration speakerFormConfiguration) {
		this.speakerFormConfiguration = speakerFormConfiguration;
	}
	
	private boolean allowPublicPseudonymizedAudioExport=false;
	
	@Column(name="allow_pub_pseudonymized_export")
	@ResourceKey("project.allowPublicPseudonymizedAudioExport")
	public boolean isAllowPublicPseudonymizedAudioExport() {
		return allowPublicPseudonymizedAudioExport;
	}

	public void setAllowPublicPseudonymizedAudioExport(boolean allowPublicPseudonymizedAudioExport) {
		this.allowPublicPseudonymizedAudioExport = allowPublicPseudonymizedAudioExport;
	}
	
	private Date lastVisitOfProjectAdmin=null;
	
	/**
	 * Approximate time when an project administrator visited (selected) this project.
	 * It is intended to determine projects that have not been used for a long time. 
	 * @return Date object of approximate time when an project administrator visited (selected) this project
	 */
	@ResourceKey("last_visit_of_project_admin")
	@Temporal(TemporalType.TIMESTAMP)
	@XmlTransient
	public Date getLastVisitOfProjectAdmin() {
		return lastVisitOfProjectAdmin;
	}

	public void setLastVisitOfProjectAdmin(Date lastLoginOfProjectAdmin) {
		this.lastVisitOfProjectAdmin = lastLoginOfProjectAdmin;
	}

	public Project() {
		this(null);
	}

	public Project(String name) {
		super();
		this.name = name;
		getAllowedSpeechRecorderClients().add(SpeechRecorderClient.HTML5_ANGULAR);
	}

	@Id
	@Column(name = "name", unique = true, nullable = false, length = 100)
	@ResourceKey("name")
	@Input(required=true)
	@XmlID
	public String getName() {
		return this.name;
	}

	public void setName(String name) {
		this.name = name;
	}
	
	@Column(name = "uuid",length=36, unique = true, nullable = true, updatable=false)
	@ResourceKey("uuid")
	@TooltipResourceKey("uuid.tooltip")
	public String getUuid() {
		return uuid;
	}

	public void setUuid(String uuid) {
		this.uuid = uuid;
	}

	@Column(name = "description", length = 1000)
	@ResourceKey("description")
	@ipsk.util.annotations.TextAreaView
	public String getDescription() {
		return this.description;
	}

	public void setDescription(String description) {
		this.description = description;
	}
	
	@Column()
	@Temporal(TemporalType.TIMESTAMP)
	@ResourceKey("registered")
	public Date getRegistered() {
		return this.registered;
	}

	public void setRegistered(Date registered) {
		this.registered=registered;
	}
	
	@ManyToOne
	@JoinColumn(name = "registered_by_account")
	@ResourceKey("registered_by_account")
	@XmlTransient
	public Account getRegisteredByAccount() {
		return registeredByAccount;
	}

	public void setRegisteredByAccount(Account registeredByAccount) {
		this.registeredByAccount = registeredByAccount;
	}
	
	
	@Column(name = "script_selection_mode", length = 32)
	@Enumerated(EnumType.STRING)
	@ResourceKey("script.selection.mode")
	@TooltipResourceKey("script.selection.mode.tooltip")
	public ScriptSelectionMode getScriptSelectionMode() {
		return scriptSelectionMode;
	}

	public void setScriptSelectionMode(ScriptSelectionMode scriptSelectionMode) {
		this.scriptSelectionMode = scriptSelectionMode;
	}
	
	@Column(name = "export_format_version")
	@ResourceKey("export.format.version")
	public Integer getExportFormatVersion() {
		return exportFormatVersion;
	}

	public void setExportFormatVersion(Integer exportFormatVersion) {
		this.exportFormatVersion = exportFormatVersion;
	}

//	@Column(name = "hosts")
//	@ResourceKey("hosts")
//	@XmlTransient
//	public String getHosts() {
//		return this.hosts;
//	}
//
//	public void setHosts(String hosts) {
//		this.hosts = hosts;
//	}

	@OneToMany( fetch = FetchType.LAZY, mappedBy = "project")
	@ResourceKey("sessions")
	@TooltipResourceKey("project.sessions.tooltip")
//	@XmlIDREF
	@XmlTransient
	public Set<Session> getSessions() {
		return this.sessions;
	}

	public void setSessions(Set<Session> sessions) {
		this.sessions = sessions;
	}
	
	@OneToMany( fetch = FetchType.LAZY, mappedBy = "project",cascade = CascadeType.REMOVE)
	@ResourceKey("consents.informed")
	@XmlTransient
	public Set<InformedConsent> getInformedConsents() {
		return informedConsents;
	}

	public void setInformedConsents(Set<InformedConsent> informedConsents) {
		this.informedConsents = informedConsents;
	}


	
	@ManyToMany(fetch = FetchType.LAZY)
	@JoinTable(
		        name="project_audio_device",
		        joinColumns={@JoinColumn(name="project")},
		        //inverseJoinColumns={@JoinColumn(name="ad_name",referencedColumnName="name"),@JoinColumn(name="ad_is_playback",referencedColumnName="is_playback"),@JoinColumn(name="ad_api",referencedColumnName="api"),@JoinColumn(name="ad_regex",referencedColumnName="regex")}
		        inverseJoinColumns={@JoinColumn(name="audio_device_id",referencedColumnName="audio_device_id")}
		        )
    @ResourceKey("audio.devices")
    @OrderColumn
	public List<AudioDevice> getAudioDevices() {
		return this.audioDevices;
	}


// @OrderColumn does not work with a ManyToMany relationship with EclipseLink JPA
// see 
// https://bugs.eclipse.org/bugs/show_bug.cgi?id=309039
//
// Storage of the ordered list works, but the SQL SELECT command for reading from DB does not use an ORDER BY clause	
	
//	/**
//	 * Get list of audio devices
//	 * @return
//	 */
//	@ManyToMany(fetch = FetchType.EAGER)
//	@JoinTable(
//		        name="project_audio_device",
//		        joinColumns={@JoinColumn(name="project")},
//		        inverseJoinColumns={@JoinColumn(name="ad_name",referencedColumnName="name"),@JoinColumn(name="ad_is_playback",referencedColumnName="is_playback"),@JoinColumn(name="ad_api",referencedColumnName="api"),@JoinColumn(name="ad_regex",referencedColumnName="regex")})
//	@OrderColumn(name="index",insertable=true,updatable=true)	        
//    @ResourceKey("audio.devices")
//	public List<AudioDevice> getAudioDevices() {
//		return this.audioDevices;
//	}
//
//	public void setAudioDevices(List<AudioDevice> audioDevices) {
//		this.audioDevices = audioDevices;
//	}
	
	
	private List<AudioDevice> filterAudioDeviceList(Collection<AudioDevice> adColl,boolean hasJExt,boolean isRegex){
		ArrayList<AudioDevice> filteredList=new ArrayList<AudioDevice>();
		for(AudioDevice ad:adColl){
			String jExt=ad.getJextension();
			boolean devHasJExt=(jExt!=null && ! jExt.equals(""));
//			boolean devIsRegex=ad.getId().isRegex();
			boolean devIsRegex=ad.isRegex();
			if((devHasJExt == hasJExt) && (devIsRegex == isRegex)){
				filteredList.add(ad);
			}
		}
		return filteredList;
	}
//// Workaround for EclipseLink bug:
//// Order the devices: First order rule is:
////	First devices which need an extension (e.g. DSJavaSound).
////	Next rule:
////	Non regular expressions first, then regular expressions named devices.	
//	
//	@Transient
//	public List<AudioDevice> getSpeechrecorderOrderedAudioDevices(){
//		Set<AudioDevice> aDevSet=getAudioDevices();
//		ArrayList<AudioDevice> aDevList=new ArrayList<AudioDevice>();
//		
//		aDevList.addAll(filterAudioDeviceList(aDevSet, true, false));
//		aDevList.addAll(filterAudioDeviceList(aDevSet, true, true));
//		aDevList.addAll(filterAudioDeviceList(aDevSet, false, false));
//		aDevList.addAll(filterAudioDeviceList(aDevSet, false, true));
//		return aDevList;
//	}
	
	public void setAudioDevices(List<AudioDevice> audioDevices) {
		this.audioDevices = audioDevices;
	}
	
	@ManyToMany(fetch = FetchType.LAZY)
	@JoinTable(
		        name="project_dialect_region",
		        joinColumns={@JoinColumn(name="project")},
		        inverseJoinColumns={@JoinColumn(name="dialect_region")}
		    )
    @ResourceKey("dialect_regions")
    @XmlTransient
	public Set<DialectRegion> getDialectRegions() {
		return this.dialectRegions;
	}

	public void setDialectRegions(Set<DialectRegion> dialectRegions) {
		this.dialectRegions = dialectRegions;
	}
	
	public String toString(){
		return name;
	}
	
//	@ManyToMany(fetch = FetchType.LAZY)
//	@JoinTable(
//		        name="project_speakers",
//		        joinColumns={@JoinColumn(name="project")},
//		        inverseJoinColumns={@JoinColumn(name="speaker")}
//		    )
//	@DOMCollectionElement(collectionElementName="speakers")
//    @ResourceKey("speakers")
//    @XmlTransient
//	public Set<Speaker> getSpeakers() {
//		return speakers;
//	}
//
//	public void setSpeakers(Set<Speaker> speakers) {
//		this.speakers = speakers;
//	}

	@ManyToMany(fetch = FetchType.LAZY)
	@JoinTable(
		        name="project_organisations",
		        joinColumns={@JoinColumn(name="project")},
		        inverseJoinColumns={@JoinColumn(name="organisation")}
		    )
	@DOMCollectionElement(collectionElementName="organisation")
    @ResourceKey("organisations")
    @XmlTransient
	public Set<Organisation> getOrganisations() {
		return organisations;
	}

	public void setOrganisations(Set<Organisation> organisations) {
		this.organisations = organisations;
	}
	
	@ManyToMany(cascade = {CascadeType.REMOVE})
	@XmlTransient
	@ResourceKey("invitation.requests")
	public Set<InvitationRequest> getInvitationRequests() {
		return invitationRequests;
	}

	public void setInvitationRequests(Set<InvitationRequest> invitationRequests) {
		this.invitationRequests = invitationRequests;
	}

	
	@ElementCollection(targetClass=SpeechRecorderClient.class)
    @Enumerated(EnumType.STRING)
    @CollectionTable(name="project_spr_client")
    @Column(name="spr_client",length = 100) 
	@ResourceKey("speechrecorder.allowedClients")
	@XmlTransient
	public List<SpeechRecorderClient> getAllowedSpeechRecorderClients() {
	
		return allowedSpeechRecorderClients;
	}

	public void setAllowedSpeechRecorderClients(List<SpeechRecorderClient> allowedSpeechRecorderClients) {
		this.allowedSpeechRecorderClients = allowedSpeechRecorderClients;
	}

	@Column(name = "context_path",length = 20)
	@ResourceKey("context.path")
	@XmlTransient
	public String getContextPath() {
		return contextPath;
	}

	public void setContextPath(String contextPath) {
		this.contextPath = contextPath;
	}

	@ManyToOne
	@JoinColumn(name = "audioformat")
	@ResourceKey("audio.format")
	public AudioFormat getAudioFormat() {
		return audioFormat;
	}

	public void setAudioFormat(AudioFormat audioFormat) {
		this.audioFormat = audioFormat;
	}
	

//	@Column(name = "audio_channel_count")
//	@ResourceKey("audio.channels")
//	public Integer getAudioChannelCount() {
//		return audioChannelCount;
//	}
//
//	public void setAudioChannelCount(Integer audioChannelCount) {
//		this.audioChannelCount = audioChannelCount;
//	}
	
	@ManyToMany(fetch = FetchType.LAZY)
	@JoinTable(
		        name="project_scripts",
		        joinColumns={@JoinColumn(name="project")},
		        inverseJoinColumns={@JoinColumn(name="script")}
		    )
    @ResourceKey("scripts.active")
	@TooltipResourceKey("scripts.active.tooltip")
    @XmlTransient
	public Set<Script> getScripts() {
		return scripts;
	}

	public void setScripts(Set<Script> scripts) {
		this.scripts = scripts;
	}
	
	@OneToMany(fetch = FetchType.LAZY,mappedBy="project",cascade = CascadeType.REMOVE)
	@ResourceKey("speaker")
	@XmlTransient
	public Set<Speaker> getSpeakers() {
		return speakers;
	}

	public void setSpeakers(Set<Speaker> speakers) {
		this.speakers = speakers;
	}
	
	@OneToMany(fetch = FetchType.LAZY,mappedBy="owningProject",cascade = CascadeType.REMOVE)
	@ResourceKey("scripts.owned")
	@TooltipResourceKey("scripts.owned.tooltip")
	@XmlTransient
	public Set<Script> getOwnedScripts() {
		return ownedScripts;
	}

	public void setOwnedScripts(Set<Script> ownedScripts) {
		this.ownedScripts = ownedScripts;
	}
	
	@OneToMany(fetch = FetchType.LAZY,mappedBy="owningProject",cascade = CascadeType.REMOVE)
	@ResourceKey("forms.owned")
	//@TooltipResourceKey("forms.owned.tooltip")
	@XmlTransient
	public Set<FormConfiguration> getOwnedForms() {
		return ownedForms;
	}

	public void setOwnedForms(Set<FormConfiguration> ownedForms) {
		this.ownedForms = ownedForms;
	}

	
	@ManyToMany()
	@JoinTable(
		        name="project_account",
		        joinColumns={@JoinColumn(name="project")},
		        inverseJoinColumns={@JoinColumn(name="account")}
		    )
    @ResourceKey("accounts")
    @XmlTransient
	public Set<Account> getAccounts() {
		return accounts;
	}

	public void setAccounts(Set<Account> accounts) {
		this.accounts = accounts;
	}
	
	@ManyToMany()
	@JoinTable(
		        name="project_admin_account",
		        joinColumns={@JoinColumn(name="project")},
		        inverseJoinColumns={@JoinColumn(name="admin_account")}
		    )
    @ResourceKey("accounts.admin")
    @XmlTransient
	public Set<Account> getAdminAccounts() {
		return adminAccounts;
	}

	public void setAdminAccounts(Set<Account> adminAccounts) {
		this.adminAccounts = adminAccounts;
	}
	
	@Column(name = "session_code", length = 100)
	@ResourceKey("session.code")
	@XmlTransient
	public String getSessionCode() {
		return sessionCode;
	}

	public void setSessionCode(String sessionCode) {
		this.sessionCode = sessionCode;
	}
	
	@Column(name = "default_research_purpose", length = 10000)
	//@Input(required=true)
	@ResourceKey("research.purpose.default.text")
	@TooltipResourceKey("research.purpose.default.text.tooltip")
	@ipsk.util.annotations.TextAreaView
	public String getDefaultResearchPurpose() {
		return defaultResearchPurpose;
	}

	public void setDefaultResearchPurpose(String defaultResearchPurpose) {
		this.defaultResearchPurpose = defaultResearchPurpose;
	}

	@Column(name = "default_research_purpose_lang_iso3", length = 3)
	//@Input(required=true)
	@ResourceKey("research.purpose.default.language")	
	public String getDefaultResearchPurposeLanguageISO3code() {
		return defaultResearchPurposeLanguageISO3code;
	}

	public void setDefaultResearchPurposeLanguageISO3code(String defaultResearchPurposeLanguageISO3code) {
		this.defaultResearchPurposeLanguageISO3code = defaultResearchPurposeLanguageISO3code;
	}

	@OneToMany(mappedBy="project",cascade = CascadeType.REMOVE)
	@ResourceKey("research.purpose.default.text.additional_languages")
	@TooltipResourceKey("research.purpose.default.text.additional_languages.tooltip")
	public List<LocalizedDefaultPurposeText> getDefaultResearchPurposeAdditionalLanguages() {
		return defaultResearchPurposeAdditionalLanguages;
	}

	public void setDefaultResearchPurposeAdditionalLanguages(
			List<LocalizedDefaultPurposeText> defaultResearchPurposeAdditionalLanguages) {
		this.defaultResearchPurposeAdditionalLanguages = defaultResearchPurposeAdditionalLanguages;
	}
	
	@Column(name = "default_informed_consent_text", length = 100000)
	@ResourceKey("consent.informed.text.default")
	@TooltipResourceKey("consent.informed.text.tooltip")
	@ipsk.util.annotations.TextAreaView
	public String getDefaultInformedConsentText() {
		return defaultInformedConsentText;
	}

	public void setDefaultInformedConsentText(String defaultInformedConsent) {
		this.defaultInformedConsentText = defaultInformedConsent;
	}

	@Column(name = "default_informed_consent_text_lang_iso3", length = 3)
	@ResourceKey("consent.informed.text.default.language")	
	public String getDefaultInformedConsentTextLanguageISO3code() {
		return defaultInformedConsentTextLanguageISO3code;
	}

	public void setDefaultInformedConsentTextLanguageISO3code(String defaultInformedConsentLanguageISO3code) {
		this.defaultInformedConsentTextLanguageISO3code = defaultInformedConsentLanguageISO3code;
	}
	
	@OneToMany(mappedBy="project",cascade = CascadeType.REMOVE)
	@ResourceKey("informed.consent.localized.text.additional_languages")
	public List<LocalizedInformedConsentText> getInformedConsentTextAdditionalLanguages() {
		return informedConsentTextAdditionalLanguages;
	}

	public void setInformedConsentTextAdditionalLanguages(List<LocalizedInformedConsentText> localizedInformedConsentTexts) {
		this.informedConsentTextAdditionalLanguages = localizedInformedConsentTexts;
	}
	
	@Column(name="enable_broad_consent")
	@ResourceKey("consent.informed.broad.enable")
	public boolean getEnableBroadConsent() {
		return enableBroadConsent;
	}

	public void setEnableBroadConsent(boolean enableBroadConsent) {
		this.enableBroadConsent = enableBroadConsent;
	}

	@Column(name = "show_stop_record_button")
	@ResourceKey("prompter.showStopRecordButton")
	public boolean isSpeakerWindowShowStopRecordAction() {
		return speakerWindowShowStopRecordAction;
	}

	public void setSpeakerWindowShowStopRecordAction(
			boolean speakerWindowShowStopRecordAction) {
		this.speakerWindowShowStopRecordAction = speakerWindowShowStopRecordAction;
	}
		
//	public Set<TypedPropertyDescriptor> getImmediateAnnotations() {
//		return immediateAnnotations;
//	}
//
//	public void setImmediateAnnotations(
//			Set<TypedPropertyDescriptor> immediateAnnotations) {
//		this.immediateAnnotations = immediateAnnotations;
//	}

//	@ManyToOne()
//	@JoinColumn(name = "session_finished_l_msg_id")
//    @ResourceKey("session.finishedMessage")
//	public LocalizableMessage getSessionFinishedMessage() {
//		return sessionFinishedMessage;
//	}
//
//	public void setSessionFinishedMessage(LocalizableMessage sessionFinishedMessage) {
//		this.sessionFinishedMessage = sessionFinishedMessage;
//	}
	
//	@OneToOne()
//	@JoinColumn(name = "speaker_formconfiguration_id")
//    @ResourceKey("speaker.form.configuration")
//	public FormConfiguration getSpeakerFormConfiguration() {
//		return speakerFormConfiguration;
//	}
//
//	public void setSpeakerFormConfiguration(
//			FormConfiguration speakerFormConfiguration) {
//		this.speakerFormConfiguration = speakerFormConfiguration;
//	}
}
