//    Speechrecorder
//    (c) Copyright 2012-2020
//    Institute of Phonetics and Speech Processing,
//    Ludwig-Maximilians-University, Munich, Germany
//
//
//    This file is part of Speechrecorder
//
//
//    Speechrecorder is free software: you can redistribute it and/or modify
//    it under the terms of the GNU Lesser General Public License as published by
//    the Free Software Foundation, version 3 of the License.
//
//    Speechrecorder is distributed in the hope that it will be useful,
//    but WITHOUT ANY WARRANTY; without even the implied warranty of
//    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
//    GNU Lesser General Public License for more details.
//
//    You should have received a copy of the GNU Lesser General Public License
//    along with Speechrecorder.  If not, see <http://www.gnu.org/licenses/>.

package ipsk.apps.speechrecorder.script;

import java.io.File;
import java.io.IOException;
import java.io.InputStream;
import java.net.MalformedURLException;
import java.net.URISyntaxException;
import java.net.URL;
import java.net.URLConnection;
import java.util.List;
import java.util.Set;
import java.util.Vector;
import java.util.logging.Logger;

import javax.sound.sampled.AudioFileFormat;
import javax.sound.sampled.AudioSystem;
import javax.sound.sampled.UnsupportedAudioFileException;
import javax.swing.JOptionPane;
import javax.xml.parsers.ParserConfigurationException;

import org.xml.sax.SAXException;

import ipsk.apps.speechrecorder.storage.StorageManagerException;
import ipsk.audio.ThreadSafeAudioSystem;
import ipsk.db.speech.script.PromptItem;
import ipsk.db.speech.script.Recording;
import ipsk.db.speech.script.Script;
import ipsk.db.speech.script.Section;
import ipsk.db.speech.script.prompt.Mediaitem;
import ipsk.io.StreamCopy;
import ipsk.net.MIMEType;
import ipsk.net.URLContext;
import ipsk.persistence.IntegerSequenceGenerator;
import ipsk.text.ParserException;
import ipsk.xml.DOMConverterException;

/**
 * RecScriptManager loads the recording script and manages the progress through
 * the recording. It keeps track of the items recorded within the current
 * sessionid, and it knows which item to record next.
 * 
 * RecScriptManager is the model for the ProgressViewer. ProgressViewer displays
 * the recording script in a table. The column names are defined in the resource
 * file of the GUI. Each table row represents a recording item.
 * 
 * @author Christoph Draxler
 * @version 2.0 Feb. 2004
 * 
 *  
 */

public class RecScriptManager extends Object{
    
    public static final int AUTOMATIC = 0;
    public static final int MANUAL = 1;
    public static final int SEQUENTIAL = 2;
    public static final int RANDOM = 3;
    
    // arbitrary identifier required to set a custom cell renderer
    public static final String RECORDED_COL_ID="progress.table.col.recorded";
    
    public static final int ERROR_MSG_MAX_ITEMS=20;
    
    private Logger logger;

    private RecscriptHandler recScriptHandler;
    private URL scriptURL;
    public URL getScriptURL() {
        return scriptURL;
    }

    private Script script;
    private boolean scriptSaved=true;

    private URL context=null;
    private boolean defaultSpeakerDisplay;
    private Section.Mode defaultMode;
    private int defaultPreDelay;
    private int defaultPostDelay;
    private boolean defaultAutomaticPromptPlay=true;
    private boolean setIndexActionsEnabled=false;

    private boolean progresToNextUnrecorded=false;
   
    private Vector<RecscriptManagerListener> listeners=new Vector<RecscriptManagerListener>();
    
    // NOT shuffled list of promptitems, only used for validation
    private List<PromptItem> promptItemsList=null;
    private ItemCodeValidator itemCodeValidator;
   
    /**
     * RecScriptManager loads the recording script and organizes the sequence of
     * recordings. A recording script can be either a text file (the use of text
     * files is deprecated) or an XML file defined in a DTD or XML-Schema.
     * Furthermore, RecScriptManager pre-fetches all resources referred to via
     * URLs so that they can be displayed in the prompt window without delay.
     * 
     * The sequence of recordings can be either automatic mode in sequence or in
     * random order, or it can be manual. For selecting prompts in manual mode
     * the RecTransporter or the ProgressViewer are used.
     * 
     * RecScriptManager is implemented as a singleton because there can be only
     * a single such manager for a given recording sessionid.
     *  
     */

    public RecScriptManager() {
        super();
        recScriptHandler=new RecscriptHandler();
        itemCodeValidator = new ItemCodeValidator();
        logger = Logger.getLogger("ipsk.apps.speechrecorder");
    }

    public void setSequenceGenerator(IntegerSequenceGenerator sequenceGenerator) {
        recScriptHandler.setSeqGen(sequenceGenerator);
    }
    
    public IntegerSequenceGenerator getSequenceGenerator() {
        return recScriptHandler.getSeqGen();
    }

    public Set<String> getExistingCodes() {
        return itemCodeValidator.getExistingCodes();
    }


    public boolean isNewVersionOfDTDFileRequired() throws MalformedURLException{
       return recScriptHandler.isNewVersionOfDTDFileRequired();
    }

    public void createDTDFileIfRequired() throws IOException{
        recScriptHandler.createDTDFileIfRequired();
    }

    /**
     * Get currently used script.
     * @return script
     */
    public Script getScript() {
        return script;
    }

    private void applyDefaults(){
        if(script!=null){
            script.defaultSectionMode(defaultMode);
            script.defaultSpeakerDisplay(defaultSpeakerDisplay);
            script.defaultPrerecdelay(defaultPreDelay);
            script.defaultPostrecdelay(defaultPostDelay);
            script.defaultAutoplay(defaultAutomaticPromptPlay);
            script.applyInternalDefaults();
        }
    }
    /**
     * Set the script to use.
     * @param script script
     */
    public void setScript(Script script){
        this.script = script;
        applyDefaults();
        
    
//      TODO: right place to check for duplicate item codes?
        //cd, 24.11.2005
        //Klausj, moved to set script
        
        itemCodeValidator.clear();
        if(script==null){
            promptItemsList=null;
        }else{
            promptItemsList=script.promptItemsList();
            if (validItemCodes()) {
                logger.info("Item codes unique.");
            } else {
                logger.warning("Warning: NON-UNIQUE item codes!");          
            }
            
            // TODO this should be done async
            analyzeScriptResources();
        }
        
        fireRecscriptManagerUpdate(new RecScriptChangedEvent(this));
    }
    
    public void shuffleItems(){
        if(script!=null){
            script.shuffleItems();
        }
    }
    
    /**
     * Loads a recording script from a URL. The format of the recording
     * script file is expected to be XML, if the the filename extension is
     * not <i>txt</i> or <i>TXT</i>. <br>
     * Note that the use of text files is deprecated.
     * 
     * @param promptURL The location of the recording script
     * @throws RecscriptManagerException script manager failed
     */
    public void load(URL promptURL) throws RecscriptManagerException{
        load(promptURL,false);
    }
    
    /**
     * Loads a recording script from a URL. The format of the recording
     * script file is expected to be XML, if the the filename extension is
     * not <i>txt</i> or <i>TXT</i>. <br>
     * Note that the use of text files is deprecated.
     * 
     * @param promptURL The location of the recording script
     * @param force if true forces loading script even if its has a newer DTD schema version
     * @throws RecscriptManagerException script manager failed
     */
    public void load(URL promptURL,boolean force) throws RecscriptManagerException{
    
        String promptURLString=promptURL.toExternalForm();
        
        try {
            promptURL=URLContext.getContextURL(context,promptURLString);
        } catch (MalformedURLException e) {
            throw new RecscriptManagerException(e);
        }
 
        Script script=readRecScriptXMLFile(promptURL,force);
        setScriptSaved(true);
        scriptURL=promptURL;
        setScript(script);
    }
    
    /**
     * Loads a recording script from a URL. The format of the recording
     * script file is expected to be XML, if the the filename extension is
     * not <i>txt</i> or <i>TXT</i>. <br>
     * Note that the use of text files is deprecated.
     * 
     * @param promptURL The location of the recording script
    * @throws RecscriptManagerException script manager failed
     */
    public void loadWithoutDTD(URL promptURL) throws RecscriptManagerException{
    
        String promptURLString=promptURL.toExternalForm();
    
        try {
            scriptURL=URLContext.getContextURL(context,promptURLString);
        } catch (MalformedURLException e) {
            throw new RecscriptManagerException(e);
        }
        
        Script script=readRecScriptXMLFileWithoutDTD(scriptURL);
        setScriptSaved(true);
        setScript(script);
    }
    
    public void storeToFile(File scriptFile) throws IOException {
        if(scriptURL!=null) {
            InputStream scriptStream=scriptURL.openStream();
            StreamCopy.copy(scriptStream, scriptFile, true);
        }else {
            throw new IllegalStateException();
        }
    }

    
    /**
     * reads recording and prompting instructions from an XML-Document.
     * After that, a vector with the prompt file contents can be obtained via
     * <code>getRecSections()</code>.
     * 
     * @param recScriptURL URL
     * @return the script
     * @throws RecscriptManagerException script manager failed
     */
    public Script readRecScriptXMLFile(URL recScriptURL) throws RecscriptManagerException {
       return readRecScriptXMLFile(recScriptURL,false);
    }

    /**
     * Reads recording and prompting instructions from an XML-Document.
     * After that, a vector with the prompt file contents can be obtained via
     * <code>getRecSections()</code>.
     * 
     * @param recScriptURL URL
     * @param force if true forces loading script even if its has a newer DTD schema version
     * @return the script
     * @throws RecscriptManagerException script manager failed
     */
    public Script readRecScriptXMLFile(URL recScriptURL,boolean force) throws RecscriptManagerException {
        logger.entering("readRecScriptXMLFile", "recScriptURLString");
        
        try {
            return recScriptHandler.readRecScriptXMLFile(recScriptURL,force);
        }catch (RecscriptHandlerException e) {
            e.printStackTrace();
            throw new RecscriptManagerException(e);
        }
    }
    
    /**
     * Reads recording and prompting instructions from an XML-Document.
     * After that, a vector with the prompt file contents can be obtained via
     * <code>getRecSections()</code>.
     * 
     * @param recScriptURL URL
     * @return the script
     * @throws RecscriptManagerException script manager failed
     */
    public Script readRecScriptXMLFileWithoutDTD(URL recScriptURL) throws RecscriptManagerException {
        logger.entering("readRecScriptXMLFile", "recScriptURLString");
        try {
            return recScriptHandler.readRecScriptXMLFileWithoutDTD(recScriptURL);
		} catch (RecscriptHandlerException e) {
            e.printStackTrace();
            throw new RecscriptManagerException(e);
        }

    }


    
    /**
     * Returns the recording section corresponding to a given
     * recording item index
     * 
     * @param itemIndex recording index in the range of 0 and (number of items - 1)
     * @return RecSection
     */
    public Section getRecSectionForItem(int itemIndex) {
        int index = itemIndex;
        Section recSection = null;
        if (script != null) {
            List<Section> sections = script.getSections();
            if (sections != null) {
                for (int i = 0; i < sections.size(); i++) {
                    Section tmpRecSection = sections.get(i);

                    if (index >= tmpRecSection.getGroups().size()) {
                        index = index - tmpRecSection.getGroups().size();
                    } else {
                        recSection = tmpRecSection;
                        break;
                    }
                }
            }
        }
        return recSection;
    }
    
    
    /**
     * Returns the (possibly shuffled) prompt item corresponding to the given recording index
     * @param promptIndex index of prompt
     * @return prompt item
     */
    public ipsk.db.speech.script.PromptItem shuffledPromptItem(int promptIndex) {
        ipsk.db.speech.script.PromptItem promptItem = null;
        int index = promptIndex;
        if(script!=null){
            List<Section> sections=script.getSections();
            if(sections!=null){
                for (int i = 0; i < sections.size(); i++) {
                    Section s=sections.get(i);
                    List<PromptItem> pis=s.getShuffledPromptItems();
                    int pisSize=pis.size();
                    if (index >= pisSize) {
                        index = index - pisSize;
                    } else {
                        promptItem = pis.get(index);
                        break;
                    }
                }
            }
        }
        return promptItem;
    }
    

    /**
     * Checks whether all recording codes are unique; if not,
     * a warning is displayed on the screen
     * 
     * @return boolean true if all item codes of the recording script
     * are unique, false otherwise
     */
    private boolean validItemCodes() {
       
        boolean allValid = true;
        StringBuffer errorMessage = new StringBuffer("Invalid item codes!\n");
        int invalidItemCount=0;
        int maxIdx=getMaxIndex();
        for (int i = 0; i < maxIdx; i++) {
            ipsk.db.speech.script.PromptItem pi=promptItemsList.get(i);
            if (pi instanceof Recording){
                String itemCode = ((Recording)pi).getItemcode();
                String validationMessage=itemCodeValidator.validateItemCode(itemCode);
                if(validationMessage==null){
                    itemCodeValidator.getExistingCodes().add(itemCode);
                }else{
                    allValid=false;
                    if(invalidItemCount<ERROR_MSG_MAX_ITEMS){
                        errorMessage.append("item #" + i+" code \"" + itemCode + "\": "+validationMessage);
                        errorMessage.append("\n");

                    }
                    invalidItemCount++;
                }
            }
        }
        if(invalidItemCount>ERROR_MSG_MAX_ITEMS){
            int moreInvalidItems=invalidItemCount-ERROR_MSG_MAX_ITEMS;
            errorMessage.append("...and "+moreInvalidItems+" more invalid item code(s).");
            errorMessage.append('\n');
        }

        if (! allValid) {
            errorMessage.append("Please correct the recording script.");
            JOptionPane.showMessageDialog(null,errorMessage.toString(), "Warning", JOptionPane.WARNING_MESSAGE);
        }
        return allValid;
    }
    
    
    public void analyzeScriptResources(){

        int maxAudioPromptChs=0;

        if(script!=null){
            List<PromptItem> prItemsList=script.promptItemsList();
            for(PromptItem pi :prItemsList){
                List<Mediaitem> mis=pi.getMediaitems();
                for(Mediaitem mi:mis){
                    String srcURLString=mi.getSrcStr();
                    String mimeStr=mi.getNNMimetype();
                    if(srcURLString!=null){
                    URL srcURL;
                    try {
                        srcURL = URLContext.getContextURL(context,srcURLString);

                        MIMEType mime=null;
                        String mimeMajorType=null;
                        try {
                            mime=MIMEType.parse(mimeStr);
                            mimeMajorType=mime.getType();
                        } catch (ParserException e) {
                            e.printStackTrace();
                        }

                        if(MIMEType.TYPE_AUDIO.equalsIgnoreCase(mimeMajorType)){

                            // try to get audio channel count
                            try {
                                AudioFileFormat aff=ThreadSafeAudioSystem.getAudioFileFormat(srcURL);
                                int chs=aff.getFormat().getChannels();
                                if(chs!=AudioSystem.NOT_SPECIFIED && chs>maxAudioPromptChs){
                                    maxAudioPromptChs=chs;
                                }
                            }catch (IOException e) {
                                // TODO Auto-generated catch block
                                e.printStackTrace();
                            } catch (UnsupportedAudioFileException e) {
                                // TODO Auto-generated catch block
                                e.printStackTrace();
                            }
                        }else{

                            // check URL
                            URLConnection urlConn;
                            try {
                                urlConn = srcURL.openConnection();
                                urlConn.connect();
                                InputStream is=urlConn.getInputStream();
                                is.close();
                            } catch (IOException e1) {
                                // TODO Auto-generated catch block
                                e1.printStackTrace();
                            }
                        }

                    } catch (MalformedURLException e2) {
                        e2.printStackTrace();
                    }
                }
                }
            }

            //System.out.println("Max audio channels of audio prompts: "+maxAudioPromptChs);
        }
    }

    /**
     * Returns the number of items in the recording script. If the
     * index has been computed once it is not recomputed for this script.
     * 
     * @return int prompt item count
     */
    // TODO method name is misleading. returns number of prompt items 
    public int getMaxIndex() {
        if (script==null){
            return 0;
        }else{
            return promptItemsList.size();
        }
           
    }

    public String getSystemIdBase() {
        return recScriptHandler.getSystemIdBase();
    }

    public void setSystemIdBase(String string) {
        recScriptHandler.setSystemIdBase(string);
    }

    /**
     * @return Returns the script id attribute.
     */
    public String getScriptID() {
        return script.getName();
    }

    /**
     * Set the URL context (usually the project directory in the workspace). 
     * @param context context
     */
    public void setContext(URL context) {
        this.context=context;
        
    }
    /** 
     * Get the workspace project context.
     * @return the URL workspace context
     */
    public URL getContext() {
        return context;
    }

    /**
     * @return logger
     */
    public Logger getLogger() {
        return logger;
    }

    /**
     * Resets the manager to the initial state
     */
    public void doClose(){
        setScript(null);
        fireRecscriptManagerUpdate(new RecScriptManagerClosedEvent(this));
    }

    public boolean isDefaultSpeakerDisplay() {
        return defaultSpeakerDisplay;
    }

    public void setDefaultSpeakerDisplay(boolean defaultSpeakerDisplay){
        this.defaultSpeakerDisplay = defaultSpeakerDisplay;
        applyDefaults();
        fireRecscriptManagerUpdate(new RecScriptChangedEvent(this));
    }

    public Section.Mode getDefaultMode() {
        return defaultMode;
    }

    public void setDefaultMode(Section.Mode defaultMode){
        this.defaultMode = defaultMode;
        applyDefaults();
        fireRecscriptManagerUpdate(new RecScriptChangedEvent(this));
    }

    public int getDefaultPreDelay() {
        return defaultPreDelay;
    }

    public void setDefaultPreDelay(int defaultPreDelay) {
        this.defaultPreDelay = defaultPreDelay;
    }

    public int getDefaultPostDelay() {
        return defaultPostDelay;
    }

    public void setDefaultPostDelay(int defaultPostDelay) {
        this.defaultPostDelay = defaultPostDelay;
    }
    
      /**
     * Add listener.
     * 
     * @param acl
     *            new listener
     */
    public synchronized void addRecscriptManagerListener(RecscriptManagerListener acl) {
        if (acl != null && !listeners.contains(acl)) {
            listeners.addElement(acl);
        }
    }

    /**
     * Remove listener.
     * 
     * @param acl
     *            listener to remove
     */
    public synchronized void removeRecscriptManagerListener(RecscriptManagerListener acl) {
        if (acl != null) {
            listeners.removeElement(acl);
        }
    }

    protected synchronized void fireRecscriptManagerUpdate(RecscriptManagerEvent event){
       
        for( RecscriptManagerListener listener:listeners){
            listener.update(event);
        }
    }

    public boolean isScriptSaved() {
        return scriptSaved;
    }

    public void setScriptSaved(boolean scriptSaved){
        this.scriptSaved = scriptSaved;
        fireRecscriptManagerUpdate(new RecScriptStoreStatusChanged(this));
    }

    public boolean isSetIndexActionsEnabled() {
        return setIndexActionsEnabled;
    }

    public void setSetIndexActionsEnabled(boolean setIndexActionsEnabled) {
        this.setIndexActionsEnabled = setIndexActionsEnabled;
    }

    public boolean isDefaultAutomaticPromptPlay() {
        return defaultAutomaticPromptPlay;
    }

    public void setDefaultAutomaticPromptPlay(boolean automaticPromptPlayDefault) {
        this.defaultAutomaticPromptPlay = automaticPromptPlayDefault;
    }

    public String getSystemId() {
        return recScriptHandler.getSystemId();
    }

    public void setSystemId(String scriptDTD) {
        recScriptHandler.setSystemId(scriptDTD);
    }

    public boolean isProgresToNextUnrecorded() {
        return progresToNextUnrecorded;
    }

    public void setProgresToNextUnrecorded(boolean progresToNextUnrecorded) {
        this.progresToNextUnrecorded = progresToNextUnrecorded;
    }

}