//    Speechrecorder
//    (c) Copyright 2009-2020
// 	  Institute of Phonetics and Speech Processing,
//    Ludwig-Maximilians-University, Munich, Germany
//
//
//    This file is part of Speechrecorder
//
//
//    Speechrecorder is free software: you can redistribute it and/or modify
//    it under the terms of the GNU Lesser General Public License as published by
//    the Free Software Foundation, version 3 of the License.
//
//    Speechrecorder is distributed in the hope that it will be useful,
//    but WITHOUT ANY WARRANTY; without even the implied warranty of
//    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
//    GNU Lesser General Public License for more details.
//
//    You should have received a copy of the GNU Lesser General Public License
//    along with Speechrecorder.  If not, see <http://www.gnu.org/licenses/>.

/*
 * Date  : Jun 2, 2004
 * Author: K.Jaensch, klausj@phonetik.uni-muenchen.de
 */

package ipsk.apps.speechrecorder.config.ui;

import ips.annot.BundleAnnotationPersistorServiceDescriptor;
import ips.annot.autoannotator.AutoAnnotationServiceDescriptor;
import ipsk.apps.speechrecorder.annotation.auto.AutoAnnotationPluginManager;
import ipsk.apps.speechrecorder.config.Annotation;
import ipsk.apps.speechrecorder.config.Control;
import ipsk.apps.speechrecorder.config.LoggingConfiguration;
import ipsk.apps.speechrecorder.config.ProjectConfiguration;
import ipsk.apps.speechrecorder.config.SpeakersConfiguration;
import ipsk.apps.speechrecorder.project.ActiveProjectManager;
import ipsk.audio.AudioController2;
import ipsk.swing.JDialogPanel;

import java.awt.Dimension;
import java.awt.event.ActionListener;
import java.awt.event.WindowEvent;
import java.awt.event.WindowListener;
import java.net.URL;
import java.util.List;

import javax.help.HelpBroker;
import javax.swing.Action;
import javax.swing.JOptionPane;
import javax.swing.JTabbedPane;


/**
 * Panel for Speechrecorder project configuration.
 * @author K.Jaensch, klausj@phonetik.uni-muenchen.de
 *
 */
public class ProjectConfigurationView extends JDialogPanel implements ActionListener, WindowListener {

	private static final long serialVersionUID = 1L;
	private SpeakersView speakersView;
	private ProjectConfiguration project;
	private JTabbedPane tabPane;

	private ProjectConfigurationPanel projectPanel;
	private RecordingView recView;
	private PlaybackView playView;
    private TransportView transportView;
	private PromptConfigurationView promptingView;
	private AnnotationPanel annotationPanel;
    private LoggingConfigurationView loggingView;
    
 
	public ProjectConfigurationView(ProjectConfiguration project,AudioController2 audioController,Action[] actions,List<BundleAnnotationPersistorServiceDescriptor> bapsds,AutoAnnotationPluginManager autoAnnotationPluginManager,ActiveProjectManager activeProjectManager,HelpBroker helpBroker) {
		super(Options.OK_CANCEL,(helpBroker!=null));
		this.project = project;
		
		setFrameTitle("Project \""+project.getName()+"\" configuration");
		if(helpBroker!=null){
		    helpBroker.enableHelpOnButton(getHelpButton(), "subsection-30", null);
		}
		tabPane = new JTabbedPane();
		projectPanel = new ProjectConfigurationPanel(audioController);
		speakersView = new SpeakersView();
		recView = new RecordingView(audioController);
		playView=new PlaybackView(audioController);
        transportView=new TransportView(actions);
		promptingView = new PromptConfigurationView(audioController,transportView.getKeyInputMapView(),activeProjectManager);
		annotationPanel=new AnnotationPanel(bapsds,autoAnnotationPluginManager);
        loggingView=new LoggingConfigurationView();
		tabPane.addTab("Project", projectPanel);
		tabPane.addTab("Speakers", speakersView);
		tabPane.addTab("Recording", recView);
		tabPane.addTab("Playback",playView);
        tabPane.addTab("Control", transportView);
		tabPane.addTab("Prompting", promptingView);
		tabPane.addTab("Annotation", annotationPanel);
        tabPane.addTab("Logging", loggingView);
        setContentPane(tabPane);
        setProjectConfiguration(project);
	}
	
	
	public void setProjectConfiguration(ProjectConfiguration projectConfiguration){
	    this.project=projectConfiguration;
	   
	    setFrameTitle("Project \""+project.getName()+"\" configuration");
	    projectPanel.setProjectConfiguration(project);
        SpeakersConfiguration speakersConfig=project.getSpeakers();
        speakersView.setSpeakersConfiguration(speakersConfig);
        recView.setProjectConfiguration(project);
        playView.setProjectConfiguration(project);
        Control control= project.getControl();
        transportView.setControl(control);
        promptingView.setProjectConfiguration(project);
        Annotation anno=project.getAnnotation();
        annotationPanel.setAnnotationConfig(anno);
        LoggingConfiguration loggingConfiguration=project.getLoggingConfiguration();
        loggingView.setLoggingConfiguration(loggingConfiguration);
	}
	
	public ProjectConfiguration getProjectConfiguration(){
	    ProjectConfiguration pc=new ProjectConfiguration();
	    projectPanel.applyValues(pc);
	    speakersView.applyValues(pc.getSpeakers());
	    recView.applyValues(pc);
	    playView.applyValues(pc);
	    transportView.applyValues(pc.getControl());
	    promptingView.applyValues(pc);
	    annotationPanel.applyValues(pc.getAnnotation());
	    loggingView.applyValues(pc.getLoggingConfiguration());
	 
	    return pc;
	}
	
	public Dimension getMinimumSize(){
		return getPreferredSize();
	}
	
	public void windowActivated(WindowEvent arg0) {
	}

	public void windowClosed(WindowEvent arg0) {
		setValue(JOptionPane.CANCEL_OPTION);
		
	}

	public void windowClosing(WindowEvent arg0) {
	}

	public void windowDeactivated(WindowEvent arg0) {
	}

	public void windowDeiconified(WindowEvent arg0) {
	}

	public void windowIconified(WindowEvent arg0) {
	}

	public void windowOpened(WindowEvent arg0) {
	}

	public void setProjectContext(URL projectContext) {
		promptingView.setProjectContext(projectContext);
	}
}
