//    Speechrecorder
// 	  (c) Copyright 2017-2020
// 	  Institute of Phonetics and Speech Processing,
//    Ludwig-Maximilians-University, Munich, Germany
//
//
//    This file is part of Speechrecorder
//
//
//    Speechrecorder is free software: you can redistribute it and/or modify
//    it under the terms of the GNU Lesser General Public License as published by
//    the Free Software Foundation, version 3 of the License.
//
//    Speechrecorder is distributed in the hope that it will be useful,
//    but WITHOUT ANY WARRANTY; without even the implied warranty of
//    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
//    GNU Lesser General Public License for more details.
//
//    You should have received a copy of the GNU Lesser General Public License
//    along with Speechrecorder.  If not, see <http://www.gnu.org/licenses/>.



package ipsk.apps.speechrecorder.session;

import java.awt.event.ActionEvent;
import java.awt.event.ActionListener;
import java.io.File;
import java.io.FileWriter;
import java.io.IOException;
import java.net.URISyntaxException;
import java.net.URL;
import java.text.MessageFormat;
import java.text.NumberFormat;
import java.util.ArrayList;
import java.util.List;
import java.util.Locale;
import java.util.logging.Logger;

import javax.sound.sampled.AudioFormat;
import javax.sound.sampled.UnsupportedAudioFileException;
import javax.swing.Action;
import javax.swing.JOptionPane;
import javax.swing.Timer;

import ips.annot.BundleAnnotationPersistor;
import ips.annot.autoannotator.AutoAnnotator;
import ips.annot.io.BundleAnnotationFilePersistor;
import ips.annot.model.AnnotatedAudioClip;
import ips.annot.model.db.Bundle;
import ips.annot.model.db.Session;
import ipsk.apps.speechrecorder.RecStatus;
import ipsk.apps.speechrecorder.RecWindow;
import ipsk.apps.speechrecorder.SpeechRecorderException;
import ipsk.apps.speechrecorder.SpeechRecorderUI;
import ipsk.apps.speechrecorder.UIResources;
import ipsk.apps.speechrecorder.actions.AdvanceToNextAction;
import ipsk.apps.speechrecorder.actions.BackwardAction;
import ipsk.apps.speechrecorder.annotation.AnnotationManager;
import ipsk.apps.speechrecorder.annotation.auto.AutoAnnotationManager;
import ipsk.apps.speechrecorder.annotation.auto.AutoAnnotationWorker;
import ipsk.apps.speechrecorder.annotation.auto.impl.PromptAutoAnnotator;
import ipsk.apps.speechrecorder.annotation.auto.impl.TemplateAutoAnnotator;
import ipsk.apps.speechrecorder.config.RecordingConfiguration.CaptureScope;
import ipsk.apps.speechrecorder.monitor.StartStopSignal;
import ipsk.apps.speechrecorder.prompting.PromptViewer;
import ipsk.apps.speechrecorder.prompting.PromptViewerListener;
import ipsk.apps.speechrecorder.prompting.PrompterException;
import ipsk.apps.speechrecorder.prompting.StartPromptPlaybackAction;
import ipsk.apps.speechrecorder.prompting.StopPromptPlaybackAction;
import ipsk.apps.speechrecorder.prompting.event.PromptViewerEvent;
import ipsk.apps.speechrecorder.prompting.event.PromptViewerOpenedEvent;
import ipsk.apps.speechrecorder.prompting.event.PromptViewerPresenterClosedEvent;
import ipsk.apps.speechrecorder.prompting.event.PromptViewerStartedEvent;
import ipsk.apps.speechrecorder.prompting.event.PromptViewerStoppedEvent;
import ipsk.apps.speechrecorder.prompting.presenter.PromptPresenterException;
import ipsk.apps.speechrecorder.script.RecScriptChangedEvent;
import ipsk.apps.speechrecorder.script.RecscriptManagerEvent;
import ipsk.apps.speechrecorder.session.action.CloseSpeakerDisplayAction;
import ipsk.apps.speechrecorder.session.action.ContinueAutoRecordingAction;
import ipsk.apps.speechrecorder.session.action.ContinuePlaybackAction;
import ipsk.apps.speechrecorder.session.action.ForwardAction;
import ipsk.apps.speechrecorder.session.action.PauseAutoRecordingAction;
import ipsk.apps.speechrecorder.session.action.PausePlaybackAction;
import ipsk.apps.speechrecorder.session.action.RecTransporterActions;
import ipsk.apps.speechrecorder.session.action.SetIndexAction;
import ipsk.apps.speechrecorder.session.action.StartAutoRecordingAction;
import ipsk.apps.speechrecorder.session.action.StartPlaybackAction;
import ipsk.apps.speechrecorder.session.action.StartRecordAction;
import ipsk.apps.speechrecorder.session.action.StopNonrecordingAction;
import ipsk.apps.speechrecorder.session.action.StopPlaybackAction;
import ipsk.apps.speechrecorder.session.action.StopRecordAction;
import ipsk.apps.speechrecorder.session.progress.ProgressManager;
import ipsk.apps.speechrecorder.session.progress.ProgressManagerEvent;
import ipsk.apps.speechrecorder.session.progress.ProgressManagerException;
import ipsk.apps.speechrecorder.session.progress.ProgressManagerListener;
import ipsk.apps.speechrecorder.session.progress.SessionPositionChangedEvent;
import ipsk.apps.speechrecorder.storage.ActiveSessionStorageManager;
import ipsk.apps.speechrecorder.storage.StorageManagerException;
import ipsk.audio.AudioController2.AudioController2Listener;
import ipsk.audio.AudioController2.AudioControllerEvent;
import ipsk.audio.AudioController4;
import ipsk.audio.AudioControllerException;
import ipsk.audio.AudioFormatNotSupportedException;
import ipsk.audio.AudioSource;
import ipsk.audio.AudioSourceException;
import ipsk.audio.ConvenienceFileAudioSource;
import ipsk.audio.FileAudioSource;
import ipsk.audio.PluginChain;
import ipsk.audio.URLAudioSource;
import ipsk.audio.ajs.AJSAudioSystem;
import ipsk.audio.capture.BufferOverrunException;
import ipsk.audio.capture.event.CaptureCloseEvent;
import ipsk.audio.capture.event.CaptureErrorEvent;
import ipsk.audio.capture.event.CaptureEvent;
import ipsk.audio.capture.event.CaptureRecordedEvent;
import ipsk.audio.capture.event.CaptureRecordingFileTransitEvent;
import ipsk.audio.capture.event.CaptureStartCaptureEvent;
import ipsk.audio.capture.event.CaptureStartRecordEvent;
import ipsk.audio.dsp.LevelInfo;
import ipsk.audio.dsp.speech.SpeechFinalSilenceDetector;
import ipsk.audio.dsp.speech.SpeechFinalSilenceDetectorEvent;
import ipsk.audio.dsp.speech.SpeechFinalSilenceDetectorListener;
import ipsk.audio.dsp.speech.vad.VoiceActivityDetector;
import ipsk.audio.dsp.speech.vad.impl.VoicedSpeechDetector;
import ipsk.audio.player.Player;
import ipsk.audio.player.PlayerException;
import ipsk.audio.player.PlayerListener;
import ipsk.audio.player.event.PlayerCloseEvent;
import ipsk.audio.player.event.PlayerEvent;
import ipsk.audio.player.event.PlayerPauseEvent;
import ipsk.audio.player.event.PlayerStartEvent;
import ipsk.audio.player.event.PlayerStopEvent;
import ipsk.audio.plugins.VolumeControlPlugin;
import ipsk.awt.ProgressListener;
import ipsk.awt.WorkerException;
import ipsk.awt.event.ProgressEvent;
import ipsk.db.speech.LocalizedText;
import ipsk.db.speech.script.Nonrecording;
import ipsk.db.speech.script.PromptItem;
import ipsk.db.speech.script.Recording;
import ipsk.db.speech.script.Script;
import ipsk.db.speech.script.Section;
import ipsk.db.speech.script.prompt.Mediaitem;
import ipsk.text.EncodeException;
import ipsk.text.ParserException;
import ipsk.util.SystemHelper;

/**
 * @author klausj
 *
 */
public class SessionManager implements ActionListener,
AudioController2Listener, ProgressManagerListener, PromptViewerListener, ProgressListener,  PlayerListener, SpeechFinalSilenceDetectorListener, AnnotationManager{

    public final static boolean DEBUG=false;
    public final boolean USE_MAX_REC_TIMER=true;
    private final double MIN_EXPECTED_REC_LEN_TOLERANCE=0.9;
    private final int MIN_EXPECTED_REC_LEN_LNE_ACTIVATION_MS_DEFAULT=250;
    // time to wait before a new recording attempt will be made after recording
    // error
    private long RECORD_RETRY_DELAY = 3000;
    
    private static final boolean DEBUG_AUDIO_QUALITY_FAKE_CHECK_FAIL = false;
    
    private UIResources uiString;
    
    private boolean debugSinusTest = false;
    
    private StartRecordAction startRecordAction;

    private StopRecordAction stopRecordAction;
    
    private StopNonrecordingAction stopNonrecordingAction;
    


    private StartAutoRecordingAction startAutoRecordingAction;

    private PauseAutoRecordingAction pauseAutoRecordingAction;

    private ContinueAutoRecordingAction continueAutoRecordingAction;

    private AdvanceToNextAction advanceToNextAction;

    private ForwardAction forwardAction;

    private BackwardAction backwardAction;

    private StartPlaybackAction startPlaybackAction;

    public StartPlaybackAction getStartPlaybackAction() {
        return startPlaybackAction;
    }

    private PausePlaybackAction pausePlaybackAction;

    private StopPlaybackAction stopPlaybackAction;

    private ContinuePlaybackAction continuePlaybackAction;
    
    private SetIndexAction setIndexAction;
    public SetIndexAction getSetIndexAction() {
        return setIndexAction;
    }
    
    private CloseSpeakerDisplayAction closeSpeakerDisplayAction;
    
    public Action[] actions=null;
    
    public Action[] getActions() {
        return actions;
    }

    /**
     * Indicates request to repeat the recording item.
     * 
     * @author klausj
     *
     */
    
    public class RepeatRequest{
        private String messageTitle;
        private String message;
        public RepeatRequest() {
            super();
          
        }
        public RepeatRequest(String messageTitle, String message) {
            super();
            this.messageTitle = messageTitle;
            this.message = message;
        }
        public String getMessageTitle() {
            return messageTitle;
        }
        public String getMessage() {
            return message;
        }
    }
    private Logger logger;
    private Logger timeLogger;
//    private RecLogger recLogger;
   
    private SessionManagerListener listener;
    
    public SessionManagerListener getListener() {
        return listener;
    }


    public void setListener(SessionManagerListener listener) {
        this.listener = listener;
    }

    private SpeechRecorderUI speechRecorderUI;
    
    public SpeechRecorderUI getSpeechRecorderUI() {
        return speechRecorderUI;
    }


    public void setSpeechRecorderUI(SpeechRecorderUI ui) {
        this.speechRecorderUI = ui;
        StartPromptPlaybackAction startPromptAction=speechRecorderUI.getPrompter().getStartPromptPlaybackAction();
        StopPromptPlaybackAction stopPromptAction=speechRecorderUI.getPrompter().getStopPromptPlaybackAction();
        actions= new Action[] {startRecordAction, stopRecordAction,
                startAutoRecordingAction, pauseAutoRecordingAction,
                continueAutoRecordingAction, startPromptAction,stopPromptAction, advanceToNextAction,
                forwardAction, backwardAction, startPlaybackAction,
                pausePlaybackAction, stopPlaybackAction, continuePlaybackAction,closeSpeakerDisplayAction};
    }

    private RecStatus recStat;
    private boolean useUploadCache;
    
    public boolean isUseUploadCache() {
        return useUploadCache;
    }


    public void setUseUploadCache(boolean useUploadCache) {
        this.useUploadCache = useUploadCache;
    }

    private SessionActions sessionActions;
    private RecTransporterActions recTransporterActions;
    
    public RecTransporterActions getRecTransporterActions() {
        return recTransporterActions;
    }


    private AudioController4 audioController;
    
    public AudioController4 getAudioController() {
        return audioController;
    }
    
    private boolean audioEnabled=false;

    public boolean isAudioEnabled() {
        return audioEnabled;
    }


    public void setAudioEnabled(boolean audioEnabled) {
        this.audioEnabled = audioEnabled;
    }

    private URL beepURL;
    public URL getBeepURL() {
        return beepURL;
    }


    public void setBeepURL(URL beepURL) {
        this.beepURL = beepURL;
    }

    private Player beepPlayer;

    public Player getBeepPlayer() {
        return beepPlayer;
    }


    public void setBeepPlayer(Player beepPlayer) {
        this.beepPlayer = beepPlayer;
    }


    public void setAudioController(AudioController4 audioController) {
        this.audioController = audioController;
    }
    
//    private ProgressManager progressManager;

    private ActiveSessionStorageManager storageManager;
    
    public ActiveSessionStorageManager getStorageManager() {
        return storageManager;
    }


    public void setStorageManager(ActiveSessionStorageManager storageManager) {
        this.storageManager = storageManager;
        progressManager.setStorageManager(storageManager);
    }
    
    private AutoAnnotationManager autoAnnotationManager=new AutoAnnotationManager();
    
    private boolean overwrite;

    public boolean isOverwrite() {
        return overwrite;
    }


    public void setOverwrite(boolean overwrite) {
        this.overwrite = overwrite;
    }
    
    private boolean overwriteWarning;
   

    public boolean isOverwriteWarning() {
        return overwriteWarning;
    }


    public void setOverwriteWarning(boolean overwriteWarning) {
        this.overwriteWarning = overwriteWarning;
    }

    private boolean defaultShowSpeakerWindow;
    
    
    public boolean isDefaultShowSpeakerWindow() {
        return defaultShowSpeakerWindow;
    }


    public void setDefaultShowSpeakerWindow(boolean defaultShowSpeakerWindow) {
        this.defaultShowSpeakerWindow = defaultShowSpeakerWindow;
    }

    private ProgressManager progressManager;
    
    /**
     * @return the progressManager
     */
    public ProgressManager getProgressManager() {
        return progressManager;
    }
    
    private AnnotatedAudioClip audioClip=new AnnotatedAudioClip();

    public AnnotatedAudioClip getAudioClip() {
        return audioClip;
    }



    public void setAudioClip(AnnotatedAudioClip audioClip) {
        this.audioClip = audioClip;
    }

    private CaptureScope captureScope=CaptureScope.ITEM;
    
    public CaptureScope getCaptureScope() {
        return captureScope;
    }


    public void setCaptureScope(CaptureScope captureScope) {
        this.captureScope = captureScope;
    }

    private RepeatRequest repeatRequest=null;
    
    private AutoAnnotationWorker autoAnnotationWorker;
    private boolean seamlessAutoRecording=false;
   
    
    public boolean isSeamlessAutoRecording() {
        return seamlessAutoRecording;
    }


    public void setSeamlessAutoRecording(boolean seamlessAutoRecording) {
        this.seamlessAutoRecording = seamlessAutoRecording;
    }
    
    private Section.Mode defaultSectionMode;
    
    
    public Section.Mode getDefaultSectionMode() {
        return defaultSectionMode;
    }


    public void setDefaultSectionMode(Section.Mode defaultSectionMode) {
        this.defaultSectionMode = defaultSectionMode;
    }

    private boolean resetPeakOnRecording;
    
    public boolean isResetPeakOnRecording() {
        return resetPeakOnRecording;
    }


    public void setResetPeakOnRecording(boolean resetPeakOnRecording) {
        this.resetPeakOnRecording = resetPeakOnRecording;
    }
    
    private boolean progressToNextUnrecorded;
    

    public boolean isProgressToNextUnrecorded() {
        return progressToNextUnrecorded;
    }


    public void setProgressToNextUnrecorded(boolean progressToNextUnrecorded) {
        this.progressToNextUnrecorded = progressToNextUnrecorded;
    }

    private AudioSource beepAudioSource;

    private VoiceActivityDetector voiceDetector;
    private SpeechFinalSilenceDetector silenceDetector;
    
    private List<AutoAnnotator> enabledAutoAnnotators=new ArrayList<AutoAnnotator>();
    
    public List<AutoAnnotator> getEnabledAutoAnnotators() {
        return enabledAutoAnnotators;
    }


    public void setEnabledAutoAnnotators(List<AutoAnnotator> enabledAutoAnnotators) {
        this.enabledAutoAnnotators = enabledAutoAnnotators;
    }
    
    private List<BundleAnnotationPersistor> bundleAnnotationPersistorList=new ArrayList<>();

    public List<BundleAnnotationPersistor> getBundleAnnotationPersistorList() {
        return bundleAnnotationPersistorList;
    }


    public void setBundleAnnotationPersistorList(List<BundleAnnotationPersistor> bundleAnnotationPersistorList) {
        this.bundleAnnotationPersistorList = bundleAnnotationPersistorList;
    }

    private PromptAutoAnnotator promptAutoAnnotator;
    
    private TemplateAutoAnnotator templateAutoAnnotator;
    
    
    private Section.Mode sectionMode=null;
    private ipsk.db.speech.script.PromptItem promptItem;
    private String promptItemCode=null;
    
    private boolean progressPaused;
    
    private boolean advanceToNextRequest=false;
    private boolean incrementIndexRequest=false;
    private boolean decrementIndexRequest=false;
    private Integer skipToRequest=null;
    
    private boolean recDisplayValid=false;
    private boolean speakerWindow;
    /**
     * returns true if the speaker window is shown
     * 
     * @return true if the speaker window is shown
     */
    public boolean isSpeakerWindowShowing() {
        return speakerWindow;
    }

    public SessionManager(){
        super();
        logger = Logger.getLogger("ipsk.apps.speechrecorder");
        timeLogger = Logger.getLogger("time");
        recStat = RecStatus.getInstance();
        progressManager=new ProgressManager();
      
        
        // Progress is paused until the start button is pressed
        // in auto recording mode
        progressPaused = true;
        
        uiString = UIResources.getInstance();
        
        startRecordAction = new StartRecordAction(this, uiString
                .getString("RecordingButtonText"));
        startRecordAction.setEnabled(false);
        stopRecordAction = new StopRecordAction(this, uiString
                .getString("StopButtonText"));
        stopRecordAction.setEnabled(false);
        stopNonrecordingAction = new StopNonrecordingAction(this, uiString
                .getString("NextButtonText"));
        stopNonrecordingAction.setEnabled(false);
        startAutoRecordingAction = new StartAutoRecordingAction(this, uiString
                .getString("StartButtonText"));
        pauseAutoRecordingAction = new PauseAutoRecordingAction(this, uiString
                .getString("PauseButtonText"));
        continueAutoRecordingAction = new ContinueAutoRecordingAction(this,
                uiString.getString("ContinueButtonText"));
        advanceToNextAction = new AdvanceToNextAction(this, uiString
                .getString("AdvanceToNextButtonText"));
        forwardAction = new ForwardAction(this, uiString
                .getString("ForwardButtonText"));
        backwardAction = new BackwardAction(this, uiString
                .getString("BackwardButtonText"));
        startPlaybackAction = new StartPlaybackAction(this, uiString
                .getString("PlayButtonText"));
        pausePlaybackAction = new PausePlaybackAction(this, uiString
                .getString("PlayButtonText")
                + "-" + uiString.getString("PauseButtonText"));
        stopPlaybackAction = new StopPlaybackAction(this, uiString
                .getString("StopButtonText"));
        continuePlaybackAction = new ContinuePlaybackAction(this, uiString
                .getString("PlayButtonText"));
        
        closeSpeakerDisplayAction=new CloseSpeakerDisplayAction(this,"Close speaker display");
        setIndexAction = new SetIndexAction(this, "Set index");
        progressManager.setSetIndexAction(setIndexAction);
        
        recTransporterActions = new RecTransporterActions();
        
        recTransporterActions.startRecordAction = startRecordAction;
        recTransporterActions.stopRecordAction = stopRecordAction;
        recTransporterActions.stopNonrecordingAction=stopNonrecordingAction;
        recTransporterActions.startAutoRecordingAction = startAutoRecordingAction;
        recTransporterActions.pauseAutoRecordingAction = pauseAutoRecordingAction;
        recTransporterActions.continueAutoRecordingAction = continueAutoRecordingAction;
        recTransporterActions.advanceToNextAction = advanceToNextAction;
        recTransporterActions.forwardAction = forwardAction;
        recTransporterActions.backwardAction = backwardAction;
        recTransporterActions.startPlaybackAction = startPlaybackAction;
        recTransporterActions.pausePlaybackAction = pausePlaybackAction;
        recTransporterActions.stopPlaybackAction = stopPlaybackAction;
        recTransporterActions.continuePlaybackAction = continuePlaybackAction;

        sessionActions=new SessionActions();
        sessionActions.recTransporterActions=recTransporterActions;
        sessionActions.closeSpeakerDisplayAction=closeSpeakerDisplayAction;
        sessionActions.setIndexAction=setIndexAction;
        
   
        
        debugSinusTest = (System.getProperty("debug.sinustest") != null);
    }
    
    private Script script=null;
    
    public Script getScript() {
        return script;
    }
    
    public List<AutoAnnotator> getAutoAnnotators(){
        if(autoAnnotationWorker!=null){
            return autoAnnotationWorker.getAutoAnnotators();
        }else{
            return null;
        }
    }

    private Session annotationSession=null;
    
    private boolean sessionOverwriteWarning=true;
    private boolean recManualPlay=false;

    public boolean isRecManualPlay() {
        return recManualPlay;
    }


    public void setRecManualPlay(boolean recManualPlay) {
        this.recManualPlay = recManualPlay;
    }

    private boolean allRecordingsDoneNotified;

    private boolean lastSpeakerWindowRequest;

    private javax.swing.Timer preRecTimer;

    private javax.swing.Timer postRecTimer;

    private javax.swing.Timer maxRecTimer;
    
    private Timer nonRecordingTimer;
    
    public boolean useMaxRecTimer=USE_MAX_REC_TIMER;
    private Double beepVolume;
    private boolean forcePostRecPhase;
    private boolean itemPlayable;
    /**
     * Returns whether the current selected recording can be played.
     * 
     * @return true if the recording is available
     */
    public boolean isItemPlayable() {
        return itemPlayable;
    }

    private boolean annotatingEnabled=false;
    
    public synchronized void actionPerformed(ActionEvent e) {
        Object src=e.getSource();
        if (src== maxRecTimer) {
            if(DEBUG)System.out.println("Max rec timer event !");
            preRecTimer.stop();
           
            if(forcePostRecPhase){
                startPostRecordingPhase();
            }else{
            try {
                stopRecording();
            } catch (AudioControllerException ex) {
                speechRecorderUI.displayError("Audiocontroller error",
                        "Error on stop of audio recording\n"
                                + ex.getLocalizedMessage());
                ex.printStackTrace();
                logger.severe(ex.getMessage());
            }
            }
        } else if (src == preRecTimer) {
            if(DEBUG)System.out.println("Precrec timer event !");
            if (recStat.getStatus() == RecStatus.PRERECWAITING){
                startRecordingPhase();
                if(DEBUG)System.out.println("Recording phase  !");
            }else if (recStat.getStatus() == RecStatus.POSTRECWAITING)
                if(annotatingEnabled){
                startAnnotation();
                }else{
                setIdle();
                startPrompt();
                }
        } else if (src == postRecTimer) {
            if(DEBUG)System.out.println("Postrec timer event !");
            try {
                stopRecording();
            } catch (AudioControllerException ex) {
                speechRecorderUI.displayError("Audio controller error ",
                        "Error on stop of audio recording\n"
                                + ex.getLocalizedMessage());
                logger.severe(ex.getMessage());
            }
        }else if (src == nonRecordingTimer) {
            setIdle();
            if(isAutoRecording()){
                continueSession();
            }
        }
    }
    
    public boolean isAutoRecording(){
        return (Section.Mode.AUTORECORDING.equals(sectionMode));
    }
    public boolean isAutoProgress(){
        return (isAutoRecording() || Section.Mode.AUTOPROGRESS.equals(sectionMode));
    }
    
    public void setProgressPaused(boolean progressPaused) {
        this.progressPaused = progressPaused;
        speechRecorderUI.setProgressPaused(progressPaused);
    }

    public boolean getProgressPaused() {
        return progressPaused;
    }


    
    
  
    
    public boolean isForcePostRecPhase() {
        return forcePostRecPhase;
    }


    public void setForcePostRecPhase(boolean forcePostRecPhase) {
        this.forcePostRecPhase = forcePostRecPhase;
    }


    private void setEditingEnabled(boolean b){
        if(this.listener!=null){
            this.listener.setEditingEnabled(b);
        }
    }
    
    private int preRecDelay(Recording r){
        return r.getNNPrerecdelay();
    }
    
    private int postRecDelay(Recording r){
        return r.getNNPostrecdelay();
    }
    
    private int minRecLengthMs(Recording r){
        int preRecDelay=preRecDelay(r);
        int postRecDelay=postRecDelay(r);
        return preRecDelay+postRecDelay;
    }
    
    

    private Bundle loadBundle(BundleAnnotationFilePersistor bafp,File annoFile,Bundle bundle){
        if(annoFile.exists()){
            bafp.setFile(annoFile);

            try{
                Bundle annotBundle=bafp.load(bundle);
                return annotBundle;
            }catch(IOException ioe){
                speechRecorderUI.displayError("Could not read annotation file \""+annoFile+"\"", ioe);
            }catch(ParserException pe){
                speechRecorderUI.displayError("Annotation parsing error","Could not read/parse annotation file \""+annoFile+"\":\n"+pe.getMessage());
            }
        }
        return null;
    }

    /**
     * Refresh the signal display.
     */
    private synchronized void setRecDisplay() {
        
        Integer recIndex=progressManager.getRecIndex();
        
        if (recIndex!= null && progressManager.getRecCounter(recIndex) > 0) {
            // URL[] recUrls = storageManager.getAudioFiles();
            if (useUploadCache) {
                try {
                    // InputStream[] recIss =
                    // uploadCache.getCachedInputStream(recUrls);

//                  InputStream[] recIss = storageManager
//                          .getCachedInputStreams();
                    File[] recFiles=storageManager.getCachedInputFiles();
                    if (recFiles != null) {
//                      BufferedInputStream[] biss = new BufferedInputStream[recIss.length];
//                      AudioInputStream[] aiss = new AudioInputStream[numLines];
//                      for (int i = 0; i < numLines; i++) {
//                          biss[i] = new BufferedInputStream(recIss[i]);
//                          aiss[i] = AudioSystem.getAudioInputStream(biss[i]);
//                      }
                        
                        audioClip.setBundle(null);
                        audioClip.setAudioSource(new FileAudioSource(recFiles[0]));
                        //speechRecorderUI.setRecDisplay(recUrls);
//                      for (int i = 0; i < numLines; i++) {
//                          aiss[i].close();
//                          biss[i].close();
//                          recIss[i].close();
//                      }
                    } else {
                        // Do not download files
                        // speechRecorderUI.getRecDisplay().setDisplay(recUrls);
                        //speechRecorderUI.clearRecDisplay();
                        audioClip.setBundle(null);
                        audioClip.setAudioSource(null);
                    }
                } catch (Exception e) {
                    speechRecorderUI.displayError("Audio system error",
                            "Cannot get audio stream.");
                }

            } else {
                URL[] audioFileURLs = null;
                try {
                    audioFileURLs = storageManager.generateAudioFileURLs();
                } catch (StorageManagerException e) {
                    e.printStackTrace();
                    String msg = "Storage Exception: "
                            + e.getLocalizedMessage();
                    logger.severe(msg);
                    speechRecorderUI.displayError("Storage manager error", msg);
                }
                try {
                    //speechRecorderUI.getRecDisplay().setDisplay(audioFileURLs);
                    if(audioFileURLs.length>1){
                        throw new AudioSourceException("Mutiple lines are currently not supported.");
                    }
                    File[] audioFiles=null;
				    if(audioFileURLs!=null){
						audioFiles=new File[audioFileURLs.length];
						for(int i=0;i<audioFileURLs.length;i++){
							audioFiles[i]=ActiveSessionStorageManager.fileURLToFile(audioFileURLs[i]);
						}
					}
				  
				    Bundle bundle=buildBaseBundle(audioFiles);
                    String bundleRootFn=storageManager.getRootFileName();
                    URL sessURL=storageManager.getSessionURL();
                    File sessDir=ActiveSessionStorageManager.fileURLToFile(sessURL);
                    
                    if(sessDir!=null){
                        Bundle annotBundle=null;
                        for(BundleAnnotationPersistor bap:bundleAnnotationPersistorList){
                            if(bap instanceof BundleAnnotationFilePersistor && bap.isLossless()){
                                BundleAnnotationFilePersistor bafp=(BundleAnnotationFilePersistor)bap;
                                File annoFile=new File(sessDir,bundleRootFn+bafp.getPreferredFilenameSuffix()+"."+bafp.getPreferredFileExtension());
                                annotBundle=loadBundle(bafp, annoFile, bundle);
                               
                            }else{
                                // currently only local files
                            }
                            // stop if lossless persistor loaded complete annotation
                            if(annotBundle!=null){
                                break;
                            }
                        }
                        if(annotBundle==null){
                            // no lossless persistor found try others
                            for(BundleAnnotationPersistor bap:bundleAnnotationPersistorList){
                                if(bap instanceof BundleAnnotationFilePersistor){
                                    BundleAnnotationFilePersistor bafp=(BundleAnnotationFilePersistor)bap;
                                    File annoFile=new File(sessDir,bundleRootFn+bafp.getPreferredFilenameSuffix()+"."+bafp.getPreferredFileExtension());
                                    annotBundle=loadBundle(bafp, annoFile, bundle);
                                }else{
                                    // currently only local files
                                }
                            }
                        }
                        if(annotBundle!=null){
                            // set annotated bundle as current bundle
                            bundle=annotBundle;
                        }
                    }
                   
                    audioClip.setAudioSource(new URLAudioSource(audioFileURLs[0]));
                    audioClip.setBundle(bundle);
                } catch (Exception ex) {
                    ex.printStackTrace();
                    String msg = "Exception: " + ex.getLocalizedMessage();
                    logger.severe(msg);
                    speechRecorderUI.displayError("Display error", msg);
                }

            }
        } else {
            //speechRecorderUI.getRecDisplay().clearDisplay();
            audioClip.setBundle(null);
            audioClip.setAudioSource(null);
        }
    }
    
    
    public void open(){
        lastSpeakerWindowRequest = false;
        if(audioController!=null){
            audioController.addAudioController2Listener(this);
        }
        if(beepPlayer!=null){
            beepPlayer.addPlayerListener(this);
        }
        autoAnnotationWorker=new AutoAnnotationWorker();
        List<AutoAnnotator> wAas=autoAnnotationWorker.getAutoAnnotators();
        wAas.clear();
        for(AutoAnnotator aa:enabledAutoAnnotators){
            if(aa instanceof PromptAutoAnnotator){
                promptAutoAnnotator=(PromptAutoAnnotator)aa;
            }
            if(aa instanceof TemplateAutoAnnotator){
                templateAutoAnnotator=(TemplateAutoAnnotator)aa;
            }
            wAas.add(aa);
        }
        autoAnnotationWorker.addProgressListener(this);
    }
    
    /**
     * Start a recording session. The application must be configured,
     * initialized and a speaker must be chosen to start a session.
     * @param sessionId session ID
     * @param speakerCode speaker code
     * @throws AudioControllerException audio controller exception
     */
    public void start(int sessionId,String speakerCode) throws AudioControllerException {
        if(DEBUG)System.out.println("Start session.");
        sessionOverwriteWarning=true;
        
        progressManager.addSessionManagerListener(this);
        
        if (storageManager != null) {
            storageManager.setSessionID(sessionId);
            annotationSession=new Session();
            NumberFormat sessFmt=storageManager.getSessionIDFormat();
            String sessNm=sessFmt.format(sessionId);
            annotationSession.setName(sessNm);
            storageManager
                    .setSpeakerCode(speakerCode);
        }
        try {
            storageManager.createSessionDirectory();
            if (progressManager != null) {
                progressManager.resetItemMarkers();
                progressManager.updateItemMarkers();
            }
        } catch (ProgressManagerException e1) {
            e1.printStackTrace();
            return;
        } catch (StorageManagerException e) {
            e.printStackTrace();
            return;
        }
        
        if(autoAnnotationWorker!=null){
            try {
                autoAnnotationWorker.open();
            } catch (WorkerException e1) {
                // TODO Auto-generated catch block
                e1.printStackTrace();
            }
            autoAnnotationWorker.start();
        }

        if (audioEnabled){
            if(seamlessAutoRecording){
                URL recSessInfUrl;
                try {
                    recSessInfUrl = storageManager.getRecordingSessionInfoFile();
                    File recSessInfFile=new File(recSessInfUrl.toURI().getPath());
                    audioController.setRecordingSessionInfoFile(recSessInfFile);
                } catch (StorageManagerException e) {
                    // TODO
                  //throw new SpeechRecorderException(e);
                } catch (URISyntaxException e) {
                    // TODO Auto-generated catch block
                    e.printStackTrace();
                }
                
            }
            int itemCount=progressManager.numberOfPromptItems();
            if(itemCount>0){
                progressManager.setRecIndex(0);
            }else{
                // set undefined
                progressManager.setRecIndex(null);
            }
            
            beepAudioSource = null;
            if(progressManager.needsBeep()){
            
                URLAudioSource orgBeepAs=new URLAudioSource(beepURL);
               
                if(beepVolume!=null){
                    PluginChain beepPc=new PluginChain(orgBeepAs);
                    VolumeControlPlugin vcp=new VolumeControlPlugin();
                    try {
                        beepPc.add(vcp);
                    } catch (AudioFormatNotSupportedException e) {
                        e.printStackTrace();
                        String eMsg="Could not set beep volume: "+e.getMessage();
                        logger.severe(eMsg);
                        throw new AudioControllerException(eMsg);
                    }
                    vcp.setGainRatio(beepVolume);
                    beepAudioSource=beepPc;
                }else{
                    beepAudioSource=orgBeepAs;
                }
            }
            
            boolean silenceDetection=progressManager.needsSilenceDetector();
            if(silenceDetection && silenceDetector==null){
                voiceDetector=new VoicedSpeechDetector();
                silenceDetector=new SpeechFinalSilenceDetector(voiceDetector,this);
            }
            
            if(CaptureScope.SESSION.equals(captureScope)){
                if(silenceDetection){
                    audioController.addCaptureFloatAudioOutputStream(voiceDetector);
                }
                if(DEBUG)System.out.println("Opening capture.");
                try{
                    audioController.openCapture();
                    audioController.startCapture();
                }catch(AudioControllerException ace){
                    // Display error, close the audio controller, but then proceed here
                    // (The user should be able to fix the project configuration)
                    speechRecorderUI.displayError("Audio controller error", ace);
                    closeAudioController();
                }
            }
        }
    }
    
    
    public Double getBeepVolume() {
        return beepVolume;
    }


    public void setBeepVolume(Double beepVolume) {
        this.beepVolume = beepVolume;
    }


    public void startItem(){
        repeatRequest=null;
        ipsk.db.speech.script.PromptItem promptItem = progressManager
        .getCurrentPromptItem();
        boolean isRecording =(promptItem instanceof Recording);
        // check overwrite first
        if (isRecording) {
         
            File recFile;
            List<File> annotationFiles=new ArrayList<File>(0);
            try {
                recFile = storageManager.getNewRecordingFiles()[0];
                if(!useUploadCache && bundleAnnotationPersistorList!=null && bundleAnnotationPersistorList.size()>0){
                    annotationFiles=currentAnnotationFiles();
                }
            } catch (StorageManagerException e) {
                  JOptionPane.showMessageDialog(speechRecorderUI.getDialogTarget(), e.getMessage(),"Storage error!",JOptionPane.ERROR_MESSAGE);
                    e.printStackTrace();
                    return;
            }
            if(DEBUG)System.out.println("Recfile: "+recFile);
            
            boolean overWriteConfirmed=overwrite;
            boolean projectOverwriteWarning=overwriteWarning;
            if(projectOverwriteWarning && sessionOverwriteWarning && !useUploadCache){
                overWriteConfirmed=false;
                boolean recFileExists=recFile.exists();
                int existingAnnoFileCnt=0;
                for(File annoFile:annotationFiles){
                    if(annoFile.exists()){
                        existingAnnoFileCnt++;
                    }
                }
                if(recFileExists){
                    //              Object[] options=new Object[]{"No","Yes","Yes to all in this session","Yes to all in this project"};
                    Object[] options=new Object[]{"No","Yes","Yes to all in this session","Yes to all in this project"};
                    String msg="Recording file ";
                    if(existingAnnoFileCnt>0){
                        msg=msg.concat("and "+existingAnnoFileCnt+" annotation ");
                        if(existingAnnoFileCnt==1){
                            msg=msg.concat("file ");
                        }else{
                            msg=msg.concat("files ");
                        }
                        msg=msg.concat("already exist!");
                    }else{
                        msg=msg.concat("already exists!");
                    }
                    msg=msg.concat("\nDo you want to overwrite?");
                    int selOpt=JOptionPane.showOptionDialog(speechRecorderUI.getDialogTarget(),msg, "Overwrite warning",JOptionPane.DEFAULT_OPTION, JOptionPane.WARNING_MESSAGE, null,options,options[0]);
                    if(selOpt==JOptionPane.CLOSED_OPTION || selOpt==0){
                        overWriteConfirmed=false;
                        setProgressPaused(true);
                        setIdle();
                       
                        return;
                    }else if(selOpt==2){
                        overWriteConfirmed=true;
                        sessionOverwriteWarning=false;
                    }else if(selOpt==3){
                        
                        // need a callback here
                        overWriteConfirmed=true;
                        if(this.listener!=null){
                            this.listener.requestProjectConfigDisableOverwriteWarning();
                        }
//                       recConfig.setOverwriteWarning(false);
//                       projectConfigurationSaved=false;
                    }
                   
                }
            }
            
            if(overWriteConfirmed){
                //  delete annotation
                for(File annoFile:annotationFiles){
                    if(annoFile.exists()){
                        // TODO with Java 7: get exception 
                        boolean deleted=annoFile.delete();
                        if(!deleted){
                            JOptionPane.showMessageDialog(speechRecorderUI.getDialogTarget(), "Could not delete annotation file "+annoFile.getName(),"Delete error",JOptionPane.ERROR_MESSAGE );
                        }
                    }
                }
            }
        }
       
        setEditingEnabled(false);
        sessionActions.setIndexAction.setEnabled(false);
        recTransporterActions.startAutoRecordingAction.setEnabled(false);
        recTransporterActions.continueAutoRecordingAction.setEnabled(false);
        recTransporterActions.startRecordAction.setEnabled(false);
        recTransporterActions.stopRecordAction.setEnabled(false);
        recTransporterActions.advanceToNextAction.setEnabled(false);
        recTransporterActions.forwardAction.setEnabled(false);

        recTransporterActions.backwardAction.setEnabled(false);

        recTransporterActions.startPlaybackAction.setEnabled(false);
        recTransporterActions.stopPlaybackAction.setEnabled(false);
        recTransporterActions.pausePlaybackAction.setEnabled(false);
        
        if (isAutoRecording()) {
            recTransporterActions.pauseAutoRecordingAction.setEnabled(false);
            recTransporterActions.continueAutoRecordingAction.setEnabled(false);
            recTransporterActions.startAutoRecordingAction.setEnabled(false);
        }
        if (Section.PromptPhase.IDLE.equals(progressManager.currentPromptPhase()) || !isRecording) {
            boolean blocked = true;
            if (promptItemCode != null) {
                timeLogger.info("PLAY_PROMPT: " + promptItemCode);
            }else{
                timeLogger.info("PLAY_PROMPT:");
            }
            if(isRecording){
                Recording recording=(Recording)promptItem;

                blocked = recording.getNNBlocked();
                
                recStat.setStatus(RecStatus.PLAY_PROMPT);
                if(!blocked){
                    // drop playing beep here
                    // currently we have only an audio media prompter 
                    // the direct audio line does not support mixing so we cannot play a beep
                    // Check if we can play a beep
                    List<Mediaitem> mis=recording.getMediaitems();
                    boolean containsAudio=false;
                    for(Mediaitem mi:mis){

                        String mimeType=mi.getNNMimetype();
                        int mimeSep=mimeType.indexOf("/");
                        if(mimeSep>0){
                            mimeType=mimeType.substring(0,mimeSep);
                        }
                        if(mimeType.equalsIgnoreCase("audio")){
                            containsAudio=true;
                            break;
                        }
                    }
                    if(containsAudio){
                        startPreRecWaiting();
                    }else{
                        try {
                            startBeep();
                        } catch (SpeechRecorderException e) {
                            e.printStackTrace();
                            speechRecorderUI.displayError("Beep playback error", e);
                            setIdle();
                            startPrompt();
                        }
                    }
                }
            }else{
                recStat.setStatus(RecStatus.PLAY_PROMPT);
            }
            speechRecorderUI.setPromptStartControlEnabled(recManualPlay);
            try {
                speechRecorderUI.openPlayPrompt();
            } catch (PrompterException e) {
                e.printStackTrace();
                setIdle();
            }

        } else {
            
            try {
                startBeep();
            } catch (SpeechRecorderException e) {
                e.printStackTrace();
                speechRecorderUI.displayError("Beep playback error", e);
                setIdle();
            }
        }
    }
    
    private void startNonRecording(){
        ipsk.db.speech.script.PromptItem promptItem = progressManager
        .getCurrentPromptItem();
        Integer duration=null;
        if(promptItem instanceof Nonrecording){
        Nonrecording nr=(Nonrecording)promptItem;
          duration=nr.getDuration();
        }
        if(duration!=null){
            if(DEBUG)System.out.println("Start non recording");
           
            timeLogger.info("NON_RECORDING:");
            
            recStat.setStatus(RecStatus.NON_RECORDING_WAIT);
            nonRecordingTimer = new javax.swing.Timer(duration, this);
            nonRecordingTimer.setRepeats(false);
            nonRecordingTimer.start();
            
        }else{
            setIdle();
            if(isAutoRecording()){
                continueSession();
            }
        }
    }
    
    public void startBeep() throws SpeechRecorderException{
        ipsk.db.speech.script.PromptItem promptItem = progressManager
        .getCurrentPromptItem();
        boolean playBeep=false;
        if(promptItem instanceof Recording){
        Recording recording=(Recording)promptItem;
           playBeep=new Boolean(recording.getBeep());
        }
        if(playBeep){
            if(DEBUG)System.out.println("Start beep");
            if (promptItemCode != null) {
                timeLogger.info("PLAY_BEEP: " + promptItemCode);
            }else{
                timeLogger.info("PLAY_BEEP:");
            }
            recStat.setStatus(RecStatus.PLAY_BEEP);
            try {
                
                beepPlayer.setAudioSource(beepAudioSource);
               
                if(beepPlayer.isOpen()){
                    System.err.println("Beep player still open!");
                }
                beepPlayer.open();
                beepPlayer.play();
            }  catch (PlayerException e) {
                e.printStackTrace();
                try {
                    stopRecording();
                } catch (AudioControllerException e1) {
                    throw new SpeechRecorderException(e1);
                }
                speechRecorderUI.displayError("Beep playback error", e);
                startPreRecWaiting();
            }
            
        }else{
            startPreRecWaiting();
        }
    }

    public void startPreRecWaiting() {

        setEditingEnabled(false);
        sessionActions.setIndexAction.setEnabled(false);
        speechRecorderUI
        .setLevelMeterMode(SpeechRecorderUI.LEVEL_METER_RECORDING);
        if (resetPeakOnRecording) {
            LevelInfo[] lis=audioController.getCaptureLevelInfos();
            if(lis!=null){
                for(LevelInfo li:lis){
                    li.setPeakLevelHold(0);
                }
            }
            speechRecorderUI.getLevelMeter().resetPeakHold();
        }
        if (promptItemCode != null) {
            timeLogger.info("PRERECORDING: " + promptItemCode);
        }
       
        recDisplayValid=false;

        recStat.setStatus(RecStatus.PRERECWAITING);
        speechRecorderUI.setRecMonitorsStatus(StartStopSignal.State.PRERECORDING);
        try {
            if(DEBUG)System.out.println("Start recording ...");
            startRecording();

        } catch (AudioControllerException e) {
            speechRecorderUI.displayError("Audiocontroller error",
                    "Error on start of audio recording\n"
                    + e.getLocalizedMessage());
            repeatRequest=new RepeatRequest();
            e.printStackTrace();
            //            setIdle();
            setProgressPaused(true);
            
            continueSession();
            return;
        } catch (StorageManagerException e) {
            speechRecorderUI.displayError("Storage error",
                    "Error on start of audio recording\n"
                    + e.getLocalizedMessage());
            repeatRequest=new RepeatRequest();
            e.printStackTrace();
            //            setIdle();
            setProgressPaused(true);
            continueSession();
            return;
        }

        RecWindow rw=speechRecorderUI.getRecWindow();
        if(rw!=null){
            PromptViewer pv0=rw.getPromptViewer();
            if(pv0!=null){
                pv0.setInstructionsEmphased(true);
            }
        }
        PromptViewer pv1=speechRecorderUI.getPromptViewer();
        if(pv1!=null){
            pv1.setInstructionsEmphased(true);
        }
        if (Section.PromptPhase.PRERECORDING.equals(progressManager.currentPromptPhase()) ||
        		Section.PromptPhase.PRERECORDINGONLY.equals(progressManager.currentPromptPhase())){
            speechRecorderUI.setShowPromptViewers(true);
            speechRecorderUI.setPromptStartControlEnabled(recManualPlay);
            try {
                speechRecorderUI.openPlayPrompt();
            } catch (PrompterException e) {
                e.printStackTrace();
                setIdle();
            }
        }

    }
    
    
    public void startRecordingPhase(){
        
        sessionActions.setIndexAction.setEnabled(false);
        if (promptItemCode != null) {
            timeLogger.info("RECORDING: " + promptItemCode);
        }
        speechRecorderUI.getPromptViewer().setInstructionsEmphased(false);
        speechRecorderUI.getRecWindow().getPromptViewer().setInstructionsEmphased(false);
        
        if (Section.PromptPhase.PRERECORDINGONLY.equals(progressManager.currentPromptPhase())) {
        	speechRecorderUI.setShowPromptViewers(false);
        }else {
        	speechRecorderUI.getPromptViewer().setPromptEmphased(true);
        	speechRecorderUI.getRecWindow().getPromptViewer().setPromptEmphased(true);
        }
        if (Section.PromptPhase.RECORDING.equals(progressManager.currentPromptPhase())) {
            speechRecorderUI.setShowPromptViewers(true);
            speechRecorderUI.setPromptStartControlEnabled(false);
            try {
                speechRecorderUI.openPlayPrompt();
            } catch (PrompterException e) {
                e.printStackTrace();
                setIdle();
            }
        
            // new behaviour with prompt phase recording and prompt blocking:
            // the traffic light switches to green after the prompt is played
            if(!progressManager.currentPromptBlocking()){
                speechRecorderUI.setRecMonitorsStatus(StartStopSignal.State.RECORDING);
            }
                
        }else{
            speechRecorderUI.setRecMonitorsStatus(StartStopSignal.State.RECORDING);
        }
        recStat.setStatus(RecStatus.RECORDING);
    }
    
    
    public void startPostRecordingPhase(){
        if(DEBUG)System.out.println("Starting post recording phase");
        sessionActions.setIndexAction.setEnabled(false);
        preRecTimer.stop();
        if(maxRecTimer!=null)maxRecTimer.stop();
        if (promptItemCode != null) {
            timeLogger.info("POSTRECORDING: " + promptItemCode);
        }
        speechRecorderUI.getPromptViewer().stop();
        speechRecorderUI.getRecWindow().getPromptViewer().stop();
        int postRecDelay=postRecDelay((Recording)promptItem);
        postRecTimer = new javax.swing.Timer(postRecDelay, this);
        postRecTimer.setRepeats(false);
        postRecTimer.start();
        
        recStat.setStatus(RecStatus.POSTRECWAITING);
        speechRecorderUI.setRecMonitorsStatus(StartStopSignal.State.POSTRECORDING);
    }
    
    public void startRecording() throws AudioControllerException,
    StorageManagerException {
ipsk.db.speech.script.PromptItem promptItem = progressManager
        .getCurrentPromptItem();
if (promptItem instanceof Recording) {
    Recording pi = (Recording) promptItem;

    int preRecDelay=preRecDelay(pi);
    
    preRecTimer = new javax.swing.Timer(preRecDelay, this);
    // System.out.println(
    // "maxRecTime: " + recScriptManager.getMaxRecTimeMillis() + " ms");
    
    // Create max rec timer if required
    maxRecTimer=null;
    
    
    // Do not use the prompt item method anymore
    // it does not consider project default values
    
    //Integer totalRecTime=pi.getTotalRecTime();
//   if(totalRecTime!=null){
//          maxRecTimer = new javax.swing.Timer(totalRecTime, this);
//          maxRecTimer.setDelay(totalRecTime);
//          maxRecTimer.setRepeats(false);
//    }
    
    Integer recDuration=pi.getRecduration();
    if(recDuration!=null){
       // RecordingConfiguration recCfg=project.getRecordingConfiguration();
        // calculate max recording time
        int postRecDelay=pi.getNNPostrecdelay();
       
        long totalRecTime=preRecDelay+recDuration+postRecDelay;
        long recTime=totalRecTime;
        if(forcePostRecPhase){
            recTime=preRecDelay+recDuration;
        }
        if(useMaxRecTimer || seamlessAutoRecording){
            // seamless autorecording only works with timer limited recording
            maxRecTimer = new javax.swing.Timer((int)recTime, this);
            maxRecTimer.setDelay((int)recTime);
            maxRecTimer.setRepeats(false);
        }else{
            // capture engine limits the maximum recording length
            // has the advantage that the recordings have exactly maximum frame length
            // of the recduration attribute
            
            // (currently not used)
            float frameRate=audioController.getAudioFileFormat().getFormat().getFrameRate();
            long maxFrameLength=((long)((float)totalRecTime*frameRate))/1000;
            audioController.setMaxRecordingFrameLength(maxFrameLength);
        }
    }
    
    preRecTimer.setDelay(preRecDelay);
    preRecTimer.setRepeats(false);
    
    audioClip.setAudioSource(null);
    File recFile=storageManager.getCurrentItemRecordingFiles()[0];
    
    audioController.setRecordingFile(recFile);
    
    boolean silenceDetection=pi.needsSilenceDetector();
    Integer finalSilence=pi.getFinalsilence();
    
    if(silenceDetection){
        silenceDetector.setSilencelength((double)finalSilence/1000.0);
    }
    // levelMeter.resetPeak();
//  System.gc();
//  Thread.yield();
    if (debugSinusTest)
        if(DEBUG)System.out.println("Start recording item: "
                + pi.getItemcode());
    
    if(!audioController.isCaptureOpen()){
        if(DEBUG)System.out.println("Opening capture.");
        if(silenceDetection){
            audioController.addCaptureFloatAudioOutputStream(voiceDetector);
        }
        audioController.openCapture();
        audioController.startRecording();
    }else{
//      if(!seamlessAutoRecording){
            audioController.startRecording();
            
//      }
    }
    if(silenceDetection){
//      voiceDetector.start();
        silenceDetector.start();
    }
    if(DEBUG)System.out.println("Recording started");

}
}
    
    public synchronized void stopRecording() throws AudioControllerException {
//      boolean repeat = false;
        if(DEBUG)System.out.println("Stop recording");
        if(preRecTimer!=null)preRecTimer.stop();
        if(maxRecTimer!=null)maxRecTimer.stop();
        if(isAutoRecording() && seamlessAutoRecording && ! progressPaused){
        	 progressManager.incrementRecCounter(progressManager
                     .getRecIndex());
            continueSession();
           
        }else{
        try {
            boolean continueCapture=CaptureScope.SESSION.equals(captureScope);
            audioController.stopRecording(continueCapture);
            // TODO
            // setLogEntries();
            // String labelFilename =
            // new String(
            // filePrefix
            // + String.valueOf(speakerID)
            // + recScriptManager.getPromptCode()
            // + labelFileExtension);
            //
            // recLogger.createLabelFile(labelFilename);

            // recScriptManager.incrementRecCounter(
            // recScriptManager.getRecIndex());
            // //URL[] recUrls = storageManager.getAudioFiles();
            // if (useUploadCache) {
            // //Upload[] uploads = new Upload[numLines];
            // storageManager.upload();
            // //uploadCache.upload(uploads);
            // }
        } catch (AudioControllerException e) {
            speechRecorderUI
                    .displayError(
                            "Audiocontroller error",
                    "Technical error: \n"
                                    + e.getLocalizedMessage()
                            + "\nPlease press OK.\nthe recording will be repeated.");
            logger.severe(e.getMessage());
//          repeat = true;
        }
        // } catch (StorageManagerException e) {
        // speechRecorderUI.displayError(
        // "Storage error",
        // "Storage error: " + e.getLocalizedMessage());
        // logger.severe(e.getMessage());
        // repeat = false;
        // }
        // continueSession(repeat);

        logger.fine("Recording stopped");
        if(DEBUG)System.out.println("Recording stopped");
        }
    }
    
    public synchronized void stopNonrecording(){
        if(nonRecordingTimer!=null){
            nonRecordingTimer.stop();
        }
        if(speechRecorderUI.isPromptClosed()){
            if(DEBUG)System.out.println("Non recording stop event. Continue session...");
            continueSession();
        }else{
            if(DEBUG)System.out.println("Non recording stop. wait for prompt viewer....");
            speechRecorderUI.closePrompt();
        }
    }

    
    
    public void startAnnotation(){
        if(promptItem!=null){
            if (promptItem instanceof Recording) {

                timeLogger.info("ANNOTATE: " + promptItemCode);
            } else if(promptItem instanceof Nonrecording){
                timeLogger.info("ANNOTATE: Nonrecording");
            }
        }
        setEditingEnabled(true);
        speechRecorderUI.idle();
        speechRecorderUI
        .setLevelMeterMode(SpeechRecorderUI.LEVEL_METER_DISABLE);
        int recVersions=0;
        if(promptItem instanceof Recording){
            try {
                recVersions = storageManager.getRecordedVersions();
            } catch (StorageManagerException e) {
                e.printStackTrace();
            }
        }
        itemPlayable = (recVersions > 0);
        setRecDisplay();
    }
    
    
    /**
     * When all recordings of a session have been performed, the user is
     * informed that the session is over. Otherwise, if the automatic_recording
     * mode is on, then the next recording is started. If automatic_recording is
     * off, the application waits for the user to continue the recording
     * session.
     * 
     */
    public void continueSession() {

        if (progressManager.allRecordingsDone() && repeatRequest==null) {
            setIdle();
            if (!allRecordingsDoneNotified) {
                JOptionPane.showMessageDialog(speechRecorderUI
                        .getDialogTarget(), uiString
                        .getString("DialogRecordingsCompleteText"), uiString
                        .getString("DialogRecordingsCompleteTitle"),
                        JOptionPane.INFORMATION_MESSAGE);
                allRecordingsDoneNotified = true;
            }
        } else {
            boolean autoProgress = isAutoProgress();
            boolean autoRecording = isAutoRecording();

            if(repeatRequest!=null){

                String title=repeatRequest.getMessageTitle();
                String msg=repeatRequest.getMessage();
                setIdle();
                if(msg!=null){
                    JOptionPane.showMessageDialog(speechRecorderUI
                            .getDialogTarget(),msg,title,
                            JOptionPane.WARNING_MESSAGE);
                }

                if (autoRecording && !progressPaused) {
                    MessageFormat form = new MessageFormat(uiString
                            .getString("prompt_repeat_info"));
                    JOptionPane.showMessageDialog(speechRecorderUI
                            .getDialogTarget(), form
                            .format(new Object[] { new Float(
                                    RECORD_RETRY_DELAY / 1000) }), uiString
                            .getString("prompt_repeat"),
                            JOptionPane.INFORMATION_MESSAGE);
                    try {
                        Thread.sleep(RECORD_RETRY_DELAY);
                    } catch (InterruptedException e) {
                        // no problem
                    }
                    startItem();
                }else{
                    JOptionPane.showMessageDialog(speechRecorderUI
                            .getDialogTarget(),uiString.getString("prompt_repeat") , uiString
                            .getString("prompt_repeat"),
                            JOptionPane.INFORMATION_MESSAGE);

                }
            }else{
                if (autoProgress) {
                    if (progressToNextUnrecorded) {
                        progressManager.advanceToNextRecording();
                    } else {
                        progressManager.incrementIndex();
                    }
                    // Update autoRecording flag
                    // Bug fix ID0032
                    autoRecording=isAutoRecording();
                }else{
                    setIdle();
                }
                if (autoRecording && !progressPaused) {
                    startItem();
                }
            }
        }
    }
    
    /**
     * Initialize the recorder. All components get ready, but are still
     * disabled.
     */
    public void init() {
        allRecordingsDoneNotified = false;
        progressManager.setRecIndex(null);
        annotationSession=null;
        
        recStat.setStatus(RecStatus.INIT);
        setEditingEnabled(true);
        speechRecorderUI.init();
    }
    
    public void applyItem() throws PromptPresenterException{
        promptItem = progressManager
                .getCurrentPromptItem();
        if(promptItem!=null){
        if (promptItem !=null && promptItem instanceof Recording) {
            promptItemCode=((Recording) promptItem).getItemcode();
        }else{
            promptItemCode=null;
        }
        storageManager.setPromptCode(promptItemCode);
        
        speechRecorderUI.setRecIndex(progressManager.getRecIndex());
        speechRecorderUI.setPromptItem(promptItem);
        
        Section currentRecSection=progressManager.getCurrentRecSection();
        if(currentRecSection!=null){
         // project mode is overwritten by section mode
           
            Section.Mode scriptMode=currentRecSection.getMode();
            if (scriptMode != null) {
                sectionMode=scriptMode;
            }else{
                sectionMode=defaultSectionMode;
            }
      
        boolean currentSpeakerWindowRequest = currentRecSection.getNNSpeakerDisplay();
       
        if (lastSpeakerWindowRequest != currentSpeakerWindowRequest) {
            setSpeakerWindowShowing(currentSpeakerWindowRequest);
        }
        lastSpeakerWindowRequest = currentSpeakerWindowRequest;
        }else{
            sectionMode=null;
        }
        
        speechRecorderUI.setAutoRecording(isAutoRecording());
        }else{
            
        }
        recDisplayValid=false;
    }
    
    public void setPromptErrorState(){
        if(promptItem!=null){
            if (promptItem instanceof Recording) {
                timeLogger.info("ERROR: " + promptItemCode);
            } else if(promptItem instanceof Nonrecording){
                timeLogger.info("ERROR: Nonrecording");
            }
        }
        setEditingEnabled(true);
        speechRecorderUI.idle();
        speechRecorderUI
                .setLevelMeterMode(SpeechRecorderUI.LEVEL_METER_DISABLE);
        //  should be done if speaker/script  changes
//        storageManager.setSpeakerCode(speakerManager.getSpeaker()
//                .getCode());
        //Script script = recScriptManager.getScript();
        storageManager.setScriptID(script.getName());
        if(!recDisplayValid){
            int recVersions=0;
            if(promptItem instanceof Recording){
            try {
                recVersions = storageManager.getRecordedVersions();
                  // set to last recorded version
                storageManager.setRecVersion(recVersions - 1);
                } catch (StorageManagerException e) {
                    e.printStackTrace();
                }
            }
                itemPlayable = (recVersions > 0);
                speechRecorderUI.setPlaybackEnabled(itemPlayable);
                setRecDisplay();
                recDisplayValid=true;
          
        }
        speechRecorderUI.setShowPromptViewers(false);
        speechRecorderUI.setRecMonitorsStatus(ipsk.apps.speechrecorder.monitor.StartStopSignal.State.OFF);
        recStat.setStatus(RecStatus.ITEM_ERROR);
        if(isAutoRecording()){
            setProgressPaused(true);
        }
       
    }
    
    public void setIdle(){
        repeatRequest=null;
        if(promptItem!=null){
            if (promptItem instanceof Recording) {
                timeLogger.info("IDLE: " + promptItemCode);
            } else if(promptItem instanceof Nonrecording){
                timeLogger.info("IDLE: Nonrecording");
            }
        }
        setEditingEnabled(true);
        speechRecorderUI.idle();
        if(CaptureScope.SESSION.equals(captureScope)){
            speechRecorderUI
            .setLevelMeterMode(SpeechRecorderUI.LEVEL_METER_CAPTURE);
        }else{
            speechRecorderUI
                .setLevelMeterMode(SpeechRecorderUI.LEVEL_METER_DISABLE);
        }
        // should be done if speaker/script  changes
//        storageManager.setSpeakerCode(speakerManager.getSpeaker()
//                .getCode());
//        Script script = recScriptManager.getScript();
        storageManager.setScriptID(script.getName());
        // TODO set meta data
        // storageManager.setMetadata(script.getMetaData());

        // TODO should be done if session is created
        if (!useUploadCache) {
            try {
                storageManager.createSessionDirectory();
            } catch (StorageManagerException e) {
                e.printStackTrace();
                return;
            }
        }
        if(!recDisplayValid){
            int recVersions=0;
            if(promptItem instanceof Recording){
            try {
                recVersions = storageManager.getRecordedVersions();
            } catch (StorageManagerException e) {
                e.printStackTrace();
//                  return;
            }
            // set to last recorded version
            storageManager.setRecVersion(recVersions - 1);
            }
            // itemPlayable = storageManager.isRecorded();
            itemPlayable = (recVersions > 0);
            speechRecorderUI.setPlaybackEnabled(itemPlayable);
            setRecDisplay();
            recDisplayValid=true;
        }
        
        if(promptItem instanceof Recording){
            if (!Section.PromptPhase.IDLE.equals(progressManager.currentPromptPhase())){
                speechRecorderUI.setShowPromptViewers(false);
            }
            speechRecorderUI.setRecMonitorsStatus(StartStopSignal.State.IDLE);
            recStat.setStatus(RecStatus.IDLE);
        }else if(promptItem instanceof Nonrecording){
            speechRecorderUI.setRecMonitorsStatus(StartStopSignal.State.OFF);
            recStat.setStatus(RecStatus.NON_RECORDING);
        }
    }
    
    private void startPrompt(){
        boolean promptStartEnabled=(!isAutoRecording() || getProgressPaused());
        if(promptItem instanceof Recording){
            
            if (Section.PromptPhase.IDLE.equals(progressManager.currentPromptPhase())){
                speechRecorderUI.setShowPromptViewers(true);
                try {
                    speechRecorderUI.startPromptAutoplay();
                } catch (PrompterException e) {
                    e.printStackTrace();
                    setIdle();
                }
                speechRecorderUI.setPromptStartControlEnabled(promptStartEnabled);
            }
        }else if(promptItem instanceof Nonrecording){
            speechRecorderUI.setShowPromptViewers(true);
            try {
                speechRecorderUI.startPromptAutoplay();
            } catch (PrompterException e) {
                e.printStackTrace();
                setIdle();
            }
            speechRecorderUI.setPromptStartControlEnabled(promptStartEnabled);
        }

        
    }
    
    
	private Bundle buildBaseBundle(File[] recFiles) throws IOException, UnsupportedAudioFileException{
		   
		// build Bundle
		Bundle bundle=new Bundle();
		bundle.setSession(annotationSession);
		String targetRootFn=storageManager.getNewRootFileName();
		bundle.setName(targetRootFn);


		if(recFiles!=null && recFiles.length>0){
			File masterFile=recFiles[0];
			bundle.setAnnotates(masterFile.getName());
			List<String> sigPathes=new ArrayList<String>();
			for(File rf:recFiles){
				sigPathes.add(rf.getAbsolutePath());
			}
			bundle.setSignalpaths(sigPathes);
			try {
				ConvenienceFileAudioSource cfas=new ConvenienceFileAudioSource(masterFile);
				AudioFormat af=cfas.getFormat();
				long fl=cfas.getFrameLength();
				bundle.setSampleRate(af.getSampleRate());
				bundle.setFrameLength(fl);
			} catch (AudioSourceException e) {

				e.printStackTrace();
				// OK could not retrieve the sample rate and frame length
			}

		}

		return bundle;

	}
	
	private Bundle prepareAutoAnnotation() throws IOException, UnsupportedAudioFileException{
	    if(promptAutoAnnotator!=null){
	        promptAutoAnnotator.setPromptText(null);
	    }
	    if(templateAutoAnnotator!=null){
	        templateAutoAnnotator.setTemplateText(null);
	    }
	    File[] recFiles;
	    Bundle bundle=null;
       
	    recFiles=storageManager.getCurrentItemRecordingFiles();
	    bundle=buildBaseBundle(recFiles);
       
        String prDescr=promptItem.getDescription();

        if(promptAutoAnnotator!=null){
            promptAutoAnnotator.setPromptText(prDescr);
        }
       
        LocalizedText annoTempl=promptItem.annotationTemplateLocalizedText();
        if(templateAutoAnnotator!=null){
            if(annoTempl!=null){
                String annoTempltext=annoTempl.getText();
                templateAutoAnnotator.setTemplateText(annoTempltext);
            }
        }

        return bundle;
	    
	}
    
 
    
    private void itemFinished(){
        RecStatus st=RecStatus.getInstance();
        int status = st.getStatus();
        if (status == RecStatus.PRERECWAITING
                || status == RecStatus.RECORDING
                || status == RecStatus.POSTRECWAITING) {
            st.setStatus(RecStatus.RECORDED);
            if (useUploadCache) {
                try {
                    storageManager.upload();

                } catch (StorageManagerException e) {
                    speechRecorderUI
                            .displayError("Storage error",
                                    "Storage error: "
                                            + e.getLocalizedMessage());
                    logger.severe(e.getMessage());
                }
            }
            if(CaptureScope.SESSION.equals(captureScope) || !audioController.isCaptureOpen()){
                if(speechRecorderUI.isPromptClosed()){

                    if(DEBUG)System.out.println("Capture finished event. Continue session...");
                   
                    if(repeatRequest==null){
                        // check if write any annotation files
                        if(bundleAnnotationPersistorList!=null && bundleAnnotationPersistorList.size()>0){
                            // build a new annotation bundle
                            Bundle bundle=null;
                            try {
                                bundle = prepareAutoAnnotation();
                            } catch (IOException e1) {
                                // TODO Auto-generated catch block
                                e1.printStackTrace();
                            } catch (UnsupportedAudioFileException e1) {
                                // TODO Auto-generated catch block
                                e1.printStackTrace();
                            }
                            if(annotatingEnabled){
                                // start user annotation
                                startAnnotation();
                            }
                            // already persist this bundle
                            try {
                                persistBundle(bundle);
                                // start auto annotators (asynchron)
                                startAutoAnnotation(bundle);
                            } catch (StorageManagerException e) {
                                // TODO Auto-generated catch block
                                e.printStackTrace();
                            } catch (IOException e) {
                                // TODO Auto-generated catch block
                                e.printStackTrace();
                            } catch (EncodeException e) {
                                // TODO Auto-generated catch block
                                e.printStackTrace();
                            }
                           
                            
                        }
                    }
                      
                    continueSession();
                    
                }else{
                    if(DEBUG)System.out.println("Capture finished. Wait for prompt viewer....");
                    speechRecorderUI.closePrompt();
                }
            }else{
                System.err.println("Capture still open !");
            }
        }
    }
    
    /*
     * (non-Javadoc)
     * 
     * @see ipsk.audio.AudioController2.AudioController2Listener#update(ipsk.audio.AudioController2.AudioControllerEvent)
     */
    public void update(AudioControllerEvent ace) {

        if(DEBUG)System.out.println("Audio controller event: "+ace);
        if (ace instanceof PlayerEvent) {
            if (ace instanceof PlayerStopEvent) {
                if (ace instanceof PlayerPauseEvent) {
                	sessionActions.setIndexAction.setEnabled(true);
                    recStat.setStatus(RecStatus.PLAYPAUSE);
                } else {
                    try {
                        audioController.closePlayback();
                    } catch (AudioControllerException e) {
                        speechRecorderUI.displayError(
                                "Audio controller close playback error", e
                                        .getLocalizedMessage());

                        e.printStackTrace();
                    }
                }
            } else if (ace instanceof PlayerStartEvent) {
                sessionActions.setIndexAction.setEnabled(true);
                speechRecorderUI
                        .setLevelMeterMode(SpeechRecorderUI.LEVEL_METER_PLAYBACK);
                recStat.setStatus(RecStatus.PLAY);
            }else if(ace instanceof PlayerCloseEvent){
            	speechRecorderUI.updateView();
            	if(incrementIndexRequest) {
            		incrementIndexRequest=false;
            		progressManager.incrementIndex();
            	}else if(decrementIndexRequest) {
            		decrementIndexRequest=false;
            		progressManager.decrementIndex();
            	}else if(advanceToNextRequest) {
            		advanceToNextRequest=false;
            		progressManager.advanceToNextRecording();
            	}else if(skipToRequest!=null) {
            		int skipTo=skipToRequest;
            		skipToRequest=null;
            		progressManager.setRecIndex(skipTo);
            	}
                setIdle();
                startPrompt();
            }
        } else if (ace instanceof CaptureEvent) {
            // String status = ce.getStatus();
            if (ace instanceof CaptureStartCaptureEvent) {
                if(DEBUG)System.out.println("Capture start capture event.");
            }else if (ace instanceof CaptureStartRecordEvent) {
                if(DEBUG)System.out.println("Capture start record event. start timers");
                preRecTimer.start();
                if(maxRecTimer!=null)maxRecTimer.start();
            } else if (ace instanceof CaptureRecordingFileTransitEvent) {
                if(DEBUG)System.out.println("Capture recording file transit");
               
                preRecTimer.start();
                if(maxRecTimer!=null)maxRecTimer.start();
                
            }  else if (ace instanceof CaptureRecordedEvent) {
                progressManager.incrementRecCounter(progressManager
                        .getRecIndex());
                if(DEBUG)System.out.println("Capture recorded event.");
                if(promptItem instanceof Recording){
                    // check recorded file
                    long recordedFrameLength=audioController.getCaptureFramePosition();
                    if(DEBUG)System.out.println("Recorded frames: "+recordedFrameLength);
                    float sampleRate=audioController.getAudioFileFormat().getFormat().getSampleRate();
                    double recordedMs=recordedFrameLength*1000.0/sampleRate;
                    Recording r=(Recording)promptItem;
                    int minRecordLenMs=minRecLengthMs(r);
                    // check if audio file has minimum length 
                    // since we are working with timers the file length may not be accurate
                    // therefore set a tolerance of 0.75
                    // this check was added to workaround problems with 
                    // Steinberg/Yamaha USB drivers 1.8.6,1.9.2,... (?)
                    // which returned in some rare cases (0.3%) no audio data

                    // Update: Startup time of capture line on Mac OS X needs sometimes about 1500 ms
                    // to activate, therefore we add 1000ms for Mac OS X here
                    // TODO Use line activation event ! (Planned for 2.14.x)
                    int lineActivateTolerance=MIN_EXPECTED_REC_LEN_LNE_ACTIVATION_MS_DEFAULT;
                    if(SystemHelper.getInstance().isMacOSX()){
                        lineActivateTolerance+=1000;
                    }
                    if(DEBUG){
                        System.out.println("Recorded/minExpected: "+recordedMs+"/"+minRecordLenMs+" ms");
                        System.out.println("Tolerance: Factor: "+MIN_EXPECTED_REC_LEN_TOLERANCE+", activation:"+lineActivateTolerance+" ms");
                    }


                    if((recordedMs+lineActivateTolerance)<(minRecordLenMs*MIN_EXPECTED_REC_LEN_TOLERANCE) || DEBUG_AUDIO_QUALITY_FAKE_CHECK_FAIL){
                        // TEST !!!!
                        //if(recordedMs<2000){
                        String msg=("Recording length "+recordedMs+" ms is shorter than "+MIN_EXPECTED_REC_LEN_TOLERANCE+" of minimum expected length of "+minRecordLenMs+" ms");
                        System.err.println(msg);
                        repeatRequest=new RepeatRequest("Audio quality check failed", msg);
                    }
                    
                }
                if(silenceDetector!=null){
                    silenceDetector.stop();
                }
               
                if(CaptureScope.SESSION.equals(captureScope)){
                    itemFinished();
                }else{
                    try {

                        if(DEBUG)System.out.println("Closing capture.");
                        audioController.closeCapture();
                    } catch (AudioControllerException e1) {
                        speechRecorderUI.displayError(
                                "Audio controller close capture error", e1
                                .getLocalizedMessage());

                        e1.printStackTrace();
                    }
                }

            } else if (ace instanceof CaptureCloseEvent) {
                if(voiceDetector!=null){
                    audioController.removeCaptureFloatAudioOutputStream(voiceDetector);
                }
                if(silenceDetector!=null){
                    silenceDetector.stop();
                }
                itemFinished();
            } else if (ace instanceof CaptureErrorEvent) {
                CaptureErrorEvent cErrEv = (CaptureErrorEvent) ace;
                Exception cause = cErrEv.getCause();
                String errMsg="Unknown capture error";
                String locErrMsg="Unknown capture error";
                if(cause!=null){
                    errMsg=cause.getMessage();
                    locErrMsg=cause.getLocalizedMessage();
                }
                if(preRecTimer!=null){
                    preRecTimer.stop();
                }
                if(maxRecTimer!=null){
                    maxRecTimer.stop();
                }
                
                if(voiceDetector!=null){
//                    voiceDetector.stop();
                    audioController.removeCaptureFloatAudioOutputStream(voiceDetector);
                }
                if(silenceDetector!=null){
                    silenceDetector.stop();
                }
                logger.severe(errMsg);
                if (cause instanceof BufferOverrunException && RecStatus.IDLE!=recStat.getStatus()) {
                    repeatRequest=new RepeatRequest();
                }
                speechRecorderUI.displayError("Audio controller error", locErrMsg);
                try {
                    if(DEBUG)System.out.println("Closing capture.");
                    audioController.closeCapture();
                } catch (AudioControllerException e) {
                    speechRecorderUI.displayError(
                            "Audio controller close capture error", e
                            .getLocalizedMessage());
                }
            }
        }

    }

    
    private void persistBundle(Bundle bundle) throws StorageManagerException, IOException, EncodeException{
        // persist bundle ( not session scoped, annotation worker has scope project)
        Session sessOfBundle=bundle.getSession();
        if(sessOfBundle!=null){

            String rootFn=bundle.getName();
            File storageDir=ActiveSessionStorageManager.fileURLToFile(storageManager.getStorageURL());
            String sessNm=sessOfBundle.getName();
            if(sessNm!=null && ! "".equals(sessNm)){
                File sessDir=new File(storageDir,sessNm);

                if(sessDir!=null && sessDir.isDirectory()){
                    for(BundleAnnotationPersistor bap:bundleAnnotationPersistorList){
                        if(bap instanceof BundleAnnotationFilePersistor){
                            BundleAnnotationFilePersistor bafp=(BundleAnnotationFilePersistor)bap;
                            File annoFile=new File(sessDir,rootFn+bafp.getPreferredFilenameSuffix()+"."+bafp.getPreferredFileExtension());
                            bafp.setFile(annoFile);
                            if (bap.isWritable(bundle)) {
                               
                                try {
                                    bap.write(bundle);
                                } catch (EncodeException ee) {
                                    ee.printStackTrace();
                                    // Continue with other file writers
                                    // TextGrid cannot handle empty annotations
                                    // (no tiers)
                                    // TextGrid writer throws exception here
                                    // we have to catch it here to continue with
                                    // other writers

                                }
                            }else{
                                // Log this 
                                // Only happens when tried to write TextRid file without at least one level
                                
                            }
                            if(DEBUG)System.out.println("Wrote bundle "+bundle.getName()+" to "+annoFile);
                        }else{
                            // currently no other protocols than file
                        }
                    }
                }
            }
        }
    }



    /**
     * Start auto annotation.
     * Method puts request(s) to the annotation worker.
     * The worker is running background therefore Speechrecorder does not change its state.
     * @param bundle 
     *  
     */
    private void startAutoAnnotation(Bundle bundle) {
        if(autoAnnotationWorker!=null){
        File[] recFiles=null;
        
        try {
            recFiles = storageManager.getRecordingFiles();
            PromptItem pi=progressManager.getCurrentPromptItem();
            LocalizedText annoTempl=pi.annotationTemplateLocalizedText();
            if(annoTempl!=null){
                Locale templLocale=annoTempl.getLocale();
                if(templLocale==null){
                    templLocale=java.util.Locale.getDefault();
                }
                bundle.setLocale(templLocale);
            }
//            
//            List<Mediaitem> mis=pi.getMediaitems();
//            
//            String orthoGraphy=null;
//            for(Mediaitem mi:mis){
//                if(mi.getAnnotationTemplate()){
//                    // media item is annotation template
//                    java.util.Locale lLoc;
//                    String langCode=mi.getLanguageISO639code();
//                    if(langCode!=null){
//                        lLoc= new java.util.Locale(langCode);
//                    }else{
//                        // default current locale
//                        lLoc=java.util.Locale.getDefault();
//                    }
//                    lLoc.getISO3Language();
//                    bundle.setLocale(lLoc);
////                    System.out.println("Lang code: "+langCode);
//                    orthoGraphy=mi.getText();
//                    if(orthoGraphy!=null){
//                        // found media item text
//                        break;
//                    }
//                }
//            }
            AutoAnnotator.AnnotationRequest ar=new AutoAnnotator.AnnotationRequest(bundle);
            autoAnnotationWorker.request(ar);
        } catch (StorageManagerException e) {
            // TODO Auto-generated catch block
            e.printStackTrace();
        }
        
        }
    }

    
    public void annotate(Object annotatedObject, String annotationName,String propertyName,
            Object annotation) {
        if(DEBUG)System.out.println("Annotation: "+propertyName+" "+annotation);
        FileWriter annoWriter=null;
        try {
            File annotationFile=storageManager.getAnnoationFile();
            //FileOutputStream annoFos=new FileOutputStream(annotationFile);
            annoWriter=new FileWriter(annotationFile);
            annoWriter.write("Annotation: "+propertyName+" "+annotation);
            annoWriter.close();
            if(useUploadCache)storageManager.uploadAnnotation();
        } catch (Exception e) {
            e.printStackTrace();
            if(annoWriter!=null){
                try {
                    annoWriter.close();
                } catch (IOException ioe) {
                    ioe.printStackTrace();
                    speechRecorderUI.displayError("Annotation storage error", ioe);
                }
            }
            speechRecorderUI.displayError("Annotation storage error", e);
            
        }
        continueSession();
    }

    /**
     * @param script script
     */
    public void setScript(Script script) {
        this.script=script;
        progressManager.setScript(script);
    }
    
    
    private boolean playbackActive() {
    	int st=recStat.getStatus();
    	return(st==RecStatus.PLAY||st==RecStatus.PLAYPAUSE);
    }
    
    public void setRecIndex(int skipTo) {
    	if(playbackActive()) {
    		skipToRequest=skipTo;
    		try {
				stopPlayback();
			} catch (AudioControllerException e) {
				// TODO Auto-generated catch block
				e.printStackTrace();
			}
    	}else {
    		progressManager.setRecIndex(skipTo);
    	}
    }

    /**
     * 
     */
    public void advanceToNextRecording() {
    	if(playbackActive()) {
    		advanceToNextRequest=true;
    		try {
				stopPlayback();
			} catch (AudioControllerException e) {
				// TODO Auto-generated catch block
				e.printStackTrace();
			}
    	}else {
    		getProgressManager().advanceToNextRecording();
    	}
    }

    /**
     * 
     */
    public void decrementIndex() {
    	if(playbackActive()) {
    		decrementIndexRequest=true;
    		try {
				stopPlayback();
			} catch (AudioControllerException e) {
				// TODO Auto-generated catch block
				e.printStackTrace();
			}
    	}else {
    		getProgressManager().decrementIndex();
    	}
    }

    /**
     * 
     */
    public void incrementIndex() {
    	if(playbackActive()) {
    		incrementIndexRequest=true;
    		try {
				stopPlayback();
			} catch (AudioControllerException e) {
				// TODO Auto-generated catch block
				e.printStackTrace();
			}
    	}else {
    		getProgressManager().incrementIndex();
    	}
        
    }
    
    
    public void startPlayback() throws AudioControllerException,
    StorageManagerException {
       startPlayback(0,AJSAudioSystem.NOT_SPECIFIED);
    }

	public void startPlayback(long start, long stop) throws AudioControllerException, StorageManagerException {

		try {
			audioController.setPlaybackFile(storageManager.getRecordingFiles()[0]);
			audioController.openPlayback();
			audioController.setPlaybackStartFramePosition(start);
			audioController.setPlaybackStopFramePosition(stop);
			audioController.startPlayback();
			setEditingEnabled(false);
		} catch (AudioControllerException | StorageManagerException e) {
			audioController.closePlayback();
			throw e;
		}

	}

    /**
     * Stops playback.
     * @throws AudioControllerException audio controller failed
     */
    public void stopPlayback() throws AudioControllerException {
        audioController.stopPlayback();
    }

    /**
     * Pauses playback.
     * @throws AudioControllerException audio controller failed
     */
    public void pausePlayback() throws AudioControllerException {
        audioController.pausePlayback();
    }

    /**
     * Continues playback after pause.
     */
    public void continuePlayback() {

        try {
            audioController.startPlayback();
        } catch (AudioControllerException e) {
            speechRecorderUI.displayError("AudioController Error", e
                    .getLocalizedMessage());
            e.printStackTrace();
        }

    }
    


    
    private List<File> currentAnnotationFiles() throws StorageManagerException{
        List<File> annoFileList=new ArrayList<File>();
        File sessDir=ActiveSessionStorageManager.fileURLToFile(storageManager.getSessionURL());
        String rootFn=storageManager.getRootFileName();
        if(sessDir!=null){
            for(BundleAnnotationPersistor bap:bundleAnnotationPersistorList){
                if(bap instanceof BundleAnnotationFilePersistor){
                    BundleAnnotationFilePersistor bafp=(BundleAnnotationFilePersistor)bap;
                    File annoFile=new File(sessDir,rootFn+bafp.getPreferredFilenameSuffix()+"."+bafp.getPreferredFileExtension());
                    annoFileList.add(annoFile);
                }else{
                    // currently no other protocols than file
                }
            }
        }
        return annoFileList;
    }
    
    /**
     * if true, a separate window is shown for the speaker prompts
     * 
     * @param v set true to show speaker window
     */
    public void setSpeakerWindowShowing(boolean v) {
        speakerWindow = v;
        speechRecorderUI.setSpeakerWindowShowing(speakerWindow);
        
    }
    
    

//    
//    public void setLogEntries() {
//        recLogger.setLogEntry("LHD: ", System.getProperty("LHD"));
//        recLogger.setLogEntry("DBN: ", System.getProperty("DBN"));
//        recLogger.setLogEntry("REP: ", System.getProperty("REP"));
//        recLogger.setLogEntry("RSW: ", uiString.getString("QTSpeechRecorder"));
//        recLogger.setLogEntry("MIT: ", System.getProperty("MIT"));
//        recLogger.setLogEntry("MIP: ", System.getProperty("MIP"));
//        progressManager.setLogEntries();
//    }
//    
    
    public void stop() throws SpeechRecorderException{

        if(autoAnnotationWorker!=null){
            try {
                autoAnnotationWorker.cancel();
                autoAnnotationWorker.close();
            } catch (WorkerException e1) {
                throw new SpeechRecorderException(e1);
            }
        }
        
        closeAudioController();

    }
    
    
    private void closeAudioController() {
        if (audioController != null) {
            audioController.removeAudioController2Listener(this);
            try {
                audioController.closePlayback();
                if(DEBUG)System.out.println("Closing capture.");
                audioController.closeCapture();
            } catch (AudioControllerException e) {
                speechRecorderUI.displayError("Audiocontroller error", e
                        .getLocalizedMessage());
                e.printStackTrace();
            }
        }
        if(beepPlayer !=null){
         beepPlayer.removePlayerListener(this);
         try {
            beepPlayer.close();
        } catch (PlayerException e) {
            speechRecorderUI.displayError("Beep player close error", e
                    .getLocalizedMessage());
            e.printStackTrace();
        }
        }

    }
    
    
    /**
     * Close a session.
     * @throws SpeechRecorderException speech recorder exception
     */
    public void close() throws SpeechRecorderException{
        
        if(recStat.getStatus()==RecStatus.CLOSE ) return;
        stop();
        recStat.setStatus(RecStatus.TERMINATE);
        if(autoAnnotationWorker!=null){
            // TODO warn user ?
            autoAnnotationWorker.cancel();
            try {
                autoAnnotationWorker.close();
                autoAnnotationWorker=null;
            } catch (WorkerException e1) {
                // TODO Auto-generated catch block
                e1.printStackTrace();
            }
        }
        annotationSession=null;
        closeAudioController();
        setEditingEnabled(false);
        speechRecorderUI.closeSession();

        
        audioEnabled = false;
        audioClip.setBundle(null);
        audioClip.setAudioSource(null);
        setIndexAction.setEnabled(false);
        speechRecorderUI
        .setLevelMeterMode(SpeechRecorderUI.LEVEL_METER_DISABLE);
        speechRecorderUI.setRecMonitorsStatus(ipsk.apps.speechrecorder.monitor.StartStopSignal.State.OFF);
        progressManager.doClose();
        
        recStat.setStatus(RecStatus.CLOSE);
    }

    
    
   
    public void update(RecscriptManagerEvent e){
        if(e instanceof RecScriptChangedEvent){
            if(getScript()!=null && !(recStat.getStatus()==RecStatus.INIT || recStat.getStatus()==RecStatus.CLOSE)){
                try {
                    applyItem();
                } catch (PromptPresenterException e1) {
                    setPromptErrorState();
                    return;
                }
                setRecDisplay();
                setIdle();
                startPrompt();
            }
        }
    }
    


    public void update(PromptViewerEvent promptViewerEvent) {
        if (promptViewerEvent instanceof PromptViewerOpenedEvent){
            // always start if prompt viewers opened
             speechRecorderUI.startPlayPrompt();
        }else if (promptViewerEvent instanceof PromptViewerStartedEvent){
            int currStat=recStat.getStatus();
            if(DEBUG)System.out.println("Curr status: "+RecStatus.getStatusName(currStat));
            if(currStat==RecStatus.NON_RECORDING || currStat==RecStatus.IDLE){
                sessionActions.setIndexAction.setEnabled(false);
                setEditingEnabled(false);
                recStat.setStatus(RecStatus.PLAY_PROMPT_PREVIEW);
            }

        }else if (promptViewerEvent instanceof PromptViewerStoppedEvent){

        }else if (promptViewerEvent instanceof PromptViewerPresenterClosedEvent){
            int status=recStat.getStatus();
            if(DEBUG)System.out.println("Prompt viewer closed event.");
            if(status == RecStatus.PLAY_PROMPT){
                ipsk.db.speech.script.PromptItem promptItem = progressManager
                .getCurrentPromptItem();

                if(DEBUG)System.out.println("Prompt played.");
                if(promptItem instanceof Recording){
                    if(DEBUG)System.out.println("Starting beep ...");
                    try {
                        startBeep();
                    } catch (SpeechRecorderException e) {
                        speechRecorderUI.displayError("Beep playback error",e);
                        setIdle();
                    }
                }else if(promptItem instanceof Nonrecording){
                    startNonRecording();
                }else{
                    setIdle();
                    if(isAutoRecording()){
                        continueSession();
                    }
                }
            }else if(status == RecStatus.PLAY_PROMPT_PREVIEW){
                setIdle();
            }else if (status == RecStatus.PRERECWAITING
                    || status == RecStatus.RECORDING
                    || status == RecStatus.POSTRECWAITING) {

                if (Section.PromptPhase.RECORDING.equals(progressManager.currentPromptPhase())
                        && progressManager.currentPromptBlocking()){
                    speechRecorderUI.setRecMonitorsStatus(StartStopSignal.State.RECORDING);
                }
                if(DEBUG)System.out.println("Prompt viewer close. wait for capture to finish...");

            }else if(status==RecStatus.RECORDED){
                if(DEBUG)System.out.println("Prompt viewer closed. Continue.");
                if(speechRecorderUI.isPromptClosed()){
                    if(DEBUG)System.out.println("Continue triggered by Prompt viewer close.");
                    continueSession();
                }else{
                    System.err.println("Prompter not closed. (Internal error)");
                }
            }
        }
    }



    public void update(PlayerEvent playerEvent) {
        Player p=(Player)playerEvent.getSource();
        if(playerEvent instanceof PlayerStopEvent){
            if(DEBUG)System.out.println("Beep-Player stop event");   
            try {
                p.close();
                
            } catch (PlayerException e) {
                e.printStackTrace();
                try {
                    stopRecording();
                } catch (AudioControllerException e1) {
                   
                    e1.printStackTrace();
                }
                speechRecorderUI.displayError("Beep playback error", e);
            }
        }else if(playerEvent instanceof PlayerCloseEvent){
            if(DEBUG)System.out.println("Beep-Player closed event");   
            startPreRecWaiting();
        }
    }


    
    
    public void update(ProgressManagerEvent e){
        if(e instanceof SessionPositionChangedEvent){
            SessionPositionChangedEvent rspce=(SessionPositionChangedEvent)e;
            Integer newPos=rspce.getPosition();
            if(newPos==null){
                init();
            }else{
                try {
                    applyItem();
                } catch (PromptPresenterException e1) {
                    setPromptErrorState();
                    return;
                }
                setIdle();
                startPrompt();
            }
        }
    }
    
    /* (non-Javadoc)
     * @see ipsk.awt.ProgressListener#update(ipsk.awt.event.ProgressEvent)
     */
    @Override
    public void update(ProgressEvent progressEvent) {
        
        Object src=progressEvent.getSource();
        if(src==autoAnnotationWorker){
            // update from auto annotation worker
            if(progressEvent instanceof AutoAnnotationWorker.BundleAnnotatedEvent){
                AutoAnnotationWorker.BundleAnnotatedEvent bae=(AutoAnnotationWorker.BundleAnnotatedEvent)progressEvent;
                Bundle bundle=bae.getAnnotatedBundle();
                try {
                    persistBundle(bundle);
                    // TODO how to handle (asynchron) errors ? Log file ?
                } catch (StorageManagerException e) {
                    // TODO Auto-generated catch block
                    e.printStackTrace();
                } catch (IOException e) {
                    // TODO Auto-generated catch block
                    e.printStackTrace();
                } catch (EncodeException e) {
                    // TODO Auto-generated catch block
                    e.printStackTrace();
                }
                String currViewedBundle=storageManager.getRootFileName();
                
                if(currViewedBundle.equals(bundle.getName())){
                    audioClip.setBundle(bundle);
                }
            }
        }
    }
    
    /* (non-Javadoc)
     * @see ipsk.audio.dsp.speech.SpeechFinalSilenceDetectorListener#update(ipsk.audio.dsp.speech.SpeechFinalSilenceDetectorEvent)
     */
    @Override
    public void update(SpeechFinalSilenceDetectorEvent event) {
        if(event.isFinalSilenceDetected()){
            startPostRecordingPhase();
        }
    }

    
    
    
    
}
