//    Speechrecorder
//    (c) Copyright 2019
// 	  Institute of Phonetics and Speech Processing,
//    Ludwig-Maximilians-University, Munich, Germany
//
//
//    This file is part of Speechrecorder
//
//
//    Speechrecorder is free software: you can redistribute it and/or modify
//    it under the terms of the GNU Lesser General Public License as published by
//    the Free Software Foundation, version 3 of the License.
//
//    Speechrecorder is distributed in the hope that it will be useful,
//    but WITHOUT ANY WARRANTY; without even the implied warranty of
//    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
//    GNU Lesser General Public License for more details.
//
//    You should have received a copy of the GNU Lesser General Public License
//    along with Speechrecorder.  If not, see <http://www.gnu.org/licenses/>.

package ipsk.apps.speechrecorder.workspace.ui;

import java.awt.Component;
import java.awt.GridBagConstraints;
import java.awt.GridBagLayout;
import java.awt.Insets;
import java.awt.event.ActionEvent;
import java.awt.event.ActionListener;
import java.util.List;

import javax.swing.JButton;
import javax.swing.JDialog;
import javax.swing.JLabel;
import javax.swing.JOptionPane;
import javax.swing.JPanel;
import javax.swing.JTextArea;
import javax.swing.JTextField;
import javax.swing.event.DocumentEvent;
import javax.swing.event.DocumentListener;
import javax.swing.text.Document;

import ipsk.apps.speechrecorder.workspace.WorkspaceManager;
import ipsk.io.FilenameValidator;

/**
 * Dialog to duplicate project.
 * 
 * @author K.Jaensch, klausj@phonetik.uni-muenchen.de
 *
 */
public class ProjectDuplicateDialog extends JPanel implements ActionListener, DocumentListener {

	private static final long serialVersionUID = 1L;
	private JTextField nameField;
	private static JButton okButton;
	private static JButton cancelButton;
	private static JOptionPane selPane;
	private Document nameDoc;
	private JTextField statusLabel;
	private DuplicateModel duplicateModel;
	private String originalName;
	private WorkspaceManager workspaceManager;
	
	public static class DuplicateModel {
		private String originalName;
		private String newName;

		public DuplicateModel(String originalName) {
			super();
			this.originalName = originalName;
		}

		public String getNewName() {
			return newName;
		}

		public void setNewName(String newName) {
			this.newName = newName;
		}

		public String getOriginalName() {
			return originalName;
		}

		public boolean changed() {
			if (originalName != null) {
				return !originalName.equals(newName);
			}
			return false;
		}
	}

	public ProjectDuplicateDialog(WorkspaceManager workspaceManager, DuplicateModel duplicateModel) {
		super(new GridBagLayout());
		this.workspaceManager=workspaceManager;
		
		this.duplicateModel = duplicateModel;
		this.originalName = duplicateModel.getOriginalName();
		okButton = new JButton("OK");
		okButton.setEnabled(false);
		okButton.addActionListener(this);
		cancelButton = new JButton("Cancel");
		cancelButton.setEnabled(true);
		cancelButton.addActionListener(this);
		GridBagConstraints c = new GridBagConstraints();
		c.fill = GridBagConstraints.HORIZONTAL;
		c.insets = new Insets(5, 10, 5, 10);
		c.gridx = 0;
		c.gridy = 0;
		add(new JLabel("Project name of duplicate"), c);
		c.gridx++;
		c.weightx = 1;
		nameField = new JTextField(20);

		nameDoc = nameField.getDocument();
		nameDoc.addDocumentListener(this);
		add(nameField, c);

		c.gridy++;
		c.gridx = 0;
		c.gridwidth = 2;
		c.weightx = 2.0;
		c.fill = GridBagConstraints.HORIZONTAL;
		statusLabel = new JTextField(30);
		statusLabel.setEditable(false);
		add(statusLabel, c);
		
		c.gridx=0;
		c.gridy++;
		c.gridwidth=2;
		JTextArea noteArea=new JTextArea("Note: The duplication process will only copy the project configuration, the recording script and media resources referenced by the script.\nThe speaker database, the recordings and other additional files of the original project are not copied.");
		noteArea.setEditable(false);
		add(noteArea, c);
		
		nameField.setText(originalName);
	}

	public static Object showDialog(Component parent, WorkspaceManager workspaceManager, DuplicateModel duplicateModel) {
		ProjectDuplicateDialog pv = new ProjectDuplicateDialog(workspaceManager, duplicateModel);
		selPane = new JOptionPane(pv, JOptionPane.PLAIN_MESSAGE, JOptionPane.OK_CANCEL_OPTION, null,
				new Object[] { okButton, cancelButton });
		JDialog dialog = selPane.createDialog(parent, "Duplicate Project");
		dialog.setVisible(true);
		return selPane.getValue();
	}

	/*
	 * (non-Javadoc)
	 * 
	 * @see
	 * java.awt.event.ActionListener#actionPerformed(java.awt.event.ActionEvent)
	 */
	public void actionPerformed(ActionEvent ae) {
		Object src = ae.getSource();
		if (src == okButton) {
			selPane.setValue(Integer.valueOf(JOptionPane.OK_OPTION));
		} else if (src == cancelButton) {
			selPane.setValue(Integer.valueOf(JOptionPane.CANCEL_OPTION));
		}
	}

	public String getNewName() {
		return null;
	}

	private void documentChanged(Document d) {
		if (d == nameDoc) {
			String name = nameField.getText();
			List<String> existingNames=workspaceManager.workspaceProjectNames();
			if ("".equals(name)) {
				statusLabel.setText("Project name must not be empty.");
				okButton.setEnabled(false);
			} else if (originalName.equals(name)) {
				statusLabel.setText("Name of duplicate project must differ.");
				okButton.setEnabled(false);
			} else if (existingNames.contains(name)) {
				statusLabel.setText("Project with this name already exists.");
				okButton.setEnabled(false);
			} else {
				FilenameValidator.ValidationResult vr = FilenameValidator.validate(name);
				if (vr.isValid()) {
					statusLabel.setText("");
					duplicateModel.setNewName(name);
					okButton.setEnabled(true);
				} else {
					ipsk.util.LocalizableMessage lm = vr.getMessage();
					String locText = lm.localize();
					statusLabel.setText(locText);
					okButton.setEnabled(false);
				}
			}
		}
	}

	/*
	 * (non-Javadoc)
	 * 
	 * @see javax.swing.event.DocumentListener#changedUpdate(javax.swing.event.
	 * DocumentEvent)
	 */
	public void changedUpdate(DocumentEvent arg0) {
		documentChanged(arg0.getDocument());

	}

	/*
	 * (non-Javadoc)
	 * 
	 * @see javax.swing.event.DocumentListener#insertUpdate(javax.swing.event.
	 * DocumentEvent)
	 */
	public void insertUpdate(DocumentEvent arg0) {

		documentChanged(arg0.getDocument());
	}

	/*
	 * (non-Javadoc)
	 * 
	 * @see javax.swing.event.DocumentListener#removeUpdate(javax.swing.event.
	 * DocumentEvent)
	 */
	public void removeUpdate(DocumentEvent arg0) {

		documentChanged(arg0.getDocument());
	}

}
