//    IPS Java Utils
//    (c) Copyright 2009-2010
//    Institute of Phonetics and Speech Processing,
//    Ludwig-Maximilians-University, Munich, Germany
//
//
//    This file is part of IPS Java Utils
//
//
//    IPS Java Utils is free software: you can redistribute it and/or modify
//    it under the terms of the GNU Lesser General Public License as published by
//    the Free Software Foundation, version 3 of the License.
//
//    IPS Java Utils is distributed in the hope that it will be useful,
//    but WITHOUT ANY WARRANTY; without even the implied warranty of
//    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
//    GNU Lesser General Public License for more details.
//
//    You should have received a copy of the GNU Lesser General Public License
//    along with IPS Java Utils.  If not, see <http://www.gnu.org/licenses/>.

package ipsk.persistence;

import ips.beans.ExtBeanInfo;
import ips.beans.PersistenceIntrospector;
import ipsk.text.ParserException;

import java.beans.IntrospectionException;
import java.util.HashMap;
import java.util.Map;


/**
 * Container class for an ID object.
 * Objects of this class contain information to identify a persistence object. 
 * @author Klaus Jaensch
 *
 */
public class PersistenceObjectIdentifier {
	public final static String KEY_TARGET_CLASS = "_oi_target_class";
	public final static String KEY_ID = "_oi_id";
	private final Object idObject;
	private final Class<?> targetClass;
	
	public PersistenceObjectIdentifier(Class<?> targetClass,Object idObject){
		this.targetClass=targetClass;
		this.idObject=idObject;
	}

	public Object getIdObject() {
		return idObject;
	}

	public Class<?> getTargetClass() {
		return targetClass;
	}
	
	public String toString(){
		return "ID: "+getIdObject();
	}
	
	/**
	 * Convert to URL query map.
	 * @return query map
	 */
	public Map<String,String[]> toQueryMap(){
		HashMap<String,String[]> queryMap=new HashMap<String,String[]>();
		queryMap.put(KEY_TARGET_CLASS, new String[]{targetClass.getName()});
		queryMap.put(KEY_ID, new String[]{idObject.toString()});
		return queryMap;
	}
	
	/**
	 * Convert query map to object identifier.
	 * @param queryMap
	 * @return the object identifier
	 * @throws ParserException
	 * @throws IntrospectionException
	 */
	public static PersistenceObjectIdentifier parseQueryMap(Map<String,String[]> queryMap) throws ParserException, IntrospectionException{
		
		String[] targetClassArr=queryMap.get(KEY_TARGET_CLASS);
		if(targetClassArr!=null){
			if(targetClassArr.length!=1){
				throw new ParserException("Ambigous keys: "+KEY_TARGET_CLASS);
			}
			String targetClassName=targetClassArr[0];
			Class<?> targetClass;
			try {
				targetClass = Class.forName(targetClassName);
			} catch (ClassNotFoundException e) {
				
				e.printStackTrace();
				throw new IntrospectionException("Class not found: "+targetClassName);
			} 
			String[] idArr=queryMap.get(KEY_ID);
			if(idArr==null){
				throw new ParserException("Missing key: "+KEY_ID);
			}
			if(idArr.length!=1){
				throw new ParserException("Ambigous keys: "+KEY_ID);
			}
			String idStr=idArr[0];
			ExtBeanInfo beanInfo=PersistenceIntrospector.getPersistenceBeanInfo(targetClass,true);
			Object idObj;
			try {
				idObj = beanInfo.createIdValueByString(idStr);
			} catch (Exception e) {
				e.printStackTrace();
				throw new IntrospectionException("Could not get ID object for class "+targetClassName+" for ID string "+idStr);
			} 
			return new PersistenceObjectIdentifier(targetClass,idObj);
		}
		
		return null;
	}
	
}
