package ipsk.db.speech.utils;

import java.io.File;
import java.io.FileNotFoundException;
import java.io.FileOutputStream;
import java.io.IOException;
import java.io.PrintWriter;
import java.io.UnsupportedEncodingException;
import java.nio.file.Files;
import java.nio.file.Path;
import java.util.HashMap;
import java.util.Map;
import java.util.Set;
import java.util.UUID;

import javax.persistence.EntityManagerFactory;
import javax.persistence.Persistence;
import javax.xml.bind.JAXBContext;
import javax.xml.bind.JAXBException;
import javax.xml.bind.Marshaller;

import org.w3c.dom.Document;

import ips.annot.model.PredefinedLevelDefinition;
import ips.annot.model.db.AttributeDefinition;
import ips.annot.model.db.LevelDefinition;
import ips.annot.model.db.LinkDefinition;
import ips.annot.model.emu.EmuBundleAnnotationPersistor;
import ips.annot.model.emu.EmuDB;
import ips.annot.model.emu.webapp.EMUwebAppConfig;
import ipsk.apps.speechrecorder.annotation.auto.AutoAnnotationManager;
import ipsk.apps.speechrecorder.annotation.auto.impl.PromptAutoAnnotator;
import ipsk.apps.speechrecorder.annotation.auto.impl.TemplateAutoAnnotator;
import ipsk.apps.speechrecorder.project.ProjectManagerException;
import ipsk.awt.WorkerException;
import ipsk.beans.DOMCodec;
import ipsk.beans.DOMCodecException;
import ipsk.db.speech.DialectRegion;
import ipsk.db.speech.Organisation;
import ipsk.db.speech.Project;
import ipsk.db.speech.Session;
import ipsk.persistence.EntityManagerWorker;
import ipsk.util.LocalizableMessage;
import ipsk.util.ProgressStatus;
import ipsk.util.zip.ZipPackerWorker;
import ipsk.xml.DOMConverter;
import ipsk.xml.DOMConverterException;


public class ProjectEmuExporter extends EntityManagerWorker{
	
	private final static String RESOURCE_BUNDLE_NAME="Messages";
	private String projectName;
	private File outputBaseDir;
	protected ProjectExportConfiguration projectConfiguration;
	private SessionEmuExporter sessionExporter=null;
	private ZipPackerWorker zipPacker;
	
	private File exportArchiveFile;
	
	boolean link=true;

	//private volatile ProgressStatus progressStatus=new ProgressStatus();
	//private volatile boolean cancel=false;
	
	public ProjectEmuExporter(EntityManagerFactory entityManagerFactory,ProjectExportConfiguration configuration) {
		super(entityManagerFactory);
		this.projectConfiguration=configuration;
	}

	@SuppressWarnings("unchecked")
	public void doWork() throws WorkerException{
		if(sessionExporter==null){
			sessionExporter=new SessionEmuExporter(entityManager,projectConfiguration);
		}else{
			sessionExporter.setEntityManager(entityManager);
		}
	
		Project project=(Project)entityManager.find(Project.class,projectName);
	
		  String prjNm=project.getName();
		    Path emuDBDir=outputBaseDir.toPath().resolve(prjNm+"_emuDB");
		    if(Files.exists(emuDBDir)) {
		        throw new ExportException("EmuDB directory "+emuDBDir+" already exists!");
		    }
		    
		    try {
				Files.createDirectories(emuDBDir);
			} catch (IOException e1) {
				e1.printStackTrace();
				throw new ExportException(e1);
			}
		  
		    // config file
		    Path emuDBCfgFile=emuDBDir.resolve(prjNm+"_DBconfig.json");
		    
		    EmuDB emuDB=new EmuDB();
		    emuDB.setName(project.getName());  
		    //Hmmm set same UUID ??
		    String prjUUIDStr=project.getUuid();
		    UUID prjUuid=UUID.fromString(prjUUIDStr);
		    if(prjUuid==null){
		        prjUuid=UUID.randomUUID();
		    }
		    emuDB.setUuid(prjUuid);
		    
		    LevelDefinition prtLvlDef=new LevelDefinition(PredefinedLevelDefinition.PRT);
		    AttributeDefinition prtAttrDef=new AttributeDefinition();
		    prtAttrDef.setName(PredefinedLevelDefinition.PRT.getKeyName());
		    prtAttrDef.setType(AttributeDefinition.TYPE_STRING);
		    prtLvlDef.addAttributeDefinition(prtAttrDef);
		    emuDB.getLevelDefinitions().add(prtLvlDef);
		    
		    LevelDefinition tplLvlDef=new LevelDefinition(PredefinedLevelDefinition.TPL);
		    AttributeDefinition tplAttrDef=new AttributeDefinition();
		    tplAttrDef.setName(PredefinedLevelDefinition.TPL.getKeyName());
		    tplAttrDef.setType(AttributeDefinition.TYPE_STRING);
		    tplLvlDef.addAttributeDefinition(tplAttrDef);
		    emuDB.getLevelDefinitions().add(tplLvlDef);
		    
		    if(link){
		        LinkDefinition linkPrtTplDef=new LinkDefinition();
		        linkPrtTplDef.setType(LinkDefinition.ONE_TO_MANY);
		        linkPrtTplDef.setSuperLevel(prtLvlDef);
		        linkPrtTplDef.setSubLevel(tplLvlDef);
		        emuDB.getLinkDefinitions().add(linkPrtTplDef);
		    }
		    
		    
		    emuDB.setEMUwebAppConfig(EMUwebAppConfig.defaultWithDefaultPerspective());

			Map jaxbEclipseLinkPropsmap=new HashMap();
		    jaxbEclipseLinkPropsmap.put("javax.xml.bind.context.factory","org.eclipse.persistence.jaxb.JAXBContextFactory");

		    try {
		        JAXBContext jaxbContext = org.eclipse.persistence.jaxb.JAXBContextFactory.createContext(new Class[]{EmuDB.class},jaxbEclipseLinkPropsmap);
		        Marshaller m=jaxbContext.createMarshaller();
		        m.setProperty(Marshaller.JAXB_FORMATTED_OUTPUT, true);
		        m.setProperty("eclipselink.media-type", "application/json");
		        m.marshal(emuDB,emuDBCfgFile.toFile());
		       
		    } catch (JAXBException e) {
		        e.printStackTrace();
		        throw new ExportException(e);
		    }
		    

		

		
		sessionExporter.setOutputBaseDir(emuDBDir);
		sessionExporter.setDbConfig(emuDB);
	
		
		int sessionProgressLength=99;
		int zipPackLength=0;
		if(projectConfiguration.isCreateProjectZIPArchive()){
			sessionProgressLength=33;
			zipPackLength=66;
		}
//		progressStatus.setMessage("Export of sessions...");
		Set<Session> sessions=project.getSessions();
		int sessionsSize=sessions.size();
		int sessionCount=0;
		//SessionExporter sessionExporter=new SessionExporter();
		for(Session s:sessions){
			
			if(progressStatus.hasCancelRequest()) {
				//progressStatus.setFinished(false);
				progressStatus.setMessage(new LocalizableMessage(RESOURCE_BUNDLE_NAME, "project.export.canceled", new Object[]{project}));
				return;
			}
			progressStatus.setMessage(new LocalizableMessage(RESOURCE_BUNDLE_NAME,"session.exporting",new Object[]{s}));
			sessionExporter.setSessionId(s.getSessionId());
		
			//exportSessionToDir(s, projectDir);
			sessionExporter.getProgressStatus().running();
			progressStatus.setProgress((sessionProgressLength*sessionCount)/sessions.size());
			progressStatus.setSubStatus(sessionExporter.getProgressStatus(),sessionProgressLength/sessionsSize);
			sessionExporter.doWork();
			sessionCount++;
		}
		progressStatus.setProgress(sessionProgressLength);
		// export project XML file
		Package basePack = Project.class.getPackage();
		DOMCodec ph;
		DOMConverter dc;
		try {
			ph = new DOMCodec(basePack, false);
			dc = new DOMConverter();
		} catch (DOMCodecException e) {
			e.printStackTrace();
			throw new ExportException(e);
		}
		
		
		if (hasCancelRequest())return;
		
		Document projectXMLdoc;
		PrintWriter pw=null;
		

		
		progressStatus.setProgress(1+sessionProgressLength);
		if(projectConfiguration.isCreateProjectZIPArchive()){
			progressStatus.setMessage(new LocalizableMessage(RESOURCE_BUNDLE_NAME, "zip.creating"));
			FileOutputStream exportArchivestream;
			try {
				exportArchivestream = new FileOutputStream(exportArchiveFile);
			} catch (FileNotFoundException e) {
				throw new ExportException(e);
			}
			zipPacker = new ZipPackerWorker();
//			zipPacker.setLimitTo32bitSizes(true);
			zipPacker.setSrcDir(emuDBDir.toFile());
			zipPacker.setPackRecusive(true);
			zipPacker.setOutputStream(exportArchivestream);
			
			progressStatus.setSubStatus(zipPacker.getProgressStatus(),zipPackLength);
			zipPacker.open();
			zipPacker.setRunningWithParentWorker();
			zipPacker.doWork();
			zipPacker.close();
			
			if(hasCancelRequest()) {
				progressStatus.setMessage(new LocalizableMessage(RESOURCE_BUNDLE_NAME,"project.export.canceled", new Object[]{project}));
				return;
			}
			
		}
		progressStatus.setSubStatus(null, 0);
		//progressStatus.setProgress(100);
		progressStatus.setMessage(new LocalizableMessage(RESOURCE_BUNDLE_NAME,"project.export.done", new Object[]{project}));
		
		}
	
	public void cancel(){
		if(sessionExporter!=null){
			sessionExporter.cancel();
		}
		if(zipPacker!=null){
			zipPacker.cancel();
		}
		super.cancel();
	}
	
	public void close() throws WorkerException{
		if(sessionExporter!=null){
			sessionExporter.close();
		}
		if(zipPacker!=null){
			zipPacker.close();
		}
		super.close();
	}
	
	public static void main(String[] args){
		String projectName=args[0];
		String outputDirName=args[1];
		String PERSISTENCE_UNIT_NAME="WebSpeechDBPU";
		EntityManagerFactory emf=Persistence.createEntityManagerFactory(PERSISTENCE_UNIT_NAME);
		//		EntityManager em=emf.createEntityManager();
		//		EntityTransaction tr=em.getTransaction();
		//		tr.begin();
		ProjectEmuExporter projectExporter=new ProjectEmuExporter(emf,new ProjectExportConfiguration());
		//Project project=em.find(Project.class, projectName);
		File outputDir=new File(outputDirName);


		projectExporter.setOutputBaseDir(outputDir);
		projectExporter.setProjectName(projectName);
		try {
			projectExporter.open();
			projectExporter.run();
		} catch (WorkerException e) {
			e.printStackTrace();

		}finally{

			try {
				projectExporter.close();
			} catch (WorkerException e) {
				e.printStackTrace();
			}
		}


	}

	public ProgressStatus getProgressStatus() {
		return progressStatus;
	}

	public File getOutputBaseDir() {
		return outputBaseDir;
	}

	public void setOutputBaseDir(File outputBaseDir) {
		this.outputBaseDir = outputBaseDir;
	}

	public String getProjectName() {
		return projectName;
	}

	public void setProjectName(String projectName) {
		this.projectName = projectName;
	}

	public File getExportArchiveFile() {
		return exportArchiveFile;
	}

	public void setExportArchiveFile(File exportArchiveFile) {
		this.exportArchiveFile = exportArchiveFile;
	}
	
	
		
}

