package ipsk.webapps.db.speech;

import java.net.URI;
import java.text.DateFormat;
import java.util.ArrayList;
import java.util.Date;
import java.util.HashMap;
import java.util.HashSet;
import java.util.List;
import java.util.Locale;
import java.util.Map;
import java.util.ResourceBundle;
import java.util.Set;
import java.util.UUID;

import javax.mail.internet.AddressException;
import javax.mail.internet.InternetAddress;
import javax.persistence.EntityManager;
import javax.servlet.http.HttpServletRequest;

import ipsk.beans.BeanModel;
import ipsk.beans.PropertyValidationResult;
import ipsk.beans.PropertyValidationResult.Type;
import ipsk.db.speech.Project;
import ipsk.db.speech.UserRoleId;
import ipsk.net.SendMail;
import ipsk.net.SendMailException;
import ipsk.util.LocalizableMessage;
import ipsk.webapps.BasicPersistenceBeanController;
import ipsk.webapps.ControllerException;
import ipsk.webapps.db.speech.Invitation.Realisation;

public abstract class BasicInvitationController<T extends AbstractInvitation> extends BasicPersistenceBeanController<T> {

	
	protected InvitationUtils invitationUtils;
	protected String invitationLink=null;
	protected Date invitationValidUntil=null;
	protected URI invitationMailToUri=null;
	
	public BasicInvitationController(String persistenceUnit, Class<T> queryType) {
		super(persistenceUnit, queryType);
		invitationUtils=new InvitationUtils();
	}

	protected String projectName=null;
	
	public String getProjectName() {
		return projectName;
	}
	
	private int invitationValidTimeMs=InvitationUtils.DEFAULT_INVITATION_VALID_TIME_MS;

	public int getInvitationValidTimeMs() {
		return invitationValidTimeMs;
	}

	public void setInvitationValidTimeMs(int invitationValidTimeMs) {
		this.invitationValidTimeMs = invitationValidTimeMs;
	}

	protected PropertyValidationResult checkEmailAddress(EntityManager em,Invitation inv) {
		InternetAddress ias=null;
		String emailAddressList=inv.getEmail();
		InternetAddress[] testIas;
		PropertyValidationResult emailPvr=null;
		try {
			testIas = InternetAddress.parse(emailAddressList);
			if(testIas==null || testIas.length!=1){
				emailPvr=new PropertyValidationResult(Type.ERROR);
			}else{
				ias=testIas[0];
				ias.validate();
			}

		} catch (AddressException e1) {
			emailPvr=new PropertyValidationResult(Type.ERROR, e1);
			emailPvr.setValidationMessage(new LocalizableMessage("Error checking E-Mail address \""+emailAddressList+"\": "+e1.getMessage()));
		}
		if(ias!=null && emailPvr==null){
			String emailUsername=ias.getAddress();
			if(AccountUtils.accountExists(em, emailUsername,true)){
				emailPvr=new PropertyValidationResult(Type.ERROR, new LocalizableMessage("Account with this address already exists!"));
				HashMap<String, String[]> fps=new HashMap<String,String[]>();
				fps.put("email",new String[]{emailUsername});
				emailPvr.setFailedProperties(fps);
			}
		}
		return emailPvr;
	}
	
	
	public String invitationLink(UUID invitationUuid) {
		return invitationUtils.invitationLink(currentRequest, invitationUuid);
	}
	
	
	/**
	 * Invite a WikiSpeech user by E-Mail.
	 * Expires invalid account request from the database, checks the E-Mail address, stores an invitation request to the database and sends the invitation E-Mail.
	 * @param em the entity manager required to persist the invitation account request
	 * @param sm a SendMail instance or null
	 * @param req current HTTP request
	 * @param inv invitation data
	 * @param userRoleIds user roles to apply to the new user
	 * @return a persisted invitation request object
	 * @throws ControllerException thrown on errors
	 */
	public ipsk.db.speech.account.InvitationRequest invitationRequest(EntityManager em,SendMail sm,HttpServletRequest req,Invitation inv,UserRoleId.RoleName[] userRoleIds)
			throws ControllerException {
		
		ipsk.db.speech.account.InvitationRequest prr=null;
		
		AccountController.expireInvalidAccountRequests(em);
	
		String invEmail=inv.getEmail();
		String inputEmail=invEmail.trim();
		String emailUsername=null;
		try {
			InternetAddress[] emailIas=InternetAddress.parse(inputEmail);
			if(emailIas!=null && emailIas.length==1) {
				emailUsername=emailIas[0].getAddress();
			}else{
				throw new ControllerException("Invalid e-mail address. (Error parsing address)");
			}
		} catch (AddressException e1) {
			throw new ControllerException("Invalid e-mail address: "+e1.getMessage());
		}
		if(sm==null) {
			 sm=new SendMail();
		}
		
		if(!sm.isAddressValid(inputEmail)){
			throw new ControllerException("Invalid e-mail address.");
		}else{
			// create an invitation request
			UUID uuid=UUID.randomUUID();
			Date requestDate=new Date();
			Date validUntil=new Date(requestDate.getTime()+(invitationValidTimeMs));
			
			int userRolesSize=userRoleIds.length;
			Set<String> userRolesStrSet=new HashSet<>();
			for(int i=0;i<userRolesSize;i++) {
				userRolesStrSet.add(userRoleIds[i].getName());
			}
				
			prr=new ipsk.db.speech.account.InvitationRequest();
			prr.setUuid(uuid);
			prr.setEmail(emailUsername);
			prr.setLogin(emailUsername);
			prr.setRequestDate(requestDate);
			prr.setValidUntil(validUntil);
			prr.setUserRoles(userRolesStrSet);
			
			List<Project> projects=new ArrayList<>(1);
			Project pr=em.find(Project.class, projectName);
			projects.add(pr);
			prr.setProjects(projects);
			
			// store persistent
			em.persist(prr);
			
			// merge project references
			pr.getInvitationRequests().add(prr);
			em.merge(pr);

		}
		return prr;
	}
	
	/**
	 * Invite a WikiSpeech user by E-Mail.
	 * Expires invalid account request from the database, checks the E-Mail address, stores an invitation request to the database and sends the invitation E-Mail.
	 * @param em the entity manager required to persist the invitation account  request
	 * @param req current HTTP request
	 * @param inv invitation data
	 * @param userRoleIds user roles to apply to the new user
	 * @throws ControllerException thrown on errors
	 */
	public void invite(EntityManager em,HttpServletRequest req,Invitation inv,UserRoleId.RoleName[] userRoleIds)
			throws ControllerException {
			SendMail sm=new SendMail();
			ipsk.db.speech.account.InvitationRequest prr=invitationRequest(em,sm, req, inv, userRoleIds);
			if(prr==null) {
				throw new ControllerException("Invitation failed");
			}else {
				String url=invitationUtils.invitationLink(req, prr.getUuid());

				String invEmail=inv.getEmail();
				String inputEmail=invEmail.trim();
				String invLocaleStr=inv.getLocale();
				Locale invLocale=new Locale(invLocaleStr);
				ResourceBundle rb=ResourceBundle.getBundle("Messages",invLocale);

				String subject=rb.getString("account.create.by_invitation.subject");
				//						
				// build mail content
				StringBuffer cb=new StringBuffer();
				String wikispeechSaluTation=rb.getString("wikispeech.salutation");
				cb.append(wikispeechSaluTation);
				cb.append(',');
				cb.append("\n\n");
				String invitationContent=rb.getString("account.create.by_invitation.message.content");
				cb.append(invitationContent);
				cb.append("\n\n");
				cb.append(url);
				cb.append("\n\n");
				String invitationValidUntil=rb.getString("account.create.by_invitation.message.link_valid_until");
				cb.append(invitationValidUntil);
				DateFormat df=DateFormat.getDateTimeInstance(DateFormat.DEFAULT, DateFormat.DEFAULT, invLocale);
				String fmtValidUntilDateStr=df.format(prr.getValidUntil());
				String fmtTimeZoneStr=df.getTimeZone().getDisplayName(invLocale);
				cb.append(fmtValidUntilDateStr);
				cb.append(" ");
				cb.append("("+fmtTimeZoneStr+")");
				cb.append("\n\n");
				String greeting=rb.getString("greeting");
				cb.append(greeting);
				cb.append("\n");
				String content=cb.toString();
				String wikiSpeechEmail=servletContext.getInitParameter("wikispeech.email_address");
				if(wikiSpeechEmail==null){
					String errMsg="Could not get WikiSpeech e-mail (sender) address (\"wikispeech.email_address\" in web.xml)";
					servletContext.log(errMsg);
					throw new ControllerException(errMsg);
				}
				try {
					sm.send(wikiSpeechEmail, inputEmail,subject, content);
					servletContext.log("Sent invitation E-Mail from "+wikiSpeechEmail+" to "+inputEmail);
				} catch (SendMailException e) {
					// TODO should this throw a controller exception ?
					servletContext.log("Could not send invitation E-Mail",e);
					return;
				}

			}
	}
	
	public URI getInvitationMailToUri() {
		return invitationMailToUri;
	}

	public String getInvitationLink() {
		return invitationLink;
	}
	
	
	
	public String invitationValidString() {
		String iv=null;
		AbstractInvitation spkInv;
		try {
			spkInv = getItem();
			if(spkInv!=null && invitationValidUntil!=null) {
				String invLocaleStr=spkInv.getLocale();
				Locale invLoc=new Locale(invLocaleStr);
				iv=invitationUtils.invitationValidString(invLoc, invitationValidUntil);
			}
		} catch (ControllerException e) {
			e.printStackTrace();
		}
		
		return iv;
	}

	
	
	

}
