package ipsk.webapps.db.speech;

import java.text.DecimalFormat;
import java.text.ParseException;
import java.util.List;
import java.util.Set;

import javax.mail.internet.AddressException;
import javax.mail.internet.InternetAddress;
import javax.persistence.EntityManager;
import javax.persistence.EntityTransaction;
import javax.persistence.Query;
import javax.servlet.http.HttpServletRequest;

import ipsk.beans.BeanModel;
import ipsk.beans.PropertyValidationResult;
import ipsk.beans.PropertyValidationResult.Type;
import ipsk.beans.validation.ValidationException;
import ipsk.beans.validation.ValidationResult;
import ipsk.db.speech.Account;
import ipsk.db.speech.Organisation;
import ipsk.db.speech.Person;
import ipsk.db.speech.Project;
import ipsk.db.speech.Session;
import ipsk.db.speech.Speaker;
import ipsk.db.speech.script.Recording;
import ipsk.persistence.ParameterizedQuery;
import ipsk.persistence.QueryParam;
import ipsk.util.LocalizableMessage;
import ipsk.webapps.ControllerException;
import ipsk.webapps.ProcessResult;

public class OrganisationController extends BasicWikiSpeechController<Organisation> {

	public final static String ACCOUNT_PREFIX = "org";

	public final static DecimalFormat ACCOUNT_FORMAT = new DecimalFormat(
			ACCOUNT_PREFIX + "0000");

	public OrganisationController() {
		super("WebSpeechDBPU", Organisation.class, "organisation");
		securityManager = new WikiSpeechSecurityManager(this);
	}

	@SuppressWarnings("unchecked")
	public String suggestAccountLogin() {
		// find highest account value
		EntityManager em = getEntityManager();
		EntityTransaction tx = null;
		int lastIndex = 0;
		try {
			tx = em.getTransaction();
			tx.begin();
			Query q = em
					.createQuery("SELECT a.login FROM Account a WHERE a.login LIKE '"
							+ ACCOUNT_PREFIX + "%' ORDER BY a.login DESC");
			List<String> list = q.getResultList();
			for (String l : list) {

				try {
					lastIndex = ACCOUNT_FORMAT.parse(l).intValue();
					break;
				} catch (ParseException e) {
					continue;
				}

			}

			tx.commit();
		} catch (RuntimeException e) {
			e.printStackTrace();

		} finally {

			em.close();
		}

		return ACCOUNT_FORMAT.format(lastIndex + 1);
	}

	public Organisation getOrganisation() throws ControllerException {
		return (Organisation) getItem();
	}
	
	
	
	
	@SuppressWarnings("unchecked")
	public List<Speaker> getSpeakersOfOrganisation() throws ControllerException{
		Organisation o=getOrganisation();
		if(o==null) return null;
		
		EntityManager em=getThreadEntityManager();
		Query q=em.createQuery("SELECT sp FROM Speaker AS sp WHERE :orga MEMBER OF sp.organisations");
		q.setParameter("orga", o);
		return q.getResultList();
	}
	
	public void organisationsByAccount(HttpServletRequest req) throws ControllerException{
		Project selProj=getSelectedProject(req);
		if(selProj==null) {
			allOrganisationsByAccount(req);
		}else {
			organisationsOfSelectedProjectByAccount(req);
		}
	}
	
	public void allOrganisationsByAccount(HttpServletRequest req) throws ControllerException{
		clear();
		createEmptyBeanTableModel(req);
//		EntityManager em=getThreadEntityManager();
		Account acc=getAccountByRequest(req);
		if(acc==null)return;
		
		ParameterizedQuery pq = new ParameterizedQuery(queryType);
		String jse = ParameterizedQuery.JPQL_SELECT_EXPRESSION;
		
		pq.setWhereClause("EXISTS (SELECT pr FROM Project pr WHERE :account MEMBER OF pr.adminAccounts AND "
					+ jse +" MEMBER OF pr.organisations) ");
		pq.setQueryParams(new QueryParam[] { new QueryParam("account", acc) });
		
		
		setParameterizedQuery(pq);
		//processListRequest(req);
		super.processRequest(req,pq);
	}
	
	
	public void organisationsOfSelectedProjectByAccount(HttpServletRequest req) throws ControllerException{
		clear();
		createEmptyBeanTableModel(req);
//		EntityManager em=getThreadEntityManager();
		Account acc=getAccountByRequest(req);
		if(acc==null)return;
		Project selProj=getSelectedProject(req);
		ParameterizedQuery pq = new ParameterizedQuery(queryType);
		String jse = ParameterizedQuery.JPQL_SELECT_EXPRESSION;
		if(selProj!=null){
			pq.setWhereClause("EXISTS (SELECT pr FROM Project pr WHERE :account MEMBER OF pr.adminAccounts AND "
					+ jse +" MEMBER OF pr.organisations AND pr.name = :selProjectName) ");
			// can we simplify this query?
			pq.setQueryParams(new QueryParam[]{new QueryParam("account", acc),new QueryParam("selProjectName",selProj.getName())});
		}
		
		setParameterizedQuery(pq);
		processListRequest(req);

	}

	public int getCompletSessionsCount() throws ControllerException{
		Organisation organisation=getOrganisation();
		int c=0;
		if(organisation!=null){

			Set<Session> sessions=organisation.getSessions();
			if(sessions!=null){
				for(Session s:sessions){
					List<Recording> missRecs=s.getMissingRecordingItems();
					if(missRecs!=null && missRecs.size()==0)c++;
				}
			}
		}
		return c;
	}
	
	public void deleteOrganisation(HttpServletRequest req) throws ControllerException {
		beanModel=null;
		String cmd=processCommand(req);
		
		if(CMD_DELETE.equals(cmd)) {
			if(checkSecureRequestToken) {
				secureRequestTokenProvider.checkSecureRequestToken(req);
			}
			String idStr = req.getParameter(beanInfo
					.getIdPropertyDescriptor().getName());
			try {
				setId(beanInfo.createIdValueByString(idStr));
			} catch (Exception e) {

				e.printStackTrace();
				throw new ControllerException(e);
			}
			EntityManager em=getThreadEntityManager();
			Organisation orga= em.find(queryType, id);
			em.refresh(orga);
			securityManager.checkRemovePermission(req, orga);
			beanModel =new BeanModel<Organisation>(orga);
			
		
			if(orga!=null) {
				
				for(Account orgaAcc:orga.getAccounts()) {
					Set<Speaker> orgaRegSpks=orgaAcc.getRegisteredSpeakers();
					for(Speaker orgaRegSpk:orgaRegSpks) {
						boolean removed=false;
						// Check if speaker should be deleted as well to avoid orphaned speaker data
						Set<Session> orgaRegSpkSesss=orgaRegSpk.getSessions();
						if(orgaRegSpkSesss.isEmpty()) {
							// spk has no session
							Set<Organisation> orgaRegSpkOrgas=orgaRegSpk.getOrganisations();
							if(orgaRegSpkOrgas.isEmpty() || (orgaRegSpkOrgas.size()==1 && orgaRegSpkOrgas.contains(orga))) {
								// spk has no associated (other) orga
								if(orgaRegSpk.getAccount()==null && orgaRegSpk.getSpeakerDataAccount()==null) {
									// spk has no account (this should not be possible, just to be safe)
									
									// the speaker data should be deleted because it is otherwise
									em.remove(orgaRegSpk);
									removed=true;
								}
								
							}
						}
						if(!removed) {
							orgaRegSpk.setRegisteredByAccount(null);
							em.merge(orgaRegSpk);
						}
					}
				}
				
				Set<Person> orgaPersons=orga.getPersons();
				for(Person orgaPerson:orgaPersons) {
					boolean remove=false;
					
					// Check if person/speaker should be deleted as well
					Set<Organisation> orgaRegPersOrgas=orgaPerson.getOrganisations();
					Account orgaPersAcc=orgaPerson.getAccount();
					
					// Check if this is the persons only organisation and if the person is not connected to an account 
					if((orgaRegPersOrgas.isEmpty() || (orgaRegPersOrgas.size()==1 && orgaRegPersOrgas.contains(orga))) && orgaPersAcc==null) {
						
						if(orgaPerson instanceof Speaker) {
							Speaker orgaSpk=(Speaker)orgaPerson;
							Account orgaSpkDataAccount=orgaSpk.getSpeakerDataAccount();
							Set<Session> orgaRegSpkSesss=orgaSpk.getSessions();
							if(orgaSpkDataAccount==null && orgaRegSpkSesss.isEmpty()) {
								// spk has no session
								remove=true;
							}
						}else {
							// Person will otherwise be orphaned
							remove=true;
						}
					}
					if(remove){
						// Remove entity
						em.remove(orgaPerson);
					} else {
						// ... or just disconnect from organisation
						orgaPerson.getOrganisations().remove(orga);
						em.merge(orgaPerson);
					}
				}
				
				Set<Session> orgaSessions=orga.getSessions();
				for(Session orgaSession:orgaSessions) {
					orgaSession.setOrganisation(null);
					em.merge(orgaSession);
				}

				em.remove(orga);
				processResult=new ProcessResult(ProcessResult.Type.SUCCESS);
			}else {
				processResult=new ProcessResult(ProcessResult.Type.ERROR);
			}
		}else {
			super.processRequest(req);
		}
	}
	
	public ValidationResult validate(Object o, ValidationResult validationResult)
			throws ValidationException {
		ValidationResult vr = super.validate(o, validationResult);
		if (!vr.isValid())
			return vr;

		Organisation org = (Organisation) o;
		if (org.getName().equals("")) {
			LocalizableMessage msg = new LocalizableMessage(RESOURCE_BUNDLE_NAME,
					"validation.field.must_not_be_empty");
			PropertyValidationResult namePvr = new PropertyValidationResult(
					PropertyValidationResult.Type.ERROR, msg);
			vr.putPropertyValidationResult("name", namePvr);
			vr.setType(ValidationResult.Type.ERRORS);
		}
		
		// TODO Duplicate code in invitation controllers
		String emailAddress=org.getEmail();
		if(emailAddress!=null && !"".equals(emailAddress)) {
			InternetAddress[] testIas;
			PropertyValidationResult emailPvr=null;
			try {
				testIas = InternetAddress.parse(emailAddress);
				// do not accept not parsable address or a list of addresses
				if(testIas==null || testIas.length!=1){
					emailPvr=new PropertyValidationResult(Type.ERROR);
				}else{
					// validate
					testIas[0].validate();
				}

			} catch (AddressException e1) {
				// if validation fails 
				emailPvr=new PropertyValidationResult(Type.ERROR, e1);
			}
			if(emailPvr!=null) {
				vr.putPropertyValidationResult("email", emailPvr);
				vr.setType(ValidationResult.Type.ERRORS);
			}
		}
		return vr;

	}

}
