package ipsk.webapps.db.servlets;

import java.io.File;
import java.io.IOException;
import java.security.Principal;

import javax.servlet.ServletContextEvent;
import javax.servlet.ServletContextListener;
import javax.servlet.http.HttpServletRequest;
import javax.servlet.http.HttpServletResponse;
import javax.servlet.http.HttpSession;

import ipsk.db.speech.Project;
import ipsk.db.speech.UserRoleId;
import ipsk.webapps.db.speech.ProjectController;

/**
 * Exports resource files.
 * @author klausj
 *
 */
public class FileServer extends ips.servlet.http.FileServer implements ServletContextListener{
	
	public final static boolean DEBUG=true;
	public final static String BASE_DIR_PARAMETER_KEY="baseDir";
	
	/*
	 * Directory structures:
	 * 
	 * project resources:
	 * 
	 * /contextpath/projects/projectname/resources/
	 * e.g.
	 * /wikispeech/projects/VOYS/resources/
	 * 
	 * user resources:
	 * 
	 * /contextpath/users/username/resources/
	 * e.g.
	 * /wikispeech/users/org0004/resources/ 
	 * 
	 */
	public final static String USERS_DIR="users";
	public final static String EXPORT_DIR="export";
	public final static String PROJECTS_DIR="projects";
	public final static String SESSION_BASE_PATH="session";
	public final static String RESOURCES_DIR="resources";
	public final static String TMP_DIR="tmp";
	public final static String SESSIONS_DIR="http_sessions";
	
	
	private static File baseDir=null;
	
	private static String principalName(HttpServletRequest req) {
		Principal p=req.getUserPrincipal();
		if(p!=null) {
			return p.getName();
		}
		return null;
	}
	
	
	public static File getUserResourceDir(HttpServletRequest req){
		String userName=principalName(req);
		if(userName==null)return null;
		File usersBaseDir = new File(baseDir,USERS_DIR);	
		File userBaseDir=new File(usersBaseDir,userName);
		File userResDir=new File(userBaseDir,RESOURCES_DIR);
		return userResDir;
	}
	
	public static File getTempSessionDirectory(HttpServletRequest req){
		File tempsessionDir=null;
		File tmpBaseDir = new File(baseDir,TMP_DIR);
		File tmpSessBaseDir = new File(tmpBaseDir,SESSIONS_DIR);
		HttpSession session=req.getSession();
		if(session!=null){
			String sessId=session.getId();
			if(sessId!=null){
				tempsessionDir=new File(tmpSessBaseDir,sessId);
			}
		}
		return tempsessionDir;
	}
	
	public static String getTempSessionServerPath(HttpServletRequest req){
		
		String sessionTempPath=null;
		HttpSession session=req.getSession();
		if(session!=null){
			String sessId=session.getId();
			if(sessId!=null){
				sessionTempPath="/"+TMP_DIR+"/"+SESSIONS_DIR+"/"+sessId;
			}
		}
		return sessionTempPath;
	}
	
	
	
	protected void doGet(HttpServletRequest req, HttpServletResponse res)
			throws IOException {
		if (DEBUG)
			log("File server...");
//		ServletContext servletContext = getServletContext();
//		String baseDirPath = servletContext
//				.getInitParameter(BASE_DIR_PARAMETER_KEY);
		String reqPath = req.getServletPath() + req.getPathInfo();
		
		// Remove redundant slashes for path compare
		String parsedReqPath=reqPath.replaceAll("/{2,}", "/");
		
		boolean accessChecked=false;
		
		// check if this is a shared resource: /resources/*
		if(parsedReqPath.startsWith("/"+RESOURCES_DIR)){
			// access granted for all authenticated
			accessChecked=(req.getUserPrincipal()!=null);
		}else if(parsedReqPath.startsWith("/"+SESSION_BASE_PATH+"/"+RESOURCES_DIR)){
			parsedReqPath=parsedReqPath.replaceAll("^/"+SESSION_BASE_PATH,"");
			accessChecked=(req.getUserPrincipal()!=null);
		}else if(parsedReqPath.startsWith("/"+USERS_DIR)){
		
			String userName=principalName(req);
			if(userName!=null){
				String usersResourcesPath="/"+USERS_DIR+"/"+userName+"/"+RESOURCES_DIR;
				accessChecked=(parsedReqPath.startsWith(usersResourcesPath));
			}
			
		}else if(parsedReqPath.startsWith("/"+TMP_DIR+"/"+SESSIONS_DIR)){
			// temporary HTTP session directory
			String sessionTempPath=getTempSessionServerPath(req);
			accessChecked=(sessionTempPath!=null && parsedReqPath.startsWith(sessionTempPath));	
		}else if(parsedReqPath.startsWith("/"+PROJECTS_DIR)){
		
		String projectName =null;
		
		// admin is allowed to access all project resources
		accessChecked=req.isUserInRole(UserRoleId.RoleName.ADMIN.getName());

			if (!accessChecked) {

				ProjectController prjController = new ProjectController();
				prjController.open();
				Project project = prjController.getSelectedProject(req);


				projectName=project.getName();
				prjController.close();
				String resourcePath = "/" + PROJECTS_DIR + "/" + projectName + "/"
				+ RESOURCES_DIR;
				
				accessChecked = (parsedReqPath.startsWith(resourcePath));
			}
		}else if(parsedReqPath.startsWith("/"+EXPORT_DIR)){
			// only admin
			accessChecked=(req.isUserInRole(UserRoleId.RoleName.ADMIN.name()));
		}
		
		
		
		if (accessChecked) {
			// if allowed transmit the requested file
//			File baseDir = new File(baseDirPath);
			File resourceFile = new File(baseDir, parsedReqPath);
			serve(resourceFile, res);
			log("Resource file " + resourceFile + " served.");
		} else {
			log("Permission denied for resource: " + reqPath);
			res.sendError(403);
		}
	}

	public void contextDestroyed(ServletContextEvent sce) {
		// Nothing TODO
	}

	public void contextInitialized(ServletContextEvent sce) {
		String baseDirPath = sce.getServletContext().getInitParameter(BASE_DIR_PARAMETER_KEY);
		if(baseDirPath!=null){
			baseDir=new File(baseDirPath);
		}
	}

	public static File getBaseDir() {
		return baseDir;
	}
}
