package ipsk.webapps.db.speech;

import java.beans.IntrospectionException;
import java.text.MessageFormat;
import java.util.Date;
import java.util.List;
import java.util.Locale;
import java.util.ResourceBundle;
import java.util.UUID;

import javax.persistence.EntityManager;
import javax.persistence.Query;
import javax.servlet.http.HttpServletRequest;

//import org.jsoup.Jsoup;
//import org.jsoup.nodes.Element;
import org.owasp.html.HtmlPolicyBuilder;
import org.owasp.html.PolicyFactory;

import ipsk.beans.MapConverter;
import ipsk.beans.PropertyValidationResult;
import ipsk.beans.PropertyValidationResult.Type;
import ipsk.beans.validation.ValidationException;
import ipsk.beans.validation.ValidationResult;
import ipsk.db.speech.Account;
import ipsk.db.speech.DialectRegion;
import ipsk.db.speech.FormConfiguration;
import ipsk.db.speech.Organisation;
import ipsk.db.speech.Person;
import ipsk.db.speech.Project;
import ipsk.db.speech.Project.TextFormat;
import ipsk.db.speech.Speaker;
import ipsk.db.speech.project.LocalizedDefaultPurposeText;
import ipsk.db.speech.project.LocalizedInformedConsentText;
import ipsk.jsp.fmt.LocaleSupport;
import ipsk.text.html.HTMLTextEncoder;
import ipsk.util.LocalizableMessage;

public class SpeakerInformedConsentController extends BasicWikiSpeechController<Speaker>{

	public SpeakerInformedConsentController() {
		super("WebSpeechDBPU", Speaker.class, "speaker");
	}
	
	protected void setPropertiesOnNew(Object bean) {
		super.setPropertiesOnNew(bean);
		if(bean instanceof Person) {
			Person p=(Person)bean;
			p.setUuid(UUID.randomUUID().toString());
		}
	}
	
	
	private boolean formConfigurationFromProject=false;
	

	public boolean isFormConfigurationFromProject() {
		return formConfigurationFromProject;
	}


	public void setFormConfigurationFromProject(boolean formConfigurationFromProject) {
		this.formConfigurationFromProject = formConfigurationFromProject;
		
		
	}
	
	public FormConfiguration getProjectSpeakerFormConfiguration(){
		FormConfiguration spkFormConfig=null;
		Project project= getSelectedProject();
		if(project!=null){
//			Set<FormConfiguration> fCfgs=project.getFormConfigurations();
//			for(FormConfiguration fCfg:fCfgs){
			FormConfiguration fCfg=project.getSpeakerFormConfiguration();
				if(fCfg!=null && Speaker.class.equals(fCfg.getBeanClass())){
					spkFormConfig=fCfg;
//					break;
				}
//			}
		}
		return  spkFormConfig;
	}
	
	public static void validateSpeakerCode(ValidationResult vr,String spkCode) {
		
		if(spkCode !=null) {
			if(!spkCode.isEmpty()) {

				if(!spkCode.matches("^[\\w\\p{L}-:! \\.]*$")) {
					if(vr==null) {
						vr=new ValidationResult();
					}
					vr.setType(ValidationResult.Type.ERRORS);
					var msg=new LocalizableMessage("Illegal character(s) in speaker code. Allowed are alphanumeric characters and - and _");
					PropertyValidationResult namePropVr=new PropertyValidationResult(Type.ERROR,msg);
					vr.putPropertyValidationResult("code", namePropVr);
					//vr.setPropertyValidationResults(propVrMap);
				}
				if(spkCode.matches(".*\\.{2,}?.*")) {
					if(vr==null) {
						vr=new ValidationResult();
					}
					vr.setType(ValidationResult.Type.ERRORS);
					var msg=new LocalizableMessage("Speaker code must not contain dot sequences");
					PropertyValidationResult namePropVr=new PropertyValidationResult(Type.ERROR,msg);
					vr.putPropertyValidationResult("code", namePropVr);
					//vr.setPropertyValidationResults(propVrMap);
				}
				if(spkCode.matches("^\\s+.*")) {
					if(vr==null) {
						vr=new ValidationResult();
					}
					vr.setType(ValidationResult.Type.ERRORS);
					var msg=new LocalizableMessage("Speaker code must not start with blank characters");
					PropertyValidationResult namePropVr=new PropertyValidationResult(Type.ERROR,msg);
					vr.putPropertyValidationResult("code", namePropVr);
					//vr.setPropertyValidationResults(propVrMap);
				}
				if(spkCode.matches(".*\\s+$")) {
					if(vr==null) {
						vr=new ValidationResult();
					}
					vr.setType(ValidationResult.Type.ERRORS);
					var msg=new LocalizableMessage("Speaker code must not end with blank characters");
					PropertyValidationResult namePropVr=new PropertyValidationResult(Type.ERROR,msg);
					vr.putPropertyValidationResult("code", namePropVr);
					//vr.setPropertyValidationResults(propVrMap);
				}
			}
		}
	}
	
	
	public ValidationResult validate(Object o,ValidationResult validationResult) throws ValidationException{
		ValidationResult vr=super.validate(o, validationResult);
		Speaker spk=(Speaker)o;
		
		//Check code
		String spkCode=spk.getCode();
		
		SpeakerInformedConsentController.validateSpeakerCode(vr, spkCode);
		
		Project selProj=getSelectedProject();
		if(selProj!=null && selProj.isInformedConsentPaperForm() && ! spk.getInformedConsentInPaperFormSignedDelegate()){
			vr.putPropertyValidationResult("informedConsentInPaperFormSignedDelegate",new PropertyValidationResult(PropertyValidationResult.Type.ERROR));
			vr.setType(ValidationResult.Type.ERRORS);
		}
		return vr;
	}
	
	private String resourceTextWithProjectParam(String key) {
		Locale loc=LocaleSupport.getLocale(currentRequest);
		ResourceBundle rb=ResourceBundle.getBundle("Messages",loc);
		String p=rb.getString(key);
		Project selProject=getSelectedProject();
		String t;
		if(selProject!=null) {
			String prjNm=selProject.getName();
			t=MessageFormat.format(p,new Object[]{prjNm});
		}else {
			t=p;
		}
		return t;
	}
	
	public String localizedDefaultResearchPurpose(Locale loc,Project prj) {
		
		// Init with default
		String defResearchPupose=prj.getDefaultResearchPurpose();
		String defRpLc=prj.getDefaultResearchPurposeLanguageISO3code();
		List<LocalizedDefaultPurposeText> addLangs=prj.getDefaultResearchPurposeAdditionalLanguages();
		String iso3Lang=loc.getISO3Language();
		if(addLangs.size()>0 && !iso3Lang.equals(defRpLc)) {
			for(LocalizedDefaultPurposeText ldpt:addLangs) {
				String ldptLc=ldpt.getLanguageISO3code();
				if(iso3Lang.equals(ldptLc)) {
					defResearchPupose=ldpt.getLocalizedText();
					break;
				}
			}
		}
		return defResearchPupose;
	}
	
	public LocalizedInformedConsentText localizedinformedConsent(Locale loc,Project prj) {
		
		// Init with default
		LocalizedInformedConsentText lic=null;
		
		String informedConsentText=prj.getDefaultInformedConsentText();
		String defIctLc=prj.getDefaultInformedConsentTextLanguageISO3code();
		if(informedConsentText!=null && !informedConsentText.isBlank()) {
			LocalizedInformedConsentText defLic=new LocalizedInformedConsentText();
			//defLic.setTextFormat(prj.getInformedConsentTextFormat());
			defLic.setLanguageISO3code(defIctLc);
			defLic.setLocalizedText(informedConsentText);
			defLic.setProject(prj);

			lic=defLic;
		}
		
		List<LocalizedInformedConsentText> licts=prj.getInformedConsentTextAdditionalLanguages();
		String iso3Lang=loc.getISO3Language();
		if(licts.size()>0 && !iso3Lang.equals(defIctLc)) {
			for(LocalizedInformedConsentText lict:licts) {
				String lictLc=lict.getLanguageISO3code();
				if(iso3Lang.equals(lictLc)) {
					lic=lict;
					break;
				}
			}
		}
		return lic;
	}
	
	public String localizedinformedConsentText(Locale loc,Project prj) {
		
		// Init with default
		String locInformedConsentText=null;
		String informedConsentText=prj.getDefaultInformedConsentText();
		if(informedConsentText!=null && !"".equals(informedConsentText)) {
			locInformedConsentText=informedConsentText;
		}
		String defIctLc=prj.getDefaultInformedConsentTextLanguageISO3code();
		List<LocalizedInformedConsentText> licts=prj.getInformedConsentTextAdditionalLanguages();
		String iso3Lang=loc.getISO3Language();
		if(licts.size()>0 && !iso3Lang.equals(defIctLc)) {
			for(LocalizedInformedConsentText lict:licts) {
				String lictLc=lict.getLanguageISO3code();
				if(iso3Lang.equals(lictLc)) {
					locInformedConsentText=lict.getLocalizedText();
					break;
				}
			}
		}
		return locInformedConsentText;
	}
	
	public String getConsentInformText(){
		
		Locale loc=LocaleSupport.getLocale(currentRequest);
		ResourceBundle rb=ResourceBundle.getBundle("Messages",loc);
		
		Project selProject=getSelectedProject();
		
		if(selProject!=null) {
			// Get informed consent stored for project
			String consentInformedText=localizedinformedConsentText(loc, selProject);
			if(consentInformedText==null) {
				// if not set use a default text which contain the default research purpose
				String prjNm=selProject.getName();
				String defResearchPupose=localizedDefaultResearchPurpose(loc,selProject);
				String p=rb.getString("consent.inform.text");
				consentInformedText=MessageFormat.format(p,new Object[]{prjNm,defResearchPupose});
			}
			return consentInformedText;
			
		}
		return null;
	}
	
	public String getConsentInformHtml(){
		
		Locale loc=LocaleSupport.getLocale(currentRequest);
		
		Project selProject=getSelectedProject();

		if(selProject!=null) {
			// Get informed consent stored for project

			LocalizedInformedConsentText lic=localizedinformedConsent(loc, selProject);

			if(lic!=null) {
				String txt=lic.getLocalizedText();
				TextFormat tf=lic.getTextFormat();
				if(txt!=null) {
					if(TextFormat.HTML5_SUBSET.equals(tf)){
						
						//org.jsoup.nodes.Document hd=Jsoup.parseBodyFragment(txt);
						
						// "Shift" header downwards
						// h1,h2 and h3 are used by WikiSpeech page
						
						// Change in 2.58.2:
						// Do not replace tags, allow only h4-h6
//						hd.select("h5").replaceAll((e)->{
//							Element ne=new Element("h6");
//							ne.appendChildren(e.childNodes());
//							return ne;
//						});
//						hd.select("h4").replaceAll((e)->{
//							Element ne=new Element("h6");
//							ne.appendChildren(e.childNodes());
//							return ne;
//						});
//						hd.select("h3").replaceAll((e)->{
//							Element ne=new Element("h6");
//							ne.appendChildren(e.childNodes());
//							return ne;
//						});
//						hd.select("h2").replaceAll((e)->{
//							Element ne=new Element("h5");
//							ne.appendChildren(e.childNodes());
//							return ne;
//						});
//						hd.select("h1").replaceAll((e)->{
//							Element ne=new Element("h4");
//							ne.appendChildren(e.childNodes());
//							return ne;
//						});
//						
						// Sanitize "shifted" doc
						PolicyFactory policy = new HtmlPolicyBuilder()
							    .allowElements("p")
							    .allowElements("h4", "h5", "h6")
							    .allowElements("b", "i")
							    .toFactory();
							String safeHTML = policy.sanitize(txt);
							
						return "<div>"+safeHTML+"</div>";
					}else {
						return "<p class=\"tableTextEmp\">"+HTMLTextEncoder.encode(txt)+"</p>";
					}
				}
			}else {
				String consentInformTxt=getConsentInformText();
				if(consentInformTxt!=null) {
					String consentInformHtml=HTMLTextEncoder.encode(consentInformTxt);
					return consentInformHtml;
				}
			}

		}
		return null;
	}
	
//	public String getConsentInformHtml(){
//		String txt=getConsentInformText();
//		if(txt!=null) {
//			return HTMLTextEncoder.encode(txt);
//		}
//		return null;
//	}
//	
	public String getConsentInformedText(){
		Locale loc=LocaleSupport.getLocale(currentRequest);
		Project selProject=getSelectedProject();
		String consentInformedText=localizedinformedConsentText(loc, selProject);
		if(consentInformedText==null) {
			return resourceTextWithProjectParam("consent.informed.text");
		}else{
			// custom informed consent text
			return "";
		}
	}
	
	public String getBroadConsentInformedText(){
		return resourceTextWithProjectParam("consent.informed.broad.purpose.text");
	}
	
	public ipsk.beans.form.FormConfiguration getFormConfiguration(){

		FormConfiguration spkFmCfg=null;
		Project selProj=getSelectedProject();
		
		if(formConfigurationFromProject){
			spkFmCfg=getProjectSpeakerFormConfiguration();
			//return spkFmCfg;
			
			
				SpeakerFormConfigForceInformedConsent spkFrfed=null;
				try {
					spkFrfed = new SpeakerFormConfigForceInformedConsent(spkFmCfg,selProj.isInformedConsentPaperForm());
				} catch (IntrospectionException e) {
					// should never happen
					e.printStackTrace();
				}
				return spkFrfed;
		
		}
		return null;

	}

	protected void setPropertiesOnCreate(HttpServletRequest request,EntityManager em,Object bean) {
		
		Speaker speaker=(Speaker)bean;
	
		speaker.setRegistered(new Date());
		
		// Should be already set by speaker form ...
		Account regAcc=speaker.getRegisteredByAccount();
		if(regAcc==null){
			Account acc=getAccountByRequest(request);
			speaker.setRegisteredByAccount(acc);
		}
		
		String drIdStr=request.getParameter("dialectRegion."+MapConverter.OBJECT_ID);
		if (drIdStr!=null){
			DialectRegion dr=em.find(DialectRegion.class, Integer.parseInt(drIdStr));
			if(dr!=null){
				speaker.setDialectRegion(dr);
			}
		}
	
	}
	
	protected void setPropertiesOnApply(HttpServletRequest request,EntityManager em,Object bean) {

		Speaker speaker=(Speaker)bean;
		speaker.applyCurrentProject(getSelectedProject());

		Account acc=getAccountByRequest(request);

		Project sPrj=getSelectedProject();
		boolean isAdmOfPrj=(acc.getAdminOfProjects().contains(sPrj));

		if(!isAdmOfPrj && acc.getProjects().contains(sPrj)) {
			// Subject, add speaker data to account to account
			acc.getSpeakerData().add(speaker);
			speaker.setSpeakerDataAccount(acc);
			em.merge(speaker);
		}

	}
	
	/**
	 * Returns alphabetical ordered list of dialect regions of project. Records marked as alternative are always sorted to the end. 
	 * @return
	 */
	@SuppressWarnings("unchecked")
	public List<DialectRegion> getOrderedDialectRegionsForSelectedProject(){
		List<DialectRegion> list=null;
		EntityManager em = getThreadEntityManager();

		Project selectedProject=getSelectedProject();
		
		//Query q = em.createQuery("SELECT dr FROM DialectRegion dr WHERE :selectedProject MEMBER OF dr.projects ORDER BY dr.position");
		Query q = em.createQuery("SELECT dr FROM DialectRegion dr WHERE :selectedProject MEMBER OF dr.projects ORDER BY dr.alternative DESC,dr.name,dr.position");
		q.setParameter("selectedProject", selectedProject);
		list=q.getResultList();
		return list;
	}
	
	public Organisation getAssociatedOrganisation(){
//		EntityManager em=getThreadEntityManager();
		Account acc=getAccountByRequest(currentRequest);
		return acc.getOrganisation();
	}
	
	public Organisation getAssociatedOrganisation(HttpServletRequest req){
//		EntityManager em=getThreadEntityManager();
		Account acc=getAccountByRequest(req);
		return acc.getOrganisation();
	}

	public void setSpeaker(Speaker sp) {
		
	}
	

	

}
