package ipsk.webapps.db.speech;

import java.security.MessageDigest;
import java.security.NoSuchAlgorithmException;
import java.util.ArrayList;
import java.util.List;
import java.util.Locale;
import java.util.Set;
import java.util.UUID;
import java.util.regex.Matcher;
import java.util.regex.Pattern;

import javax.persistence.EntityManager;
import javax.persistence.TypedQuery;
import javax.persistence.criteria.CriteriaBuilder;
import javax.persistence.criteria.CriteriaQuery;
import javax.persistence.criteria.Expression;
import javax.persistence.criteria.Predicate;
import javax.persistence.criteria.Root;
import javax.servlet.http.HttpServletRequest;

import org.jasypt.util.password.StrongPasswordEncryptor;
import org.jasypt.util.password.rfc2307.RFC2307SSHAPasswordEncryptor;

import edu.vt.middleware.password.AlphabeticalSequenceRule;
import edu.vt.middleware.password.LengthRule;
import edu.vt.middleware.password.NumericalSequenceRule;
import edu.vt.middleware.password.Password;
import edu.vt.middleware.password.PasswordData;
import edu.vt.middleware.password.PasswordValidator;
import edu.vt.middleware.password.QwertySequenceRule;
import edu.vt.middleware.password.RepeatCharacterRegexRule;
import edu.vt.middleware.password.Rule;
import edu.vt.middleware.password.RuleResult;
import ipsk.db.speech.Account;
import ipsk.db.speech.Message;
import ipsk.db.speech.Project;
import ipsk.db.speech.Speaker;
import ipsk.db.speech.UserRole;
import ipsk.db.speech.UserRoleId;
import ipsk.db.speech.account.AccountRequest;
import ipsk.util.LocalizableMessage;
import ipsk.util.RadixConverters;
import ipsk.webapps.ControllerException;
import ipsk.webapps.ProcessResult;

public class AccountUtils {
	
	private RFC2307SSHAPasswordEncryptor rfcPasswordEncryptor=null;
	public RFC2307SSHAPasswordEncryptor getRfcPasswordEncryptor() {
		return rfcPasswordEncryptor;
	}

	private StrongPasswordEncryptor strongPasswordEncryptor=null;
	public StrongPasswordEncryptor getStrongPasswordEncryptor() {
		return strongPasswordEncryptor;
	}

	private MessageDigest sha5Digest=null;
	
	public MessageDigest getSha5Digest() {
		return sha5Digest;
	}


	public AccountUtils() {
		super();
		
		rfcPasswordEncryptor = new RFC2307SSHAPasswordEncryptor();
		strongPasswordEncryptor = new StrongPasswordEncryptor();
		try {
			// for Oracle JVM
			sha5Digest = MessageDigest.getInstance("SHA-512");
		} catch (NoSuchAlgorithmException e) {
			// for IBM JVM
			try {
				sha5Digest = MessageDigest.getInstance("SHA5");
			} catch (NoSuchAlgorithmException e1) {
				// ignore
			}
		}
	}

	/**
	 * Validate username (login) for new user. 
	 * @param login the username
	 * @return null if username is good or a localized Message object describing the problem
	 */
	public static LocalizableMessage validateUsername(String login) {
		LocalizableMessage errMsg=null;
		if(login==null) {
			errMsg=new LocalizableMessage("Username must not be null");
		}else if(login.isEmpty()){
			errMsg=new LocalizableMessage("Username must not be empty");
		}else if(login.isBlank()){
			errMsg=new LocalizableMessage("Username must not be blank");
		}else {
			Pattern invalidCharsPattern = Pattern.compile("[^a-zA-Z0-9._@-]");
			Matcher usernameMatcher = invalidCharsPattern.matcher(login);
			if(usernameMatcher.find()) {
				errMsg=new LocalizableMessage("Allowed characters for username: a-z,A-Z,0-9,.,_,@,-");
			}
		}
		return errMsg;
	}
	
	public static boolean accountExists(EntityManager em,String login) {
		return AccountUtils.accountExists(em, login, true);
	}
	
	public static boolean accountExists(EntityManager em,String login,boolean caseInsensitive) {
		if(caseInsensitive) {
			List<Account> accs=accountsByLoginCaseInsensitive(em, login);
			return !(accs.isEmpty() || accs.size()==0);
		}else {
			Account exAcc=em.find(Account.class, login);
			return (exAcc!=null);
		}
	}
	
	private PasswordValidator buildPasswordValidator() {
		// password check using vt-password library
		
		// password must be between 8 and 64 chars long
		LengthRule lengthRule = new LengthRule(8, 64);
		
		// don't allow alphabetical sequences
		AlphabeticalSequenceRule alphaSeqRule = new AlphabeticalSequenceRule();

		// don't allow numerical sequences of length 5
		NumericalSequenceRule numSeqRule = new NumericalSequenceRule(5, false);

		// don't allow qwerty sequences
		QwertySequenceRule qwertySeqRule = new QwertySequenceRule();

		// don't allow 4 repeat characters
		RepeatCharacterRegexRule repeatRule = new RepeatCharacterRegexRule(4);


		// group all rules together in a List
		List<Rule> ruleList = new ArrayList<Rule>();

		ruleList.add(lengthRule);
		ruleList.add(alphaSeqRule);
		ruleList.add(numSeqRule);
		ruleList.add(qwertySeqRule);
		ruleList.add(repeatRule);

		PasswordValidator validator = new PasswordValidator(ruleList);
		return validator;
	}
	

	public LocalizableMessage checkPassword(String passw1,String passw2){
		LocalizableMessage lmErrMsg=null;
		if(passw1 != null && passw2 !=null){
			if(passw1.equals(passw2)){
				if(!"".equals(passw1)){
					PasswordValidator validator=buildPasswordValidator();
					PasswordData passwordData = new PasswordData(new Password(passw1));
					RuleResult result = validator.validate(passwordData);

					if (!result.isValid()) {
						// Bad password

						// TODO Password validator messages are not localized. Use RuleResult.getDetails() to create localized messages
						
//						List<RuleResultDetail> resDetails=result.getDetails();
//						for(RuleResultDetail resDetail:resDetails) {
//							
//							String errCode=resDetail.getErrorCode();
//							// Example for length rule
//							if(LengthRule.ERROR_CODE_MIN.equals(errCode)) {
//								// build localized message
//							}
//						}
					
						
						// default message
						String message="Password validation failed.";

						List<String> validatorMsgs=validator.getMessages(result);

						if(validatorMsgs.size()>0) {
							// simply use the first one for now
							message=validatorMsgs.get(0);
						}

						lmErrMsg=new LocalizableMessage(message);
					}
				}else{
					// password empty
					lmErrMsg=new LocalizableMessage("Messages", "password.is_empty");	
				}
			}else{
				lmErrMsg=new LocalizableMessage("Messages", "password.passwords_not_equal");
			}
		}
		
		// Return null if password is OK
		return lmErrMsg;
	}
	
	public void applyDigestPassword(Account a,String plainTextPassword){
		
		String rfcPasswd = rfcPasswordEncryptor.encryptPassword(plainTextPassword);
		a.setRfc2307Password(rfcPasswd);

		if(sha5Digest!=null){
			byte[] ba=plainTextPassword.getBytes();
			String digestedPassw=RadixConverters.bytesToHex(sha5Digest.digest(ba));
			a.setSha5HexPassword(digestedPassw);
		}

		String strongPasswd = strongPasswordEncryptor
				.encryptPassword(plainTextPassword);
		a.setStrongPassword(strongPasswd);
	}
	
	public static boolean accountHasUserRole(Account acc,UserRoleId.RoleName userRoleName) {
		Set<UserRole> urs=acc.getUserRoles();
		for(UserRole ur:urs) {
			if(userRoleName.equals(ur.getId().getRoleName())) {
				return true;
			}
		}
		return false;
	}
	
	public void applyAccountEMail(Account a,String email){
		a.setEmail(email);
		if(a.getLogin().equalsIgnoreCase(a.getEmail())) {
			a.setLoginCaseInsensitiv(true);
		}
	}
	
	public AccountRequest accountRequestByUUID(EntityManager em,UUID uuid) {

		AccountRequest ar=null;
		CriteriaBuilder cb = em.getCriteriaBuilder();
		CriteriaQuery<AccountRequest> cq = cb.createQuery(AccountRequest.class);
		Root<AccountRequest> qr = cq.from(AccountRequest.class);
		cq.select(qr);
		Predicate eqUuid=cb.equal(qr.get("uuid"),uuid);
		cq.where(eqUuid);
		TypedQuery<AccountRequest> aq = em.createQuery(cq);
		List<AccountRequest> arList=aq.getResultList();
		// should be unique
		if(arList.size()>0) {
			ar=arList.get(0);
		}
		return ar;
	}
	
	public static List<Account> accountsByLoginCaseInsensitive(EntityManager em,String login) {
		String reqLoginLc=login.toLowerCase(Locale.ENGLISH);
		CriteriaBuilder cb = em.getCriteriaBuilder();
		CriteriaQuery<Account> cq = cb.createQuery(Account.class);
		Root<Account> qr = cq.from(Account.class);
		cq.select(qr);
		Expression<String> loginLc=cb.lower(qr.<String>get("login"));
		Predicate loginEqualsCaseInsensitive=cb.equal(loginLc,reqLoginLc);
		cq.where(loginEqualsCaseInsensitive);
		TypedQuery<Account> aq = em.createQuery(cq);
		List<Account> accsList=aq.getResultList();
		return accsList;
	}
	

	
	public void deleteAccount(EntityManager em, Account acc) {

		Set<Project> admPrjs = acc.getAdminOfProjects();
		for (Project admPrj : admPrjs) {
			// project is owning side, we have to merge
			admPrj.getAdminAccounts().remove(acc);
			em.merge(admPrj);
		}
		acc.getAdminOfProjects().clear();

		Set<Project> prjs = acc.getProjects();
		for (Project prj : prjs) {
			// project is owning side, we have to merge
			prj.getAccounts().remove(acc);
			em.merge(prj);
		}
		acc.getProjects().clear();
		
		Set<Speaker> spkDatas=acc.getSpeakerData();
		for (Speaker spkData : spkDatas) {
			// Disconnect (pseudonymize) speaker data
			spkData.setSpeakerDataAccount(null);
			em.merge(spkData);
		}

		Set<Speaker> regSpks = acc.getRegisteredSpeakers();
		for (Speaker regSpk : regSpks) {
			regSpk.setRegisteredByAccount(null);
			em.merge(regSpk);
		}
		
		// Messages from this account are cascade removed
		// Update: Cascade remove does not work for messages that have reply of column set.
		Set<Message> messagesFromThisAccount = acc.getMessagesForFromLogin();
		for (Message messageFromThisAcc : messagesFromThisAccount) {
			
			messageFromThisAcc.setReplyOf(null);
			em.merge(messageFromThisAcc);
		}
		
		for (Message messageFromThisAcc : messagesFromThisAccount) {
			em.remove(messageFromThisAcc);
		}
		
		// Messages to this account:
		Set<Message> messagesToThisAccount = acc.getMessagesForToLogin();
		for (Message messageToThisAcc : messagesToThisAccount) {
			// remove this account as recipient
			messageToThisAcc.setAccountByToLogin(null);
			messageToThisAcc.setReplyOf(null);
			em.merge(messageToThisAcc);
		}
		 messagesToThisAccount.clear();
	
		// Finally remove the account
		em.remove(acc);
		
	}
}
