package ipsk.webapps.db.speech;

import java.io.IOException;
import java.io.LineNumberReader;
import java.io.StringReader;
import java.util.ArrayList;
import java.util.HashSet;
import java.util.List;
import java.util.Locale;
import java.util.Map;
import java.util.Set;

import javax.mail.internet.AddressException;
import javax.mail.internet.InternetAddress;
import javax.persistence.EntityManager;
import javax.servlet.http.HttpServletRequest;

import ips.beans.PersistenceMapConverter;
import ipsk.beans.BeanModel;
import ipsk.beans.MapConverterException;
import ipsk.beans.PropertyValidationResult;
import ipsk.beans.PropertyValidationResult.Type;
import ipsk.beans.validation.ValidationResult;
import ipsk.db.speech.Project;
import ipsk.db.speech.UserRoleId;
import ipsk.util.LocalizableMessage;
import ipsk.webapps.ControllerException;
import ipsk.webapps.ProcessResult;

public class MultipleSpeakerInvitationsController extends BasicSpeakerInvitationController<MultipleSpeakerInvitations> {

	public static final int DEFAULT_SPEAKER_INVITATION_VALID_TIME_MS=1000*60*60*24*5;  // 5 days
	
	public MultipleSpeakerInvitationsController() {
		super("WebSpeechDBPU",MultipleSpeakerInvitations.class);
		setInvitationValidTimeMs(DEFAULT_SPEAKER_INVITATION_VALID_TIME_MS);
	}

	
	
	public void inviteMultipleSpeakers(HttpServletRequest req)
			throws ControllerException {
		processResult=null;
		beanModel=null;
		String cmd=processCommand(req,new String[]{"invite","cancel"});
		Map<String,String[]> paramMap=req.getParameterMap();
		String[] projectNames=paramMap.get("project.name");

		Project pr=null;
		if(projectNames!=null && projectNames.length==1){
			projectName=projectNames[0];
		}
		if("invite".equals(cmd)){
			secureRequestTokenProvider.checkSecureRequestToken(req);
			EntityManager em = getThreadEntityManager();
			ValidationResult vr=new ValidationResult();

			pr=em.find(Project.class,projectName);
			if(pr==null){
				throw new ControllerException("Cannot find project.");
			}
			PersistenceMapConverter<MultipleSpeakerInvitations> mc=new PersistenceMapConverter<MultipleSpeakerInvitations>();


			MultipleSpeakerInvitations spkInvitdata=new MultipleSpeakerInvitations();

			try {
				mc.setBeanProperties(spkInvitdata,paramMap);
			} catch (MapConverterException mce) {
				throw new ControllerException(mce);
			}

			// Parse email list string
			Set<String> duplicateCheckSet=new HashSet<>();
			List<SpeakerInvitation> spkInvList=new ArrayList<>();
			try (StringReader sr=new StringReader(spkInvitdata.getEmailList());
					LineNumberReader lnr=new LineNumberReader(sr);
					){
				PropertyValidationResult emailPvr=null;
				String line;
				int lineNumber=0;
				while((line=lnr.readLine())!=null) {
					String trimmedLine=line.trim();
					lineNumber++;
					// Ignore blank lines
					if(trimmedLine.isBlank()) {
						continue;
					}

					// Parse address line
					InternetAddress ias=null;
					String emailUsername=null;
					try {
						InternetAddress[] iass = InternetAddress.parse(trimmedLine);
						if(iass==null || iass.length!=1){
							emailPvr=new PropertyValidationResult(Type.ERROR);
						}else{
							ias=iass[0];
							ias.validate();
							// use only address without optional personal name
							emailUsername=ias.getAddress();
						}
					} catch (AddressException e1) {
						String msg="E-Mail address syntax error in line "+lineNumber+" "+e1.getMessage();
						emailPvr=new PropertyValidationResult(Type.ERROR, new LocalizableMessage(msg));
						// stop further processing of the list
						break;
					}
					if(emailUsername!=null){
						if(AccountUtils.accountExists(em, emailUsername,true)){
							emailPvr=new PropertyValidationResult(Type.ERROR, new LocalizableMessage("Account for the address "+emailUsername+" in line "+lineNumber+" already exists!"));
							break;
						}
					}

					// lower case E-Mail string to check duplicates
					String lowerCaseEmailUsername=emailUsername.toLowerCase(Locale.US);
					if(duplicateCheckSet.contains(lowerCaseEmailUsername)) {
						emailPvr=new PropertyValidationResult(Type.ERROR);
						emailPvr.setValidationMessage(new LocalizableMessage("Found duplicate E-Mail address \""+emailUsername+"\" in line "+lineNumber+"."));
						break;
					}
					duplicateCheckSet.add(lowerCaseEmailUsername);
					SpeakerInvitation spkInv=new SpeakerInvitation();
					spkInv.setEmail(emailUsername);
					spkInv.setLocale(spkInvitdata.getLocale());
					emailPvr=checkEmailAddress(em, spkInv);

					if(emailPvr!=null){
						emailPvr.setValidationMessage(new LocalizableMessage("Error checking E-Mail address \""+spkInv.getEmail()+"\" in line "+lineNumber+"."));
						break;
					}
					// OK, add to parsed list
					spkInvList.add(spkInv);
				}
				
				if(emailPvr!=null) {
					vr.putPropertyValidationResult("emailList",emailPvr);
					vr.setType(ValidationResult.Type.ERRORS);
					if(beanModel==null){
						beanModel=new BeanModel<MultipleSpeakerInvitations>(spkInvitdata);
					}
					beanModel.setValidationResult(vr);
				}

				if(vr.isValid()) {
					// All E-Mail lines are valid, send invitation to each of them
					for(SpeakerInvitation spkInvit:spkInvList) {
						UserRoleId.RoleName[] roles=new UserRoleId.RoleName[] {UserRoleId.RoleName.SUBJECT};
						invite(em, req, spkInvit,roles );
					}
				}

			} catch (IOException e) {
				throw new ControllerException(e);
			}
			processResult=new ProcessResult(vr);
		}else if("cancel".equals(cmd)){
			super.processRequest(req);
		}else{
			beanModel=new BeanModel<MultipleSpeakerInvitations>(new MultipleSpeakerInvitations());
		}
	}
	
	public void processRequest(HttpServletRequest req){
		Map<String, String[]> paramMap=req.getParameterMap();
		String[] projectNames=(String[]) paramMap.get("project.name");
		
		Project pr=null;
		if(projectNames!=null && projectNames.length==1){
			projectName=projectNames[0];
		}
		beanModel=new BeanModel<MultipleSpeakerInvitations>(new MultipleSpeakerInvitations());
		processResult=null;
	}

}
