package ipsk.webapps.db.speech.ws;

import java.io.StringWriter;
import java.util.Collection;
import java.util.List;
import java.util.Set;

import javax.persistence.EntityManager;
import javax.persistence.TypedQuery;
import javax.persistence.criteria.CriteriaBuilder;
import javax.persistence.criteria.CriteriaQuery;
import javax.persistence.criteria.Expression;
import javax.persistence.criteria.Order;
import javax.persistence.criteria.Root;
import javax.servlet.http.HttpServletRequest;
import javax.ws.rs.GET;
import javax.ws.rs.HEAD;
import javax.ws.rs.Path;
import javax.ws.rs.PathParam;
import javax.ws.rs.Produces;
import javax.ws.rs.QueryParam;
import javax.ws.rs.core.Context;
import javax.ws.rs.core.GenericEntity;
import javax.ws.rs.core.MediaType;
import javax.ws.rs.core.Response;
import javax.ws.rs.core.Response.Status;
import javax.xml.parsers.ParserConfigurationException;

import ipsk.apps.speechrecorder.script.RecscriptHandler;
import ipsk.db.speech.Project;
import ipsk.db.speech.Session;
import ipsk.db.speech.script.Script;
import ipsk.persistence.EntityManagerProvider;
import ipsk.webapps.EntityManagerFactoryInitializer;
import ipsk.webapps.PermissionDeniedException;
import ipsk.webapps.db.speech.WikiSpeechSecurityManager;
import ipsk.xml.DOMConverterException;

public class ProjectScriptResource {

	public enum OrderDirection {ASC,DESC};
	
//	public class OrderDirectionList {
//		  private List<OrderDirection> orderDirectionList;
//
//		  public OrderDirectionList(String toParse) {
//			  
//		  }
//
//		  public List<OrderDirection> getOrderDirectionList() {
//		    return this.orderDirectionList;
//		  }
//		}
	
	private String projectName;

	public ProjectScriptResource(String projectName) {
		this.projectName = projectName;
	}

	private Script tryGetScript(HttpServletRequest req, int scriptId) throws PermissionDeniedException {
		final EntityManager em = EntityManagerFactoryInitializer.getEntityManagerFactory().createEntityManager();
		WikiSpeechSecurityManager securityManager = new WikiSpeechSecurityManager(new EntityManagerProvider() {
			@Override
			public EntityManager getThreadEntityManager() {
				return em;
			}
		});

		try {
			Script sc = em.find(Script.class, scriptId);
			securityManager.checkReadPermission(req, sc);
			return sc;
		} catch (PermissionDeniedException pde) {
			throw pde;
		} finally {
			em.close();
		}
	}

	@GET
	@Produces({ MediaType.APPLICATION_JSON })
	@Path("/{scriptId}")
	public Response getScript(@Context HttpServletRequest req, @PathParam("scriptId") int scriptId) {
		try {
			Script s = tryGetScript(req, scriptId);
			return Response.ok(s).build();
		} catch (PermissionDeniedException e) {
			return Response.status(Status.FORBIDDEN).build();
		}
	}

	@GET
	@Produces({ MediaType.APPLICATION_XML })
	@Path("/{scriptId}")
	public Response getScriptXML(@Context HttpServletRequest req, @PathParam("scriptId") int scriptId) {
		Script sc;
		try {
			sc = tryGetScript(req, scriptId);
			// Speechrecorder script has its own legacy XML conversion
			// methods
			RecscriptHandler rsh = new RecscriptHandler();
			StringWriter sw = new StringWriter();
			try {
				rsh.writeXML(sc, sw);
				return Response.ok(sw.toString()).build();
			} catch (DOMConverterException | ParserConfigurationException e) {

				return Response.serverError().build();
			}
		} catch (PermissionDeniedException e1) {
			return Response.status(Status.FORBIDDEN).build();
		}
	}

	private Response projectScripts(HttpServletRequest req, boolean content, Integer idx,String orderBy,OrderDirection orderDirection,Integer limit) {
		final EntityManager em = EntityManagerFactoryInitializer.getEntityManagerFactory().createEntityManager();
		WikiSpeechSecurityManager securityManager = new WikiSpeechSecurityManager(new EntityManagerProvider() {
			@Override
			public EntityManager getThreadEntityManager() {
				return em;
			}
		});
		try {
			Project p = em.find(Project.class, this.projectName);

			CriteriaBuilder cb = em.getCriteriaBuilder();
			CriteriaQuery<Script> cq = cb.createQuery(Script.class);
			Root<Script> rt = cq.from(Script.class);
			cq.select(rt);
			Expression<Collection<Project>> projects = rt.get("projects");
			cq.where(cb.isMember(p, projects));
			if("sessions._size".equals(orderBy)){
				Expression<Collection<Session>> sessions = rt.get("sessions");
				Expression<Integer> sessSizeExpr=cb.size(sessions);
				Order order;
				if(OrderDirection.DESC.equals(orderDirection)){
					order=cb.desc(sessSizeExpr);
				}else{
					order=cb.asc(sessSizeExpr);
				}
				cq.orderBy(order);
			}
			
			TypedQuery<Script> q = em.createQuery(cq.distinct(true));
			if(limit!=null){
				q.setMaxResults(limit);
			}
			List<Script> projScripts = q.getResultList();
			String sizeStr = Integer.toString(projScripts.size());
			if (idx != null) {
				if(idx>projScripts.size()-1 || idx<0){
					return Response.status(Status.BAD_REQUEST).build();
				}
				Script idxSc = projScripts.get(idx);
				securityManager.checkReadPermission(req, idxSc);
				if (content) {
					
					return Response.ok(idxSc).header("X-Total-Count", sizeStr).build();
				} else {
					return Response.noContent().header("X-Total-Count", sizeStr).build();
				}
			} else {
				for (Script sc : projScripts) {
					securityManager.checkReadPermission(req, sc);
				}
				
			
				if (content) {
					final GenericEntity<List<Script>> listEntity = new GenericEntity<List<Script>>(projScripts) {
					};
					return Response.ok(listEntity).header("X-Total-Count", sizeStr).build();
				} else {
					return Response.noContent().header("X-Total-Count", sizeStr).build();
				}
			}

		} catch (PermissionDeniedException e) {
			return Response.status(Status.FORBIDDEN).build();
		} finally {
			em.close();
		}
	}

	@GET
	@Produces({ MediaType.APPLICATION_JSON })
	@Path("/")
	public Response getScripts(@Context HttpServletRequest req, @QueryParam("listIndex") Integer listIndex,@QueryParam("order-by") String orderBy,@QueryParam("order-direction") List<OrderDirection> orderDirectionList,@QueryParam("limit") Integer limit) {
		
	
		
		if(orderBy!=null){
			// Currently the only supported order by clause
			// The query "/project/[projectName]/script?order-by=sessions._size&order-direction=DESC"
			// should have the same meaning as "script with lowest usage"
			
			if(!"sessions._size".equals(orderBy)){
				return Response.status(Status.BAD_REQUEST).build();
			}
		}
		
		if(limit!=null && limit<0){
			Response.status(Status.BAD_REQUEST).build();
		}
		OrderDirection orderDirection=OrderDirection.ASC;
		if(orderDirectionList!=null){
			if(orderDirectionList.size()>1){
				Response.status(Status.BAD_REQUEST).build();
			}else if(orderDirectionList.size()==1){
				orderDirection=orderDirectionList.get(0);
			}
			
		}
		return projectScripts(req, true, listIndex,orderBy,orderDirection,limit);
	}

	@HEAD
	@Produces({ MediaType.APPLICATION_JSON })
	@Path("/")
	public Response headScripts(@Context HttpServletRequest req) {
		return projectScripts(req, false, null,null,null,null);
	}

}
