package ipsk.webapps.db.speech.ws;

import java.io.InputStream;
import java.util.Collection;
import java.util.HashSet;
import java.util.Set;
import java.util.UUID;

import javax.persistence.EntityManager;
import javax.persistence.EntityTransaction;
import javax.persistence.TypedQuery;
import javax.persistence.criteria.CriteriaBuilder;
import javax.persistence.criteria.CriteriaQuery;
import javax.persistence.criteria.Expression;
import javax.persistence.criteria.Order;
import javax.servlet.ServletContext;
import javax.servlet.http.HttpServletRequest;
import javax.ws.rs.Consumes;
import javax.ws.rs.GET;
import javax.ws.rs.HEAD;
import javax.ws.rs.PATCH;
import javax.ws.rs.POST;
import javax.ws.rs.PUT;
import javax.ws.rs.Path;
import javax.ws.rs.PathParam;
import javax.ws.rs.Produces;
import javax.ws.rs.QueryParam;
import javax.ws.rs.core.Context;
import javax.ws.rs.core.GenericEntity;
import javax.ws.rs.core.MediaType;
import javax.ws.rs.core.Response;
import javax.ws.rs.core.Response.Status;
import javax.ws.rs.core.SecurityContext;
import javax.ws.rs.core.UriBuilder;
import javax.ws.rs.core.UriInfo;

import ipsk.beans.test.Root;
import ipsk.db.speech.Account;
import ipsk.db.speech.Person;
import ipsk.db.speech.Project;
import ipsk.db.speech.Session;
import ipsk.db.speech.SessionDTO;
import ipsk.db.speech.Speaker;
import ipsk.db.speech.UserRoleId;
import ipsk.db.speech.script.Script;
import ipsk.persistence.EntityManagerProvider;
import ipsk.webapps.EntityManagerFactoryInitializer;
import ipsk.webapps.db.speech.WikiSpeechSecurityManager;
import ipsk.webapps.db.speech.ws.ProjectScriptResource.OrderDirection;


public class ProjectSessionResource extends WikispeechBasicResource<Session>{

	private WikispeechBasicResource<Speaker> relSpeakerRes=new WikispeechBasicResource<Speaker>(Speaker.class);
	
	private String projectName;

	public ProjectSessionResource(String projectName) {
		super(Session.class);
		this.projectName = projectName;
	}

	
	
	@GET
	@Produces({ MediaType.APPLICATION_JSON, MediaType.APPLICATION_XML})
	@Path("/{sessionIdOrUUID}")
	public Response getSession(@Context HttpServletRequest req, @PathParam("sessionIdOrUUID") String sessionIdStr) {
		Integer entId=null;
		UUID entUUID=null;
		try{
			entId=Integer.parseInt(sessionIdStr);
		}catch(NumberFormatException nfe){
			entId=null;
			try{
				entUUID=UUID.fromString(sessionIdStr);
			}catch(IllegalArgumentException iae){
				return Response.status(Status.BAD_REQUEST).build();
			}
		}
		Session ent=null;
		final EntityManager em=EntityManagerFactoryInitializer.getEntityManagerFactory().createEntityManager();
		try{
			if(entId!=null){
				ent=em.find(entityClass, entId);
			}else if(entUUID!=null){
				ent=entityByUUID(em, entUUID);
			}
			if(ent==null){
				return Response.status(Status.NOT_FOUND).build();
			}
			
			Project sessProject=ent.getProject();
			if(sessProject==null || !sessProject.getName().equals(projectName)){
				// requested session not in this project context
				return Response.status(Status.NOT_FOUND).build();
			}

			WikiSpeechSecurityManager securityManager = new WikiSpeechSecurityManager(new EntityManagerProvider() {
				@Override
				public EntityManager getThreadEntityManager() {
					return em;
				}
			});
			if(!securityManager.getReadPermission(req, ent)){
				return Response.status(Status.FORBIDDEN).build();
			}

			return Response.ok(ent).build();
		}finally{
			em.close();
		}
	}


	
	
	private Response projectSessions(HttpServletRequest req,boolean content,Boolean exportForAnnotation) {
		final EntityManager em = EntityManagerFactoryInitializer.getEntityManagerFactory().createEntityManager();
		
		try {

			Set<Session> projSesss=new HashSet<Session>();
			Project p=em.find(Project.class, this.projectName);
			
			Account acc=getAccountByRequest(req, em);
		
			if(req.isUserInRole(UserRoleId.RoleName.ADMIN.name()) || 
			    (req.isUserInRole(UserRoleId.RoleName.PROJECT_ADMIN.name()) && p.getAdminAccounts().contains(acc)) ||
			    (req.isUserInRole(UserRoleId.RoleName.PROJECT_ANNOTATOR.name()) && p.getAccounts().contains(acc)))
			{


				if(exportForAnnotation!=null) {

					CriteriaBuilder cb = em.getCriteriaBuilder();
					CriteriaQuery<Session> cq = cb.createQuery(Session.class);
					javax.persistence.criteria.Root<Session> rt = cq.from(Session.class);
					cq.select(rt);
					Expression<Project> sPrj = rt.get("project");
					Expression<Boolean> prjExpr=cb.equal(sPrj,p);
					Expression<Boolean> expExpr=rt.get("exportForAnnotation");
					Expression<Boolean> forAnnExp=cb.equal(expExpr,exportForAnnotation);
					Expression<Boolean> cond=cb.and(prjExpr,forAnnExp);
					
					cq.where(cond);

					TypedQuery<Session> q = em.createQuery(cq.distinct(true));
					projSesss.addAll(q.getResultList());

				}else {
					projSesss.addAll(p.getSessions());
				}


			}else{
				// add own sessions
				Person accPers=acc.getPerson();
				if(accPers instanceof Speaker){
					Speaker accSpk=(Speaker)accPers;
					Set<Session> accSpkSesss=accSpk.getSessions();
					for(Session accSpksess:accSpkSesss){
						if(exportForAnnotation!=null) {
							if(exportForAnnotation != accSpksess.isExportForAnnotation()) {
								continue;
							}
						}
						if(p.equals(accSpksess.getProject())){
							
							projSesss.add(accSpksess);
						}
					}
				}
				
				Set<Speaker> accSpkDatas=acc.getSpeakerData();
				for(Speaker accSpkData:accSpkDatas) {
					if(p.equals(accSpkData.getProject())) {
						Set<Session> accSpkSesss=accSpkData.getSessions();
						for(Session accSpksess:accSpkSesss){
							if(exportForAnnotation!=null) {
								if(exportForAnnotation != accSpksess.isExportForAnnotation()) {
									continue;
								}
							}
							if(p.equals(accSpksess.getProject())){

								projSesss.add(accSpksess);
							}
						}
					}
				}
				
			}
			
			if (content) {
				final GenericEntity<Set<Session>> listEntity = new GenericEntity<Set<Session>>(projSesss) {};
				return Response.ok(listEntity).header("X-Total-Count", projSesss.size()).build();
			} else {
				return Response.noContent().header("X-Total-Count", projSesss.size()).build();
			}
		} finally {
			em.close();
		}
	}
	
	
	

	@GET
	@Produces({ MediaType.APPLICATION_JSON })
	public Response getProjectSessions(@Context HttpServletRequest req,@QueryParam("exportForAnnotation") Boolean exportForAnnotation) {
		
		return projectSessions(req,true,exportForAnnotation);
	}

	@HEAD
	@Produces({ MediaType.APPLICATION_JSON })
	public Response headProjectSessions(@Context HttpServletRequest req,@QueryParam("exportForAnnotation") Boolean exportForAnnotation) {
		return projectSessions(req,false,exportForAnnotation);
	}
	
	private Response storeSession(UUID sessionUUID,UriBuilder uriBuilder,SessionDTO sessionDto){
	EntityManager em=EntityManagerFactoryInitializer.getEntityManagerFactory().createEntityManager();
	try{
		Session existingSess=entityByUUID(em, sessionUUID);
		if(existingSess!=null){
			return Response.status(Status.CONFLICT).build();
		}
		
		String sessDtoProj=sessionDto.getProject();
		if(sessDtoProj!=null && !sessDtoProj.equals(projectName)){
			// project (name) mismatch
			return Response.status(Status.BAD_REQUEST).build();
		}
		Session session=sessionDto.toSession();
		Project p=em.find(Project.class,projectName);
		
		EntityTransaction tr=em.getTransaction();
		tr.begin();
		
		em.persist(session);
		
		
		session.setProject(p);
		em.merge(p);
		
		int scriptId=Integer.parseInt(sessionDto.getScript());
		Script sc=em.find(Script.class, scriptId);
		session.setScript(sc);
		em.merge(sc);

		for(String spkIdStr:sessionDto.getSpeakers()){
			Speaker spk=null;
			try{
				UUID speakerUUID=UUID.fromString(spkIdStr);
//				spk=byUUID(em, speakerUUID);
				spk=relSpeakerRes.entityByUUID(em,speakerUUID);
				if(spk==null){
					tr.rollback();
					return Response.status(Status.BAD_REQUEST).build();
				}
			}catch(IllegalArgumentException iae){
				try{
				int spkId=Integer.parseInt(spkIdStr);
					spk=em.find(Speaker.class, spkId);
				}catch(NumberFormatException nfe){
					tr.rollback();
					return Response.status(Status.BAD_REQUEST).build();
				}
			}
		
			session.getSpeakers().add(spk);
			spk.getSessions().add(session);
			em.merge(spk);
		}
		
		em.merge(session);
		
		tr.commit();
		
	    // builder.path(sessionUUID.toString());
		return Response.created(uriBuilder.build()).entity(session).build();
		
	}finally{
		em.close();
	}
	}
	
	
	@POST
	@Consumes(MediaType.APPLICATION_JSON)
	@Produces(MediaType.APPLICATION_JSON)
	@Path("/{sessionUUID}")
	public Response postSession(@PathParam("sessionUUID") String sessionUUIDStr, @Context HttpServletRequest req, @Context UriInfo uriInfo,SessionDTO sessionDto){
		
		

		String sessionDtoUuid=sessionDto.getUuid();
		
		UUID sessionUUID=null;
		
		try{
		// try to get from path
		if(sessionUUIDStr!=null){
			sessionUUID=UUID.fromString(sessionUUIDStr);
			// if UUID is given in data object as well the UUIDs must match
			if(sessionDtoUuid!=null && !sessionDtoUuid.equals(sessionUUID.toString())){
				// Mismatch of UUIDs 
				return Response.status(Status.BAD_REQUEST).build();
			}
		}else{
			if(sessionDtoUuid!=null){
				// try to get from DTO
				sessionUUID=UUID.fromString(sessionDtoUuid);
			}else{
				// no UUID transmitted, create one
				sessionUUID=UUID.randomUUID();
			}
		}
		}catch(IllegalArgumentException iae){
			// UUID string could not be parsed
			return Response.status(Status.BAD_REQUEST).build();
		}
		
		if(sessionDtoUuid==null){
			// set UUID to object
			sessionDto.setUuid(sessionUUID.toString());
		}
		
		UriBuilder uriBuilder = uriInfo.getAbsolutePathBuilder();
		
		return storeSession(sessionUUID, uriBuilder,sessionDto);
		
		
	}
	
	
	@POST
	@Consumes(MediaType.APPLICATION_JSON)
	@Produces(MediaType.APPLICATION_JSON)
	public Response postSession(@Context HttpServletRequest req, @Context UriInfo uriInfo,SessionDTO sessionDto){

		String sessionDtoUuid=sessionDto.getUuid();
		
		UUID sessionUUID=null;
		
		try{
			if(sessionDtoUuid!=null){
				// try to get from DTO
				sessionUUID=UUID.fromString(sessionDtoUuid);
			}else{
				// no UUID transmitted, create one
				sessionUUID=UUID.randomUUID();
			}
		}catch(IllegalArgumentException iae){
			// UUID string could not be parsed
			return Response.status(Status.BAD_REQUEST).build();
		}
		
		if(sessionDtoUuid==null){
			// set UUID to object
			sessionDto.setUuid(sessionUUID.toString());
		}
		
		UriBuilder uriBuilder = uriInfo.getAbsolutePathBuilder();
		// UUID not yet part of Location header
		uriBuilder.path(sessionUUID.toString());
		return storeSession(sessionUUID, uriBuilder,sessionDto);
		
		
	}
	
	@PATCH
	@Path("/{sessionIDOrUUIDStr}")	
	@Consumes({ MediaType.APPLICATION_JSON})
	public Response recordingFilePut(@Context ServletContext sc, @Context SecurityContext sec,
			@Context HttpServletRequest req, @PathParam("sessionIDOrUUIDStr") String sessionIDOrUUIDStr, SessionDTO sessionDto) {
		EntityManager em=EntityManagerFactoryInitializer.getEntityManagerFactory().createEntityManager();
		WikiSpeechSecurityManager securityManager = new WikiSpeechSecurityManager(new EntityManagerProvider() {
			@Override
			public EntityManager getThreadEntityManager() {
				return em;
			}
		});

		Integer sessionId=null;
		UUID sessionUUID=null;
		try{
			sessionId=Integer.parseInt(sessionIDOrUUIDStr);
		}catch(NumberFormatException nfe){
			sessionId=null;
			try{
				sessionUUID=UUID.fromString(sessionIDOrUUIDStr);
			}catch(IllegalArgumentException iae){
				return Response.status(Status.BAD_REQUEST).build();
			}
		}
		
		Session existingSess=null;
		EntityTransaction tr=null;
		try{
			tr=em.getTransaction();
			tr.begin();
			if(sessionId!=null) {
				existingSess=em.find(Session.class,sessionId);
			}else if(sessionUUID!=null) {
				existingSess=entityByUUID(em, sessionUUID);
			}else {
				throw new WSTransactionException(Status.BAD_REQUEST);
			}
			
			if(existingSess==null){
				throw new WSTransactionException(Status.BAD_REQUEST);
			}
			
			// check project
			Project prj=existingSess.getProject();
			if(prj==null || !prj.getName().equals(projectName)) {
				throw new WSTransactionException(Status.BAD_REQUEST);
			}
			
			if(!securityManager.getMergePermission(req, existingSess,null)){
				return Response.status(Status.FORBIDDEN).build();
			}
			
			existingSess.applyDTO(sessionDto,false);
			
			em.merge(existingSess);
	
			tr.commit();
			
			return Response.ok(existingSess).build();
		}catch(WSTransactionException wstre) {
			if(tr!=null) {
				tr.rollback();
			}
			return Response.status(wstre.getResponseStatus()).build();
		}finally{
			em.close();
		}
	}


	
	@PUT
	@Consumes(MediaType.APPLICATION_JSON)
	@Produces(MediaType.APPLICATION_JSON)
	@Path("/{sessionIDOrUUID}")
	public Response updateSession(@Context HttpServletRequest req,@PathParam("sessionIDOrUUID") String sessionIDOrUUIDStr,SessionDTO sessionDto){
		EntityManager em=EntityManagerFactoryInitializer.getEntityManagerFactory().createEntityManager();
		WikiSpeechSecurityManager securityManager = new WikiSpeechSecurityManager(new EntityManagerProvider() {
			@Override
			public EntityManager getThreadEntityManager() {
				return em;
			}
		});

		Integer sessionId=null;
		UUID sessionUUID=null;
		try{
			sessionId=Integer.parseInt(sessionIDOrUUIDStr);
		}catch(NumberFormatException nfe){
			sessionId=null;
			try{
				sessionUUID=UUID.fromString(sessionIDOrUUIDStr);
			}catch(IllegalArgumentException iae){
				return Response.status(Status.BAD_REQUEST).build();
			}
		}
		
		Session existingSess=null;
		EntityTransaction tr=null;
		try{
			tr=em.getTransaction();
			tr.begin();
			if(sessionId!=null) {
				existingSess=em.find(Session.class,sessionId);
			}else if(sessionUUID!=null) {
				existingSess=entityByUUID(em, sessionUUID);
			}else {
				throw new WSTransactionException(Status.BAD_REQUEST);
			}
			
			if(existingSess==null){
				throw new WSTransactionException(Status.BAD_REQUEST);
			}
			
			// check project
			Project prj=existingSess.getProject();
			if(prj==null || !prj.getName().equals(projectName)) {
				throw new WSTransactionException(Status.BAD_REQUEST);
			}
			
			if(!securityManager.getMergePermission(req, existingSess,null)){
				return Response.status(Status.FORBIDDEN).build();
			}
			
			existingSess.applyDTO(sessionDto);
			
			em.merge(existingSess);
	
			tr.commit();
			
			return Response.ok(existingSess).build();
		}catch(WSTransactionException wstre) {
			if(tr!=null) {
				tr.rollback();
			}
			return Response.status(wstre.getResponseStatus()).build();
		}finally{
			em.close();
		}
	}
	
	
	
	
	
	
	
	
	/**
	 * Recording  file sub resource upload
	 * @param sc
	 * @param sec
	 * @param sessionIdStr
	 * @return
	 */
	@Path("/{sessionIdOrUUID}/recfile")
	public SessionRecfileResource postRecfile(@PathParam("sessionIdOrUUID") String sessionIdStr) {

		SessionRecfileResource refFileRs = new SessionRecfileResource(sessionIdStr);
		return refFileRs;

	}
	
	/**
	 * Recording file sub resource upload
	 
	 * @param sessionIdStr
	 * @return
	 */
	@Path("/{sessionIdOrUUID}/recordingfile")
	public SubRecordingfileResource delegateRecordingfile(@PathParam("sessionIdOrUUID") String sessionIdStr) {
		return  new SubRecordingfileResource(sessionIdStr);
	}
	

}
