package ipsk.webapps.db.speech.ws;

import java.io.StringWriter;
import java.util.List;

import javax.persistence.EntityManager;
import javax.persistence.TypedQuery;
import javax.persistence.criteria.CriteriaBuilder;
import javax.persistence.criteria.CriteriaQuery;
import javax.persistence.criteria.Root;
import javax.servlet.http.HttpServletRequest;
import javax.ws.rs.GET;
import javax.ws.rs.HEAD;
import javax.ws.rs.Path;
import javax.ws.rs.PathParam;
import javax.ws.rs.Produces;
import javax.ws.rs.core.Context;
import javax.ws.rs.core.GenericEntity;
import javax.ws.rs.core.MediaType;
import javax.ws.rs.core.Response;
import javax.ws.rs.core.Response.Status;
import javax.xml.parsers.ParserConfigurationException;

import ipsk.apps.speechrecorder.script.RecscriptHandler;
import ipsk.db.speech.Session;
import ipsk.db.speech.script.Script;
import ipsk.persistence.EntityManagerProvider;
import ipsk.webapps.EntityManagerFactoryInitializer;
import ipsk.webapps.PermissionDeniedException;
import ipsk.webapps.db.speech.WikiSpeechSecurityManager;
import ipsk.xml.DOMConverterException;

@Path("/script")
public class ScriptResource {

	
	private Script tryGetScript(HttpServletRequest req,int scriptId) throws PermissionDeniedException{
		final EntityManager em=EntityManagerFactoryInitializer.getEntityManagerFactory().createEntityManager();
		WikiSpeechSecurityManager securityManager=new WikiSpeechSecurityManager(new EntityManagerProvider() {
			@Override
			public EntityManager getThreadEntityManager() {
				return em;
			}
		});
		
		try {
			Script sc=em.find(Script.class, scriptId);
			securityManager.checkReadPermission(req, sc);
			return sc;
		}catch(PermissionDeniedException pde){
			throw pde;
		} finally {
			em.close();
		}
	}
	
	
	@GET
	@Produces({MediaType.APPLICATION_JSON})
	@Path("/{scriptId}")
	public Response getScript(@Context HttpServletRequest req,@PathParam("scriptId") int scriptId){
		try {
			Script s=tryGetScript(req, scriptId);
			return Response.ok(s).build();
		} catch (PermissionDeniedException e) {
			return Response.status(Status.FORBIDDEN).build();
		}
	}
	
	@GET
	@Produces({MediaType.APPLICATION_XML})
	@Path("/{scriptId}")
	public Response getScriptXML(@Context HttpServletRequest req,@PathParam("scriptId") int scriptId){
			Script sc;
			try {
				sc = tryGetScript(req, scriptId);
				// Speechrecorder script has its own legacy XML conversion
				// methods
				RecscriptHandler rsh = new RecscriptHandler();
				StringWriter sw = new StringWriter();
				try {
					rsh.writeXML(sc, sw);
					return Response.ok(sw.toString()).build();
				} catch (DOMConverterException | ParserConfigurationException e) {

					return  Response.serverError().build();
				}
			} catch (PermissionDeniedException e1) {
				return Response.status(Status.FORBIDDEN).build();
			}
	}
	
	
	private Response allScripts(HttpServletRequest req,boolean content){
		final EntityManager em=EntityManagerFactoryInitializer.getEntityManagerFactory().createEntityManager();
		WikiSpeechSecurityManager securityManager=new WikiSpeechSecurityManager(new EntityManagerProvider() {
			@Override
			public EntityManager getThreadEntityManager() {
				return em;
			}
		});
		try{	
			CriteriaBuilder cb = em.getCriteriaBuilder();
			CriteriaQuery<Script> cq = cb.createQuery(Script.class);
			Root<Script> rt = cq.from(Script.class);
			cq.select(rt);
			TypedQuery<Script> q = em.createQuery(cq.distinct(true));
			List<Script> allScripts = q.getResultList();
			for(Script sc:allScripts){
				securityManager.checkReadPermission(req, sc);
			}
			String sizeStr=Integer.toString(allScripts.size());
			if(content){
				final GenericEntity<List<Script>> listEntity= new GenericEntity<List<Script>>(allScripts){};
				
				return Response.ok(listEntity).header("X-Total-Count", sizeStr).build();
			}else{
				return Response.noContent().header("X-Total-Count",sizeStr).build();
			}
			
		} catch (PermissionDeniedException e) {
			return Response.status(Status.FORBIDDEN).build();
		}finally{
			em.close();
		}
	}
	
	@GET
	@Produces({MediaType.APPLICATION_JSON})
	public Response getScripts(@Context HttpServletRequest req){
		return allScripts(req, true);
	}

	@HEAD
	@Produces({MediaType.APPLICATION_JSON})
	public Response headScripts(@Context HttpServletRequest req){
		return allScripts(req, false);
	}

	
	
	
}
