package ipsk.db.speech.utils;

import java.io.File;
import java.io.FileNotFoundException;
import java.io.FileOutputStream;
import java.io.PrintWriter;
import java.io.UnsupportedEncodingException;
import java.util.Set;

import javax.persistence.EntityManagerFactory;
import javax.persistence.Persistence;

import org.w3c.dom.Document;

import ipsk.awt.WorkerException;
import ipsk.beans.DOMCodec;
import ipsk.beans.DOMCodecException;
import ipsk.db.speech.DialectRegion;
import ipsk.db.speech.Organisation;
import ipsk.db.speech.Project;
import ipsk.db.speech.Session;
import ipsk.persistence.EntityManagerWorker;
import ipsk.util.LocalizableMessage;
import ipsk.util.ProgressStatus;
import ipsk.util.zip.ZipPackerWorker;
import ipsk.xml.DOMConverter;
import ipsk.xml.DOMConverterException;


public class ProjectExporter extends EntityManagerWorker{
	
	private final static String RESOURCE_BUNDLE_NAME="Messages";
	private String projectName;
	private File outputBaseDir;
	protected ProjectExportConfiguration projectConfiguration;
	private SessionExporter sessionExporter=null;
	private ZipPackerWorker zipPacker;
	
	private File exportArchiveFile;

	//private volatile ProgressStatus progressStatus=new ProgressStatus();
	//private volatile boolean cancel=false;
	
	public ProjectExporter(EntityManagerFactory entityManagerFactory,ProjectExportConfiguration configuration) {
		super(entityManagerFactory);
		this.projectConfiguration=configuration;
	}

	public void doWork() throws WorkerException{
		if(sessionExporter==null){
			sessionExporter=new SessionExporter(entityManager,projectConfiguration);
		}else{
			sessionExporter.setEntityManager(entityManager);
		}
	
		
		Project project=(Project)entityManager.find(Project.class,projectName);
		//reset();
		
//		progressStatus.setMessage("Creating export dir...");
		File projectDir=new File(outputBaseDir,project.getName());
		//exportArchiveFile=new File(outputBaseDir,project.getName()+".zip");
		if(projectDir.exists()){
			throw new ExportException("project export directory "+projectDir+" already exists!");
		}
		projectDir.mkdirs();
		sessionExporter.setOutputBaseDir(projectDir);
		//progressStatus.setProgress(1);
		
		
		int sessionProgressLength=99;
		int zipPackLength=0;
		if(projectConfiguration.isCreateProjectZIPArchive()){
			sessionProgressLength=33;
			zipPackLength=66;
		}
//		progressStatus.setMessage("Export of sessions...");
		Set<Session> sessions=project.getSessions();
		int sessionsSize=sessions.size();
		int sessionCount=0;
		//SessionExporter sessionExporter=new SessionExporter();
		for(Session s:sessions){
			
			if(progressStatus.hasCancelRequest()) {
				//progressStatus.setFinished(false);
				progressStatus.setMessage(new LocalizableMessage(RESOURCE_BUNDLE_NAME, "project.export.canceled", new Object[]{project}));
				return;
			}
			progressStatus.setMessage(new LocalizableMessage(RESOURCE_BUNDLE_NAME,"session.exporting",new Object[]{s}));
			sessionExporter.setSessionId(s.getSessionId());
		
			//exportSessionToDir(s, projectDir);
			sessionExporter.getProgressStatus().running();
			progressStatus.setProgress((sessionProgressLength*sessionCount)/sessions.size());
			progressStatus.setSubStatus(sessionExporter.getProgressStatus(),sessionProgressLength/sessionsSize);
			sessionExporter.doWork();
			sessionCount++;
		}
		progressStatus.setProgress(sessionProgressLength);
		// export project XML file
		Package basePack = Project.class.getPackage();
		DOMCodec ph;
		DOMConverter dc;
		try {
			ph = new DOMCodec(basePack, false);
			dc = new DOMConverter();
		} catch (DOMCodecException e) {
			e.printStackTrace();
			throw new ExportException(e);
		}
		
		
		if (hasCancelRequest())return;
		
		Document projectXMLdoc;
		PrintWriter pw=null;
		
//		progressStatus.setMessage("Creating project description file ...");
		try {
			projectXMLdoc = ph.createDocument(project);
			File projectXMLFile=new File(projectDir,project.getName()+".xml");
			// TODO UTF-8 ??
			pw=new PrintWriter(projectXMLFile,"UTF-8");
			dc.writeXML(projectXMLdoc, pw);
			pw.close();
			
			File organisationsDir=new File(projectDir,"organisations");
			organisationsDir.mkdirs();
			for(Organisation orga:project.getOrganisations()){
				Document orgaDoc=ph.createDocument(orga);
				File orgaXMLFile=new File(organisationsDir,orga.getOrganisationId()+".xml");
				pw=new PrintWriter(orgaXMLFile,"UTF-8");
				dc.writeXML(orgaDoc, pw);
				pw.close();
			}
			File drsDir=new File(projectDir,"dialectregions");
			drsDir.mkdirs();
			for(DialectRegion dr:project.getDialectRegions()){
				Document drDoc=ph.createDocument(dr);
				File drXMLFile=new File(drsDir,dr.getDialectRegionId()+".xml");
				pw=new PrintWriter(drXMLFile,"UTF-8");
				dc.writeXML(drDoc, pw);
				pw.close();
			}
			
		} catch (DOMCodecException e) {
			e.printStackTrace();
			throw new ExportException(e);
		} catch (FileNotFoundException e) {
			e.printStackTrace();
			throw new ExportException(e);
		} catch (DOMConverterException e) {
			e.printStackTrace();
			throw new ExportException(e);
		} catch (UnsupportedEncodingException e) {
			e.printStackTrace();
			throw new ExportException(e);
		}finally{
			if(pw!=null)pw.close();
		}
		
		progressStatus.setProgress(1+sessionProgressLength);
		if(projectConfiguration.isCreateProjectZIPArchive()){
			progressStatus.setMessage(new LocalizableMessage(RESOURCE_BUNDLE_NAME, "zip.creating"));
			FileOutputStream exportArchivestream;
			try {
				exportArchivestream = new FileOutputStream(exportArchiveFile);
			} catch (FileNotFoundException e) {
				throw new ExportException(e);
			}
			zipPacker = new ZipPackerWorker();
//			zipPacker.setLimitTo32bitSizes(true);
			zipPacker.setSrcDir(projectDir);
			zipPacker.setPackRecusive(true);
			zipPacker.setOutputStream(exportArchivestream);
			
			progressStatus.setSubStatus(zipPacker.getProgressStatus(),zipPackLength);
			zipPacker.open();
			zipPacker.setRunningWithParentWorker();
			zipPacker.doWork();
			zipPacker.close();
			
			if(hasCancelRequest()) {
				progressStatus.setMessage(new LocalizableMessage(RESOURCE_BUNDLE_NAME,"project.export.canceled", new Object[]{project}));
				return;
			}
			
		}
		progressStatus.setSubStatus(null, 0);
		//progressStatus.setProgress(100);
		progressStatus.setMessage(new LocalizableMessage(RESOURCE_BUNDLE_NAME,"project.export.done", new Object[]{project}));
		
		}
	
	public void cancel(){
		if(sessionExporter!=null){
			sessionExporter.cancel();
		}
		if(zipPacker!=null){
			zipPacker.cancel();
		}
		super.cancel();
	}
	
	public void close() throws WorkerException{
		if(sessionExporter!=null){
			sessionExporter.close();
		}
		if(zipPacker!=null){
			zipPacker.close();
		}
		super.close();
	}
	
	public static void main(String[] args){
		String projectName=args[0];
		String outputDirName=args[1];
		String PERSISTENCE_UNIT_NAME="WebSpeechDBPU";
		EntityManagerFactory emf=Persistence.createEntityManagerFactory(PERSISTENCE_UNIT_NAME);
		//		EntityManager em=emf.createEntityManager();
		//		EntityTransaction tr=em.getTransaction();
		//		tr.begin();
		ProjectExporter projectExporter=new ProjectExporter(emf,new ProjectExportConfiguration());
		//Project project=em.find(Project.class, projectName);
		File outputDir=new File(outputDirName);


		projectExporter.setOutputBaseDir(outputDir);
		projectExporter.setProjectName(projectName);
		try {
			projectExporter.open();
			projectExporter.run();
		} catch (WorkerException e) {
			e.printStackTrace();

		}finally{

			try {
				projectExporter.close();
			} catch (WorkerException e) {
				e.printStackTrace();
			}
		}


	}

	public ProgressStatus getProgressStatus() {
		return progressStatus;
	}

	public File getOutputBaseDir() {
		return outputBaseDir;
	}

	public void setOutputBaseDir(File outputBaseDir) {
		this.outputBaseDir = outputBaseDir;
	}

	public String getProjectName() {
		return projectName;
	}

	public void setProjectName(String projectName) {
		this.projectName = projectName;
	}

	public File getExportArchiveFile() {
		return exportArchiveFile;
	}

	public void setExportArchiveFile(File exportArchiveFile) {
		this.exportArchiveFile = exportArchiveFile;
	}
	
	
		
}

